﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nim/nim_Result.h>
#include <nn/ns/detail/ns_Log.h>
#include "ns_CleanupUtil.h"
#include "ns_TaskUtil.h"

namespace nn { namespace ns { namespace srv {
namespace
{
Result HasPatchTask(bool* outValue, ApplicationInstallTask* task) NN_NOEXCEPT
{
    int offset = 0;
    for (;;)
    {
        const int CountKeys = 8;
        ncm::StorageContentMetaKey keys[CountKeys];
        int count;
        // TTask を作ってからここまでの間に自動コミットされることを考えて、TaskNotFound をハンドルする
        // この時点でタスクが無くなっていたら、タスクはないと判定できる
        NN_RESULT_TRY(task->ListKey(&count, keys, CountKeys, offset))
            NN_RESULT_CATCH(nim::ResultTaskNotFound)
            {
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY
        for (int i = 0; i < count; ++i)
        {
            if (keys[i].key.type == ncm::ContentMetaType::Patch)
            {
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
        }
        if (count != CountKeys)
        {
            break;
        }
        offset += count;
    }
    *outValue = false;
    NN_RESULT_SUCCESS;
}

}

    Result CleanupTask(ncm::ApplicationId id, IntegratedContentManager* integrated, RequestServer::ManagedStop&& stopper, TaskType taskType) NN_NOEXCEPT
    {
        auto scoped = std::move(stopper);

        if (taskType == TaskType::Download || taskType == TaskType::All)
        {
            ApplicationInstallTask task(id);
            if (task.IsValid())
            {
                NN_RESULT_DO(task.Destroy());
            }
        }
        if (taskType == TaskType::ApplyDelta || taskType == TaskType::All)
        {
            ApplicationApplyDeltaTask task(id);
            if (task.IsValid())
            {
                NN_RESULT_DO(task.Destroy());
            }
        }

        {
            auto result = CleanupAllPlaceHolderAndFragmentsIfNoTask(integrated);
            if (result.IsFailure())
            {
                NN_DETAIL_NS_TRACE("[CleanupTask] Failed to cleanup all place holder as 0x%08x\n", result.GetInnerValueForDebug());
            }
        }

        NN_RESULT_SUCCESS;
    }
    Result CanCreateNetworkInstallTask(bool* outValue, ncm::ApplicationId id, ncm::ContentMetaType type, bool strictly) NN_NOEXCEPT
    {
        // ns_DownloadTaskListManager の FilterKeys では、Patch に対してのみ判定を行うという知識を使っている
        // その前提を崩す修正をする場合は、そちらも一緒に修正が必要
        if (type != ncm::ContentMetaType::Patch)
        {
            *outValue = true;
            NN_RESULT_SUCCESS;
        }
        {
            ApplicationInstallTask task(id);
            if (task.IsValid())
            {
                if (strictly)
                {
                    bool has;
                    NN_RESULT_DO(HasPatchTask(&has, &task));
                    if (has)
                    {
                        NN_DETAIL_NS_TRACE("[CanCreateNetworkInstallTask] Patch download task exists, cannot create task\n");
                        *outValue = false;
                        NN_RESULT_SUCCESS;
                    }
                }
                else
                {
                    NN_DETAIL_NS_TRACE("[CanCreateNetworkInstallTask] Download task exists, cannot create task\n");
                    *outValue = false;
                    NN_RESULT_SUCCESS;
                }
            }
        }
        {
            ApplicationApplyDeltaTask task(id);
            if (task.IsValid())
            {
                NN_DETAIL_NS_TRACE("[CanCreateNetworkInstallTask] Apply delta task exists, cannot create task\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    int64_t CalculateThroughput(int64_t downloadSize, TimeSpan elapsedTime) NN_NOEXCEPT
    {
        // 以下の場合は正しいスループットではないだろうとして、0 を返す
        // - 計測を開始してから 1 秒経過していない
        // - 毎秒ダウンロード容量が 0 以下
        if (elapsedTime < TimeSpan::FromSeconds(1))
        {
            return 0;
        }

        int64_t downloadPerMillisecond = downloadSize / elapsedTime.GetMilliSeconds();
        if (downloadPerMillisecond <= 0)
        {
            return 0;
        }
        else
        {
            return downloadPerMillisecond;
        }
    }

}}}
