﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ncm/ncm_AutoBuffer.h>
#include <nn/ncm/ncm_ContentMeta.h>
#include <nn/ncm/ncm_ContentMetaUtil.h>
#include <nn/ncm/ncm_ContentInfoData.h>
#include <nn/ncm/ncm_ContentInfoUtil.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/srv/ns_IntegratedContentManager.h>
#include <nn/util/util_Optional.h>
#include "ns_CleanupUtil.h"
#include "ns_ProgramIndexUtil.h"
#include "ns_ForEachUtil.h"

namespace nn { namespace ns { namespace srv {

namespace {
    bool IsApplicationProgramContentMetaType(ncm::ContentMetaType type) NN_NOEXCEPT
    {
        switch (type)
        {
        case ncm::ContentMetaType::Application:
        case ncm::ContentMetaType::Patch:
            return true;
        default:
            return false;
        }
    }
}
    void IntegratedContentManager::Register(ncm::StorageId storageId, ncm::ContentMetaDatabase&& db, ncm::ContentStorage&& storage) NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        for (auto& unit : m_List)
        {
            if (!unit.IsRegistered())
            {
                unit.Initialize(storageId, std::move(db), std::move(storage));
                return;
            }
        }

        NN_ABORT("must not come here");
    }

    void IntegratedContentManager::Unregister(ncm::StorageId storageId) NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        for (auto& unit : m_List)
        {
            if (unit.storageId == storageId)
            {
                unit.Reset();
                return;
            }
        }
    }

    int IntegratedContentManager::ListInstalledApplication(ncm::ApplicationContentMetaKey outValue[], int count) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        int outCount = 0;
        for (auto& unit : m_List)
        {
            if (unit.IsInstallableStorage())
            {
                auto countOnce = count - outCount;
                auto outCountOnce = unit.db.ListApplication(&outValue[outCount], countOnce);
                outCount += outCountOnce.listed;
                if (outCount == count)
                {
                    break;
                }
            }
        }

        return outCount;
    }

    int IntegratedContentManager::ListInstalledApplicationContent(ncm::ContentMetaKey outValue[], int count, ncm::ApplicationId id, ncm::ContentMetaType metaType, ncm::ContentInstallType installType) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        int outCount = 0;
        for (auto& unit : m_List)
        {
            if (unit.IsInstallableStorage())
            {
                auto countOnce = count - outCount;
                auto outCountOnce = unit.db.ListContentMeta(&outValue[outCount], countOnce, metaType, id,
                    0x0, 0xffffffffffffffff, installType);
                outCount += outCountOnce.listed;
                if (outCount == count)
                {
                    break;
                }
            }
        }

        return outCount;
    }

    int IntegratedContentManager::ListContent(ncm::ContentMetaKey outValue[], int count, ncm::StorageId storageId, ncm::ContentMetaType metaType, ncm::ContentInstallType installType) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        int outCount = 0;

        for (auto& unit : m_List)
        {
            if (unit.IsTargetStorage(storageId))
            {
                auto countOnce = count - outCount;
                auto outCountOnce = unit.db.ListContentMeta(
                    &outValue[outCount], countOnce, metaType, ncm::ContentMetaDatabase::AnyApplicationId, 0x0, 0xffffffffffffffff, installType);
                outCount += outCountOnce.listed;
                if (outCount == count)
                {
                    break;
                }
            }
        }

        return outCount;
    }

    Result IntegratedContentManager::GetContentId(ncm::ContentId* outValue, const ncm::ContentMetaKey& key, ncm::ContentType type, uint8_t programIndex) const NN_NOEXCEPT
    {
        // contentInfo.idOffset == programIndex とする
        return GetContentIdImpl(outValue, key, type, programIndex, SearchOption::None, ncm::StorageId::Any);
    }

    Result IntegratedContentManager::GetContentIdByStorageId(ncm::ContentId* outValue, const ncm::ContentMetaKey& key, ncm::ContentType type, uint8_t programIndex, ncm::StorageId storageId) const NN_NOEXCEPT
    {
        // contentInfo.idOffset == programIndex とする
        return GetContentIdImpl(outValue, key, type, programIndex, SearchOption::Only, storageId);
    }

    Result IntegratedContentManager::CalculateOccupiedSize(int64_t* outValue, const ncm::ContentMetaKey& key, ncm::StorageId storageId) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        int64_t occupiedSize = 0;
        for (auto& unit : m_List)
        {
            if (unit.storageId == storageId)
            {
                bool hasContentMeta;
                NN_RESULT_DO(unit.db.Has(&hasContentMeta, key));
                if (hasContentMeta)
                {
                    NN_RESULT_DO(CalculateStorage(&occupiedSize, unit, key));
                }
                break;
            }
        }

        *outValue = occupiedSize;
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::DeleteContentMeta(const ncm::ContentMetaKey& key, const ncm::ContentMetaKey* latest, ncm::StorageId storageId) NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        for (auto& unit : m_List)
        {
            if (unit.IsInstallableStorage() && unit.IsTargetStorage(storageId))
            {
                bool hasContentMeta;
                NN_RESULT_DO(unit.db.Has(&hasContentMeta, key));
                if (hasContentMeta)
                {
                    bool hasLatest = false;
                    if (latest)
                    {
                        NN_RESULT_DO(unit.db.Has(&hasLatest, *latest));
                    }

                    ncm::ContentManagerAccessor accessor(&unit.db, &unit.storage);
                    NN_RESULT_DO(accessor.DeleteRedundant(key, hasLatest ? latest : nullptr));

                    unit.needsCommit = true;
                }
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::DeleteDuplicatedContentMeta(const ncm::ContentMetaKey& key) NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        ContentManagerUnit* pRemindUnit = nullptr;
        for (auto& unit : m_List)
        {
            if (unit.IsInstallableStorage())
            {
                bool hasContentMeta;
                NN_RESULT_DO(unit.db.Has(&hasContentMeta, key));
                if (hasContentMeta)
                {
                    ApplicationStoragePriority priority(pRemindUnit ? pRemindUnit->storageId : ncm::StorageId::None);
                    ApplicationStoragePriority storagePriority(unit.storageId);
                    ContentManagerUnit* pTargetUnit = (storagePriority < priority) ? pRemindUnit : &unit;
                    pRemindUnit = (storagePriority < priority) ? &unit : pRemindUnit;

                    if (pTargetUnit)
                    {
                        ncm::ContentManagerAccessor accessor(&pTargetUnit->db, &pTargetUnit->storage);
                        NN_RESULT_DO(accessor.DeleteRedundant(key, nullptr));
                        pTargetUnit->needsCommit = true;
                    }
                }
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::GetContentPath(ncm::Path* outValue, ncm::ContentId contentId) const NN_NOEXCEPT
    {
        return GetContentPathImpl(outValue, contentId);
    }

    Result IntegratedContentManager::GetContentPathByStorageId(ncm::Path* outValue, ncm::ContentId contentId, ncm::StorageId storageId) const NN_NOEXCEPT
    {
        return GetContentPathImpl(outValue, contentId, SearchOption::Only, storageId);
    }

    Result IntegratedContentManager::HasContentMetaKey(bool* outValue, const ncm::ContentMetaKey& key) const NN_NOEXCEPT
    {
        ncm::StorageId storage;
        NN_RESULT_TRY(GetContentMetaStorageImpl(&storage, key))
            NN_RESULT_CATCH(ncm::ResultContentMetaNotFound)
            {
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::HasContentMetaKey(bool* outValue, const ncm::ContentMetaKey& key, ncm::StorageId storageId) const NN_NOEXCEPT
    {
        ncm::StorageId storage;
        auto searchOption = storageId == ncm::StorageId::Any ? SearchOption::None : SearchOption::Only;
        NN_RESULT_TRY(GetContentMetaStorageImpl(&storage, key, searchOption, storageId))
            NN_RESULT_CATCH(ncm::ResultContentMetaNotFound)
            {
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::GetContentMetaStorage(ncm::StorageId* outValue, const ncm::ContentMetaKey& key) const NN_NOEXCEPT
    {
        return GetContentMetaStorageImpl(outValue, key);
    }

    Result IntegratedContentManager::GetInstalledContentMetaStorage(ncm::StorageId* outValue, const ncm::ContentMetaKey& key) const NN_NOEXCEPT
    {
        return GetContentMetaStorageImpl(outValue, key, SearchOption::Exclude, ncm::StorageId::Card);
    }

    Result IntegratedContentManager::GetLatest(ncm::ContentMetaKey* outValue, nn::Bit64 id) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        ncm::ContentMetaKey latest = {};
        for (auto& unit : m_List)
        {
            if (unit.IsRegistered())
            {
                ncm::ContentMetaKey key;
                NN_RESULT_TRY(unit.db.GetLatest(&key, id))
                    // ContentMetaNotFound 以外のエラーは失敗扱いとする
                    NN_RESULT_CATCH(ncm::ResultContentMetaNotFound)
                    {
                        continue;
                    }
                NN_RESULT_END_TRY;

                if (latest.id == 0 || latest.version <= key.version)
                {
                    latest = key;
                }
            }
        }

        if (latest.id != id)
        {
            NN_RESULT_THROW(ncm::ResultContentMetaNotFound());
        }

        *outValue = latest;
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::Get(size_t* outSize, void* outValue, size_t bufferSize, const ncm::ContentMetaKey key) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
        const ContentManagerUnit* pUnit;
        NN_RESULT_DO(FindContentManagerUnitByKey(&pUnit, key));
        NN_RESULT_DO(pUnit->db.Get(outSize, outValue, bufferSize, key));
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::GetSize(size_t* outSize, const ncm::ContentMetaKey& key) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
        const ContentManagerUnit* pUnit;
        NN_RESULT_DO(FindContentManagerUnitByKey(&pUnit, key));
        NN_RESULT_DO(pUnit->db.GetSize(outSize, key));
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::GetRequiredSystemVersion(uint32_t* out, const ncm::ContentMetaKey& key) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
        const ContentManagerUnit* pUnit;
        NN_RESULT_DO(FindContentManagerUnitByKey(&pUnit, key));
        NN_RESULT_DO(pUnit->db.GetRequiredSystemVersion(out, key));
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::GetPatchId(ncm::PatchId* out, const ncm::ContentMetaKey& key) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
        const ContentManagerUnit* pUnit;
        NN_RESULT_DO(FindContentManagerUnitByKey(&pUnit, key));
        NN_RESULT_DO(pUnit->db.GetPatchId(out, key));
        NN_RESULT_SUCCESS;
    }

    bool IntegratedContentManager::IsRegisteredStorage(ncm::StorageId storageId) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
        for (auto& unit : m_List)
        {
            if (unit.storageId == storageId)
            {
                return true;
            }
        }
        return false;
    }

    Result IntegratedContentManager::Commit() NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
        for (auto& unit : m_List)
        {
            if (unit.needsCommit)
            {
                NN_RESULT_DO(unit.db.Commit());
                unit.needsCommit = false;
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::GetContentMetaStorageImpl(ncm::StorageId* outValue, const ncm::ContentMetaKey& key, SearchOption option, ncm::StorageId optionTarget) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        const ContentManagerUnit* pUnit;
        NN_RESULT_DO(FindContentManagerUnitByKey(&pUnit, key, option, optionTarget));

        *outValue = pUnit->storageId;
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::GetContentIdImpl(ncm::ContentId* outValue, const ncm::ContentMetaKey& key, ncm::ContentType type, uint8_t idOffset, SearchOption option, ncm::StorageId optionTarget) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        ApplicationStoragePriority priority;
        util::optional<ncm::ContentId> found;
        for (auto& unit : m_List)
        {
            if (IsTargetStorage(unit, option, optionTarget, priority))
            {
                ncm::ContentId contentId;
                NN_RESULT_TRY(unit.db.GetContentIdByTypeAndIdOffset(&contentId, key, type, idOffset))
                    NN_RESULT_CATCH(ncm::ResultContentMetaNotFound)
                    {
                        continue;
                    }
                NN_RESULT_END_TRY;
                found = contentId;
                priority = ApplicationStoragePriority(unit.storageId);
            }
        }
        NN_RESULT_THROW_UNLESS(found, ncm::ResultContentMetaNotFound());

        *outValue = *found;
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::GetContentPathImpl(ncm::Path* outValue, ncm::ContentId contentId, SearchOption option, ncm::StorageId optionTarget) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        const ContentManagerUnit* pUnit;
        NN_RESULT_DO(FindContentManagerUnitById(&pUnit, contentId, option, optionTarget));

        pUnit->storage.GetPath(outValue, contentId);
        NN_RESULT_SUCCESS;
    }

    bool IntegratedContentManager::IsTargetStorage(const ContentManagerUnit& unit, SearchOption option, ncm::StorageId optionTarget, ApplicationStoragePriority currentPriority) const NN_NOEXCEPT
    {
        ApplicationStoragePriority storagePriority(unit.storageId);
        if (unit.IsRegistered() && storagePriority < currentPriority)
        {
            return IsSearchTarget<ncm::StorageId>(option, unit.storageId, optionTarget);
        }
        else
        {
            return false;
        }
    }

    Result IntegratedContentManager::GetContentIdFileSize(int64_t* pOutSize, ncm::ContentId contentId, ncm::StorageId targetStorage) const NN_NOEXCEPT
    {
        return GetContentIdFileSizeImpl(pOutSize, contentId, SearchOption::Only, targetStorage);
    }

    Result IntegratedContentManager::GetContentIdFileSizeImpl(int64_t* pOutSize, ncm::ContentId contentId, SearchOption option, ncm::StorageId optionTarget) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        const ContentManagerUnit* pUnit;
        NN_RESULT_DO(FindContentManagerUnitById(&pUnit, contentId, option, optionTarget));

        NN_RESULT_DO(pUnit->storage.GetSize(pOutSize, contentId));
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::ListContentInfo(int* pOutCount, ncm::ContentInfo* pOutInfo, int numInfo, const ncm::ContentMetaKey& key, int offset) const NN_NOEXCEPT
    {
        return ListContentInfoImpl(pOutCount, pOutInfo, numInfo, key, offset, SearchOption::None, ncm::StorageId::Any);
    }

    Result IntegratedContentManager::ListContentInfo(int* pOutCount, ncm::ContentInfo* pOutInfo, int numInfo, const ncm::ContentMetaKey& key, int offset, ncm::StorageId targetStorage) const NN_NOEXCEPT
    {
        return ListContentInfoImpl(pOutCount, pOutInfo, numInfo, key, offset, SearchOption::Only, targetStorage);
    }

    Result IntegratedContentManager::ListContentInfoImpl(int* pOutCount, ncm::ContentInfo* pOutInfo, int numInfo, const ncm::ContentMetaKey& key, int offset, SearchOption option, ncm::StorageId optionTarget) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        *pOutCount = 0;
        ApplicationStoragePriority priority;
        for (auto& unit : m_List)
        {
            if (IsTargetStorage(unit, option, optionTarget, priority))
            {
                bool hasKey;
                NN_RESULT_DO(unit.db.Has(&hasKey, key));
                if (hasKey)
                {
                    int count;
                    NN_RESULT_DO(unit.db.ListContentInfo(&count, pOutInfo, numInfo, key, offset));
                    if (count > 0)
                    {
                        *pOutCount = count;
                        priority = ApplicationStoragePriority(unit.storageId);
                    }
                }
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::GetContentInfo(ncm::ContentInfo* pOutInfo, const ncm::ContentMetaKey& key, ncm::ContentType type, uint8_t programIndex, ncm::StorageId targetStorage) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        ApplicationStoragePriority priority;
        util::optional<ncm::ContentInfo> resultInfo;
        auto option = ncm::IsUniqueStorage(targetStorage) ? SearchOption::Only : SearchOption::None;

        for (auto& unit : m_List)
        {
            if (IsTargetStorage(unit, option, targetStorage, priority))
            {
                bool hasKey;
                NN_RESULT_DO(unit.db.Has(&hasKey, key));
                if (hasKey)
                {
                    NN_RESULT_DO(ForEachContentInfo(&unit.db, key, [&](bool* outIsEnd, const ncm::ContentInfo& info) NN_NOEXCEPT -> Result
                    {
                        *outIsEnd = false;
                        if (GetProgramIndex(info) == programIndex &&
                            info.GetType() == type)
                        {
                            resultInfo = info;
                            priority = ApplicationStoragePriority(unit.storageId);
                            *outIsEnd = true;
                        }
                        NN_RESULT_SUCCESS;
                    }));
                }
            }
        }

        NN_RESULT_THROW_UNLESS(resultInfo, ncm::ResultContentNotFound());
        *pOutInfo = *resultInfo;

        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::ReadContentIdFile(void* buffer, size_t bufferSize, ncm::ContentId contentId, int64_t offset, ncm::StorageId targetStorage) const NN_NOEXCEPT
    {
        return ReadContentIdFileImpl(buffer, bufferSize, contentId, offset, SearchOption::Only, targetStorage);
    }

    Result IntegratedContentManager::ReadContentIdFileImpl(void* buffer, size_t bufferSize, ncm::ContentId contentId, int64_t offset, SearchOption option, ncm::StorageId optionTarget) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        const ContentManagerUnit* pUnit = nullptr;
        NN_RESULT_DO(FindContentManagerUnitById(&pUnit, contentId, option, optionTarget));

        NN_RESULT_DO(pUnit->storage.ReadContentIdFile(buffer, bufferSize, contentId, offset));

        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::WriteContentIdFile(ncm::ContentId contentId, int64_t offset, const void* buffer,size_t bufferSize, ncm::StorageId targetStorage) NN_NOEXCEPT
    {
        return WriteContentIdFileImpl(contentId, offset, buffer, bufferSize, SearchOption::Only, targetStorage);
    }

    Result IntegratedContentManager::WriteContentIdFileImpl(ncm::ContentId contentId, int64_t offset, const void* buffer,size_t bufferSize, SearchOption option, ncm::StorageId optionTarget) NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        const ContentManagerUnit* pUnit;
        NN_RESULT_DO(FindContentManagerUnitById(&pUnit, contentId, option, optionTarget));

        // TORIAEZU: pUnit が const ポインタで、 WriteContentForDebug を呼べないので、再度探す
        for (auto& unit : m_List)
        {
            if (unit.storageId == pUnit->storageId)
            {
                NN_RESULT_DO(unit.storage.WriteContentForDebug(contentId, offset, buffer, bufferSize));
                NN_RESULT_SUCCESS;
            }
        }

        NN_ABORT("must not come here");
    }

    Result IntegratedContentManager::GetContentIdFileHash(ncm::Hash* pOutHash, ncm::ContentId contentId, ncm::ContentId metaContentId, ncm::StorageId targetStorage) const NN_NOEXCEPT
    {
        return GetContentIdFileHashImpl(pOutHash, contentId, metaContentId, SearchOption::Only, targetStorage);
    }

    Result IntegratedContentManager::GetContentRightsId(fs::RightsId* outValue, const ncm::ContentMetaKey& key, ncm::ContentType contentType, uint8_t programIndex, ncm::StorageId storageId) const NN_NOEXCEPT
    {
        uint8_t tmpKeyGeneration;
        NN_RESULT_DO(GetContentRightsId(outValue, &tmpKeyGeneration, key, contentType, programIndex, storageId));

        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::GetContentRightsId(fs::RightsId* outRightsId, uint8_t* outKeyGeneration, const ncm::ContentMetaKey& key, ncm::ContentType contentType, uint8_t programIndex, ncm::StorageId storageId) const NN_NOEXCEPT
    {
        ncm::ContentId contentId;
        NN_RESULT_DO(GetContentIdByStorageId(&contentId, key, contentType, programIndex, storageId));

        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
        for (auto& unit : m_List)
        {
            if (unit.storageId == storageId)
            {
                ncm::RightsId rightsId;
                NN_RESULT_DO(unit.storage.GetRightsId(&rightsId, contentId));

                *outRightsId = rightsId.id;
                *outKeyGeneration = rightsId.keyGeneration;
                NN_RESULT_SUCCESS;
            }
        }

        NN_RESULT_THROW(ncm::ResultContentNotFound());
    }

    Result IntegratedContentManager::CleanupOrphanContents() NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
        for (auto& unit : m_List)
        {
            if (!unit.IsRegistered() || unit.storageId == nn::ncm::StorageId::Card)
            {
                continue;
            }

            NN_RESULT_DO(srv::CleanupOrphanContents(unit.storageId));
        }

        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::CleanupAllPlaceHolder(ncm::StorageId storageId) NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
        for (auto& unit : m_List)
        {
            if (unit.IsTargetStorage(storageId))
            {
                NN_RESULT_DO(unit.storage.CleanupAllPlaceHolder());
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::CleanupFragments(ncm::StorageId storageId) NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
        for (auto& unit : m_List)
        {
            // 気持ちとしては All と書きたい
            if (unit.IsInstallableStorage() && unit.IsTargetStorage(storageId))
            {
                NN_RESULT_DO(srv::CleanupFragments(unit.storageId));
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::RepairInvalidFileAttribute(ncm::StorageId storageId) NN_NOEXCEPT
    {
        // 現時点では、SD 以外で呼ばれることは想定しない
        NN_ABORT_UNLESS(storageId == ncm::StorageId::SdCard);

        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
        for (auto& unit : m_List)
        {
            // 気持ちとしては All と書きたい
            if (unit.IsInstallableStorage() && unit.IsTargetStorage(storageId))
            {
                NN_RESULT_DO(unit.storage.RepairInvalidFileAttribute());
            }
        }
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::GetContentIdFileHashImpl(ncm::Hash* pOutHash, ncm::ContentId contentId, ncm::ContentId metaContentId, SearchOption option, ncm::StorageId optionTarget) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        const ContentManagerUnit* pUnit;
        NN_RESULT_DO(FindContentManagerUnitById(&pUnit, contentId, option, optionTarget));
        NN_RESULT_DO(ReadHash(pOutHash, contentId, *pUnit, metaContentId));
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::CalculateStorage(int64_t* outValue, const ContentManagerUnit& unit, const ncm::ContentMetaKey& key) const NN_NOEXCEPT
    {
        int64_t size = 0;
        NN_RESULT_DO(ForEachContentInfo(&unit.db, key, [&unit, &size](bool* outIsEnd, const ncm::ContentInfo& info) NN_NOEXCEPT -> Result
        {
            *outIsEnd = false;
            bool hasContent;
            NN_RESULT_DO(unit.storage.Has(&hasContent, info.id));
            if (hasContent)
            {
                size += info.GetSize();
            }
            NN_RESULT_SUCCESS;
        }));

        *outValue = size;
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::ReadHash(ncm::Hash* outValue, ncm::ContentId contentId, const ContentManagerUnit& unit, ncm::ContentId metaContentId) const NN_NOEXCEPT
    {
        ncm::Path path;
        unit.storage.GetPath(&path, metaContentId);
        ncm::AutoBuffer buffer;
        NN_RESULT_DO(ncm::ReadContentMetaPath(&buffer, path.string));
        ncm::PackagedContentMetaReader reader(buffer.Get(), buffer.GetSize());

        int numContent = reader.CountContent();

        for (int i = 0; i < numContent; i++)
        {
            const ncm::PackagedContentInfo* info = reader.GetContentInfo(i);

            if (info->info.GetId() == contentId)
            {
                std::memcpy(outValue, info->hash.data, sizeof(info->hash.data));
                NN_RESULT_SUCCESS;
            }
        }

        NN_RESULT_THROW(ncm::ResultContentNotFound());
    }

    template <class T>
    bool IntegratedContentManager::IsSearchTarget(SearchOption option, const T a, const T b) const NN_NOEXCEPT
    {
        if (option == SearchOption::Exclude && a == b)
        {
            return false;
        }

        if (option == SearchOption::Only && a != b)
        {
            return false;
        }

        return true;
    }

    int IntegratedContentManager::GetRegisteredStorages(ncm::StorageId* outValue, int numStorage) const NN_NOEXCEPT
    {
        int count = 0;
        for (auto& unit : m_List)
        {
            if (count >= numStorage)
            {
                return count;
            }
            else if (unit.IsRegistered())
            {
                outValue[count++] = unit.storageId;
            }
        }

        return count;
    }

    Result IntegratedContentManager::FindContentManagerUnitByKey(const ContentManagerUnit** outValue, const ncm::ContentMetaKey& key, SearchOption option, ncm::StorageId optionTarget) const NN_NOEXCEPT
    {
        ApplicationStoragePriority priority;
        const ContentManagerUnit* found = nullptr;

        for (auto& unit : m_List)
        {
            if (IsTargetStorage(unit, option, optionTarget, priority))
            {
                bool hasKey;
                NN_RESULT_DO(unit.db.Has(&hasKey, key));
                if (hasKey)
                {
                    found = &unit;
                    priority = ApplicationStoragePriority(unit.storageId);
                }
            }
        }
        NN_RESULT_THROW_UNLESS(found, ncm::ResultContentMetaNotFound());

        *outValue = found;
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::FindContentManagerUnitById(const ContentManagerUnit** outValue, const ncm::ContentId& id, SearchOption option, ncm::StorageId optionTarget) const NN_NOEXCEPT
    {
        ApplicationStoragePriority priority;
        const ContentManagerUnit* found = nullptr;

        for (auto& unit : m_List)
        {
            if (IsTargetStorage(unit, option, optionTarget, priority))
            {
                bool hasContent;
                NN_RESULT_DO(unit.storage.Has(&hasContent, id));
                if (hasContent)
                {
                    found = &unit;
                    priority = ApplicationStoragePriority(unit.storageId);
                }
            }
        }
        NN_RESULT_THROW_UNLESS(found, ncm::ResultContentNotFound());

        *outValue = found;
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::GetRequiredApplicationVersion(uint32_t* outValue, const ncm::ContentMetaKey& key) const NN_NOEXCEPT
    {
        return GetRequiredApplicationVersionImpl(outValue, key);
    }

    Result IntegratedContentManager::GetRequiredApplicationVersionByStorageId(uint32_t* outValue, const ncm::ContentMetaKey& key, ncm::StorageId storageId) const NN_NOEXCEPT
    {
        return GetRequiredApplicationVersionImpl(outValue, key, SearchOption::Only, storageId);
    }

    Result IntegratedContentManager::GetRequiredApplicationVersionImpl(uint32_t* outValue, const ncm::ContentMetaKey& key, SearchOption option, ncm::StorageId optionTarget) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
        const ContentManagerUnit* pUnit;
        NN_RESULT_DO(FindContentManagerUnitByKey(&pUnit, key, option, optionTarget));
        NN_RESULT_DO(pUnit->db.GetRequiredApplicationVersion(outValue, key));
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::HasOrphanContents(bool* outValue) const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

        for (auto& unit : m_List)
        {
            if (!unit.IsRegistered() || unit.storageId == nn::ncm::StorageId::Card)
            {
                continue;
            }

            int offset = 0;
            while (NN_STATIC_CONDITION(true))
            {
                int count;
                NN_RESULT_DO(unit.storage.ListContentId(&count, m_ContentList, MaxContentCount, offset));
                if (count == 0)
                {
                    break;
                }

                NN_RESULT_DO(unit.db.LookupOrphanContent(m_OrphanList, m_ContentList, count));
                for (int i = 0; i < count; i++)
                {
                    if (m_OrphanList[i])
                    {
                        *outValue = true;
                        NN_RESULT_SUCCESS;
                    }
                }

                offset += count;
            }
        }

        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::EstimateRequiredSize(int64_t* outValue, const ncm::ContentMetaKey& key) NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
        const ContentManagerUnit* pUnit;
        NN_RESULT_DO(FindContentManagerUnitByKey(&pUnit, key, SearchOption::None, ncm::StorageId::Any));
        NN_RESULT_DO(ncm::EstimateRequiredSize(outValue, key, &(pUnit->db)));
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::GetMinimumProgramIndex(uint8_t* outValue, const ncm::ContentMetaKey& key) const NN_NOEXCEPT
    {
        if (!IsApplicationProgramContentMetaType(key.type))
        {
            *outValue = 0;
            NN_RESULT_SUCCESS;
        }

        util::optional<uint8_t> programIndex;
        NN_RESULT_DO(ForEachContentInfo(this, key, [&programIndex](bool* outIsEnd, const ncm::ContentInfo& info) NN_NOEXCEPT -> Result
        {
            *outIsEnd = false;
            if (info.type == ncm::ContentType::Program)
            {
                auto newProgramIndex = GetProgramIndex(info);
                programIndex = programIndex ? std::min(*programIndex, newProgramIndex) : newProgramIndex;
                // 最適化
                if (*programIndex == 0)
                {
                    *outIsEnd = true;
                }
            }
            NN_RESULT_SUCCESS;
        }));

        NN_RESULT_THROW_UNLESS(programIndex, ResultApplicationContentNotFound());

        *outValue = *programIndex;
        NN_RESULT_SUCCESS;
    }

    Result IntegratedContentManager::ListProgramIndex(int* outValue, uint8_t* outList, int numList, const ncm::ContentMetaKey& key, ncm::ContentType type, int offset) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(offset >= 0);
        int foundCount = 0;

        NN_RESULT_DO(ForEachContentInfo(this, key, [&](bool* outIsEnd, const ncm::ContentInfo& info) NN_NOEXCEPT -> Result
        {
            *outIsEnd = false;
            if (info.type == type)
            {
                auto index = foundCount - offset;
                if ((index >= 0) && (index < numList))
                {
                    // 同じ ContentType で同じ ProgramIndex は存在しないはず
                    outList[index] = GetProgramIndex(info);
                }
                foundCount++;
                if (foundCount - offset >= numList)
                {
                    *outIsEnd = true;
                }
            }
            NN_RESULT_SUCCESS;
        }));

        *outValue = (foundCount - offset > 0) ? foundCount - offset : 0;
        NN_RESULT_SUCCESS;
    }
}}}
