﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ns/srv/ns_DevelopInterfaceServer.h>
#include <nn/ns/srv/ns_IProgramLaunchObserver.h>

#include <mutex>

#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_StaticAssert.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/pm/pm_ShellApi.h>
#include <nn/ns/srv/ns_Shell.h>
#include <nn/ns/ns_Result.h>
#include <nn/arp/arp_Api.h>
#include "ns_ApplicationPathRedirectionUtil.h"
#include "ns_LogUtil.h"
#include "ns_ProgramIndexUtil.h"

#include <nn/lr/lr_Service.h>
#include <nn/lr/lr_Result.h>
#include <nn/lr/lr_LocationResolver.h>
#include <nn/lr/lr_RegisteredLocationResolver.h>

namespace nn { namespace ns { namespace srv {

    IEventDispatcher* DevelopInterfaceServer::s_pEventDispatcher;

    void DevelopInterfaceServer::ShellEventState::HandleEvent()
    {
        auto cause = GetEvent()->GetCause();
        if (cause == pm::ProcessEvent_Exit)
        {
            this->state = EventState_Exited;
            this->pClientEvent->Signal();
        }
        else if (cause == pm::ProcessEvent_Started)
        {
            this->state = EventState_Started;
            this->pClientEvent->Signal();
            GetEvent()->Clear();
            s_pEventDispatcher->AddEventHandler(this);
        }
        else if (cause == pm::ProcessEvent_DebugRunning)
        {
            this->state = EventState_DebugRunning;
            this->pClientEvent->Signal();
            GetEvent()->Clear();
            s_pEventDispatcher->AddEventHandler(this);
        }
        else if (cause == pm::ProcessEvent_Exception)
        {
            this->state = EventState_Exception;
            this->pClientEvent->Signal();
            GetEvent()->Clear();
            s_pEventDispatcher->AddEventHandler(this);
        }
    }

    DevelopInterfaceServer::DevelopInterfaceServer()
        : m_ClientEvent(os::EventClearMode_AutoClear, true), m_ObserverRegisteredEvent(os::EventClearMode_ManualClear)
    {
        for( auto& ses : m_States )
        {
            ses.pClientEvent = &m_ClientEvent;
        }
    }

    Result DevelopInterfaceServer::PrepareLaunchProgramFromHost(sf::Out<ProgramLaunchProperty> out, sf::InBuffer programPathBuffer) NN_NOEXCEPT
    {
        auto programPath = programPathBuffer.GetPointerUnsafe();
        size_t length = strnlen(programPath, programPathBuffer.GetSize());
        lr::Path path;
        NN_RESULT_THROW_UNLESS(length < sizeof(path.string), ns::ResultBufferNotEnough());

        std::strcpy(path.string, programPath);

        // W/A : hoge.nspd/program0.ncd/ -> hoge.nspd/
        if (length > 13 && strncmp(path.string + length - 13, "program0.ncd/", 13) == 0)
        {
            path.string[length - 13] = '\0';
        }

        // アプリケーションメタを読み取り、正しいアプリケーション ID でリダイレクトし直す
        lr::LocationResolver hostLocationResolver;
        NN_RESULT_DO(lr::OpenLocationResolver(&hostLocationResolver, ncm::StorageId::Host));

        ncm::ProgramId programId;
        util::optional<ncm::ApplicationId> applicationId;
        uint32_t version;
        uint8_t programIndex;
        NN_RESULT_DO(LoadHostProgram(&programId, &programIndex, &applicationId, &version, path.string, &hostLocationResolver));
        auto extType = GetExtensionType(path.string);
        if(applicationId && (extType == Nspd || extType == Nsp))
        {
            m_ObserverRegisteredEvent.Wait();
        }

        if (applicationId)
        {
            // パッチの Redirection を消しておく
            lr::RegisteredLocationResolver registeredLocationResolver;
            NN_RESULT_DO(lr::OpenRegisteredLocationResolver(&registeredLocationResolver));
            NN_RESULT_DO(registeredLocationResolver.UnregisterProgramPath(programId));
            NN_RESULT_DO(registeredLocationResolver.UnregisterHtmlDocumentPath(programId));

            // Host 起動の際に、ROM がインストールされているものを参照する機能をサポートするために、
            // NANDやSDのリゾルバを設定する
            if (m_pProgramLaunchObserver)
            {
                NN_NS_TRACE_RESULT_IF_FAILED(m_pProgramLaunchObserver->RedirectApplicationProgramForDevelop(*applicationId, programIndex), "[DevelopInterfaceServer] failed to redirect installed application program.\n");
            }
        }

        *out = { programId, version, ncm::StorageId::Host, programIndex, applicationId };

        NN_RESULT_SUCCESS;
    }

    Result DevelopInterfaceServer::LaunchProgram(sf::Out<nn::os::ProcessId> out, const ProgramLaunchProperty& launchProperty, int32_t flags) NN_NOEXCEPT
    {
        os::ProcessId id;
        ncm::ProgramLocation location = ncm::MakeProgramLocation(launchProperty.storageId, launchProperty.programId);

        NN_RESULT_DO(ns::srv::LaunchProgram(&id, location, flags));
        bool isSuccess = false;
        NN_UTIL_SCOPE_EXIT
        {
            if (!isSuccess)
            {
                srv::TerminateProgram(launchProperty.programId);
            }
        };


        if (m_pProgramLaunchObserver && launchProperty.isApplication)
        {
            ApplicationLaunchInfo info =
            {
                GetApplicationId(launchProperty.programId, launchProperty.programIndex),
                launchProperty.version,
                flags,
                launchProperty.storageId,
                ncm::StorageId::None,
                static_cast<Bit8>(ApplicationLaunchInfoFlag::None)
            };

            NN_RESULT_DO(m_pProgramLaunchObserver->AfterLaunchProgram(id, info, launchProperty.programIndex));
        }

        isSuccess = true;
        out.Set(id);

        NN_RESULT_SUCCESS;
    }

    Result DevelopInterfaceServer::LaunchApplication(sf::Out<nn::os::ProcessId> out, ncm::ApplicationId applicationId, int32_t flags) NN_NOEXCEPT
    {
        return LaunchApplicationWithStorageId(
            out, applicationId, flags, ncm::StorageId::Any, ncm::StorageId::Any);
    }

    Result DevelopInterfaceServer::LaunchApplicationWithStorageId(
        sf::Out<nn::os::ProcessId> out,
        ncm::ApplicationId applicationId,
        int32_t flags,
        ncm::StorageId appStorageId,
        ncm::StorageId patchStorageId) NN_NOEXCEPT
    {
        m_ObserverRegisteredEvent.Wait();
        ApplicationLaunchInfo info;
        NN_RESULT_DO(m_pProgramLaunchObserver->GetApplicationLaunchInfoForDevelop(&info, applicationId));
        info.applicationStorageId = ncm::IsUniqueStorage(appStorageId) ? appStorageId : info.applicationStorageId;
        info.patchStorageId = ncm::IsUniqueStorage(patchStorageId) ? patchStorageId : info.patchStorageId;
        info.launchFlags = flags;

        nn::os::ProcessId processId;
        NN_RESULT_DO(m_pProgramLaunchObserver->LaunchApplicationForDevelop(&processId, info));

        out.Set(processId);

        NN_RESULT_SUCCESS;
    }

    Result DevelopInterfaceServer::GetShellEventHandle(os::NativeHandle* pOut) NN_NOEXCEPT
    {
        *pOut = m_ClientEvent.GetReadableHandle();
        return ResultSuccess();
    }

    Result DevelopInterfaceServer::RegisterState(Event** pOut, os::ProcessId processId) NN_NOEXCEPT
    {
        ShellEventState* pses = FindState(os::ProcessId::GetInvalidId());
        if (!pses)
        {
            ForceFetchShellEvent();
            pses = FindState(os::ProcessId::GetInvalidId());
            NN_ABORT_UNLESS_NOT_NULL(pses);
        }

        pses->Clear();
        pses->processId = processId;
        *pOut = pses->GetEvent();
        s_pEventDispatcher->AddEventHandler(pses);
        return ResultSuccess();
    }

    Result DevelopInterfaceServer::GetShellEventInfo(ShellEventInfo* pOut) NN_NOEXCEPT
    {
        for( auto& ses : m_States )
        {
            if( ses.state == EventState_Exited )
            {
                pOut->event     = ShellEvent_Exit;
                pOut->processId = ses.processId;

                ses.Release();
                return ResultSuccess();
            }
            if( ses.state == EventState_Started )
            {
                pOut->event     = ShellEvent_Started;
                pOut->processId = ses.processId;

                ses.Clear();
                return ResultSuccess();
            }
            if( ses.state == EventState_DebugRunning )
            {
                pOut->event     = ShellEvent_DebugRunning;
                pOut->processId = ses.processId;

                ses.Clear();
                return ResultSuccess();
            }
            if( ses.state == EventState_Exception )
            {
                pOut->event     = ShellEvent_Exception;
                pOut->processId = ses.processId;

                ses.Clear();
                return ResultSuccess();
            }
        }

        return ResultNoShellInfo();
    }

    Result DevelopInterfaceServer::TerminateProcess(os::ProcessId id) NN_NOEXCEPT
    {
        return srv::TerminateProcess(id);
    }

    Result DevelopInterfaceServer::TerminateProgram(ncm::ProgramId id) NN_NOEXCEPT
    {
        return srv::TerminateProgram(id);
    }

    Result DevelopInterfaceServer::TerminateApplication() NN_NOEXCEPT
    {
        return srv::TerminateApplication();
    }

    Result DevelopInterfaceServer::IsSystemMemoryResourceLimitBoosted(nn::sf::Out<bool> pOut) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_BoostedSystemMemoryResourceLimitValueMutex)> lk(m_BoostedSystemMemoryResourceLimitValueMutex);
        *pOut = (m_CurrentBoostedSystemMemoryResourceLimitValue != 0);
        NN_RESULT_SUCCESS;
    }

    Result DevelopInterfaceServer::GetRunningApplicationProcessId(sf::Out<nn::os::ProcessId> out) NN_NOEXCEPT
    {
        auto pid = srv::GetRunningApplicationProcessId();
        NN_RESULT_THROW_UNLESS(pid, ResultApplicationNotRunning());

        out.Set(*pid);

        NN_RESULT_SUCCESS;
    }

    void DevelopInterfaceServer::SetCurrentBoostedSystemMemoryResourceLimitValue(int64_t boostSize) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_BoostedSystemMemoryResourceLimitValueMutex)> lk(m_BoostedSystemMemoryResourceLimitValueMutex);
        this->m_CurrentBoostedSystemMemoryResourceLimitValue = boostSize;
    }

    void DevelopInterfaceServer::ForceFetchShellEvent() NN_NOEXCEPT
    {
        // TORIAEZU: あぶれる程 ShellEvent が溜まるという事は読み取りに来るプロセスがいないということだとし、
        //           全部フェッチしてしまう

        NN_DETAIL_NS_TRACE("[DevelopInterfaceServer] ShellEvents are fetched forcibly.\n");
        while (NN_STATIC_CONDITION(true))
        {
            ShellEventInfo info;
            auto result = GetShellEventInfo(&info);
            NN_UNUSED(info);
            if (result.IsFailure())
            {
                if (!ResultNoShellInfo::Includes(result))
                {
                    NN_DETAIL_NS_TRACE("[DevelopInterfaceServer] ForceFetchShellEvent() encountered an unexpected result. (result=0x%08x)\n", result.GetInnerValueForDebug());
                }
                break;
            }
        }
    }
}}}

