﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/account_Types.h>
#include <nn/es/es_ELicenseTypes.h>
#include <nn/ec/system/ec_TicketApi.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/nim/nim_DynamicRightsTypes.h>
#include <nn/nim/nim_DynamicRightsApi.h>
#include <nn/ns/ns_ApplicationRightsTypes.h>
#include <nn/ns/ns_DynamicRightsSystemApi.h>
#include <nn/ns/srv/ns_ApplicationRecordDatabase.h>
#include <nn/ns/srv/ns_IntegratedContentManager.h>
#include <nn/ns/srv/ns_RightsEnvironment.h>
#include <nn/ns/srv/ns_RequestServer.h>
#include <nn/util/util_BitFlagSet.h>
#include "ns_AsyncImpl.h"

namespace nn { namespace ns { namespace srv {

    struct AvailableELicenseHolderHandle
    {
        Bit8 value;

        bool IsValid()
        {
            return value != 0;
        }

        bool operator==(const AvailableELicenseHolderHandle& rhs) const NN_NOEXCEPT
        {
            return value == rhs.value;
        }

        bool operator!=(const AvailableELicenseHolderHandle& rhs) const NN_NOEXCEPT
        {
            return !(*this == rhs);
        }
    };
    constexpr AvailableELicenseHolderHandle InvalidAvailableELicenseHolderHandle { 0 };

    class QueryApplicationContentRightsManager
    {
        NN_DISALLOW_COPY(QueryApplicationContentRightsManager);
        NN_DISALLOW_MOVE(QueryApplicationContentRightsManager);

    public:
        QueryApplicationContentRightsManager(
            ApplicationRecordDatabase* pDb,
            IntegratedContentManager* pIntegrated,
            ncm::ApplicationId appId,
            ApplicationRightsQueryFlags flags) NN_NOEXCEPT;

        Result Initialize() NN_NOEXCEPT;
        Result GetRightsIdList(int* outCount, es::RightsId* outList, int listCount) NN_NOEXCEPT;
        ApplicationContentType GetApplicationContentType(const es::RightsId& rightsId) const NN_NOEXCEPT;
        bool IsQueried(ApplicationContentType type) const NN_NOEXCEPT;

    private:
        bool NeedsRights(const ncm::ContentMetaKey& key) const NN_NOEXCEPT;
        ApplicationRecordDatabase* m_pRecordDb;
        IntegratedContentManager* m_pIntegrated;
        ncm::ApplicationId m_AppId;
        ApplicationRightsQueryFlags m_Flags;
        es::RightsId m_AppRightsId;
        bool m_QueryAppRights;
        bool m_QueryAocRights;
    };

    struct AssignedELicenseInfo
    {
        account::NintendoAccountId ownerId;
        struct ELicenseList
        {
            std::unique_ptr<es::ELicenseId[]> eLicenseIds;
            int eLicenseCount;
        };
        ELicenseList dynamicRightsELicense;
        ELicenseList deviceLinkedELicense;
    };

    class AsyncApplicationRightsOnServerImpl : public AsyncValueBase, private ErrorContextHolder
    {
    public:
        AsyncApplicationRightsOnServerImpl(
            ApplicationRecordDatabase* pDb,
            IntegratedContentManager* pIntegrated,
            ncm::ApplicationId id,
            const account::Uid& uid,
            ApplicationRightsQueryFlags queryFlags,
            RequestServer::ManagedStop&& stopper) NN_NOEXCEPT;

        virtual ~AsyncApplicationRightsOnServerImpl() NN_NOEXCEPT;
        Result Run() NN_NOEXCEPT;

        os::SystemEvent& GetEvent() NN_NOEXCEPT
        {
            return m_Event;
        }

        virtual Result GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = ErrorContextHolder::GetErrorContextImpl();
            NN_RESULT_SUCCESS;
        }

    private:
        virtual void CancelImpl() NN_NOEXCEPT NN_OVERRIDE;
        virtual size_t GetSizeImpl() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result GetImpl(const nn::sf::OutBuffer& buffer) NN_NOEXCEPT NN_OVERRIDE;

        Result Execute() NN_NOEXCEPT;

        bool IsCanceled() const NN_NOEXCEPT;

        void AddDebugFlags(ApplicationRightsOnServerFlag flags) NN_NOEXCEPT;

        Result QueryAvailableELicenses() NN_NOEXCEPT;
        Result SetUpAvailableELicenseHolder(AvailableELicenseHolderHandle* outValue) NN_NOEXCEPT;
        Result AddLicenseTypeFlags(const AvailableELicenseHolderHandle& handle) NN_NOEXCEPT;

        QueryApplicationContentRightsManager m_QueryRightsManager;

        ncm::ApplicationId m_AppId;
        account::Uid m_Uid;

        Result m_Result {};
        ManualClearSystemEvent m_Event;
        util::optional<nim::AsyncAvailableELicenses> m_Async;
        ApplicationRightsQueryFlags m_QueryFlags;
        mutable NonRecursiveMutex m_CancelMutex;
        util::optional<ThreadInfo> m_ThreadInfo;
        bool m_IsCanceled {};
        // TORIAEZU: Application & AddOnContent で 2 個用意しておく
        std::array<ApplicationRightsOnServer, 2> m_ApplicationRightsList {};
        int m_Count {};
        RequestServer::ManagedStop m_Stopper;
    };

    class AsyncAssignRightsImpl: public AsyncResultBase, private ErrorContextHolder
    {
    public:
        AsyncAssignRightsImpl() = default;

        virtual ~AsyncAssignRightsImpl() NN_NOEXCEPT;

        Result Initialize(
            ApplicationRecordDatabase* pRecord,
            IntegratedContentManager* pIntegrated,
            const ApplicationRightsOnServer rightsList[],
            int numList,
            RequestServer::ManagedStop&& stopper) NN_NOEXCEPT;

        Result Run() NN_NOEXCEPT;

        os::SystemEvent& GetEvent() NN_NOEXCEPT
        {
            return m_Event;
        }

        virtual Result GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = ErrorContextHolder::GetErrorContextImpl();
            NN_RESULT_SUCCESS;
        }

    private:
        virtual void CancelImpl() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result GetImpl() NN_NOEXCEPT NN_OVERRIDE;

        Result Execute() NN_NOEXCEPT;

        bool IsCanceled() const NN_NOEXCEPT;

        template <class FuncT>
        Result ForEachRightsId(FuncT func) NN_NOEXCEPT;

        Result ImportELicensesFromServer() NN_NOEXCEPT;
        Result ImportELicensesByDebugUtility() NN_NOEXCEPT;

        std::array<ApplicationRightsOnServer, 2> m_ApplicationRightsList {};
        int m_NumList {};

        ApplicationRecordDatabase* m_pRecord {};
        IntegratedContentManager* m_pIntegrated {};

        ncm::ApplicationId m_AppId {};
        account::Uid m_Uid {};
        Result m_Result {};
        ManualClearSystemEvent m_Event;
        util::optional<nim::AsyncAssignedELicenses> m_AssignAsync;
        util::optional<nim::AsyncResult> m_ResultAsync;
        util::optional<ec::system::AsyncProgressResult> m_EcAsync;
        mutable NonRecursiveMutex m_CancelMutex;
        util::optional<ThreadInfo> m_ThreadInfo;
        bool m_IsCanceled {};
        RequestServer::ManagedStop m_Stopper;
    };

    class AsyncAssignRightsToResumeImpl: public AsyncResultBase, private ErrorContextHolder
    {
    public:
        AsyncAssignRightsToResumeImpl(
            RightsEnvironmentManager* pRightsManager,
            RightsEnvironmentHandle handle,
            const account::Uid& uid,
            RequestServer::ManagedStop&& stopper) NN_NOEXCEPT;

        virtual ~AsyncAssignRightsToResumeImpl() NN_NOEXCEPT;

        Result Run() NN_NOEXCEPT;

        os::SystemEvent& GetEvent() NN_NOEXCEPT
        {
            return m_Event;
        }

        virtual Result GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = ErrorContextHolder::GetErrorContextImpl();
            NN_RESULT_SUCCESS;
        }

    private:
        virtual void CancelImpl() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result GetImpl() NN_NOEXCEPT NN_OVERRIDE;

        Result Execute() NN_NOEXCEPT;
        Result QueryAvailableELicenses(AvailableELicenseHolderHandle* outValue) NN_NOEXCEPT;
        Result AssignRights(AvailableELicenseHolderHandle handle) NN_NOEXCEPT;

        bool IsCanceled() const NN_NOEXCEPT;

        Result m_Result {};
        ManualClearSystemEvent m_Event;
        util::optional<nim::AsyncAvailableELicenses> m_AsyncAvailableELicenses;
        util::optional<nim::AsyncAssignedELicenses> m_AsyncAssignedELicenses;
        util::optional<ec::system::AsyncProgressResult> m_AsyncSyncTicket;
        util::optional<nim::AsyncResult> m_AsyncDownloadETicket;
        util::optional<nim::AsyncResult> m_AsyncSyncELicense;
        mutable NonRecursiveMutex m_CancelMutex;
        util::optional<ThreadInfo> m_ThreadInfo;
        bool m_IsCanceled {};

        RightsEnvironmentManager* m_pRightsEnvironmentManager;
        RightsEnvironmentHandle m_Handle;
        account::Uid m_Uid;
        RequestServer::ManagedStop m_Stopper;
    };

}}} // ~nn::ns::srv

