﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdlib>
#include <mutex>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_ExFatDriverManager.h>
#include <nn/os/os_Thread.h>
#include <nn/util/util_ScopeExit.h>
#include "ns_AsyncContentDeliveryImpl.h"
#include "ns_AsyncThreadAllocator.h"
#include "ns_LogUtil.h"
#include "ns_ReceiveApplicationTask.h"
#include "ns_SendApplicationTask.h"
#include "ns_SystemUpdateUtil.h"

namespace nn { namespace ns { namespace srv {

//--------------------------------------------------------------------------
//  AsyncReceiveApplicationImpl
//--------------------------------------------------------------------------
    AsyncReceiveApplicationImpl::~AsyncReceiveApplicationImpl() NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[AsyncReceiveApplicationImpl] Destructor\n");

        if (m_ThreadInfo)
        {
            CancelImpl();
            os::WaitThread(m_ThreadInfo->thread);
            os::DestroyThread(m_ThreadInfo->thread);
            GetAsyncThreadAllocator()->Free(*m_ThreadInfo);
        }

        m_AsyncResult = util::nullopt;
        ReceiveApplicationTask task(m_AppId);
        if (task.IsValid())
        {
            NN_NS_TRACE_RESULT_IF_FAILED(task.Destroy(), "[AsyncReceiveApplicationImpl] Failed to destory task.\n");
        }
    }

    Result AsyncReceiveApplicationImpl::Initialize(ncm::ApplicationId appId, uint32_t ipv4, uint16_t port, const ncm::ContentMetaKey keyList[], int numKey, ncm::StorageId storageId) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(numKey <= MaxContentMetaKeyForContentDelivery, ResultTooManyContentMetaKey());
        m_AppId = appId;
        m_Ip = ipv4;
        m_Port = port;
        std::memcpy(m_KeyList, keyList, sizeof(m_KeyList[0]) * numKey);
        m_KeyCount = numKey;
        m_StorageId = storageId;
        NN_RESULT_SUCCESS;
    }

    Result AsyncReceiveApplicationImpl::Run() NN_NOEXCEPT
    {
        ThreadInfo info;
        NN_RESULT_DO(GetAsyncThreadAllocator()->Allocate(&info));
        info.priority = NN_SYSTEM_THREAD_PRIORITY(nssrv, AsyncReceiveApplicationTask);
        NN_UTIL_SCOPE_EXIT
        {
            if (!m_ThreadInfo)
            {
                GetAsyncThreadAllocator()->Free(info);
            }
        };
        NN_RESULT_DO(os::CreateThread(info.thread, [](void* p)
        {
            auto t = reinterpret_cast<AsyncReceiveApplicationImpl*>(p);
            t->m_Result = t->Execute();
            t->m_Event.Signal();
        }, this, info.stack, info.stackSize, info.priority));
        os::SetThreadNamePointer(info.thread, NN_SYSTEM_THREAD_NAME(nssrv, AsyncReceiveApplicationTask));
        os::StartThread(info.thread);

        m_ThreadInfo = info;

        NN_RESULT_SUCCESS;
    }

    Result AsyncReceiveApplicationImpl::GetImpl() NN_NOEXCEPT
    {
        return m_Result;
    }

    void AsyncReceiveApplicationImpl::CancelImpl() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_CancelMutex);

        m_IsCanceled = true;
        if (m_AsyncResult)
        {
            m_AsyncResult->Cancel();
        }
    }

    void AsyncReceiveApplicationImpl::CreateErrorContext(err::ErrorContext* pContext, Result result) NN_NOEXCEPT
    {
        // nim では何も登録しないので、こちらで登録する
        err::LocalContentShareErrorContext context {};
        context.applicationId = m_AppId;
        context.resultInnerValue = result.GetInnerValueForDebug();
        context.ip = m_Ip;
        context.isSender = false;
        context.isApplicationRequest = true;
        NN_NS_TRACE_RESULT_IF_FAILED(ExFatDriverManager::IsExFatInstalled(&context.hasExFatDriver), "[AsyncReceiveApplicationImpl] Failed to IsExFatInstalled().\n");
        context.numKey = m_KeyCount;
        auto copyCount = std::min(m_KeyCount, static_cast<int>(NN_ARRAY_SIZE(context.keyList)));
        std::memcpy(context.keyList, m_KeyList, sizeof(context.keyList[0]) * copyCount);

        pContext->type = err::ErrorContextType::LocalContentShare;
        pContext->localContentShare = context;
    }

    Result AsyncReceiveApplicationImpl::Execute() NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());

        nim::LocalCommunicationReceiveApplicationTaskId id;
        NN_RESULT_DO(nim::CreateLocalCommunicationReceiveApplicationTask(&id, m_Ip, m_Port, m_AppId, m_KeyList, m_KeyCount, m_StorageId));

        NN_UTIL_SCOPE_EXIT
        {
            std::lock_guard<NonRecursiveMutex> guard(m_CancelMutex);
            m_AsyncResult = util::nullopt;
        };
        {
            std::lock_guard<NonRecursiveMutex> guard(m_CancelMutex);
            m_AsyncResult.emplace();
            NN_RESULT_DO(nim::RequestLocalCommunicationReceiveApplicationTaskRun(&(*m_AsyncResult), id));
        }

        // 処理が終了するのを待ち、その結果を取得する
        NN_RESULT_DO(SaveNuiShellTaskErrorContextIfFailed(*this, m_AsyncResult->Get()));

        NN_RESULT_SUCCESS;
    }

    bool AsyncReceiveApplicationImpl::IsCanceled() const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_CancelMutex);
        return m_IsCanceled;
    }

//--------------------------------------------------------------------------
//  AsyncSendApplicationImpl
//--------------------------------------------------------------------------

    AsyncSendApplicationImpl::~AsyncSendApplicationImpl() NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[AsyncSendApplicationImpl] Destructor\n");

        if (m_ThreadInfo)
        {
            CancelImpl();
            os::WaitThread(m_ThreadInfo->thread);
            os::DestroyThread(m_ThreadInfo->thread);
            GetAsyncThreadAllocator()->Free(*m_ThreadInfo);
        }

        m_AsyncResult = util::nullopt;

        SendApplicationTask task(m_AppId);
        if (task.IsValid())
        {
            NN_NS_TRACE_RESULT_IF_FAILED(task.Destroy(), "[AsyncSendApplicationImpl] Failed to destory task.\n");
        }
    }

    Result AsyncSendApplicationImpl::Initialize(ncm::ApplicationId id, uint32_t ipv4, uint16_t port) NN_NOEXCEPT
    {
        m_AppId = id;
        m_Ip = ipv4;
        m_Port = port;
        NN_RESULT_SUCCESS;
    }

    Result AsyncSendApplicationImpl::Add(const ncm::StorageContentMetaKey keyList[], int numKey) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_KeyCount + numKey < MaxContentMetaKeyForContentDelivery, ResultTooManyContentMetaKey());
        std::memcpy(&m_KeyList[m_KeyCount], keyList, numKey * sizeof(m_KeyList[0]));
        m_KeyCount += numKey;

        NN_RESULT_SUCCESS;
    }

    Result AsyncSendApplicationImpl::Run() NN_NOEXCEPT
    {
        ThreadInfo info;
        NN_RESULT_DO(GetAsyncThreadAllocator()->Allocate(&info));
        info.priority = NN_SYSTEM_THREAD_PRIORITY(nssrv, AsyncSendApplicationTask);
        NN_UTIL_SCOPE_EXIT
        {
            if (!m_ThreadInfo)
            {
                GetAsyncThreadAllocator()->Free(info);
            }
        };
        NN_RESULT_DO(os::CreateThread(info.thread, [](void* p)
        {
            auto t = reinterpret_cast<AsyncSendApplicationImpl*>(p);
            t->m_Result = t->Execute();
            t->m_Event.Signal();
        }, this, info.stack, info.stackSize, info.priority));
        os::SetThreadNamePointer(info.thread, NN_SYSTEM_THREAD_NAME(nssrv, AsyncSendApplicationTask));
        os::StartThread(info.thread);

        m_ThreadInfo = info;

        NN_RESULT_SUCCESS;
    }

    Result AsyncSendApplicationImpl::GetImpl() NN_NOEXCEPT
    {
        return m_Result;
    }


    void AsyncSendApplicationImpl::CancelImpl() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_CancelMutex);

        m_IsCanceled = true;
        if (m_AsyncResult)
        {
            m_AsyncResult->Cancel();
        }
    }

    void AsyncSendApplicationImpl::CreateErrorContext(err::ErrorContext* pContext, Result result) NN_NOEXCEPT
    {
        // nim では何も登録しないので、こちらで登録する
        err::LocalContentShareErrorContext context {};
        context.applicationId = m_AppId;
        context.resultInnerValue = result.GetInnerValueForDebug();
        context.ip = m_Ip;
        context.isSender = true;
        context.isApplicationRequest = true;
        NN_NS_TRACE_RESULT_IF_FAILED(ExFatDriverManager::IsExFatInstalled(&context.hasExFatDriver), "[AsyncSendApplicationImpl] Failed to IsExFatInstalled().\n");
        context.numKey = m_KeyCount;
        auto copyCount = std::min(m_KeyCount, static_cast<int>(NN_ARRAY_SIZE(context.keyList)));
        for (int i = 0; i < copyCount; i++)
        {
            context.keyList[i] = m_KeyList[i].key;
        }

        pContext->type = err::ErrorContextType::LocalContentShare;
        pContext->localContentShare = context;
    }

    Result AsyncSendApplicationImpl::Execute() NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceled());

        nim::LocalCommunicationSendApplicationTaskId id;
        NN_RESULT_DO(nim::CreateLocalCommunicationSendApplicationTask(&id, m_Ip, m_Port, m_KeyList, m_KeyCount, m_AppId));
        NN_UTIL_SCOPE_EXIT
        {
            std::lock_guard<NonRecursiveMutex> guard(m_CancelMutex);
            m_AsyncResult = util::nullopt;
        };
        {
            std::lock_guard<NonRecursiveMutex> guard(m_CancelMutex);
            m_AsyncResult.emplace();
            NN_RESULT_DO(nim::RequestLocalCommunicationSendApplicationTaskRun(&(*m_AsyncResult), id));
        }

        // 処理が終了するのを待ち、その結果を取得する
        NN_RESULT_DO(SaveNuiShellTaskErrorContextIfFailed(*this, m_AsyncResult->Get()));

        NN_RESULT_SUCCESS;
    }

    bool AsyncSendApplicationImpl::IsCanceled() const NN_NOEXCEPT
    {
        std::lock_guard<NonRecursiveMutex> guard(m_CancelMutex);
        return m_IsCanceled;
    }

//--------------------------------------------------------------------------
//  AsyncReceiveSystemUpdateImpl
//--------------------------------------------------------------------------

    AsyncReceiveSystemUpdateImpl::~AsyncReceiveSystemUpdateImpl() NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[AsyncReceiveSystemUpdateImpl] Destructor\n");

        if (m_ThreadInfo)
        {
            CancelImpl();
            os::WaitThread(m_ThreadInfo->thread);
            os::DestroyThread(m_ThreadInfo->thread);
            GetAsyncThreadAllocator()->Free(*m_ThreadInfo);
        }

        m_AsyncResult = util::nullopt;

        nim::LocalCommunicationReceiveSystemUpdateTaskId id;
        if (nim::ListLocalCommunicationReceiveSystemUpdateTask(&id, 1) > 0)
        {
            NN_NS_TRACE_RESULT_IF_FAILED(nim::DestroyLocalCommunicationReceiveSystemUpdateTask(id), "[AsyncReceiveSystemUpdateImpl] Failed to destroy task.\n");
        }
    }

    Result AsyncReceiveSystemUpdateImpl::Run() NN_NOEXCEPT
    {
        ThreadInfo info;
        NN_RESULT_DO(GetAsyncThreadAllocator()->Allocate(&info));
        info.priority = NN_SYSTEM_THREAD_PRIORITY(nssrv, AsyncReceiveSystemUpdateTask);
        NN_UTIL_SCOPE_EXIT
        {
            if (!m_ThreadInfo)
            {
                GetAsyncThreadAllocator()->Free(info);
            }
        };
        NN_RESULT_DO(os::CreateThread(info.thread, [](void* p)
        {
            auto t = reinterpret_cast<AsyncReceiveSystemUpdateImpl*>(p);
            t->m_Result = t->Execute();
            t->m_Event.Signal();
        }, this, info.stack, info.stackSize, info.priority));
        os::SetThreadNamePointer(info.thread, NN_SYSTEM_THREAD_NAME(nssrv, AsyncReceiveSystemUpdateTask));
        os::StartThread(info.thread);

        m_ThreadInfo = info;

        NN_RESULT_SUCCESS;
    }

    Result AsyncReceiveSystemUpdateImpl::GetImpl() NN_NOEXCEPT
    {
        return m_Result;
    }

    void AsyncReceiveSystemUpdateImpl::CancelImpl() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_CancelMutex);

        m_IsCanceled = true;
        if (m_AsyncResult)
        {
            m_AsyncResult->Cancel();
        }
    }

    void AsyncReceiveSystemUpdateImpl::CreateErrorContext(err::ErrorContext* pContext, Result result) NN_NOEXCEPT
    {
        // nim では何も登録しないので、こちらで登録する
        err::LocalContentShareErrorContext context {};
        context.applicationId = ncm::ApplicationId::GetInvalidId();
        context.resultInnerValue = result.GetInnerValueForDebug();
        context.ip = m_Ip;
        context.isSender = false;
        context.isApplicationRequest = false;
        NN_NS_TRACE_RESULT_IF_FAILED(ExFatDriverManager::IsExFatInstalled(&context.hasExFatDriver), "[AsyncReceiveSystemUpdateImpl] Failed to IsExFatInstalled().\n");
        context.numKey = 1;
        context.keyList[0] = m_Key;

        pContext->type = err::ErrorContextType::LocalContentShare;
        pContext->localContentShare = context;
    }

    Result AsyncReceiveSystemUpdateImpl::Execute() NN_NOEXCEPT
    {
        nim::SystemUpdateTaskId networkSystemUpdateTaskId;
        auto count = nim::ListSystemUpdateTask(&networkSystemUpdateTaskId, 1);
        NN_RESULT_THROW_UNLESS(count == 0, ResultSystemUpdateAlreadyHasTask());

        bool needsUpdate;
        NN_RESULT_DO(NeedsUpdate(&needsUpdate, m_Key, m_IsExFatDriverDownloaded));
        NN_RESULT_THROW_UNLESS(needsUpdate, ResultAlreadyUpToDate());

        auto config = m_IsExFatDriverDownloaded ? ncm::InstallConfig_SystemUpdateRecursive | ncm::InstallConfig_RequiresExFatDriver :
                                                  ncm::InstallConfig_SystemUpdateRecursive;
        nim::LocalCommunicationReceiveSystemUpdateTaskId id;
        NN_RESULT_DO(nim::CreateLocalCommunicationReceiveSystemUpdateTask(&id, m_Ip, m_Port, m_Key, config));

        NN_UTIL_SCOPE_EXIT
        {
            std::lock_guard<os::Mutex> guard(m_CancelMutex);
            m_AsyncResult = util::nullopt;
        };
        {
            std::lock_guard<os::Mutex> guard(m_CancelMutex);
            NN_RESULT_THROW_UNLESS(!m_IsCanceled, ResultCanceled());
            m_AsyncResult.emplace();
            NN_RESULT_DO(nim::RequestLocalCommunicationReceiveSystemUpdateTaskRun(&(*m_AsyncResult), id));
        }

        NN_RESULT_DO(SaveNuiShellTaskErrorContextIfFailed(*this, m_AsyncResult->Get()));

        NN_RESULT_SUCCESS;
    }

//--------------------------------------------------------------------------
//  AsyncSendSystemUpdateImpl
//--------------------------------------------------------------------------

    AsyncSendSystemUpdateImpl::~AsyncSendSystemUpdateImpl() NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[AsyncSendSystemUpdateImpl] Destructor\n");

        if (m_ThreadInfo)
        {
            CancelImpl();
            os::WaitThread(m_ThreadInfo->thread);
            os::DestroyThread(m_ThreadInfo->thread);
            GetAsyncThreadAllocator()->Free(*m_ThreadInfo);
        }
        m_AsyncResult = util::nullopt;
        nim::LocalCommunicationSendSystemUpdateTaskId id;
        if (nim::ListLocalCommunicationSendSystemUpdateTask(&id, 1) > 0)
        {
            NN_NS_TRACE_RESULT_IF_FAILED(nim::DestroyLocalCommunicationSendSystemUpdateTask(id), "[AsyncSendSystemUpdateImpl] Failed to destroy task.\n");
        }
    }

    Result AsyncSendSystemUpdateImpl::Run() NN_NOEXCEPT
    {
        ThreadInfo info;
        NN_RESULT_DO(GetAsyncThreadAllocator()->Allocate(&info));
        info.priority = NN_SYSTEM_THREAD_PRIORITY(nssrv, AsyncSendSystemUpdateTask);
        NN_UTIL_SCOPE_EXIT
        {
            if (!m_ThreadInfo)
            {
                GetAsyncThreadAllocator()->Free(info);
            }
        };
        NN_RESULT_DO(os::CreateThread(info.thread, [](void* p)
        {
            auto t = reinterpret_cast<AsyncSendSystemUpdateImpl*>(p);
            t->m_Result = t->Execute();
            t->m_Event.Signal();
        }, this, info.stack, info.stackSize, info.priority));
        os::SetThreadNamePointer(info.thread, NN_SYSTEM_THREAD_NAME(nssrv, AsyncSendSystemUpdateTask));
        os::StartThread(info.thread);

        m_ThreadInfo = info;

        NN_RESULT_SUCCESS;
    }

    Result AsyncSendSystemUpdateImpl::GetImpl() NN_NOEXCEPT
    {
        return m_Result;
    }

    void AsyncSendSystemUpdateImpl::CancelImpl() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_CancelMutex);

        m_IsCanceled = true;

        if (m_AsyncResult)
        {
            m_AsyncResult->Cancel();
        }
    }

    void AsyncSendSystemUpdateImpl::CreateErrorContext(err::ErrorContext* pContext, Result result) NN_NOEXCEPT
    {
        // nim では何も登録しないので、こちらで登録する
        err::LocalContentShareErrorContext context {};
        context.applicationId = ncm::ApplicationId::GetInvalidId();
        context.resultInnerValue = result.GetInnerValueForDebug();
        context.ip = m_Ip;
        context.isSender = true;
        context.isApplicationRequest = false;
        NN_NS_TRACE_RESULT_IF_FAILED(ExFatDriverManager::IsExFatInstalled(&context.hasExFatDriver), "[AsyncSendSystemUpdateImpl] Failed to IsExFatInstalled().\n");
        context.numKey = 1;
        context.keyList[0] = m_Key;

        pContext->type = err::ErrorContextType::LocalContentShare;
        pContext->localContentShare = context;
    }

    Result AsyncSendSystemUpdateImpl::Execute() NN_NOEXCEPT
    {
        nim::LocalCommunicationSendSystemUpdateTaskId id;
        NN_RESULT_DO(nim::CreateLocalCommunicationSendSystemUpdateTask(&id, m_Ip, m_Port, m_Key));

        NN_UTIL_SCOPE_EXIT
        {
            std::lock_guard<os::Mutex> guard(m_CancelMutex);
            m_AsyncResult = util::nullopt;
        };
        {
            std::lock_guard<os::Mutex> guard(m_CancelMutex);
            NN_RESULT_THROW_UNLESS(!m_IsCanceled, ResultCanceled());
            m_AsyncResult.emplace();
            NN_RESULT_DO(nim::RequestLocalCommunicationSendSystemUpdateTaskRun(&(*m_AsyncResult), id));
        }

        NN_RESULT_DO(SaveNuiShellTaskErrorContextIfFailed(*this, m_AsyncResult->Get()));

        NN_RESULT_SUCCESS;
    }

}}}
