﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/detail/ns_IAsync.sfdl.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/detail/json/ns_RapidJsonApi.h>
#include <nn/ns/srv/detail/json/ns_RapidJsonInputStream.h>
#include <nn/ns/srv/ns_ApplicationVersionManager.h>
#include <nn/ns/srv/ns_VersionManagementDatabase.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_Drm.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/util/util_ScopeExit.h>
#include "ns_AsyncImpl.h"
#include "ns_Config.h"
#include "ns_DebugUtil.h"
#include "ns_SystemUpdateUtil.h"
#include "ns_TaskUtil.h"

namespace nn { namespace ns { namespace srv {
    namespace {
        // クイックランチャーの表示数に従う
        const int MaxAutoUpdateCount = 12;

        class NotifiedVersionGetter {
        public:
            static uint32_t GetVersion(const VersionListEntry& entry) NN_NOEXCEPT
            {
                return entry.notifiedVersion;
            }
        };

        class RequiredVersionGetter {
        public:
            static uint32_t GetVersion(const RequiredVersionEntry& entry) NN_NOEXCEPT
            {
                return entry.requiredVersion;
            }
        };

        typedef sf::ObjectFactory<sf::impl::StaticOneAllocationPolicy> StaticOneFactory;

        template<typename Database, typename EntryType, typename VersionGetter>
        uint32_t GetVersion(ncm::ApplicationId applicationId, ncm::PatchId patchId, void* lockedBuffer, size_t bufferSize) NN_NOEXCEPT
        {
            Database db(VersionListMountName, lockedBuffer, bufferSize);

            // これは何だ？
            NN_RESULT_DO(db.Load());

            EntryType defaultEntry = {};
            EntryType* appEntry;
            EntryType* patchEntry;
            appEntry = db.FindOrDefault(applicationId.value, &defaultEntry);
            patchEntry = db.FindOrDefault(patchId.value, &defaultEntry);
            return std::max(VersionGetter::GetVersion(*appEntry), VersionGetter::GetVersion(*patchEntry));
        }

        Result CheckApplicationUpdateRequired(ncm::ApplicationId applicationId, ncm::PatchId patchId, uint32_t version, void* lockedBuffer, size_t bufferSize) NN_NOEXCEPT
        {
            auto requiredVersion = GetVersion<RequiredVersionDatabase, RequiredVersionEntry, RequiredVersionGetter>(
                applicationId, patchId, lockedBuffer, bufferSize);

            NN_RESULT_THROW_UNLESS(version >= requiredVersion, nn::ns::ResultApplicationUpdateRequiredByRequiredVersion());

            NN_RESULT_SUCCESS;
        }

        Result CheckApplicationUpdateRecommended(ncm::ApplicationId applicationId, ncm::PatchId patchId, uint32_t version, void* lockedBuffer, size_t bufferSize) NN_NOEXCEPT
        {
            auto notifiedVersion = GetVersion<VersionListDatabase, VersionListEntry, NotifiedVersionGetter>(
                applicationId, patchId, lockedBuffer, bufferSize);

            NN_RESULT_THROW_UNLESS(version >= notifiedVersion, nn::ns::ResultApplicationUpdateRecommended());

            NN_RESULT_SUCCESS;
        }

        template<typename ExtendedHeaderType>
        Result GetRequiredSystemVersion(uint32_t* out, Bit64 id, IntegratedContentManager& integrated) NN_NOEXCEPT
        {
            ncm::ContentMetaKey key;
            NN_RESULT_DO(integrated.GetLatest(&key, id));
            size_t contentMetaBufferSize;
            NN_RESULT_DO(integrated.GetSize(&contentMetaBufferSize, key));

            std::unique_ptr<nn::Bit8[]> contentMetaBuffer(new nn::Bit8[contentMetaBufferSize]);

            size_t readSize;
            NN_RESULT_DO(integrated.Get(&readSize, contentMetaBuffer.get(), contentMetaBufferSize, key));
            ncm::ContentMetaReader reader(contentMetaBuffer.get(), readSize);


            *out = reader.GetExtendedHeader<ExtendedHeaderType>()->requiredSystemVersion;

            NN_RESULT_SUCCESS;
        }

        Result CheckSystemUpdateRequired(uint32_t requiredSystemVersion) NN_NOEXCEPT
        {
            auto systemUpdateVersion = GetSystemUpdateVersion();

            if (systemUpdateVersion < requiredSystemVersion)
            {
                NN_RESULT_THROW(ns::ResultSystemUpdateRequired());
            }

            NN_RESULT_SUCCESS;
        }

        Result MountAndCreateSaveData() NN_NOEXCEPT
        {
            size_t saveDataSize =
                VersionListDatabaseConfig::MaxEntryCount * sizeof(VersionListEntry) +
                RequiredVersionDatabaseConfig::MaxEntryCount * sizeof(RequiredVersionEntry) +
                16 * 1024 * 4; // ヘッダファイル2つ + マージン

            // SIGLO-31130 参照
            size_t journalSize =
                std::max(VersionListDatabaseConfig::MaxEntryCount * sizeof(VersionListEntry),
                    RequiredVersionDatabaseConfig::MaxEntryCount * sizeof(RequiredVersionEntry)) +
                16 * 1024 + // ヘッダ1つ
                32 * 1024;  // 管理ブロック

            NN_RESULT_TRY(fs::CreateSystemSaveData(VersionListSaveDataId, saveDataSize, journalSize, VersionListSaveDataFlags))
                NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
                {
                    // 作成済みならエラーにしない
                }
            NN_RESULT_END_TRY;
            NN_RESULT_DO(fs::MountSystemSaveData(VersionListMountName, VersionListSaveDataId));

            NN_RESULT_SUCCESS;
        }

        class AsyncVersionListDataImpl : public AsyncValueBase
        {
        public:
            explicit AsyncVersionListDataImpl(RequestServer::ManagedStop&& stopper) NN_NOEXCEPT : m_Stopper(std::move(stopper)) {}

            Result Initialize() NN_NOEXCEPT
            {
                NN_RESULT_TRY(nim::RequestVersionList(&m_AsyncVersionListData, nim::ETag::MakeEmpty()))
                    NN_RESULT_CATCH_CONVERT(nim::ResultOutOfMaxTask, ResultOutOfMaxRunningTask())
                    NN_RESULT_CATCH_CONVERT(nim::ResultOutOfMaxRunningTask, ResultOutOfMaxRunningTask())
                NN_RESULT_END_TRY
                NN_RESULT_SUCCESS;
            }
            os::SystemEvent& GetEvent() NN_NOEXCEPT
            {
                return m_AsyncVersionListData.GetEvent();
            }

        private:
            virtual size_t GetSizeImpl() NN_NOEXCEPT NN_OVERRIDE
            {
                return static_cast<size_t>(m_AsyncVersionListData.GetSize());
            }
            virtual Result GetImpl(const nn::sf::OutBuffer& buffer) NN_NOEXCEPT NN_OVERRIDE
            {
                NN_RESULT_DO(m_AsyncVersionListData.Get());
                size_t size;
                NN_RESULT_DO(m_AsyncVersionListData.Read(&size, 0, buffer.GetPointerUnsafe(), buffer.GetSize()));
                NN_RESULT_SUCCESS;
            }
            virtual void CancelImpl() NN_NOEXCEPT NN_OVERRIDE
            {
                m_AsyncVersionListData.Cancel();
            }

            RequestServer::ManagedStop m_Stopper;
            nim::AsyncData m_AsyncVersionListData;
        };
    } // namespace

    Result  ApplicationVersionManager::PushLaunchVersion(ncm::ApplicationId id, uint32_t version) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> bufferLock(m_VersionEntryBufferLock);

        RequiredVersionDatabase db(VersionListMountName, m_VersionEntryBuffer, sizeof(m_VersionEntryBuffer));

        NN_RESULT_DO(db.Load());
        RequiredVersionEntry entry = { id.value, version };
        NN_RESULT_DO(db.Update(id.value, entry));
        NN_RESULT_DO(db.Commit());

        NN_RESULT_SUCCESS;
    }

    Result ApplicationVersionManager::UpdateVersionList(const char* versionListBuffer, size_t versionListBufferSize) NN_NOEXCEPT
    {
        // DevMenuCommand の application update-version-list 等ローカルでのインポート時に呼ばれる
        Bit8 jsonParseWorkBuffer[128];
        detail::json::MemoryInputStreamForRapidJson inputStream(jsonParseWorkBuffer, sizeof(jsonParseWorkBuffer));
        inputStream.Set(versionListBuffer, versionListBufferSize);
        detail::json::EncodedMemoryInputStreamForRapidJson encodedInputStream(inputStream);

        return UpdateVersionList(encodedInputStream);
    }

    Result ApplicationVersionManager::UpdateVersionList(nim::AsyncData* asyncData, char* workBuffer, size_t workBufferSize) NN_NOEXCEPT
    {
        // ネットワーク経由でのインポート時に呼ばれる
        detail::json::AsyncDataInputStreamForRapidJson<> inputStream(asyncData, workBuffer, workBufferSize);
        return UpdateVersionList(inputStream);
    }

    template<typename InputStreamType>
    Result ApplicationVersionManager::UpdateVersionList(InputStreamType& inputStream) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> bufferLock(m_VersionEntryBufferLock);

        VersionListDatabase db(VersionListMountName, m_VersionEntryBuffer, sizeof(m_VersionEntryBuffer));

        VersionListImporter importer(&db);
        NN_RESULT_DO(importer.Import(inputStream));
        NN_RESULT_DO(db.Commit());

        NN_RESULT_SUCCESS;
    }

    Result ApplicationVersionManager::PerformAutoUpdate() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> bufferLock(m_VersionEntryBufferLock);

        VersionListDatabase db(VersionListMountName, m_VersionEntryBuffer, sizeof(m_VersionEntryBuffer));

        NN_RESULT_DO(db.Load());

        if (settings::system::IsAutoUpdateEnabled())
        {
            PrepareInstallTask(&db);
        }

        NN_RESULT_SUCCESS;
    }

    Result ApplicationVersionManager::ListRequiredVersion(int* outCount, RequiredVersionEntry* outValue, int count) NN_NOEXCEPT
    {
        return ListTemplate<RequiredVersionDatabase>(outCount, outValue, count);
    }

    Result  ApplicationVersionManager::ListVersionList(int* outCount, VersionListEntry* outValue, int count) NN_NOEXCEPT
    {
        return ListTemplate<VersionListDatabase>(outCount, outValue, count);
    }

    Result  ApplicationVersionManager::CheckApplicationLaunchVersion(ncm::ApplicationId applicationId, ncm::PatchId patchId, uint32_t version, uint32_t requiredSystemVersion, uint32_t requiredApplicationVersion) NN_NOEXCEPT
    {
        // システムバージョンのチェック
        NN_RESULT_DO(CheckSystemUpdateRequired(requiredSystemVersion));

        {
            std::lock_guard<nn::os::Mutex> bufferLock(m_VersionEntryBufferLock);
            // 必須バージョンのチェック
            NN_RESULT_DO(CheckApplicationUpdateRequired(applicationId, patchId, version, m_VersionEntryBuffer, sizeof(m_VersionEntryBuffer)));

            // 必須アプリケーションリリースバージョンのチェック
            NN_RESULT_THROW_UNLESS(version >= requiredApplicationVersion, ResultApplicationUpdateRequiredByRequiredApplicationVersion());

            // ブラックリストによる起動制限のチェック
            NN_RESULT_THROW_UNLESS(!m_pBlackListManager->IsApplicationOnBlackList(applicationId, version), ResultApplicationUpdateRequiredByBlackList());

            // 推奨バージョンのチェック
            NN_RESULT_DO(CheckApplicationUpdateRecommended(applicationId, patchId, version, m_VersionEntryBuffer, sizeof(m_VersionEntryBuffer)));
        }

        NN_RESULT_SUCCESS;
    }

    Result ApplicationVersionManager::RequestVersionListData(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncValue>> outAsync, RequestServer::ManagedStop&& stopper) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<ns::detail::IAsyncValue, AsyncVersionListDataImpl>(std::move(stopper));
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize());

        *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
        *outAsync = emplacedRef;
        NN_RESULT_SUCCESS;
    }

    Result  ApplicationVersionManager::Initialize(ApplicationRecordDatabase* record, BlackListManager* blacklist) NN_NOEXCEPT
    {
        NN_RESULT_DO(MountAndCreateSaveData());
        m_Record = record;
        m_pBlackListManager = blacklist;

        NN_RESULT_SUCCESS;
    }

    void    ApplicationVersionManager::Finalize() NN_NOEXCEPT
    {
        fs::Unmount(VersionListMountName);
    }

    template<typename DatabaseType>
    Result ApplicationVersionManager::ListTemplate(int* outCount, typename DatabaseType::SelfEntryType* outValue, int count) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> bufferLock(m_VersionEntryBufferLock);

        DatabaseType db(VersionListMountName, m_VersionEntryBuffer, sizeof(m_VersionEntryBuffer));

        NN_RESULT_DO(db.Load());

        int index;
        for (index = 0; index < count; ++index)
        {
            auto entry = db.Get(index);
            if (entry == nullptr) {
                break;
            }
            outValue[index] = *entry;
        }

        *outCount = index;
        NN_RESULT_SUCCESS;
    }

    void ApplicationVersionManager::PrepareInstallTask(VersionListDatabase* db) NN_NOEXCEPT
    {
        // TODO: ちゃんとした同期

        ApplicationRecord list[MaxAutoUpdateCount];
        auto count = m_Record->ListSpecificAttribute(list, MaxAutoUpdateCount, ApplicationRecordAttribute::AutoUpdateEnabled);
        for (int i = 0; i < count; i++)
        {
            auto result = PrepareInstallTaskOneIfNeeded(db, list[i].id);
            if (result.IsFailure())
            {
                NN_DETAIL_NS_TRACE("[ApplicationVersionManager] Failed to prepare install task for 0x%016llx\n", list[i].id.value);
            }
        }
    }

    Result ApplicationVersionManager::PrepareInstallTaskOneIfNeeded(VersionListDatabase* db, ncm::ApplicationId id) NN_NOEXCEPT
    {
        util::optional<ncm::ContentMetaKey> mainKey;
        NN_RESULT_DO(m_Record->FindMain(&mainKey, id));
        NN_RESULT_THROW_UNLESS(mainKey, ResultMainApplicationNotFound());

        AutoUpdateInfo info;
        NN_RESULT_DO(m_Record->GetAutoUpdateInfo(&info, id));
        auto patchId = info.patchId;

        uint32_t version;
        {
            // TODO: とりあえずリマスターの自動更新は考えない
            auto entry = db->Find(patchId.value);
            if (!entry)
            {
                NN_DETAIL_NS_TRACE("[ApplicationVersionManager] No version list entry for patchId 0x%016llx\n", patchId.value);
                NN_RESULT_SUCCESS;
            }
            auto notifiedVersion = entry->notifiedVersion;
            if (notifiedVersion <= mainKey->version)
            {
                NN_DETAIL_NS_TRACE("[ApplicationVersionManager] Notified version %u <= current recorded version %u\n", notifiedVersion, mainKey->version);
                NN_RESULT_SUCCESS;
            }
            if (notifiedVersion <= info.currentVersion)
            {
                NN_DETAIL_NS_TRACE("[ApplicationVersionManager] Notified version %u <= current auto update version %u\n", notifiedVersion, info.currentVersion);
                NN_RESULT_SUCCESS;
            }

            version = entry->recommendedVersion;
            NN_DETAIL_NS_TRACE("[ApplicationVersionManager] Found recommended version %u\n", version);
        }

        bool canCreate;
        NN_RESULT_DO(CanCreateNetworkInstallTask(&canCreate, id, ncm::ContentMetaType::Patch));
        if (!canCreate)
        {
            NN_DETAIL_NS_TRACE("[ApplicationVersionManager] Skip create install task\n");
        }
        else
        {
            auto patchKey = ncm::ContentMetaKey::Make(patchId, version);
            nim::NetworkInstallTaskId taskId;

            NN_RESULT_DO(nim::CreateNetworkInstallTask(&taskId, id, &patchKey, 1, ncm::StorageId::Any));
            NN_DETAIL_NS_TRACE("[ApplicationVersionManager] Created\n");
            m_Record->NotifyUpdate();
        }

        NN_RESULT_SUCCESS;
    }

    Result  ApplicationVersionManager::GetLaunchRequiredVersion(nn::sf::Out<std::uint32_t> outValue, nn::ncm::ApplicationId id) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> bufferLock(m_VersionEntryBufferLock);

        RequiredVersionDatabase db(VersionListMountName, m_VersionEntryBuffer, sizeof(m_VersionEntryBuffer));

        NN_RESULT_DO(db.Load());

        RequiredVersionEntry defaultEntry = {};
        auto entry = db.FindOrDefault(id.value, &defaultEntry);

        *outValue = entry->requiredVersion;
        NN_RESULT_SUCCESS;
    }

    Result  ApplicationVersionManager::UpgradeLaunchRequiredVersion(nn::ncm::ApplicationId id, std::uint32_t launchRequiredVersion) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> bufferLock(m_VersionEntryBufferLock);

        RequiredVersionDatabase db(VersionListMountName, m_VersionEntryBuffer, sizeof(m_VersionEntryBuffer));

        NN_RESULT_DO(db.Load());

        RequiredVersionEntry defaultEntry = {};
        auto entry = db.FindOrDefault(id.value, &defaultEntry);

        if (launchRequiredVersion > entry->requiredVersion)
        {
            RequiredVersionEntry newEntry = { id.value, launchRequiredVersion };
            NN_RESULT_DO(db.Update(id.value, newEntry));
            NN_RESULT_DO(db.Commit());
        }

        NN_RESULT_SUCCESS;
    }

}}}
