﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nim/nim_Result.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/detail/ns_IAsync.sfdl.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_ApplicationDownloadManager.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include "ns_ApplicationInstallTask.h"
#include "ns_ApplicationApplyDeltaTask.h"
#include "ns_AsyncImpl.h"
#include "ns_TaskUtil.h"

namespace nn { namespace ns { namespace srv {

namespace
{
template<class TTask, typename TResult>
Result Resume(ncm::ApplicationId id, RequestServer* requestServer) NN_NOEXCEPT
{
    auto managedStop = requestServer->Stop();

    // RequestServer を止めた状態でタスクがあるか確認する
    TTask task(id);
    NN_RESULT_THROW_UNLESS(task.IsValid(), TResult());

    NN_RESULT_DO(task.Resume());
    NN_RESULT_SUCCESS;
}

template<class TTask, typename TResult>
Result CalculateRequiredSize(int64_t* outSize, ncm::StorageId* outStorageId, ncm::ApplicationId id) NN_NOEXCEPT
{
    TTask task(id);
    NN_RESULT_THROW_UNLESS(task.IsValid(), TResult());

    auto result = [&]() -> Result
        {
            typename TTask::TypeProgress progress;
            NN_RESULT_DO(task.GetProgress(&progress));

            if (progress.state == TTask::TypeState::NotEnoughSpace)
            {
                NN_RESULT_DO(task.CalculateRequiredSize(outSize));
                NN_RESULT_DO(task.GetRequiredStorage(outStorageId));

                NN_RESULT_SUCCESS;
            }
            NN_RESULT_THROW(ResultApplicationSeemsEnoughSpace());
        }();

    // TTask を作ったあとに自動コミットされることを考えて、TaskNotFound をハンドルする
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(nim::ResultTaskNotFound)
        {
            NN_RESULT_THROW(TResult());
        }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}



}
    typedef sf::ObjectFactory<sf::impl::StaticOneAllocationPolicy> StaticOneFactory;

    Result ApplicationDownloadManager::RequestApplicationUpdateInfo(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncValue>> outAsync, ncm::ApplicationId id) NN_NOEXCEPT
    {
        ApplicationRecord record;
        NN_RESULT_DO(m_RecordDb->Get(&record, id));

        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<ns::detail::IAsyncValue, AsyncApplicationUpdateInfoImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());

        ApplicationRecordAccessor list;
        NN_RESULT_DO(m_RecordDb->Open(&list, id));

        bool lostAny = false;
        for (int i = 0; i < list.Count(); i++)
        {
            auto& meta = list[i];
            if (meta.storageId == ncm::StorageId::Card)
            {
                continue;
            }

            bool hasEntity;
            NN_RESULT_DO(m_Integrated->HasContentMetaKey(&hasEntity, meta.key));
            if (!hasEntity)
            {
                lostAny = true;
                break;
            }
        }

        if (lostAny)
        {
            NN_RESULT_DO(emplacedRef.GetImpl().Immediate(ApplicationUpdateInfo::Updatable));
        }
        else
        {
            auto buffer = m_LockedBufferManager->Allocate(sizeof(ncm::ContentMetaKey) * MaxContentMetaCountPerApplication);
            auto keyList = buffer.Get<ncm::ContentMetaKey*>();

            int count = 0;
            for (; count < list.Count() && count < MaxContentMetaCountPerApplication; count++)
            {
                keyList[count] = list[count].key;
            }

            NN_RESULT_DO(emplacedRef.GetImpl().Initialize(id, keyList, count));
        }

        *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
        *outAsync = emplacedRef;

        NN_RESULT_SUCCESS;
    }

    Result ApplicationDownloadManager::RequestUpdateApplication(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, ncm::ApplicationId id) NN_NOEXCEPT
    {
        auto stopper = m_RequestServer->Stop();
        ApplicationRecord record;
        NN_RESULT_DO(m_RecordDb->Get(&record, id));

        {
            bool canCreate;
            NN_RESULT_DO(CanCreateNetworkInstallTask(&canCreate, id, ncm::ContentMetaType::Patch));
            NN_RESULT_THROW_UNLESS(canCreate, ResultAlreadyDownloading());
        }

        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<ns::detail::IAsyncResult, AsyncDownloadApplicationImpl>(id, DownloadApplicationMode::Update, std::move(stopper));
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());

        ApplicationRecordAccessor list;
        NN_RESULT_DO(m_RecordDb->Open(&list, id));

        for (int i = 0; i < list.GetDataCount(); i++)
        {
            auto& meta = list.GetDataAt(i);

            bool forceUpdate = false;
            if (meta.storageId != ncm::StorageId::Card)
            {
                bool hasEntity;
                NN_RESULT_DO(m_Integrated->HasContentMetaKey(&hasEntity, meta.key));
                forceUpdate = !hasEntity;
            }

            NN_RESULT_DO(emplacedRef.GetImpl().Add(&meta.key, 1, forceUpdate));
        }

        NN_RESULT_DO(emplacedRef.GetImpl().Run());

        *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
        *outAsync = emplacedRef;

        NN_RESULT_SUCCESS;
    }

    Result ApplicationDownloadManager::RequestDownloadApplication(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, ncm::ApplicationId applicationId, ncm::StorageId storageId, bool forceDirectUpdate) NN_NOEXCEPT
    {
        auto stopper = m_RequestServer->Stop();

        NN_RESULT_DO(m_RecordDb->Push(applicationId, ns::ApplicationEvent::DownloadStarted, nullptr, 0));

        ApplicationRecord record;
        NN_RESULT_DO(m_RecordDb->Get(&record, applicationId));

        {
            bool canCreate;
            NN_RESULT_DO(CanCreateNetworkInstallTask(&canCreate, applicationId, ncm::ContentMetaType::Patch));
            NN_RESULT_THROW_UNLESS(canCreate, ResultAlreadyDownloading());
        }

        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<ns::detail::IAsyncResult, AsyncDownloadApplicationImpl>(applicationId, DownloadApplicationMode::DownloadApplication, std::move(stopper), storageId, forceDirectUpdate);
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());

        ApplicationRecordAccessor list;
        NN_RESULT_DO(m_RecordDb->Open(&list, applicationId));

        for (int i = 0; i < list.Count(); i++)
        {
            auto& meta = list[i];

            if (meta.key.type == ncm::ContentMetaType::Application || meta.key.type == ncm::ContentMetaType::Patch)
            {
                bool hasEntity;
                Result result = m_Integrated->HasContentMetaKey(&hasEntity, meta.key);
                // 記録がありかつ実体があるコンテンツをリストに登録する
                if (result.IsSuccess() && hasEntity)
                {
                    NN_RESULT_DO(emplacedRef.GetImpl().Add(&meta.key, 1));
                }
            }
        }

        NN_RESULT_DO(emplacedRef.GetImpl().Run());

        *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
        *outAsync = emplacedRef;

        NN_RESULT_SUCCESS;
    }

    Result ApplicationDownloadManager::RequestDownloadAddOnContent(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, ncm::ApplicationId id, sf::InArray<ncm::ContentMetaKey> keyList, ncm::StorageId storageId) NN_NOEXCEPT
    {
        auto stopper = m_RequestServer->Stop();

        ApplicationRecord record;
        NN_RESULT_DO(m_RecordDb->Get(&record, id));

        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<ns::detail::IAsyncResult, AsyncDownloadApplicationImpl>(id, DownloadApplicationMode::DownloadAddOnContent, std::move(stopper), storageId);
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());

        for (size_t i = 0; i < keyList.GetLength(); i++)
        {
            if (keyList[i].type == ncm::ContentMetaType::AddOnContent)
            {
                bool forceUpdate = false;
                ncm::ContentMetaKey key;
                NN_RESULT_TRY(m_Integrated->GetLatest(&key, keyList[i].id))
                    NN_RESULT_CATCH(ncm::ResultContentMetaNotFound)
                    {
                        key = keyList[i];
                        forceUpdate = true;
                    }
                NN_RESULT_END_TRY

                NN_RESULT_DO(emplacedRef.GetImpl().Add(&key, 1, forceUpdate));
            }
        }

        NN_RESULT_DO(emplacedRef.GetImpl().Run());

        *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
        *outAsync = emplacedRef;

        NN_RESULT_SUCCESS;
    }

    Result ApplicationDownloadManager::CancelTask(ncm::ApplicationId id, TaskType taskType) NN_NOEXCEPT
    {
        // アプリのダウンロードがキャンセルされたら、ユーザにとって自動更新不要なはずなので、無効化する
        NN_RESULT_DO(m_RecordDb->DisableAutoUpdate(id));

        NN_RESULT_DO(CleanupTask(id, m_Integrated, m_RequestServer->Stop(), taskType));

        // キャンセルした結果、コンテンツ記録がない記録ができたらその時点で消す
        int count;
        NN_RESULT_DO(m_RecordDb->CountContentMeta(&count, id));
        if (count == 0)
        {
            NN_RESULT_DO(m_RecordDb->Delete(id));
        }

        NN_RESULT_SUCCESS;
    }

    Result ApplicationDownloadManager::ResumeTask(ncm::ApplicationId id, TaskType taskType) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(taskType == TaskType::Download || taskType == TaskType::ApplyDelta, ResultInvalidTaskTypeArgument());
        NN_RESULT_THROW_UNLESS(m_RecordDb->Has(id), ResultApplicationRecordNotFound());

        switch(taskType)
        {
        case TaskType::Download:
            NN_RESULT_DO((Resume<ApplicationInstallTask, ResultApplicationNotDownloading>(id, m_RequestServer)));
            break;
        case TaskType::ApplyDelta:
            NN_RESULT_DO((Resume<ApplicationApplyDeltaTask, ResultApplicationNotApplying>(id, m_RequestServer)));
            break;
        default:
            NN_ABORT("never come here");
            break;
        }
        NN_RESULT_SUCCESS;
    }

    Result ApplicationDownloadManager::CalculateTaskRequiredSize(sf::Out<ncm::StorageId> outStorageId, sf::Out<std::int64_t> outValue, ncm::ApplicationId id, TaskType taskType) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(taskType == TaskType::Download || taskType == TaskType::ApplyDelta, ResultInvalidTaskTypeArgument());
        NN_RESULT_THROW_UNLESS(m_RecordDb->Has(id), ResultApplicationRecordNotFound());

        switch(taskType)
        {
        case TaskType::Download:
            NN_RESULT_DO((CalculateRequiredSize<ApplicationInstallTask, ResultApplicationNotDownloading>(outValue.GetPointer(), outStorageId.GetPointer(), id)));
            break;
        case TaskType::ApplyDelta:
            NN_RESULT_DO((CalculateRequiredSize<ApplicationApplyDeltaTask, ResultApplicationNotApplying>(outValue.GetPointer(), outStorageId.GetPointer(), id)));
            break;
        default:
            NN_ABORT("never come here");
            break;
        }
        NN_RESULT_SUCCESS;
    }

    Result ApplicationDownloadManager::ResumeAll() NN_NOEXCEPT
    {
        // タスクのインデックスがずれたりしても嫌なので、一度タスクは止める
        auto stopper = m_RequestServer->Stop();

        NN_RESULT_DO(ApplicationInstallTask::ResumeAll());
        NN_RESULT_DO(ApplicationApplyDeltaTask::ResumeAll());

        NN_RESULT_SUCCESS;
    }

}}}
