﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nsd/detail/nsd_DetailApi.h>
#include <nn/nsd/nsd_ResultPrivate.h>
#include <nn/nsd/detail/nsd_BuiltinBackboneSettings.h>
#include <nn/nsd/detail/nsd_Log.h>
#include <nn/nsd/nsd_Version.h>

#include <nn/nsd/detail/util/nsd_StringUtility.h>
#include <nn/nsd/detail/config/nsd_Config.h>
#include <nn/nsd/detail/device/nsd_Device.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace nsd { namespace detail {

    namespace
    {
        const char EnrironmentIdentifierConvertedChar = '%';

        template <typename T>
        bool IsZero(const T* pValue, size_t size) NN_NOEXCEPT
        {
            const uint8_t* p = reinterpret_cast<const uint8_t*>(pValue);

            for(size_t i = 0 ; i < size ; ++i)
            {
                if(*p != 0)
                {
                    return false;
                }
                ++p;
            }
            return true;
        }

        // 時限失効チェック。期限切れだと(Sirius fqdn は除く)すべての解決 API を失敗させる
        Result CheckExpiration( int64_t expire, const nn::time::PosixTime& currentTime )
        {
            if(NN_STATIC_CONDITION(config::IsSettingsExpirationEnabled)) // 時限失効のしくみが有効か
            {
                nn::time::PosixTime expireTime = {expire};
                //NN_DETAIL_NSD_INFO("[NSD] compare time: current=%lld expire=%lld\n", currentTime.value, expire );
                NN_RESULT_THROW_UNLESS( currentTime < expireTime, nn::nsd::ResultSettingExpiredDetected() ); // Expire 判定
            }
            NN_RESULT_SUCCESS;
        }
    }

    Result Resolve(
        Fqdn *pOut,
        const Fqdn& fqdn,
        const SaveData& saveData,
        const nn::time::PosixTime& currentTime) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);

        NN_RESULT_THROW_UNLESS( saveData.isAvailable, nn::nsd::ResultSettingExpired() ); // すでに時限失効検出済かチェック
        NN_RESULT_DO( CheckExpiration(saveData.expireUnixTime, currentTime) ); // 時限失効チェック
        NN_RESULT_THROW_UNLESS(saveData.version >= nn::nsd::Version, ResultInvalidSettingVersion());

        // NAS FQDN 判定. dd1/lp1 の NAS FQDN を設定に従った FQDN へ置換する
        // ( 将来 は lp1 の設定を変換するほうだけのはず )
        if(fqdn == GetDd1NasApiFqdn() || fqdn == GetLp1NasApiFqdn())
        {
            return (GetNasApiFqdn(pOut, saveData, currentTime));
        }
        else if(fqdn == GetDd1NasRequestFqdn() || fqdn == GetLp1NasRequestFqdn())
        {
            return (GetNasRequestFqdn(pOut, saveData, currentTime));
        }

        for(auto &entry : saveData.backboneSettings.fqdnEntries)
        {
            if(fqdn == entry.src)
            {
                *pOut = entry.dst;
                NN_RESULT_SUCCESS;
            }
        }

        // applicationSettings の fqdnEntries
        for(auto &entry : saveData.applicationSettings.fqdnEntries)
        {
            if(fqdn == entry.src)
            {
                *pOut = entry.dst;
                NN_RESULT_SUCCESS;
            }
        }

        *pOut = fqdn;
        NN_RESULT_SUCCESS;
    }

    // FQDN に含まれる '%' を環境識別子へ変換します。
    Result InsertEnvironmentIdentifier(Fqdn* pOut, const Fqdn& fqdn, const EnvironmentIdentifier& identifier) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);

        *pOut = fqdn;
        const int length = util::GetLength(fqdn);

        if(length == Fqdn::Size)
        {
            pOut->value[Fqdn::Size - 1] = '\0';
            NN_RESULT_THROW(ResultFqdnIsNotNullTerminated());
        }

        const char* p = util::SearchCharWithLenght(fqdn.value, length, EnrironmentIdentifierConvertedChar);

        if(p == nullptr)
        {
            NN_RESULT_SUCCESS; // 環境識別子入れる必要がない
        }

        // 変換文字列を環境識別子に
        ptrdiff_t position = p - fqdn.value;
        char *dstPtr = pOut->value + position;
        int dstCount = static_cast<int>(Fqdn::Size) - static_cast<int>(position);
        int ret = nn::util::Strlcpy(dstPtr, identifier.value, dstCount);
        NN_RESULT_THROW_UNLESS(ret < dstCount, ResultFqdnTooLong());

        // 環境識別子以降をコピー
        dstPtr += ret;
        dstCount -= ret;
        ret = nn::util::Strlcpy(dstPtr, p + 1, dstCount);
        NN_RESULT_THROW_UNLESS(ret < dstCount, ResultFqdnTooLong());

        NN_RESULT_SUCCESS;
    }

    bool HasBadChar(const Fqdn& fqdn) NN_NOEXCEPT
    {
        const char* p = fqdn.value;
        int count = 0;
        while(*p != '\0' && count < static_cast<int>(Fqdn::Size))
        {
            if( (*p >= 'a' && *p <= 'z') ||
                (*p >= 'A' && *p <= 'Z') ||
                (*p >= '0' && *p <= '9') ||
                (*p == '-') ||
                (*p == '.') )
            {
            }
            else
            {
                return true;
            }
            p++;
            count++;
        }
        return false;
    }

    Result HandleServerError(const char* errorCode) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(errorCode);

        struct ErrorMap
        {
            char errorCode[8];
            Result result;
        };
        const ErrorMap errorMap[] =
        {
            {"0000", nn::nsd::ResultMissingParameterInRequest()},
            {"0001", nn::nsd::ResultInvalidParameterInRequest()},
            {"0002", nn::nsd::ResultPasscodeNotFound()},
            {"0003", nn::nsd::ResultForbidden()},
            {"0004", nn::nsd::ResultMethodNotAllow()},
            {"0005", nn::nsd::ResultApiVersionNotFound()},
            {"0500", nn::nsd::ResultInternalServerError()},
            {"0503", nn::nsd::ResultUnderMaintenance()},
        };
        for(auto &e : errorMap)
        {
            if(nn::util::Strncmp(e.errorCode, errorCode, static_cast<int>(sizeof(e.errorCode))) == 0)
            {
                 NN_RESULT_THROW(e.result);
            }
        }
        NN_RESULT_THROW( nn::nsd::ResultUnknownServerError() );
    }

    bool IsInitialSaveData(const SaveData& saveData) NN_NOEXCEPT
    {
        return IsZero(&saveData, sizeof(SaveData));
    }

    void GetDefaultSettings(SaveData* pOut, const DeviceId& ownDeviceId, const EnvironmentIdentifier& env) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);

        std::memset(pOut, 0, sizeof(SaveData));

        pOut->version = nn::nsd::Version;
        pOut->deviceId = ownDeviceId;
        pOut->environmentIdentifier = env;
        pOut->backboneSettings = GetBuiltinBackboneSettings(pOut->environmentIdentifier);
        pOut->isAvailable = true;
        pOut->expireUnixTime = std::numeric_limits<int64_t>::max();

        NN_SDK_ASSERT( VerifySaveData(*pOut, ownDeviceId, config::IsDeviceIdCheckEnabled).IsSuccess() );
    }


    Result VerifySaveData(
        const SaveData& saveData,
        const DeviceId& ownDeviceId,
        bool isDeviceIdCheckEnabled) NN_NOEXCEPT
    {
        bool isEmpty = false;

        if(saveData.version == 0)
        {
            NN_DETAIL_NSD_WARN("[NSD][DETAIL] Error settings. Version is zero.\n");
            isEmpty = true;
        }

        if( IsZero(&saveData.backboneSettings, sizeof(saveData.backboneSettings)) )
        {
            NN_DETAIL_NSD_WARN("[NSD][DETAIL] Error settings. Backbone settings is empty.\n");
            isEmpty = true;
        }

        if( IsZero(&saveData.environmentIdentifier, sizeof(saveData.environmentIdentifier)) )
        {
            NN_DETAIL_NSD_WARN("[NSD][DETAIL] Error settings. Environment identifier is empty.\n");
            isEmpty = true;
        }

        NN_RESULT_THROW_UNLESS(!isEmpty, ResultEmptyValueExists());

        if(isDeviceIdCheckEnabled)
        {
            NN_RESULT_THROW_UNLESS(ownDeviceId == saveData.deviceId, ResultUnexpectedDeviceId());
        }

        // FQDN 不正文字の確認(backboneSettings)
        for(auto& entry : saveData.backboneSettings.fqdnEntries)
        {
            NN_RESULT_THROW_UNLESS(
                !detail::HasBadChar(entry.src) && !detail::HasBadChar(entry.dst),
                ResultInvalidFqdnChar());
        }

        // FQDN 不正文字の確認(applicationSettings)
        for(auto& entry : saveData.applicationSettings.fqdnEntries)
        {
            NN_RESULT_THROW_UNLESS(
                !detail::HasBadChar(entry.src) && !detail::HasBadChar(entry.dst),
                ResultInvalidFqdnChar());
        }

        NN_RESULT_SUCCESS;
    }

    Result GetNasRequestFqdn(Fqdn* pOut, const SaveData& saveData, const nn::time::PosixTime& currentTime) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);

        NN_RESULT_THROW_UNLESS( saveData.isAvailable, nn::nsd::ResultSettingExpired() ); // すでに時限失効検出済かチェック
        NN_RESULT_DO( CheckExpiration(saveData.expireUnixTime, currentTime) ); // 時限失効チェック
        NN_RESULT_THROW_UNLESS(saveData.version >= nn::nsd::Version, ResultInvalidSettingVersion());

        NN_RESULT_THROW_UNLESS(
            !util::IsEmptyString(saveData.backboneSettings.nasServiceSettings.nasRequestServiceFqdn),
            ResultNotFound());

        *pOut = saveData.backboneSettings.nasServiceSettings.nasRequestServiceFqdn;

        NN_RESULT_SUCCESS;
    }

    Result GetNasApiFqdn(Fqdn* pOut, const SaveData& saveData, const nn::time::PosixTime& currentTime) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);

        NN_RESULT_THROW_UNLESS( saveData.isAvailable, nn::nsd::ResultSettingExpired() ); // すでに時限失効検出済かチェック
        NN_RESULT_DO( CheckExpiration(saveData.expireUnixTime, currentTime) ); // 時限失効チェック
        NN_RESULT_THROW_UNLESS(saveData.version >= nn::nsd::Version, ResultInvalidSettingVersion());

        NN_RESULT_THROW_UNLESS(
            !util::IsEmptyString(saveData.backboneSettings.nasServiceSettings.nasApiServiceFqdn),
            ResultNotFound());

        *pOut = saveData.backboneSettings.nasServiceSettings.nasApiServiceFqdn;

        NN_RESULT_SUCCESS;
    }

    Result GetNasServiceSetting(
        NasServiceSetting* pOutNasServiceSetting,
        const NasServiceName& nasServiceName,
        const SaveData& saveData,
        const nn::time::PosixTime& currentTime) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutNasServiceSetting);

        NN_RESULT_THROW_UNLESS( saveData.isAvailable, nn::nsd::ResultSettingExpired() ); // すでに時限失効検出済かチェック
        NN_RESULT_DO( CheckExpiration(saveData.expireUnixTime, currentTime) ); // 時限失効チェック
        NN_RESULT_THROW_UNLESS(saveData.version >= nn::nsd::Version, ResultInvalidSettingVersion());

        bool found = false;
        for(auto &entry : saveData.backboneSettings.nasServiceSettings.entries)
        {
            if(nasServiceName == entry.nasServiceName)
            {
                *pOutNasServiceSetting = entry.nasServiceSetting;
                found = true;
            }
        }

        NN_RESULT_THROW_UNLESS(found, ResultNotFound());

        NN_RESULT_SUCCESS;
    }
}}} // nn::nsd::detail
