﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/ns/ns_DynamicRightsApi.h>
#include <nn/ns/ns_DynamicRightsSystemApi.h>
#include <nn/ns/ns_Result.h>

#include "ns_Initialize.h"

namespace nn { namespace ns {

    //--------------------------------------------------------------------------
    //  DynamicRightspi
    //--------------------------------------------------------------------------

    Result RequestApplicationRightsOnServer(
        AsyncApplicationRightsOnServerList* pOutValue,
        ncm::ApplicationId id,
        const account::Uid& uid) NN_NOEXCEPT
    {
        sf::NativeHandle nativeHandle;
        sf::SharedPointer<detail::IAsyncValue> sp;
        ApplicationRightsQueryFlags flags =
            ApplicationRightsQueryFlag_Application::Mask |
            ApplicationRightsQueryFlag_AddOnContent::Mask;
        NN_RESULT_DO(GetDynamicRightsInterface()->RequestApplicationRightsOnServer(&nativeHandle, &sp, id, uid, flags._storage[0]));
        pOutValue->Initialize(sp, nativeHandle);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }

    Result RequestApplicationRightsOnServer(
        AsyncApplicationRightsOnServerList* pOutValue,
        ncm::ApplicationId id) NN_NOEXCEPT
    {
        return RequestApplicationRightsOnServer(pOutValue, id, account::InvalidUid);
    }

    Result RequestAssignRights(
        AsyncResult* pOutValue,
        const ApplicationRightsOnServer* pList,
        int listCount) NN_NOEXCEPT
    {
        sf::NativeHandle nativeHandle;
        sf::SharedPointer<detail::IAsyncResult> sp;
        NN_RESULT_DO(GetDynamicRightsInterface()->RequestAssignRights(
            &nativeHandle, &sp, sf::InArray<ApplicationRightsOnServer>(pList, listCount)));
        pOutValue->Initialize(sp, nativeHandle);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }

    Result RequestAssignRightsToResume(
        AsyncResult* pOutValue,
        RightsEnvironmentHandle handle,
        const account::Uid& uid) NN_NOEXCEPT
    {
        sf::NativeHandle nativeHandle;
        sf::SharedPointer<detail::IAsyncResult> sp;
        NN_RESULT_DO(GetDynamicRightsInterface()->RequestAssignRightsToResume(
            &nativeHandle, &sp, handle, uid));
        pOutValue->Initialize(sp, nativeHandle);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }

    Result RegisterUserOfAccountRestrictedRights(const account::Uid& uid) NN_NOEXCEPT
    {
        return GetDynamicRightsInterface()->RegisterUserOfAccountRestrictedRights(uid);
    }

    void UnregisterAllUsersOfAccountRestrictiveRights() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetDynamicRightsInterface()->UnregisterAllUsersOfAccountRestrictedRights());
    }

    Result VerifyActivatedRightsOwners(RightsEnvironmentHandle handle) NN_NOEXCEPT
    {
        return GetDynamicRightsInterface()->VerifyActivatedRightsOwners(handle);
    }

    ApplicationRightsStatus GetApplicationRightsStatus(RightsEnvironmentHandle handle) NN_NOEXCEPT
    {
        ApplicationRightsStatus status;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetDynamicRightsInterface()->GetApplicationRightsStatus(&status, handle));
        return status;
    }

    //--------------------------------------------------------------------------
    //  ApplicationRightsOnServer
    //--------------------------------------------------------------------------

    bool ApplicationRightsOnServer::HasAvailableRights() const NN_NOEXCEPT
    {
        return _flags.Test<ApplicationRightsOnServerFlag_HasAvailableRights>();
    }

    bool ApplicationRightsOnServer::HasUnavailableRights() const NN_NOEXCEPT
    {
        return _flags.Test<ApplicationRightsOnServerFlag_HasUnavailableRights>();
    }

    bool ApplicationRightsOnServer::IsAccountRestrictedRights() const NN_NOEXCEPT
    {
        return _flags.Test<ApplicationRightsOnServerFlag_IsAccountRestrictedRights>();
    }

    bool ApplicationRightsOnServer::RecommendSyncTicket() const NN_NOEXCEPT
    {
        return _flags.Test<ApplicationRightsOnServerFlag_RecommendSyncTicket>();
    }

    bool ApplicationRightsOnServer::RecommendAssignRights() const NN_NOEXCEPT
    {
        return _flags.Test<ApplicationRightsOnServerFlag_RecommendAssignRights>();
    }

    //--------------------------------------------------------------------------
    //  ApplicationRightsUnavailableReason
    //--------------------------------------------------------------------------

    bool ApplicationRightsUnavailableReason::NoRights() const NN_NOEXCEPT
    {
        return _flags.Test<ApplicationRightsUnavailableFlag_NoRights>();
    }

    bool ApplicationRightsUnavailableReason::SystemUpdateRequired() const NN_NOEXCEPT
    {
        return _flags.Test<ApplicationRightsUnavailableFlag_SystemUpdateRequired>();
    }

    bool ApplicationRightsUnavailableReason::HasDeviceLinkedRightsOnlyContent() const NN_NOEXCEPT
    {
        return _flags.Test<ApplicationRightsUnavailableFlag_HasDeviceLinkedRightsOnlyContent>();
    }

    bool ApplicationRightsUnavailableReason::NotReleased() const NN_NOEXCEPT
    {
        return _flags.Test<ApplicationRightsUnavailableFlag_NotReleased>();
    }

    bool ApplicationRightsUnavailableReason::AssignableRightsLimitExceeded() const NN_NOEXCEPT
    {
        return _flags.Test<ApplicationRightsUnavailableFlag_AssignableRightsLimitExceeded>();
    }
}}
