﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <climits>

#include <nn/account/account_ApiForSystemServices.h>
#include <nn/dauth/dauth_Api.h>
#include <nn/os.h>

#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>

#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>

#include "nim_HeapUtil.h"
#include "nim_TokenStore.h"

//-----------------------------------------------------------------------------
namespace {

// デバッグコード
#if !defined(NN_SDK_BUILD_RELEASE)
#define DEBUG_TRACE(...)    NN_DETAIL_NIM_TRACE( "[TokenStore] " __VA_ARGS__ )
#else
#define DEBUG_TRACE(...)    static_cast<void>(0)
#endif

}   // ~unnamed
//-----------------------------------------------------------------------------
namespace nn { namespace nim { namespace srv {

namespace TokenStore {

//-----------------------------------------------------------------------------
namespace {

/**
 * @brief       システム TICK UP TIME の取得。
 */
NN_FORCEINLINE::nn::TimeSpan GetUptime() NN_NOEXCEPT
{
    return ::nn::os::GetSystemTick().ToTimeSpan();
}

}   // ~unnamed

//!============================================================================
//! @name TokenStore::SharedStore 実装

//-----------------------------------------------------------------------------
SharedStore* SharedStore::GetSharedInstance() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(SharedStore, s_SharedStore);
    return &s_SharedStore;
}

//!============================================================================
//! @name TokenStore::NsaId 実装

//-----------------------------------------------------------------------------
Result NsaId::AcquireToken(char* pOutToken, const size_t availableCapacity, const ::nn::account::Uid& uid, const ::nn::account::SystemProgramIdentification& info, ICanceler* const pCanceler) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutToken);
    NN_SDK_REQUIRES_NOT_NULL(pCanceler);
    NN_SDK_REQUIRES(availableCapacity >= LengthMax);

    ::nn::account::NetworkServiceAccountManager nsaManager;
    NN_RESULT_DO(::nn::account::GetNetworkServiceAccountManager(&nsaManager, uid));
    NN_RESULT_DO(nsaManager.SetSystemProgramIdentification(info));

    NN_UTIL_SCOPE_EXIT{pCanceler->Invalidate();};
    const auto pHandle = pCanceler->Emplace();
    NN_RESULT_DO(nsaManager.EnsureNetworkServiceAccountIdTokenCacheAsync(pHandle));
    ::nn::os::SystemEvent done;
    NN_RESULT_DO(pHandle->GetSystemEvent(&done));
    done.Wait();
    NN_RESULT_DO(pHandle->GetResult());

    size_t nsaTokenSize;
    NN_RESULT_DO(nsaManager.LoadNetworkServiceAccountIdTokenCache(&nsaTokenSize, pOutToken, availableCapacity));
    NN_SDK_ASSERT_EQUAL('\0', pOutToken[nsaTokenSize]); // LoadNetworkServiceAccountIdTokenCache() での保証要件の確認( Release では除去 )。
    NN_RESULT_SUCCESS;
}

//!============================================================================
//! @name TokenStore::NaId::SingleCache 実装

//-----------------------------------------------------------------------------
NaId::SingleCacheBase::SingleCacheBase() NN_NOEXCEPT
    : m_Lock()
    , m_Async()
    , m_Owner(::nn::account::InvalidUid)
    , m_Expired(0)
    , m_AvailabilityPeriod(::nn::TimeSpan::FromHours(1))
    , m_CachedLength(0)
    , m_Token("")
{
}

//-----------------------------------------------------------------------------
NaId::SingleCacheBase::SingleCacheBase(const ::nn::TimeSpan& cachedTokenAvailabilityPeriod) NN_NOEXCEPT
    : m_Lock()
    , m_Async()
    , m_Owner(::nn::account::InvalidUid)
    , m_Expired(0)
    , m_AvailabilityPeriod(cachedTokenAvailabilityPeriod)
    , m_CachedLength(0)
    , m_Token("")
{
}

//-----------------------------------------------------------------------------
NaId::SingleCacheBase::~SingleCacheBase() NN_NOEXCEPT
{
    Invalidate();
}

//-----------------------------------------------------------------------------
void NaId::SingleCacheBase::InvalidateImpl() NN_NOEXCEPT
{
    m_Token[0] = '\0';
    m_CachedLength = 0;
    m_Expired = ::nn::TimeSpan::FromNanoSeconds(0);
    m_Owner.Invalidate();
}

//-----------------------------------------------------------------------------
void NaId::SingleCacheBase::Invalidate() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    InvalidateImpl();
}

//-----------------------------------------------------------------------------
Result NaId::SingleCacheBase::AcquireToken(char* pOutToken, const size_t availableCapacity, const ::nn::account::Uid& uid, ICanceler* const pCanceler) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutToken);
    NN_SDK_REQUIRES_NOT_NULL(pCanceler);
    NN_SDK_REQUIRES(availableCapacity >= sizeof(m_Token));
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);

    OwnerUserAccount newOwner(uid);
    ::nn::account::NetworkServiceAccountManager nsaManager;
    NN_RESULT_DO(::nn::account::GetNetworkServiceAccountManager(&nsaManager, uid));
    NN_RESULT_DO(nsaManager.GetNetworkServiceAccountId(&newOwner.nsa));

    // 失効猶予期間に突入していたら再取得。
    if (m_Owner != newOwner || IsExpired())
    {
        InvalidateImpl();

        // 取得スコープ中のみの一時ヒープ利用( nn::account::RequiredBufferSizeForNintendoAccountAuthorizationRequestContext )。
        HeapUtil::OnetimeHeapSession session;
        NN_RESULT_DO(session.Acquire(::nn::os::MemoryPageSize, ::nn::account::RequiredBufferSizeForNintendoAccountAuthorizationRequestContext));

        // トークン取得要求用パラメータ取得。
        AuthParameter params;
        ::nn::nsd::NasServiceSetting shopSetting;
        NN_RESULT_DO(PrepareAuthorizationRequestSettings(&shopSetting, &params));

        NN_UTIL_SCOPE_EXIT{m_Async = ::nn::util::nullopt;};
        m_Async.emplace();
        auto& handle = *m_Async;
        NN_RESULT_DO(nsaManager.CreateNintendoAccountAuthorizationRequest(&handle
            , shopSetting.clientId
            , shopSetting.redirectUri.value
            , ::nn::nsd::NasServiceSetting::RedirectUri::Size
            , params
            , session.pTop
            , session.size
        ));
        {
            NN_UTIL_SCOPE_EXIT{pCanceler->Invalidate();};
            pCanceler->store(&handle, std::memory_order_release);
            ::nn::os::SystemEvent done;
            NN_RESULT_DO(handle.GetSystemEvent(&done));
            done.Wait();
            NN_RESULT_DO(handle.GetRawResult());
        }
        size_t tokenSize;
        NN_RESULT_DO(handle.GetIdToken(&tokenSize, m_Token, sizeof(m_Token)));
        NN_SDK_ASSERT_EQUAL('\0', m_Token[tokenSize]);    // GetIdToken() での保証要件の確認( Release では除去 )。

        m_CachedLength = static_cast<Bit32>(tokenSize);
        m_Expired = GetUptime() + m_AvailabilityPeriod;
        m_Owner = newOwner;

        DEBUG_TRACE("NA ID Token refreshed( %zu bytes ).\n", tokenSize);
    }

    // availableCapacity > strlen(m_Token) なら、null終端付きで全てコピー。
    // そうでなければ availableCapacity - 1 までコピーして null 終端。
    const int limit = (availableCapacity > INT_MAX) ? INT_MAX : static_cast<int>(availableCapacity);
    ::nn::util::Strlcpy(pOutToken, m_Token, limit);
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
bool NaId::SingleCacheBase::IsExpired() const NN_NOEXCEPT
{
    const auto expiration = m_Expired;
    return (expiration <= ::nn::TimeSpan::FromNanoSeconds(0)) || (expiration < GetUptime());
}



//!============================================================================
//! @name デバイス認証トークン管理

//-----------------------------------------------------------------------------
Result DeviceAuth::AcquireToken(ICanceler* const pCanceler, const uint64_t clientId, char* pOutToken, const size_t availableCapacity, char** pOutTokenEnd) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutToken);
    NN_SDK_REQUIRES_NOT_NULL(pCanceler);
    NN_SDK_REQUIRES(availableCapacity >= LengthMax);

    int cachedLength;       //!< トークン長( 終端文字を含めない )
    ::nn::TimeSpan expired; //!< Uptime 失効期限
    NN_UTIL_SCOPE_EXIT{pCanceler->Invalidate();};
    const auto pCancelImpl = pCanceler->Emplace();
    NN_RESULT_DO(::nn::dauth::AcquireDeviceAuthenticationToken(
        &expired
        , &cachedLength
        , pOutToken
        , availableCapacity
        , clientId
        , false
        , pCancelImpl
    ));
    if (pOutTokenEnd)
    {
        *pOutTokenEnd = &pOutToken[cachedLength];
    }
    NN_RESULT_SUCCESS;
}

}   // ~TokenStore

}}}
