﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <mutex>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Mount.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/srv/nim_SimpleDataStore.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace nim { namespace srv {
    SimpleDataStoreImpl::~SimpleDataStoreImpl() NN_NOEXCEPT
    {
        if (m_MountName)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(*m_MountName));
            fs::Unmount(*m_MountName);
        }
    }

    Result SimpleDataStoreImpl::Initialize(const char* mountName,
        fs::SystemSaveDataId id, int64_t saveDataSize, int64_t saveDataJournalSize, int saveDataFlags,
        const void* initialData, size_t dataSize) NN_NOEXCEPT
    {
        nn::fs::DisableAutoSaveDataCreation();
        NN_RESULT_TRY(fs::MountSystemSaveData(mountName, id))
            NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
            {
                NN_RESULT_DO(nn::fs::CreateSystemSaveData(id, saveDataSize, saveDataJournalSize, saveDataFlags));
                NN_RESULT_DO(nn::fs::MountSystemSaveData(mountName, id));
            }
        NN_RESULT_END_TRY;

        m_MountName.emplace();
        m_MountName->Assign(mountName);

        NN_RESULT_TRY(fs::CreateFile(MakeDataFilePath(), dataSize))
            NN_RESULT_CATCH(fs::ResultPathAlreadyExists)
            {
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        NN_RESULT_DO(Save(initialData, dataSize));
        NN_RESULT_SUCCESS;
    }

    SimpleDataStoreImpl::Path SimpleDataStoreImpl::MakeDataFilePath() NN_NOEXCEPT
    {
        Path path;
        path.AssignFormat("%s:/%s", m_MountName->Get(), "data.bin");
        return path;
    }

    Result SimpleDataStoreImpl::Save(const void* data, size_t size) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_Mutex);

        {
            fs::FileHandle file;
            NN_RESULT_DO(fs::OpenFile(&file, MakeDataFilePath(), fs::OpenMode_Write | fs::OpenMode_AllowAppend));
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

            NN_RESULT_DO(fs::WriteFile(file, 0, data, size, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
        }
        NN_RESULT_DO(fs::CommitSaveData(*m_MountName));

        NN_RESULT_SUCCESS;
    }

    Result SimpleDataStoreImpl::Load(void* data, size_t size) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_Mutex);

        fs::FileHandle file;
        NN_RESULT_TRY(fs::OpenFile(&file, MakeDataFilePath(), fs::OpenMode_Read))
            NN_RESULT_CATCH(fs::ResultPathNotFound)
            {
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        NN_RESULT_DO(fs::ReadFile(file, 0, data, size));
        NN_RESULT_SUCCESS;
    }

}}}
