﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>
#include <utility>

#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_Utility.h>
#include <nn/fs/fs_SystemSaveData.h>

#include <nn/util/util_TinyMt.h>
#include <nn/util/util_StringUtil.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include <nn/nim/nim_Result.h>
#include "nim_ShopServiceAccessDebug.h"
#include "nim_ShopServiceAccessPrivate.h"

namespace nn { namespace nim { namespace srv {

namespace {

// デバッグコード
#if !defined(NN_SDK_BUILD_RELEASE) && defined(ENABLE_DEBUG_TRACE)
#define DEBUG_TRACE(...)                        NN_DETAIL_NIM_TRACE( "[ShopServiceAccess::Debug] " __VA_ARGS__ )
#define DEBUG_TRACE_FOR_BOUNDARY_STRING(...)    LogOutForBoundaryString( __VA_ARGS__ )

// @note    IPCの引数メモリ領域( InArray<char> などで受け取る )は書き込み禁止なので使えない点に注意。
void LogOutForBoundaryString(const char* pPrefix, const char* pBuffer, const size_t bufferSize) NN_NOEXCEPT
{
    auto pRear = const_cast<char*>(&pBuffer[bufferSize - 1]);
    const auto rearCode = *pRear;
    *pRear = '\0';
    DEBUG_TRACE("%s(%zu) => `%s%c`\n", pPrefix, bufferSize, pBuffer, rearCode);
    *pRear = rearCode;
}
#else
#define DEBUG_TRACE(...)                        static_cast<void>(0)
#define DEBUG_TRACE_FOR_BOUNDARY_STRING(...)    static_cast<void>(0)
#endif

/**
 * @brief   ファイルシステムライブラリ( fs )定数要件。
 * @note    <https://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=107336120>
 */
namespace Requirement
{
    namespace FileSystem
    {
        /**
         * @brief   レコードのための最小エントリサイズ( byte )。
         * @details セーブデータ1つに対して最低限 32 KiB 必要。( フォルダ1個に 16KiB, ファイル1個に 16KiB )@n
         *          以降 170 エントリ( ファイル / ディレクトリ )ごとに 16KiB 増加。
         */
        constexpr uint32_t EntrySize = 16 * 1024;

        /**
         * @brief   システムジャーナルサイズ定数。
         * @details ファイル生成時 = ファイル生成に 16 * 2 KiB, ファイル更新に 16 * 2 KiB。
         *          ファイル更新時 = ファイル更新に 16 * 2 KiB。
         */
        namespace JournalSize
        {
            constexpr uint32_t FileCreateAtOnce = EntrySize * 4;   // ユーザデータ更新を含まない、1ファイル生成時のコミットで生じる最小ジャーナルサイズ( byte )。
            //const uint32_t FileUpdateAtOnce = EntrySize * 2;   // ユーザデータ更新を含まない、1ファイル更新時のコミットで生じる最小ジャーナルサイズ( byte )。
        };
    };

    /**
     * @brief   デバッグレスポンス登録ファイル用構成定数。
     *
     * @details ResponseStore のファイルのみのセーブデータ構成は以下設計です。@n
     *          想定: R/W用ジャーナルをファイルサイズと同じとして、ジャーナル含めて 512KiB 利用になるようにする。@n
     *              @li ActualJournalSize = (FileCreateAtOnce + JournalSize);
     *              @li SaveData = 512 - (EntrySize * 2 + ActualJournalSize);
     *
     * @note    対象セーブデータに ResponseStore 以外のファイルを生成する場合は適宜調整してください。
     */
    namespace ResponseStore
    {
        //! @brief  利用想定セーブデータ総容量。
        //! @note   ResponseStore ファイルのみのセーブデータのみで有効です。
        constexpr uint32_t   ExpectTotalSaveDataStorageUseSize = 512 * 1024;

        //! @brief  利用想定セーブデータ内ファイルサイズ。 ( 512 KiB セーブデータを目標指標とする場合、208 KiB )
        //! @note   対象セーブデータに ResponseStore 以外のファイルを生成する場合は、@ref ExpectTotalSaveDataStorageUseSize を利用せずに適宜調整値を設定してください。
        constexpr uint32_t   FileSize = (ExpectTotalSaveDataStorageUseSize - (FileSystem::EntrySize * 2 + FileSystem::JournalSize::FileCreateAtOnce)) / 2;

        //! @brief  R/W ジャーナル単位( byte )。
        constexpr uint32_t   JournalSize = FileSize;
    };

    /**
     * @brief   マウントプロパティ構成定数。
     */
    namespace Mount
    {
        //! @brief  マウントボリューム名。
        //! @note   要確認.
        constexpr char Volume[] = "nim_eca_dbg";

        //! @brief  セーブデータID。
        //! @note   要確認.
        constexpr fs::SystemSaveDataId SaveDataId = 0x8000000000000078llu;

        //! @brief  セーブデータフラグ。
        constexpr fs::SaveDataFlags SaveDataFlags = static_cast<fs::SaveDataFlags>(0);

        //! @brief  割り当てセーブデータ総容量( byte )。
        constexpr uint32_t SaveDataSize
            = Requirement::FileSystem::EntrySize                                        // ルートエントリ
            + Requirement::FileSystem::EntrySize + Requirement::ResponseStore::FileSize // ResponseStore ファイルエントリ + 実体
            ;

        //! @brief  セーブデータジャーナルサイズ( byte )。
        constexpr uint32_t SaveDataJournalSize
            = Requirement::FileSystem::JournalSize::FileCreateAtOnce    // 1ファイル生成・更新
            + Requirement::ResponseStore::JournalSize                   // ResponseStore ファイル更新領域
            ;
    };
};

/**
 * @brief   デバッグシステム用共有セーブデータマウント管理クラス。
 *
 * @details 参照カウンタによるネスト管理を行っています。@n
 *          参照カウンタがゼロ以下の状態は「アンマウント中」です。@n
 *          参照カウンタの初期化値は 0 とし、@ref Unmount() で -1, @ref Mount() で +1 されます。@n
 *          これは初期状態が「アンマウント」から始まる事を想定しています。@n
 *          また、ゼロ状態(アンマウント状態)からの「アンマウント」要求はネストされません。
 */
class MountHandle
{
    NN_DISALLOW_COPY(MountHandle);
    NN_DISALLOW_MOVE(MountHandle);

private:
    //! @brief  参照カウンタ型。
    typedef int32_t ReferenceCounter;

    //! @brief  参照カウンタ初期値。
    static constexpr ReferenceCounter InitialReferenceCount = 0;

    /**
     * @brief       セーブデータマウント要求。
     *
     * @param[in]   allowCreateSaveData     true の場合、セーブデータが存在しない場合、セーブデータを作成してマウントします。@n
     *                                      false の場合、セーブデータが既存する時のみマウントします。
     *
     * @retval  ::nn::fs::ResultTargetNotFound  マウント対象セーブデータが見つからない。( @a allowCreateSaveData が false 指定時のみ )
     */
    Result MountImpl(bool allowCreateSaveData) NN_NOEXCEPT;

    /**
     * @brief       セーブデータアンマウント要求。
     */
    void UnmountImpl(bool ignoreNestCondition = false) NN_NOEXCEPT;

    /**
     * @brief       マウント中かどうか確認します。
     */
    bool IsMountedUnsafe() const NN_NOEXCEPT;

public:
    /**
     * @brief       マウント状態のスコープセッション管理クラス。
     */
    class EnsureSession
    {
    private:
        NN_DISALLOW_COPY(EnsureSession);
        friend class MountHandle;
        MountHandle* m_pMount;

    public:
        EnsureSession() NN_NOEXCEPT : m_pMount(nullptr) {}
        ~EnsureSession() NN_NOEXCEPT;

        EnsureSession(EnsureSession&& rvalue) NN_NOEXCEPT
        {
            m_pMount = rvalue.m_pMount;
            rvalue.m_pMount = nullptr;
        }

        EnsureSession& operator=(EnsureSession&& rvalue) NN_NOEXCEPT
        {
            EnsureSession r(std::move(rvalue));
            std::swap(r.m_pMount, m_pMount);
            return *this;
        }
    };

    /**
     * @brief       デフォルトコンストラクタ。
     */
    explicit MountHandle(ReferenceCounter initialReferenceCount = InitialReferenceCount) NN_NOEXCEPT;

    /**
     * @brief       マウントスキームに従ったファイルパスを生成します。
     *
     * @param[out]  pOutValue   生成されたファイルパス文字列の受け取る格納コンテナの先頭アドレス。
     * @param[in]   capacity    生成されたファイルパス文字列を受け取る格納コンテナの受信可能容量( byte数 )。
     * @param[in]   pFilePath   マウントスキームからの相対ファイルパス。
     *
     * @return  生成されたファイルパス文字列の先頭アドレス。( 引数の @ref pOutValue と同じです )
     */
    static const char* MakeFilePath(char* pOutValue, const int capacity, const char* pFilePath) NN_NOEXCEPT;

    /**
     * @brief       セーブデータマウントセッション要求。( セーブデータがない場合は自動作成 )
     *
     * @param[out]  pOutSession     マウント成功時にマウントハンドルを格納したスコープセッションを返します。
     */
    Result MountAlongWithCreate(EnsureSession* pOutSession) NN_NOEXCEPT
    {
        return MountImplForSession(pOutSession, true);
    }

    /**
     * @brief       セーブデータマウントセッション要求。( セーブデータがない場合はエラー返却 )
     *
     * @retval  ::nn::fs::ResultTargetNotFound  マウント対象セーブデータが見つからない。
     *
     * @param[out]  pOutSession     マウント成功時にマウントハンドルを格納したスコープセッションを返します。
     */
    Result Mount(EnsureSession* pOutSession) NN_NOEXCEPT
    {
        return MountImplForSession(pOutSession, false);
    }

    /**
     * @brief       セーブデータコミット要求。
     */
    Result Commit() NN_NOEXCEPT;

private:
    /**
     * @brief       セーブデータマウントセッション要求。
     *
     * @param[out]  pOutSession             マウント成功時にマウントハンドルを格納したスコープセッションを返します。
     * @param[in]   allowCreateSaveData     true の場合、セーブデータが存在しない場合、セーブデータを作成してマウントします。@n
     *                                      false の場合、セーブデータが既存する時のみマウントします。
     *
     * @retval  ::nn::fs::ResultTargetNotFound  マウント対象セーブデータが見つからない。( @a allowCreateSaveData が false 指定時のみ )
     */
    Result MountImplForSession(EnsureSession* pOutSession, bool allowCreateSaveData) NN_NOEXCEPT;

private:
    typedef ::nn::os::SdkRecursiveMutex MutexType;

    mutable MutexType   m_Lock;
    ReferenceCounter    m_EnsureCount;  //!< マウント/アンマウント要求カウント状態
};

/**
 * @brief       ファイルハンドルスコープセッション管理クラス。
 */
class ScopedFileHandle
{
    NN_DISALLOW_COPY(ScopedFileHandle);

private:
    typedef ::nn::util::optional<::nn::fs::FileHandle> FileHandle;
    FileHandle  m_Handle;

    void Close() NN_NOEXCEPT
    {
        if (m_Handle)
        {
            ::nn::fs::CloseFile(*m_Handle);
            m_Handle = ::nn::util::nullopt;
        }
    }

public:
    ScopedFileHandle() NN_NOEXCEPT : m_Handle() {}

    ~ScopedFileHandle() NN_NOEXCEPT
    {
        Close();
    }

    ScopedFileHandle(ScopedFileHandle&& rvalue) NN_NOEXCEPT
    {
        m_Handle = rvalue.m_Handle;
        rvalue.m_Handle = ::nn::util::nullopt;
    }

    ScopedFileHandle& operator=(ScopedFileHandle&& rvalue) NN_NOEXCEPT
    {
        ScopedFileHandle r(std::move(rvalue));
        std::swap(r.m_Handle, m_Handle);
        return *this;
    }

    Result Open(const char* pPath, int mode) NN_NOEXCEPT
    {
        Close();
        ::nn::fs::FileHandle handle;
        NN_RESULT_DO(::nn::fs::OpenFile(&handle, pPath, mode));
        m_Handle = handle;
        NN_RESULT_SUCCESS;
    }

    NN_FORCEINLINE NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return static_cast<bool>(m_Handle);
    }

    NN_FORCEINLINE const ::nn::fs::FileHandle& Get() const NN_NOEXCEPT
    {
        return *m_Handle;
    }
};

/**
 * @brief   デバッグレスポンス登録ストア管理クラス。
 */
class ResponseStore
{
    NN_DISALLOW_COPY(ResponseStore);
    NN_DISALLOW_MOVE(ResponseStore);

public:
    typedef ::nn::nim::srv::ShopServiceAccessDebug::Response::EmulationResult   ResultType;
    typedef ::nn::nim::ShopServiceAccessTypes::Server                           ServerType;

private:
    //! @brief  内部作業用ヒープ領域確保上限サイズ( バイト単位 )。
    static const size_t     ExpectLimitHeapWorkSize = 4 * 1024;

    //! @brief  デバッグレスポンス照合一致発生率( 分率 )。
    static const uint16_t   HappenedRateRange = 10000u;

    //! @brief  ルートからの相対ファイルパス。
    static const char       RelativeFilePath[];

    //! @brief  ファイルヘッダ
    struct FileHeader
    {
        Bit16   registered;     // 登録済キー総数。
        Bit16   keyLengthMax;   // 登録済キー中の最大長。

        FileHeader() NN_NOEXCEPT;

        void Reset() NN_NOEXCEPT;

        FileHeader MakeNewRegistered(const Bit16 keyLength) const NN_NOEXCEPT;
    };

    //! @brief  登録レスポンスデータ登録ブロックヘッダ
    struct ItemHeader
    {
        Bit32   offsetForNextData;      // アイテムヘッダからの相対 ( 4byte aligned )
        Bit32   expectResult;           // 発生時 Result
        Bit16   happenedRate;           // 発生率( @ref HappenedRateRange 分率 )
        Bit16   keyLength;              // キー要素長( バイト単位 )
        Bit32   valueLength;            // 値要素長( バイト単位 )
        Bit8    targetServer;           // 接続先サーバー( @ref ShopServiceAccessTypes::Server )
        Bit8    padding[3];
    };

    //! @brief  内部作業用
    struct FileTop
    {
        FileHeader  header;
        ItemHeader  item;
    };

    NN_STATIC_ASSERT(Requirement::ResponseStore::FileSize > (sizeof(FileHeader) + sizeof(ItemHeader)));
    NN_STATIC_ASSERT(sizeof(ShopServiceAccessTypes::Server) == sizeof(ItemHeader::targetServer));
    NN_STATIC_ASSERT(0 == sizeof(ItemHeader) % sizeof(Bit32));
    NN_STATIC_ASSERT(0 == sizeof(FileHeader) % sizeof(Bit32));

    static Result FindRearPositionOnFile(int64_t* pOutPosition, const ::nn::fs::FileHandle& handle, FileHeader* pOutHeader = nullptr) NN_NOEXCEPT;

    Result WriteTo(const ::nn::fs::FileHandle& handle, const int64_t position, const ServerType& target, const ::nn::sf::InArray<char>& key, const ::nn::sf::InArray<char>& value, const uint32_t expectResult, const uint32_t happenedRate) NN_NOEXCEPT;

public:
    ResponseStore() NN_NOEXCEPT;

    Result Clear() NN_NOEXCEPT;

    Result Register(const ServerType& keyTarget, const ::nn::sf::InArray<char>& keyPath, const ::nn::sf::InArray<char>& valueResponse, const uint32_t expectResult, const uint32_t happenedRate) NN_NOEXCEPT;

    Result DoUseDebugResponse(ResultType* pOutResult, const char* pVerifyPath, const ServerType& verifyServer) NN_NOEXCEPT;

private:
    FileHeader      m_FileHeader;
    char            m_FilePath[47];
    bool            m_ShadowValidity;   //!< オンメモリシャドウの可用状態
};

//-----------------------------------------------------------------------------
const char ResponseStore::RelativeFilePath[] = "ResponseStore.dat";
//-----------------------------------------------------------------------------

/**
 * @brief   デバッグシステムコンテキスト
 */
struct DebugContext
{
    NN_DISALLOW_COPY(DebugContext);
    NN_DISALLOW_MOVE(DebugContext);

public:
    static MountHandle* GetMountHandle() NN_NOEXCEPT
    {
        return &GetInstance()->m_MountHandle;
    }

    static ResponseStore* GetResponseStore() NN_NOEXCEPT
    {
        return &GetInstance()->m_ResponseStore;
    }

    static ::nn::util::TinyMt* GetRandomizer() NN_NOEXCEPT
    {
        return &GetInstance()->m_Randomizer;
    }

    static Result IsAvailable() NN_NOEXCEPT
    {
        return GetInstance()->IsAvailableImpl();
    }

    static void RefreshAvailability() NN_NOEXCEPT
    {
        GetInstance()->m_FwdbgLoaded = false;
    }

    static bool IsDragonsEmulationEnabled() NN_NOEXCEPT
    {
        return GetInstance()->m_IsEnableDragons;
    }

private:
    DebugContext() NN_NOEXCEPT;
    Result IsAvailableImpl() NN_NOEXCEPT;

    static DebugContext* GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(DebugContext, s_DebugContext);
        return &s_DebugContext;
    }

    static bool GetFirmwareDebugSettings(const char* pKey) NN_NOEXCEPT
    {
        bool enabled = false;
        const auto size = ::nn::settings::fwdbg::GetSettingsItemValue(&enabled, sizeof(enabled), "nim.errorsimulate", pKey);
        return (size == sizeof(enabled) && enabled);
    }

    ResponseStore       m_ResponseStore;
    MountHandle         m_MountHandle;
    ::nn::util::TinyMt  m_Randomizer;
    bool                m_FwdbgLoaded;
    bool                m_IsAvailable;
    bool                m_IsEnableDragons;
};

//-----------------------------------------------------------------------------
DebugContext::DebugContext() NN_NOEXCEPT
    : m_FwdbgLoaded(false)
    , m_IsAvailable(false)
    , m_IsEnableDragons(false)
{
}

//-----------------------------------------------------------------------------
Result DebugContext::IsAvailableImpl() NN_NOEXCEPT
{
    if (!m_FwdbgLoaded)
    {
        m_FwdbgLoaded = true;
        m_IsAvailable = GetFirmwareDebugSettings("enable_error_simulate");
        m_IsEnableDragons = GetFirmwareDebugSettings("enable_simulate_dynamic_rights");
        if (m_IsAvailable)
        {
            m_Randomizer.Initialize(static_cast<Bit32>(::nn::os::GetSystemTick().GetInt64Value()));
        }
    }
    NN_RESULT_THROW_UNLESS(m_IsAvailable, ResultNotSupported());
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
MountHandle::MountHandle(ReferenceCounter initialReferenceCount) NN_NOEXCEPT
    : m_EnsureCount(initialReferenceCount)
{
}

//-----------------------------------------------------------------------------
const char* MountHandle::MakeFilePath(char* pOutValue, const int capacity, const char* pFilePath) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pFilePath);

    auto tail = ::nn::util::Strlcpy(pOutValue, Requirement::Mount::Volume, capacity);
    NN_SDK_ASSERT(tail < capacity);
    tail += ::nn::util::Strlcpy(&pOutValue[tail], ":/", capacity - tail);
    NN_SDK_ASSERT(tail < capacity);
    tail += ::nn::util::Strlcpy(&pOutValue[tail], pFilePath, capacity - tail);
    NN_SDK_ASSERT(tail < capacity);
    return pOutValue;
}

//-----------------------------------------------------------------------------
NN_FORCEINLINE bool MountHandle::IsMountedUnsafe() const NN_NOEXCEPT
{
    return (m_EnsureCount > 0);
}

//-----------------------------------------------------------------------------
Result MountHandle::MountImpl(bool allowCreateSaveData) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);

    // 参照カウンタチェック。
    const auto ensureCount = m_EnsureCount + 1;
    if (1 == ensureCount)
    {
        NN_RESULT_TRY(::nn::fs::MountSystemSaveData(Requirement::Mount::Volume, Requirement::Mount::SaveDataId))
            NN_RESULT_CATCH(::nn::fs::ResultTargetNotFound)
            {
                if (!allowCreateSaveData)
                {
                    NN_RESULT_RETHROW;
                }
                NN_RESULT_DO(::nn::fs::CreateSystemSaveData(Requirement::Mount::SaveDataId, Requirement::Mount::SaveDataSize, Requirement::Mount::SaveDataJournalSize, Requirement::Mount::SaveDataFlags));
                NN_RESULT_DO(::nn::fs::MountSystemSaveData(Requirement::Mount::Volume, Requirement::Mount::SaveDataId));
            }
            NN_RESULT_CATCH(::nn::fs::ResultMountNameAlreadyExists)
            {
                // 既にマウント済ならスルー
            }
        NN_RESULT_END_TRY;
    }
    m_EnsureCount = ensureCount;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
void MountHandle::UnmountImpl(bool ignoreNestCondition) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);

    // 参照カウンタチェック。
    const auto ensureCount = m_EnsureCount - 1;
    if (ignoreNestCondition || 0 == ensureCount)
    {
        // このスコープの時点でアンマウント状態への移行は確定。
        m_EnsureCount = 0;

        // セーブデータ有無・マウント中確認。
        const auto result = ::nn::fs::MountSystemSaveData(Requirement::Mount::Volume, Requirement::Mount::SaveDataId);
        if (::nn::fs::ResultTargetNotFound::Includes(result))
        {
            // なければマウントしてないので成功。
            return;
        }
        // その他エラーはマウントされている想定でアンマウント実施。
        ::nn::fs::Unmount(Requirement::Mount::Volume);
    }
    else if (0 < ensureCount)
    {
        m_EnsureCount = ensureCount;
    }
}

//-----------------------------------------------------------------------------
Result MountHandle::Commit() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    if (IsMountedUnsafe())
    {
        return ::nn::fs::CommitSaveData(Requirement::Mount::Volume);
    }
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result MountHandle::MountImplForSession(EnsureSession* pOutSession, bool allowCreateSaveData) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutSession);
    const auto result = MountImpl(allowCreateSaveData);
    if (result.IsSuccess())
    {
        pOutSession->m_pMount = this;
    }
    return result;
}

//-----------------------------------------------------------------------------
MountHandle::EnsureSession::~EnsureSession() NN_NOEXCEPT
{
    auto pMount = m_pMount;
    if (nullptr != pMount)
    {
        m_pMount = nullptr;
        pMount->UnmountImpl();
    }
}



//-----------------------------------------------------------------------------
ResponseStore::FileHeader::FileHeader() NN_NOEXCEPT : registered(0), keyLengthMax(0)
{
}

//-----------------------------------------------------------------------------
void ResponseStore::FileHeader::Reset() NN_NOEXCEPT
{
    registered = 0;
    keyLengthMax = 0;
}

//-----------------------------------------------------------------------------
ResponseStore::FileHeader ResponseStore::FileHeader::MakeNewRegistered(const Bit16 keyLength) const NN_NOEXCEPT
{
    auto t = *this;
    ++(t.registered);
    if (keyLength > t.keyLengthMax)
    {
        t.keyLengthMax = keyLength;
    }
    return t;
}

//-----------------------------------------------------------------------------
ResponseStore::ResponseStore() NN_NOEXCEPT
    : m_ShadowValidity(false)
{
    m_FileHeader.Reset();
    MountHandle::MakeFilePath(m_FilePath, sizeof(m_FilePath), RelativeFilePath);
}

//-----------------------------------------------------------------------------
Result ResponseStore::Clear() NN_NOEXCEPT
{
    const auto pMount = DebugContext::GetMountHandle();
    MountHandle::EnsureSession session;
    NN_RESULT_TRY(pMount->Mount(&session))
        NN_RESULT_CATCH(::nn::fs::ResultTargetNotFound)
        {
            // 対象セーブデータが存在しないので成功として返却。
            m_FileHeader.Reset();
            m_ShadowValidity = true;
            NN_RESULT_MARK_AS_RETHROW;
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    const auto result = ::nn::fs::DeleteFile(m_FilePath);
    NN_RESULT_THROW_UNLESS(result.IsSuccess() || ::nn::fs::ResultPathNotFound::Includes(result), result);
    NN_RESULT_DO(pMount->Commit());
    m_FileHeader.Reset();
    m_ShadowValidity = true;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ResponseStore::Register(const ::nn::nim::ShopServiceAccessTypes::Server& keyTarget, const ::nn::sf::InArray<char>& keyPath, const ::nn::sf::InArray<char>& valueResponse, const uint32_t expectResult, const uint32_t happenedRate) NN_NOEXCEPT
{
    // 無効パラメータのチェック
    NN_RESULT_THROW_UNLESS(nullptr != keyPath.GetData() && keyPath.GetLength() > 0, ResultShopServiceAccessInvalidCharacter());
    NN_RESULT_THROW_UNLESS(nullptr != valueResponse.GetData() && valueResponse.GetLength() > 0, ResultShopServiceAccessInvalidCharacter());

    // パラメータのオーバーフローチェック
    const size_t keyLength = keyPath.GetLength();
    NN_RESULT_THROW_UNLESS(keyLength <= static_cast<size_t>(std::numeric_limits<Bit16>::max()), ResultBufferNotEnough());
    const size_t valueLength = valueResponse.GetLength();
    NN_RESULT_THROW_UNLESS(valueLength <= static_cast<size_t>(std::numeric_limits<Bit32>::max()), ResultBufferNotEnough());
    const int64_t requiredSize = static_cast<int64_t>(sizeof(ItemHeader) + ::nn::util::align_up(keyLength + valueLength, sizeof(Bit32)));
    NN_RESULT_THROW_UNLESS(requiredSize <= static_cast<int64_t>(std::numeric_limits<Bit32>::max()), ResultBufferNotEnough());

    const auto pMount = DebugContext::GetMountHandle();
    MountHandle::EnsureSession session;
    NN_RESULT_DO(pMount->MountAlongWithCreate(&session));
    {
        int64_t next = 0;
        ScopedFileHandle file;
        const auto result = file.Open(m_FilePath, ::nn::fs::OpenMode_Read | ::nn::fs::OpenMode_Write);
        NN_RESULT_THROW_UNLESS(result.IsSuccess() || ::nn::fs::ResultPathNotFound::Includes(result), result);

        if (result.IsSuccess())
        {
            // rear position 算出
            NN_RESULT_DO(FindRearPositionOnFile(&next, file.Get(), &m_FileHeader));
        }
        else
        {
            // 初回生成
            NN_RESULT_DO(::nn::fs::CreateFile(m_FilePath, Requirement::ResponseStore::FileSize));
            NN_RESULT_DO(file.Open(m_FilePath, ::nn::fs::OpenMode_Read | ::nn::fs::OpenMode_Write));
            m_FileHeader.Reset();
            next = sizeof(FileHeader);
        }
        NN_RESULT_DO(WriteTo(file.Get(), next, keyTarget, keyPath, valueResponse, expectResult, happenedRate));
        NN_RESULT_DO(::nn::fs::FlushFile(file.Get()));
    }
    NN_RESULT_DO(pMount->Commit());
    m_ShadowValidity = true;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ResponseStore::WriteTo(const ::nn::fs::FileHandle& handle, const int64_t position, const ServerType& target, const ::nn::sf::InArray<char>& key, const ::nn::sf::InArray<char>& value, const uint32_t expectResult, const uint32_t happenedRate) NN_NOEXCEPT
{
    // NOTE: 本関数は private として、パラメタチェックは @ref Register() で行われている前提。
    const size_t keyLength = key.GetLength();
    const size_t valueLength = value.GetLength();
    const int64_t requiredSize = static_cast<int64_t>(sizeof(ItemHeader) + ::nn::util::align_up(keyLength + valueLength, sizeof(Bit32)));
    const int64_t capacity = static_cast<int64_t>(Requirement::ResponseStore::FileSize) - position;
    NN_RESULT_THROW_UNLESS(capacity >= requiredSize, ResultBufferNotEnough());

    ItemHeader item =
    {
        static_cast<Bit32>(requiredSize),
        expectResult,
        static_cast<Bit16>((happenedRate > HappenedRateRange) ? HappenedRateRange : happenedRate),
        static_cast<Bit16>(keyLength),
        static_cast<Bit32>(valueLength),
        static_cast<Bit8>((DebugContext::IsDragonsEmulationEnabled()) ? ShopServiceAccessDebug::Response::DragonsServer : static_cast<Bit8>(target))
    };
    const auto writeOption = fs::WriteOption::MakeValue(0);
    NN_RESULT_DO(::nn::fs::WriteFile(handle, position, &item, sizeof(ItemHeader), writeOption));

    int64_t next = position + sizeof(ItemHeader);
    NN_RESULT_DO(::nn::fs::WriteFile(handle, next, key.GetData(), keyLength, writeOption));

    next += keyLength;
    NN_RESULT_DO(::nn::fs::WriteFile(handle, next, value.GetData(), valueLength, writeOption));

    const auto newHeader = m_FileHeader.MakeNewRegistered(static_cast<Bit16>(keyLength));
    NN_RESULT_DO(::nn::fs::WriteFile(handle, 0, &newHeader, sizeof(FileHeader), writeOption));

    m_FileHeader = newHeader;
    DEBUG_TRACE("ResponseStore::WriteTo() => FileHeader {registered: %u, keyLengthMax: %u}\n", newHeader.registered, newHeader.keyLengthMax);
    DEBUG_TRACE("ResponseStore::WriteTo(%lld) => ItemHeader {\n    offsetForNextData: %lu\n    expectResult: %lx\n    happenedRate: %u\n    keyLength: %u\n    valueLength: %lu\n    targetServer: %u\n}\n",
        position, item.offsetForNextData, item.expectResult, item.happenedRate, item.keyLength, item.valueLength, item.targetServer);
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ResponseStore::FindRearPositionOnFile(int64_t* pOutPosition, const ::nn::fs::FileHandle& handle, FileHeader* pOutHeader) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutPosition);

    // NOTE: FileSize は生成時に最大値( Requirement::ResponseStore::FileSize )で生成している。
    FileTop work;
    int64_t rear = sizeof(FileHeader);
    NN_RESULT_DO(::nn::fs::ReadFile(handle, 0, &work, sizeof(FileTop)));
    int registered = work.header.registered;
    if (registered > 0)
    {
        while (NN_STATIC_CONDITION(true))
        {
            rear += work.item.offsetForNextData;
            if (0 >= (--registered))
            {
                break;
            }
            NN_RESULT_DO(::nn::fs::ReadFile(handle, rear, &work.item, sizeof(ItemHeader)));
        };
    }
    *pOutPosition = rear;

    if (nullptr != pOutHeader)
    {
        *pOutHeader = work.header;
    }
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ResponseStore::DoUseDebugResponse(ResultType* pOutResult, const char* pVerifyPath, const ServerType& verifyServer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutResult);

    pOutResult->Update();
    NN_RESULT_THROW_UNLESS(pVerifyPath, ResultSuccess());

    // オンメモリヘッダによる登録キーチェック。
    NN_RESULT_THROW_UNLESS(!m_ShadowValidity || (m_FileHeader.keyLengthMax > 0 && m_FileHeader.registered > 0), ResultSuccess());

    // オンメモリ上ではキー登録あり、なのでファイルから検索。
    const auto pMount = DebugContext::GetMountHandle();
    MountHandle::EnsureSession session;
    NN_RESULT_TRY(pMount->Mount(&session))
        NN_RESULT_CATCH(::nn::fs::ResultTargetNotFound)
        {
            // 対象セーブデータが存在しないので適用なしで返却。
            m_FileHeader.Reset();
            m_ShadowValidity = true;
            NN_RESULT_MARK_AS_RETHROW;
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    ScopedFileHandle file;
    NN_RESULT_TRY(file.Open(m_FilePath, ::nn::fs::OpenMode_Read | ::nn::fs::OpenMode_Write))
        NN_RESULT_CATCH(::nn::fs::ResultPathNotFound)
        {
            // 対象セーブデータ中にストアファイルが存在しないので適用なしで返却。
            m_FileHeader.Reset();
            m_ShadowValidity = true;
            NN_RESULT_MARK_AS_RETHROW;
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    // ファイルストレージ側のヘッダチェック。
    FileTop work;
    NN_RESULT_DO(::nn::fs::ReadFile(file.Get(), 0, &work, sizeof(FileTop)));

    m_ShadowValidity = true;
    m_FileHeader = work.header;
    int registered = work.header.registered;
    const int keyLengthMax = work.header.keyLengthMax;
    NN_RESULT_THROW_UNLESS(keyLengthMax > 0 && registered > 0, ResultSuccess());

    DEBUG_TRACE("DoUseDebugResponse: {registered: %d, keyLengthMax: %d}\n", registered, keyLengthMax);

    // 取得スコープ中のみの一時ヒープ利用.
    const size_t requiredMaxSize = (keyLengthMax + 1) * 2; // リプレース処理用に入力と出力。
    const size_t workHeapSize = (ExpectLimitHeapWorkSize > requiredMaxSize) ? ExpectLimitHeapWorkSize : requiredMaxSize;
    ShopServiceAccess::OnetimeHeapSession scopedHeap;
    NN_RESULT_DO(scopedHeap.Acquire(std::alignment_of<Bit64>::value, workHeapSize));

    auto rateThreshold = DebugContext::GetRandomizer()->GenerateRandomN(HappenedRateRange);
    const auto pHeap = static_cast<char*>(scopedHeap.pTop);
    const auto pKeyUrl = &pHeap[workHeapSize / 2];
    const auto handle = file.Get();
    int64_t currentItemSeekPosition = sizeof(FileHeader);
    while (NN_STATIC_CONDITION(true))
    {
        // カレントのキーURLを読み込み。( 非 null 終端 )
        const size_t keyLength = work.item.keyLength;
        const int64_t keyTop = static_cast<int64_t>(currentItemSeekPosition + sizeof(ItemHeader));
        NN_RESULT_DO(::nn::fs::ReadFile(handle, keyTop, pKeyUrl, keyLength));

        DEBUG_TRACE("DoUseDebugResponse: verify path() => `%s`\n", pVerifyPath);
        DEBUG_TRACE_FOR_BOUNDARY_STRING("DoUseDebugResponse: registered path", pKeyUrl, keyLength);
        const auto happenedRate = static_cast<uint32_t>(work.item.happenedRate);
        DEBUG_TRACE("DoUseDebugResponse: happened rate ( %u < %u )\n", rateThreshold, happenedRate);

        // URLリプレース
        pKeyUrl[keyLength] = '\0';  // リプレースのために必要.
        NN_RESULT_DO(pOutResult->ExecuteAliasReplaceForUrl(pHeap, workHeapSize / 2, pKeyUrl));
        DEBUG_TRACE_FOR_BOUNDARY_STRING("DoUseDebugResponse: replaced path", pHeap, keyLength);

        // 前方一致比較。
        if (verifyServer == work.item.targetServer && 0 == ::nn::util::Strncmp(pVerifyPath, pHeap, static_cast<int>(keyLength)))
        {
            // 発生率検証。
            if (rateThreshold < happenedRate)
            {
                const size_t totalResponseSize = work.item.valueLength;
                Bit8* pResponse = static_cast<Bit8*>(pOutResult->AllocateResponseStore(totalResponseSize));
                NN_RESULT_THROW_UNLESS(pResponse, ResultShopServiceAccessInsufficientWorkMemory());

                // 分散読み込みで pResponse へ転送。
                size_t remainSize = totalResponseSize;
                int64_t readPosition = currentItemSeekPosition + sizeof(ItemHeader) + keyLength;
                do
                {
                    const size_t readSize = (remainSize > workHeapSize) ? workHeapSize : remainSize;
                    NN_RESULT_DO(::nn::fs::ReadFile(handle, readPosition, pHeap, readSize));
                    std::memcpy(pResponse, pHeap, readSize);

                    remainSize -= readSize;
                    readPosition += readSize;
                    pResponse = &pResponse[readSize];
                } while (remainSize > 0);

                // レスポンスボディの転送完了したので検出済 Result の設定。
                pOutResult->Update(::nn::result::detail::ConstructResult(work.item.expectResult), true);
                DEBUG_TRACE("DoUseDebugResponse: Hit!! [result: %lx]\n", work.item.expectResult);
                NN_RESULT_SUCCESS;
            }
            else
            {
                // パス前方一致したけど発生しなかったので、次の前方一致要素での発生確率を上げる。
                rateThreshold -= happenedRate;
            }
        }

        // 違ったら次。
        currentItemSeekPosition += work.item.offsetForNextData;
        if (0 >= (--registered))
        {
            break;
        }
        NN_RESULT_DO(::nn::fs::ReadFile(handle, currentItemSeekPosition, &work.item, sizeof(ItemHeader)));
    };
    NN_RESULT_SUCCESS;
}



//-----------------------------------------------------------------------------
::nn::os::SdkMutex g_DebugInterfaceLock;

//-----------------------------------------------------------------------------
}   // ~unnamed
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
void ShopServiceAccessDebug::RefreshAvailability() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_DebugInterfaceLock)> lock(g_DebugInterfaceLock);
    DebugContext::RefreshAvailability();
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessDebug::IsAvailable() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_DebugInterfaceLock)> lock(g_DebugInterfaceLock);
    return DebugContext::IsAvailable();
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessDebug::Response::Clear() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_DebugInterfaceLock)> lock(g_DebugInterfaceLock);

    NN_RESULT_DO(DebugContext::IsAvailable());
    return DebugContext::GetResponseStore()->Clear();
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessDebug::Response::Register(const ::nn::nim::ShopServiceAccessTypes::Server& keyTarget, const ::nn::sf::InArray<char>& keyUrl, const ::nn::sf::InArray<char>& valueResponse, const uint32_t expectResult, const uint32_t happenedRate) NN_NOEXCEPT
{
    std::lock_guard<decltype(g_DebugInterfaceLock)> lock(g_DebugInterfaceLock);

    NN_RESULT_DO(DebugContext::IsAvailable());
    return DebugContext::GetResponseStore()->Register(keyTarget, keyUrl, valueResponse, expectResult, happenedRate);
}

//-----------------------------------------------------------------------------
Result ShopServiceAccessDebug::Response::DoUseDebugResponse(EmulationResult* pOutResult, const char* pVerifyPath, const ::nn::nim::ShopServiceAccessTypes::Server& verifyServer) NN_NOEXCEPT
{
    std::lock_guard<decltype(g_DebugInterfaceLock)> lock(g_DebugInterfaceLock);

    NN_RESULT_DO(DebugContext::IsAvailable());
    NN_RESULT_THROW_UNLESS(DragonsServer != verifyServer || DebugContext::IsDragonsEmulationEnabled(), ResultNotSupported());
    return DebugContext::GetResponseStore()->DoUseDebugResponse(pOutResult, pVerifyPath, verifyServer);
}

}}}
