﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "nim_ShopServiceAccessPrivate.h"

#include <nn/nim/srv/nim_ShopServiceAccessServer.h>
#include <nn/nim/srv/nim_HttpConnection.h>
#include <nn/nim/srv/nim_OsUtil.h>

#include <nn/err/err_Types.h>

namespace nn { namespace nim { namespace srv {

class ShopServiceAccessorImpl;

/**
 * @brief   ShopServiceAccessor 機能の非同期アクセスサービスオブジェクト( IShopServiceAsync )実体。
 *
 * @details クライアントからのリクエストキュー単位( このインスタンスへのポインタがキュー要素になる )です。
 *          リクエスト発生時に要求元の HttpConnection ( IShopServiceAccessor ) への参照ポインタを登録して、キューへ追加します。
 */
class ShopServiceAccessAsyncImpl
{
    NN_DISALLOW_COPY(ShopServiceAccessAsyncImpl);
    NN_DISALLOW_MOVE(ShopServiceAccessAsyncImpl);

public:
    /**
     * @brief   非同期維持リクエストパラメータ構造
     */
    class RequestParameter
    {
    public:
        typedef ShopServiceAccessTypes::FixedParams FixedParams;
        typedef ShopServiceAccess::HeapAllocator    Allocator;

        struct VariableParams
        {
        private:
            Allocator*  m_pAllocator;

        public:
            struct Memory
            {
                void*   pTop;
                size_t  size;

                NN_IMPLICIT Memory() NN_NOEXCEPT : pTop(nullptr), size(0) {}
                NN_FORCEINLINE void Reset(void* pTop_ = nullptr, size_t size_ = 0) NN_NOEXCEPT
                {
                    pTop = pTop_;
                    size = size_;
                }
                NN_FORCEINLINE bool IsAvailable() const NN_NOEXCEPT
                {
                    return (nullptr != pTop && size > 0);
                }
            };
            Memory  path;
            Memory  post;
            Memory  response;

            NN_IMPLICIT VariableParams() NN_NOEXCEPT : m_pAllocator(nullptr), path(), post(), response() {}

            void Initialize(Allocator* const pAllocator) NN_NOEXCEPT;

            bool Prepare(const ::nn::sf::InArray<char>& inPath, const ::nn::sf::InArray<char>& inPostData) NN_NOEXCEPT;

            bool PrepareResponse(const size_t expectResponseSize) NN_NOEXCEPT;

            void Finalize() NN_NOEXCEPT;

            template<typename CastType>
            NN_FORCEINLINE CastType* GetPathTop() const NN_NOEXCEPT
            {
                return static_cast<CastType*>(path.pTop);
            }

            template<typename CastType>
            NN_FORCEINLINE CastType* GetPostDataTop() const NN_NOEXCEPT
            {
                return static_cast<CastType*>(post.pTop);
            }

            template<typename CastType>
            NN_FORCEINLINE CastType* GetReponseTop() const NN_NOEXCEPT
            {
                return static_cast<CastType*>(response.pTop);
            }

        private:
            void FinalizeMemory(Allocator* const pAllocator, Memory& memory) NN_NOEXCEPT;
        };

        NN_IMPLICIT RequestParameter() NN_NOEXCEPT {}

        ~RequestParameter() NN_NOEXCEPT;

        NN_FORCEINLINE const FixedParams& GetFixedParameter() const NN_NOEXCEPT
        {
            return m_Fixed;
        }

        NN_FORCEINLINE const VariableParams& GetVariableParameter() const NN_NOEXCEPT
        {
            return m_Variable;
        }

        NN_FORCEINLINE VariableParams& GetVariableParameter() NN_NOEXCEPT
        {
            return m_Variable;
        }

        template<typename CastType>
        NN_FORCEINLINE CastType* GetPathTop() const NN_NOEXCEPT
        {
            return m_Variable.GetPathTop<CastType>();
        }

        template<typename CastType>
        NN_FORCEINLINE CastType* GetPostDataTop() const NN_NOEXCEPT
        {
            return m_Variable.GetPostDataTop<CastType>();
        }

        template<typename CastType>
        NN_FORCEINLINE CastType* GetReponseTop() const NN_NOEXCEPT
        {
            return m_Variable.GetReponseTop<CastType>();
        }

        void Apply(Allocator* const pAllocator, const FixedParams& fixed) NN_NOEXCEPT;

    private:
        VariableParams  m_Variable;
        FixedParams     m_Fixed;
    };

    /**
     * @brief   通信ハンドル実装( キャンセル可能 )。
     */
    class ConnectionImpl : public HttpConnection
    {
        NN_DISALLOW_COPY(ConnectionImpl);
        NN_DISALLOW_MOVE(ConnectionImpl);

        /**
         * @brief   基底型。
         */
        typedef HttpConnection BaseType;

        /**
         * @brief   要求制限解除待ち中断ハンドルコンテナ型。
         */
        typedef ShopServiceAccess::RequestCountRestriction::Canceler RestrictCanceler;

        /**
         * @brief   認証トークン取得中断ハンドルコンテナ型。
         */
        typedef TokenStore::CancelerSet TokenCanceler;

    public:
        /**
         * @brief   コンストラクタ。
         */
        ConnectionImpl() NN_NOEXCEPT : HttpConnection(), m_TokenCanceler(), m_RestrictCanceler() {}

        /**
         * @brief   中断要求。
         */
        void Cancel() NN_NOEXCEPT
        {
            BaseType::Cancel();
            m_TokenCanceler.Cancel();
            m_RestrictCanceler.Cancel();
        }

        /**
         * @brief   キャンセル可能状態の強制無効化。
         */
        void Invalidate() NN_NOEXCEPT
        {
            m_TokenCanceler.Invalidate();
            m_RestrictCanceler.Invalidate();
        }

        NN_FORCEINLINE TokenCanceler&  GetTokenCanceler() NN_NOEXCEPT
        {
            return m_TokenCanceler;
        }

        NN_FORCEINLINE RestrictCanceler&  GetRestrictCanceler() NN_NOEXCEPT
        {
            return m_RestrictCanceler;
        }

    private:
        TokenCanceler       m_TokenCanceler;    //!< 認証トークン取得キャンセル用。
        RestrictCanceler    m_RestrictCanceler; //!< 任意要求制限解除待ちキャンセル用。
    };

    /**
     * @brief   コンストラクタ
     */
    explicit ShopServiceAccessAsyncImpl(ShopServiceAccessorImpl* pAccessor) NN_NOEXCEPT;

    /**
     * @brief   デストラクタ
     */
    ~ShopServiceAccessAsyncImpl() NN_NOEXCEPT;

    /**
     * @brief       IPC: 非同期処理中断取得要求
     *
     * @return      要求結果
     * @retval      nn::ResultSuccess   正常に終了しました。
     */
    Result Cancel() NN_NOEXCEPT
    {
        return OnCancel();
    }

    /**
     * @brief       IPC: 非同期処理結果データサイズ取得要求
     */
    Result GetSize(::nn::sf::Out<uint64_t> outValue) NN_NOEXCEPT;

    /**
     * @brief       IPC: 非同期処理結果データ取得要求
     *
     * @param[out]  outValue    @a outBuffer に出力できた実質的なレスポンスデータサイズ( バイトサイズ )を返します。
     * @param[in]   offset      処理結果レスポンスデータに対する取得開始位置をバイト単位オフセットで指定します。
     * @param[in]   outBuffer   処理結果レスポンスデータを取得する受信バッファを指定します。
     *
     * @return      要求結果
     * @retval      nn::ResultSuccess               全レスポンスが @a outBuffer に出力されました。
     * @retval      nn::nim::ResultBufferNotEnough  出力先バッファ( @a outBuffer )の容量内に全てのレスポンスを出力できませんでした。
     *
     * @details     処理結果レスポンスデータ列を取得します。@n
     *              全レスポンスデータサイズ( totalSize )に対して、@a offset 位置からのデータ列を @a outBuffer に出力します。@n
     *              想定出力サイズ( totalSize - @a offset )が、@a outBuffer の容量を越える場合、outBuffer のサイズに切り詰めて出力し、@a nn::nim::ResultBufferNotEnough を返します。@n
     *              @a outValue には @a outBuffer に出力できた実質的なレスポンスデータサイズが返されます。
     */
    Result Read(::nn::sf::Out<uint64_t> outValue, uint64_t offset, const ::nn::sf::OutBuffer& outBuffer) const NN_NOEXCEPT;

    /**
     * @brief       IPC: 非同期処理結果エラーコード取得要求
     */
    Result GetErrorCode(::nn::sf::Out<::nn::err::ErrorCode> outCode) const NN_NOEXCEPT;

    /**
     * @brief       システムイベントへの参照の取得。
     */
    os::SystemEvent& GetEvent() NN_NOEXCEPT
    {
        return m_Event;
    }

    /**
     * @brief       通信作業スレッド上での通信処理実施メソッド。
     *
     * @details     完了時のシステムイベントへのシグナルは @ref FinishOnThread() で行ってください。
     */
    void RunOnThread() NN_NOEXCEPT;

    /**
     * @brief       通信作業スレッド上での完了通知実施メソッド。
     *
     * @details     クライアントに対しての完了通知イベントをシグナルします。
     */
    void FinishOnThread() NN_NOEXCEPT;

    /**
     * @brief       クエリパラメータのコピー登録。
     *
     * @param[in]   pAllocator  クライアントヒープアロケータ。
     * @param[in]   fixed       固定長リクエストパラメータ。
     */
    void SetParameter(ShopServiceAccess::HeapAllocator* pAllocator, const ShopServiceAccessTypes::FixedParams& fixed) NN_NOEXCEPT;

    /**
     * @brief       IPC: 非同期通信パラメータ登録要求。
     *
     * @param[in]   inPath  URLパスフィールド文字列データ。
     * @param[in]   inPost  ポストデータフィールドバイトデータ。
     */
    Result Prepare(const ::nn::sf::InArray<char>& inPath, const ::nn::sf::InArray<char>& inPost) NN_NOEXCEPT;

    /**
     * @brief   IPC: 非同期通信実施要求。
     */
    Result Request() NN_NOEXCEPT;

protected:
    /**
     * @brief       通信キャンセルを要求。
     *
     * @return      要求処理結果を返します。
     * @retval      nn::ResultSuccess   正常終了しました。
     *
     * @details     通信実行中なら通信キャンセルを要求します。
     */
    Result OnCancel() NN_NOEXCEPT;

    /**
     * @brief       通信実行。
     *
     * @param[out]  pOutResponseSize    受信したレスポンスサイズ( バイトサイズ )の取得用バッファ先頭アドレスへのポインタ。
     *
     * @return      要求処理結果を返します。
     * @retval      nn::ResultSuccess               正常終了しました。
     * @retval      nn::nim::ResultNotSupported     サポートしていない通信要求パラメータが指定されました。
     * @retval      nn::nim::ResultTaskStillRunning 要求ハンドルは処理中です。
     */
    Result OnExecute(size_t* const pOutResponseSize) NN_NOEXCEPT;

private:
    /**
     * @brief   処理継続可否確認。
     */
    Result CanContinue() NN_NOEXCEPT;

    /**
     * @brief   非同期アクセス用値コンテナ
     */
    struct ResponseValues
    {
        Bit64                   receivedSize;   //!< 受信レスポンスバイトサイズ
        ::nn::err::ErrorCode    errorCode;      //!< サービス固有エラーコード
        Result                  result;         //!< 処理結果保存コンテナ
        Bit32                   reserved;

        NN_IMPLICIT ResponseValues() NN_NOEXCEPT;
        NN_IMPLICIT ResponseValues(const Result result_) NN_NOEXCEPT;
        NN_IMPLICIT ResponseValues(const Result result_, const Bit64 receivedSize_, const ::nn::err::ErrorCode& errorCode_) NN_NOEXCEPT;
    };
    NN_STATIC_ASSERT(sizeof(ResponseValues) == 24);

    /**
     * @brief   レスポンスコンテナアトミック型宣言。
     */
    typedef detail::MutexAtomic<ResponseValues> ResponseValuesAtomic;

    /**
     * @brief   依存親型宣言。
     */
    typedef sf::SharedPointer<ShopServiceAccessorImpl>  ParentInterface;

    const ParentInterface           m_pAccessor;        //!< リクエストオーナー(親寿命管理用共有ポインタ)
    std::atomic<ConnectionImpl*>    m_pConnection;      //!< 実行中コンテキスト

    ManualClearSystemEvent          m_Event;            //!< クライアント同期システムイベント
    RequestParameter                m_Data;             //!< リクエストパラメータ
    ResponseValuesAtomic            m_ResponseValues;   //!< 処理結果保存コンテナ

    bool                            m_IsRequested;      //!< キュー要求成功済判断。( このフラグは IPC スレッド上のみの操作を想定しています )
};

}}}
