﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/nim/srv/nim_LocalCommunicationReceiveContentTaskBase.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/socket.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nim/srv/nim_LocalCommunicationDeliveryProtocol.h>
#include "nim_DebugUtil.h"
#include "nim_StringUtil.h"

namespace nn { namespace nim { namespace srv {

    void LocalCommunicationReceiveContentTaskBase::Cancel() NN_NOEXCEPT
    {
        m_Connection.Cancel();
        InstallTaskBase::Cancel();
    }

    void LocalCommunicationReceiveContentTaskBase::ResetCancel() NN_NOEXCEPT
    {
        m_Connection.ResetCancel();
        InstallTaskBase::ResetCancel();
    }

    Result LocalCommunicationReceiveContentTaskBase::Initialize(uint32_t ipv4, uint16_t port, ncm::StorageId storage, ncm::InstallTaskDataBase* data, Bit32 config) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_Connection.Initialize(ipv4, port));
        m_Data = data;
        NN_RESULT_DO(InstallTaskBase::Initialize(storage, m_Data, config));

        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationReceiveContentTaskBase::InitializeForCleanup(ncm::StorageId storage, ncm::InstallTaskDataBase* data, Bit32 config) NN_NOEXCEPT
    {
        m_Data = data;
        NN_RESULT_DO(InstallTaskBase::Initialize(storage, m_Data, config));

        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationReceiveContentTaskBase::GetInstallContentMetaInfo(ncm::InstallContentMetaInfo* outValue, const ncm::ContentMetaKey& key) NN_NOEXCEPT
    {
        /* TORIAEZU:　socket を使って適当に通信する */
        auto header = MakeLocalCommunicationDeliveryProtocolRequestHeader(LocalCommunicationDeliveryProtocolTag::GetPackagedContentInfoTag, sizeof(key));
        NN_RESULT_DO(m_Connection.SendHeader(header));

        NN_RESULT_DO(m_Connection.SendData(&key, sizeof(key), sizeof(key)));

        LocalCommunicationDeliveryProtocolHeader receiveHeader;
        ncm::PackagedContentInfo info;
        NN_RESULT_DO(m_Connection.ReceiveHeader(&receiveHeader));
        NN_RESULT_THROW_UNLESS(receiveHeader.tag == header.tag, ResultLocalCommunicationUnexpectedTag());
        NN_RESULT_THROW_UNLESS(receiveHeader.size == sizeof(info), ResultLocalCommunicationInvalidDataSize());
        NN_RESULT_DO(m_Connection.ReceiveData(&info, sizeof(info), sizeof(info)));

        *outValue = ncm::InstallContentMetaInfo::MakeVerifiable(info.info.GetId(), info.info.GetSize(), key.type, info.hash);

        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationReceiveContentTaskBase::PrepareInstallContentMetaData() NN_NOEXCEPT
    {
        auto count = CountInstallContentMetaKey();
        for (int i = 0; i < count; i++)
        {
            NN_RESULT_DO(CanContinue());
            ncm::ContentMetaKey key;
            NN_RESULT_DO(GetInstallContentMetaKey(&key, i));
            bool hasAlreadyData;
            NN_RESULT_DO(m_Data->Has(&hasAlreadyData, key.id));
            if (hasAlreadyData)
            {
                continue;
            }

            ncm::InstallContentMetaInfo info;
            NN_RESULT_DO(GetInstallContentMetaInfo(&info, key));
            NN_RESULT_DO(PrepareContentMeta(info, key));
        }

        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationReceiveContentTaskBase::OnPrepareComplete() NN_NOEXCEPT
    {
        NN_RESULT_DO(GetLocalCommunicationErrorResultForDebug());

        auto progress = GetProgress();

        /* TORIAEZU:　socket を使って適当に通信する */
        auto header = MakeLocalCommunicationDeliveryProtocolRequestHeader(LocalCommunicationDeliveryProtocolTag::SendTotalSizeTag, sizeof(progress.totalSize));
        NN_RESULT_DO(m_Connection.Send(header, &progress.totalSize, sizeof(progress.totalSize)));

        LocalCommunicationDeliveryProtocolHeader receiveHeader;
        NN_RESULT_DO(m_Connection.ReceiveHeader(&receiveHeader));
        NN_RESULT_THROW_UNLESS(receiveHeader.tag == header.tag, ResultLocalCommunicationUnexpectedTag());
        NN_RESULT_THROW_UNLESS(receiveHeader.size == 0, ResultLocalCommunicationInvalidDataSize());

        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationReceiveContentTaskBase::OnWritePlaceHolder(ncm::InstallContentInfo* data) NN_NOEXCEPT
    {
        /* TORIAEZU: socket を使って適当に通信する */
        LocalCommunicationContentInfo info {};
        info.contentId = data->info.GetId();
        info.isTemporary = data->isTemporary;
        auto header = MakeLocalCommunicationDeliveryProtocolRequestHeader(LocalCommunicationDeliveryProtocolTag::GetContentTag, sizeof(info));
        NN_RESULT_DO(m_Connection.SendHeader(header));

        NN_RESULT_DO(m_Connection.SendData(&info, sizeof(info), header.size));

        LocalCommunicationDeliveryProtocolHeader receiveHeader;
        NN_RESULT_DO(m_Connection.ReceiveHeader(&receiveHeader));
        NN_RESULT_THROW_UNLESS(receiveHeader.tag == header.tag, ResultLocalCommunicationUnexpectedTag());
        NN_RESULT_THROW_UNLESS(receiveHeader.size == data->info.GetSize(), ResultLocalCommunicationInvalidDataSize());

        auto receiveContent = [&](size_t recvSize, int64_t offset) -> Result
        {
            NN_UNUSED(offset);
            NN_RESULT_DO(WritePlaceHolderBuffer(data, m_Buffer, recvSize));
            NN_RESULT_SUCCESS;
        };
        NN_RESULT_DO(m_Connection.ReceiveData(m_Buffer, m_BufferSize, data->info.GetSize(), receiveContent));

        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationReceiveContentTaskBase::OnExecuteComplete() NN_NOEXCEPT
    {
        auto header = MakeLocalCommunicationDeliveryProtocolRequestHeader(LocalCommunicationDeliveryProtocolTag::EndSessionTag, 0);
        NN_RESULT_DO(m_Connection.SendHeader(header));

        NN_RESULT_SUCCESS;
    }
}}}
