﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <mutex>
#include <curl/curl.h>
#include <nn/ssl.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_StringView.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/srv/nim_OsUtil.h>
#include <nn/nim/srv/nim_HttpConnection.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/os/os_Tick.h>

#include "nim_DebugUtil.h"
#include "nim_HttpUtil.h"

// #define NN_NIM_TRACE_HTTP_RESPONSE_HEADER
// #define NN_NIM_TRACE_HTTP_REQUEST_HEADER
// #define NN_NIM_TRACE_HTTP_RESPONSE_DATA

#if defined NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON
#define NN_NIM_AUTO_PROXY_SUPPORTED
#endif

#define NN_NIM_DO_CURL(curl) NN_RESULT_DO(ToResult(curl))

namespace nn { namespace nim { namespace srv {

    namespace {

        template<class CallbackT>
        class CallbackHolder
        {
        public:
            CallbackHolder(util::optional<CallbackT> func, HttpConnection* connection) NN_NOEXCEPT : m_Func(func), m_Connection(connection) {}

            CallbackT Get() NN_NOEXCEPT
            {
                return *m_Func;
            }

            HttpConnection& GetConnection() NN_NOEXCEPT
            {
                return *m_Connection;
            }
        private:
            util::optional<CallbackT> m_Func;
            HttpConnection* m_Connection;
        };

        typedef CallbackHolder<HttpConnection::WriteCallback> WriteCallbackHolder;
        typedef CallbackHolder<HttpConnection::HeaderCallback> HeaderCallbackHolder;

        int CurlProgressCallback(void *clientp, curl_off_t dltotal, curl_off_t  dlnow, curl_off_t  ultotal, curl_off_t ulnow) NN_NOEXCEPT
        {
            // NN_DETAIL_NIM_TRACE("[HttpConnection] Progress callback\n");
            auto connection = static_cast<HttpConnection*>(clientp);
            connection->OnProgress(dltotal, dlnow, ultotal, ulnow);
            if (IsErrorSimulationEnabled())
            {
                auto url = connection->GetUrl();
                if (url)
                {
                    NN_RESULT_TRY(DoErrorSimulation(url))
                        NN_RESULT_CATCH_ALL
                        {
                            NN_DETAIL_NIM_TRACE("[HttpConnection] Simulate error of %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
                            connection->SetLastResult(NN_RESULT_CURRENT_RESULT);
                            return -1;
                        }
                    NN_RESULT_END_TRY;
                }
            }
            return connection->IsCanceled() || connection->IsTimeout() ? -1 : 0;
        }

        template<class CallbackHolderT>
        size_t CurlCallback(char *ptr, size_t size, size_t nmemb, CallbackHolderT* holder) NN_NOEXCEPT
        {
            auto& connection = holder->GetConnection();
            if (connection.IsCanceled())
            {
                return 0;
            }

            auto result = holder->Get()(ptr, size * nmemb);

            connection.SetLastResult(result);
            return result.IsSuccess() ? size * nmemb : 0;
        }

        size_t CurlHeaderCallback(char *ptr, size_t size, size_t nmemb, void *userdata) NN_NOEXCEPT
        {
#ifdef NN_NIM_TRACE_HTTP_RESPONSE_HEADER
            NN_DETAIL_NIM_TRACE("[HttpConnection] Response Header:\n");
            char trace[128];
            auto traceSize = std::min(sizeof(trace) - 1, size * nmemb);
            std::memcpy(trace, ptr, traceSize);
            trace[traceSize] = '\0';
            NN_DETAIL_NIM_TRACE("[HttpConnection]     %s", trace);
#endif
            return userdata ? CurlCallback(ptr, size, nmemb, static_cast<HeaderCallbackHolder*>(userdata)) : size * nmemb;
        }

        size_t CurlWriteCallback(char *ptr, size_t size, size_t nmemb, void *userdata) NN_NOEXCEPT
        {
#ifdef NN_NIM_TRACE_HTTP_RESPONSE_DATA
            const auto bufferSize = size * nmemb;
            NN_DETAIL_NIM_TRACE("[HttpConnection] Response Data %zu bytes:\n", bufferSize);
            static const size_t CharacterCountPerLine = 64;
            char trace[CharacterCountPerLine + 1];
            size_t traced = 0;
            while (traced < bufferSize)
            {
                auto traceSize = std::min(CharacterCountPerLine, bufferSize - traced);
                std::memcpy(trace, ptr + traced, traceSize);
                trace[traceSize] = '\0';
                NN_DETAIL_NIM_TRACE("%s", trace);
                traced += traceSize;
            }
            NN_DETAIL_NIM_TRACE("\n");
#endif
            return CurlCallback(ptr, size, nmemb, static_cast<WriteCallbackHolder*>(userdata));
        }

        Result GetDetailIfSslCurlError(CURL* curl, CURLcode e) NN_NOEXCEPT
        {
            CURLINFO info;
            switch (e)
            {
            case CURLE_SSL_CONNECT_ERROR: info = CURLINFO_SSL_HANDSHAKE_RESULT; break;
            case CURLE_PEER_FAILED_VERIFICATION: info = CURLINFO_SSL_VERIFYRESULT; break;
            default: NN_RESULT_SUCCESS;
            }

            long value;
            NN_NIM_DO_CURL(curl_easy_getinfo(curl, info, &value));

            Result result;
            NN_RESULT_DO(ssl::GetSslResultFromValue(&result, reinterpret_cast<char*>(&value), sizeof(value)));
            // TODO: 当面ログを出すだけ
            NN_DETAIL_NIM_TRACE("[HttpConnection] ssl detail result 0x%08x\n", result.GetInnerValueForDebug());

            NN_RESULT_SUCCESS;
        }

        bool IsHttps(const char* url) NN_NOEXCEPT
        {
            static const char Prefix[] = "https";
            if (std::strlen(url) < std::strlen(Prefix))
            {
                return false;
            }
            return std::strncmp(url, Prefix, std::strlen(Prefix)) == 0;
        }

        const char* GetCurlStringInfo(CURL* curl, CURLINFO info) NN_NOEXCEPT
        {
            char* s;
            return curl_easy_getinfo(curl, info, &s) == CURLE_OK ? s : nullptr;
        }
    }

    Result FindHttpHeader(util::optional<HttpHeaderValue>* outValue, const char* name, const char* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        util::string_view header(buffer, bufferSize);

        kvdb::BoundedString<32> headerName;
        headerName.AssignFormat("%s: ", name);

        auto pos = header.find(headerName);
        if (pos == util::string_view::npos)
        {
            *outValue = util::nullopt;
            NN_RESULT_SUCCESS;
        }

        size_t startIndex = pos + headerName.GetLength();
        for (size_t i = startIndex; i < bufferSize; i++)
        {
            if (buffer[i] == '\n' && i > 0 && buffer[i - 1] == '\r')
            {
                HttpHeaderValue value;
                size_t valueLength = i - startIndex;
                NN_RESULT_THROW_UNLESS(valueLength < sizeof(value.string), ResultUnexpectedResponseHttpHeaderTooLong());
                util::Strlcpy(value.string, &buffer[startIndex], static_cast<int>(valueLength));
                *outValue = value;
                NN_RESULT_SUCCESS;
            }
        }

        *outValue = util::nullopt;
        NN_RESULT_SUCCESS;
    }

    HttpConnection::~HttpConnection() NN_NOEXCEPT
    {
        Finalize();
    }

    Result HttpConnection::Initialize(DeviceContext* deviceContext) NN_NOEXCEPT
    {
        m_DeviceContext = deviceContext;
        return InitializeCore(m_DeviceContext->GetSslContext());
    }

    Result HttpConnection::Initialize(ssl::Context* sslContext) NN_NOEXCEPT
    {
        return InitializeCore(sslContext);
    }

    void HttpConnection::Finalize() NN_NOEXCEPT
    {
        if (m_Impl)
        {
            curl_easy_cleanup(m_Impl);
            m_Impl = nullptr;
        }
    }

    Result HttpConnection::InitializeCore(ssl::Context* sslContext) NN_NOEXCEPT
    {
        auto curl = curl_easy_init();
        NN_RESULT_THROW_UNLESS(curl, ResultCurlEasyInitFailure());
#if !defined NN_NIM_AUTO_PROXY_SUPPORTED
        NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_PROXYAUTOCONFIG, false));
        NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_PROXY, "proxy.nintendo.co.jp:8080"));
#endif
        NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 1L));
        NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 2L));
        NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_SSL_VERIFYDATE, 0L));
        NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_SSL_CONTEXT, sslContext));

        m_Impl = curl;

        NN_RESULT_SUCCESS;
    }


    Result HttpConnection::Head(const char* url, HeaderCallback headerCallback, const char * headerList[], int headerCount, int timeoutSecond) NN_NOEXCEPT
    {
        return Perform(url, nullptr, headerCallback, util::nullopt, headerList, headerCount, timeoutSecond);
    }

    Result HttpConnection::Get(const char* url, WriteCallback writeCallback, const char * headerList[], int headerCount, int timeoutSecond) NN_NOEXCEPT
    {
        return Perform(url, nullptr, util::nullopt, writeCallback, headerList, headerCount, timeoutSecond);
    }

    Result HttpConnection::Get(const char* url, HeaderCallback headerCallback, WriteCallback writeCallback, const char * headerList[], int headerCount, int timeoutSecond) NN_NOEXCEPT
    {
        return Perform(url, nullptr, headerCallback, writeCallback, headerList, headerCount, timeoutSecond);
    }

    Result HttpConnection::Post(const char* url, const char* postFields, WriteCallback writeCallback, const char * headerList[], int headerCount, int timeoutSecond) NN_NOEXCEPT
    {
        return Perform(url, postFields, util::nullopt, writeCallback, headerList, headerCount, timeoutSecond);
    }

    Result HttpConnection::Post(const char* url, const char* postFields, HeaderCallback headerCallback, WriteCallback writeCallback, const char * headerList[], int headerCount, int timeoutSecond) NN_NOEXCEPT
    {
        return Perform(url, postFields, headerCallback, writeCallback, headerList, headerCount, timeoutSecond);
    }

    Result HttpConnection::Escape(char* buffer, size_t bufferSize, const char* str) NN_NOEXCEPT
    {
        auto curl = curl_easy_init();
        NN_RESULT_THROW_UNLESS(curl, ResultCurlEasyInitFailure());
        NN_UTIL_SCOPE_EXIT{ curl_easy_cleanup(curl); };

        auto escaped = curl_easy_escape(curl, str, 0);
        NN_RESULT_THROW_UNLESS(escaped, ResultAllocationMemoryFailed());
        NN_UTIL_SCOPE_EXIT{ curl_free(escaped); };

        NN_RESULT_THROW_UNLESS(std::strlen(escaped) < bufferSize, ResultBufferNotEnough());
        std::strncpy(buffer, escaped, bufferSize);

        NN_RESULT_SUCCESS;
    }

    void HttpConnection::GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT
    {
        const char* nulltext = "<null>";
        *outValue = {};

        if (!m_Impl)
        {
            return;
        }
        // TYPE
        outValue->type = err::ErrorContextType::Http;

        // IP
        auto ip = GetCurlStringInfo(m_Impl, CURLINFO_PRIMARY_IP);
        if (!ip)
        {
            ip = nulltext;
        }
        util::Strlcpy(outValue->http.ip, ip, static_cast<int>(sizeof(outValue->http.ip)));

        // FQDN
        auto url = GetCurlStringInfo(m_Impl, CURLINFO_EFFECTIVE_URL);
        if (url)
        {
            auto p = std::strstr(url, "://");
            if (p)
            {
                auto fqdnBegin = p + std::strlen("://");
                auto fqdnEnd = std::strstr(fqdnBegin, "/");
                intptr_t fqdnLength = fqdnEnd ? fqdnEnd - fqdnBegin : std::strlen(fqdnBegin);

                util::Strlcpy(outValue->http.fqdn, fqdnBegin, std::min(static_cast<int>(sizeof(outValue->http.fqdn)), static_cast<int>(fqdnLength + 1)));
                return;
            }
        }
        // URL/FQDN 取得に失敗した場合
        util::Strlcpy(outValue->http.fqdn, nulltext, static_cast<int>(sizeof(outValue->http.fqdn)));
    }

    Result HttpConnection::Perform(const char* url, const char* postFields, util::optional<HeaderCallback> headerCallback, util::optional<WriteCallback> writeCallback, const char* headerList[], int headerCount, int timeoutSecond) NN_NOEXCEPT
    {
        NN_DETAIL_NIM_TRACE("[HttpConnection] Request: %s\n", url);
        CURL* curl = m_Impl;

        ResetProgress(timeoutSecond);
        NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_XFERINFODATA, this));
        NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_XFERINFOFUNCTION, CurlProgressCallback));
        NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0));

        NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_URL, url));
        if (postFields)
        {
            NN_DETAIL_NIM_TRACE("[HttpConnection] PostFields: %s\n", postFields);
            NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_POST, 1));
            NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_POSTFIELDS, postFields));
        }
        else
        {
            NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_POST, 0));
        }

#ifdef NN_NIM_TRACE_HTTP_REQUEST_HEADER
        NN_DETAIL_NIM_TRACE("[HttpConnection] Request Header:\n");
#endif
        curl_slist* slist = nullptr;
        NN_UTIL_SCOPE_EXIT{ curl_slist_free_all(slist); };
        if (m_DeviceContext)
        {
            auto uaHeader = IsHttps(url) ? m_DeviceContext->GetPrivateUserAgentHeader() :
                m_DeviceContext->GetPublicUserAgentHeader();
            slist = curl_slist_append(slist, uaHeader);
            NN_RESULT_THROW_UNLESS(slist, ResultCurlSlistAppendFailure());
#ifdef NN_NIM_TRACE_HTTP_REQUEST_HEADER
            NN_DETAIL_NIM_TRACE("[HttpConnection]     %s\n", uaHeader);
#endif
        }
        for (int i = 0; i < headerCount; i++)
        {
#ifdef NN_NIM_TRACE_HTTP_REQUEST_HEADER
            NN_DETAIL_NIM_TRACE("[HttpConnection]     %s\n", headerList[i]);
#endif
            auto temp = curl_slist_append(slist, headerList[i]);
            NN_RESULT_THROW_UNLESS(temp, ResultCurlSlistAppendFailure());
            slist = temp;
        }
        NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_HTTPHEADER, slist));

        HeaderCallbackHolder headerCallbackHolder(headerCallback, this);
        NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_HEADERDATA, headerCallback ? &headerCallbackHolder : nullptr));
        NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, CurlHeaderCallback));

        WriteCallbackHolder writeCallbackHolder(writeCallback, this);
        NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_WRITEDATA, writeCallback ? &writeCallbackHolder : nullptr));
        NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, writeCallback ? CurlWriteCallback : nullptr));
        NN_NIM_DO_CURL(curl_easy_setopt(curl, CURLOPT_NOBODY, writeCallback ? 0 : true));

        auto curlResult = curl_easy_perform(curl);
        NN_RESULT_THROW_UNLESS(!m_IsCanceled, ResultHttpConnectionCanceled());
        NN_RESULT_THROW_UNLESS(!m_IsTimeout, ResultHttpConnectionTimeout());
        NN_RESULT_DO(m_LastResult);
        NN_RESULT_DO(GetDetailIfSslCurlError(curl, curlResult));
        NN_NIM_DO_CURL(curlResult);

        int responseCode;
        NN_NIM_DO_CURL(curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &responseCode));

        NN_DETAIL_NIM_TRACE("[HttpConnection] Response %s %d\n", url, responseCode);
        m_LastStatusCode = responseCode;
        NN_RESULT_DO(ToResultFromHttpStatusCode(responseCode));

        NN_RESULT_SUCCESS;
    }

    void HttpConnection::ResetProgress(int timeoutSecond) NN_NOEXCEPT
    {
        m_LastResult = ResultSuccess();
        m_LastStatusCode = 200;
        m_IsTimeout = false;
        m_LastDownloaded = 0;
        m_LastUploaded = 0;
        m_LastProgressChangedTime = os::GetSystemTick().ToTimeSpan();
        m_TimeoutSecond = timeoutSecond;
    }

    void HttpConnection::OnProgress(int64_t, int64_t  dlnow, int64_t, int64_t ulnow) NN_NOEXCEPT
    {
        auto currentTime = os::GetSystemTick().ToTimeSpan();

        if (m_LastDownloaded != dlnow || m_LastUploaded != ulnow)
        {
            m_LastDownloaded = dlnow;
            m_LastUploaded = ulnow;
            m_LastProgressChangedTime = currentTime;
        }
        else if (currentTime - m_LastProgressChangedTime >= TimeSpan::FromSeconds(m_TimeoutSecond))
        {
            m_IsTimeout = true;
        }
    }
    const char* HttpConnection::GetUrl() NN_NOEXCEPT
    {
        if (m_Impl)
        {
            return GetCurlStringInfo(m_Impl, CURLINFO_EFFECTIVE_URL);
        }
        else
        {
            return nullptr;
        }
    }



    //!----------------------------------------------------------------------------
    //! @name パラメータ分散設定方式 Perform
    //! @{
    void HttpConnection::TransactionHandle::Reset() NN_NOEXCEPT
    {
        auto* const pSlist = reinterpret_cast<curl_slist*>(pHeaders);
        if (nullptr != pSlist)
        {
            pHeaders = nullptr;
            curl_slist_free_all(pSlist);
        }
        pCustomMethod[0] = '\0';
        hasPostField = false;
        isHttpsUrl = false;
    }

    Result HttpConnection::BeginTransaction(TransactionHandle* pHandle, bool withResetCancel) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHandle);
        pHandle->Reset();
        if (withResetCancel)
        {
            ResetCancel();
        }
        NN_RESULT_SUCCESS;
    }

    Result HttpConnection::SetTransactionUrl(TransactionHandle* pHandle, const char* pUrl) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHandle);
        NN_SDK_REQUIRES_NOT_NULL(pUrl);
        NN_RESULT_THROW_UNLESS(!m_IsCanceled, ResultHttpConnectionCanceled());

        CURL* const pCurl = m_Impl;
        NN_RESULT_THROW_UNLESS(pCurl, ResultCurlEasyInitFailure());

        pHandle->isHttpsUrl = IsHttps(pUrl);
        NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_URL, pUrl));
        NN_RESULT_SUCCESS;
    }

    Result HttpConnection::SetTransactionHeader(TransactionHandle* pHandle, const char* pHeader) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHandle);
        NN_SDK_REQUIRES_NOT_NULL(pHeader);
        NN_RESULT_THROW_UNLESS(!m_IsCanceled, ResultHttpConnectionCanceled());

        auto pSlist = curl_slist_append(reinterpret_cast<curl_slist*>(pHandle->pHeaders), pHeader);
        NN_RESULT_THROW_UNLESS(pSlist, ResultCurlSlistAppendFailure());
        pHandle->pHeaders = pSlist;
        NN_RESULT_SUCCESS;
    }

    Result HttpConnection::SetTransactionPostFields(TransactionHandle* pHandle, const char* pFields, const int64_t size, bool isClone) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHandle);
        NN_RESULT_THROW_UNLESS(!m_IsCanceled, ResultHttpConnectionCanceled());

        CURL* const pCurl = m_Impl;
        NN_RESULT_THROW_UNLESS(pCurl, ResultCurlEasyInitFailure());

        if (nullptr != pFields)
        {
            const auto postSize = static_cast<curl_off_t>(size);
            NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_POST, 1));
            NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_POSTFIELDSIZE_LARGE, postSize));
            NN_NIM_DO_CURL(curl_easy_setopt(pCurl, isClone ? CURLOPT_COPYPOSTFIELDS : CURLOPT_POSTFIELDS, pFields));
            pHandle->hasPostField = true;
        }
        else
        {
            pHandle->hasPostField = false;
        }
        NN_RESULT_SUCCESS;
    }

    Result HttpConnection::SetTransactionCustomMethod(TransactionHandle* pHandle, const char* pMethod) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHandle);
        NN_RESULT_THROW_UNLESS(!m_IsCanceled, ResultHttpConnectionCanceled());
        auto* pOut = pHandle->pCustomMethod;
        if (nullptr != pMethod)
        {
            std::strncpy(pOut, pMethod, sizeof(pHandle->pCustomMethod));
            pOut = &pOut[sizeof(pHandle->pCustomMethod) - 1];   // strlen(pMethod) >= sizeof(pHandle->pCustomMethod)時対策
        }
        *pOut = '\0';
        NN_RESULT_SUCCESS;
    }

    Result HttpConnection::EndTransaction(TransactionHandle* pHandle, util::optional<HeaderCallback> headerCallback, util::optional<WriteCallback> writeCallback, int timeoutSecond) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHandle);
        NN_RESULT_THROW_UNLESS(!m_IsCanceled, ResultHttpConnectionCanceled());

        CURL* const pCurl = m_Impl;
        NN_RESULT_THROW_UNLESS(pCurl, ResultCurlEasyInitFailure());

        ResetProgress(timeoutSecond);
        NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_XFERINFODATA, this));
        NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_XFERINFOFUNCTION, CurlProgressCallback));
        NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_NOPROGRESS, 0));

        // メソッド依存設定値最終調整
        if (false == pHandle->hasPostField)
        {
            NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_POST, 0));
            if (pHandle->HasCustomMethod())
            {
                // POSTデータ指定が無い && カスタムメソッドの場合は、POSTフィールドは転送対象外のため処分する。
                NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_POSTFIELDSIZE_LARGE, static_cast<curl_off_t>(-1LL)));
                NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_POSTFIELDS, nullptr));
            }
        }
        // 直前のメソッド指定に関わらず null 以外の文字列が渡されればメソッドはカスタムが優先される。
        const auto pCustom = (pHandle->HasCustomMethod()) ? pHandle->pCustomMethod : nullptr;
        NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_CUSTOMREQUEST, pCustom));

        // リクエストヘッダ準備
        NN_UTIL_SCOPE_EXIT{pHandle->Reset();};
        auto* pSlist = reinterpret_cast<curl_slist*>(pHandle->pHeaders);

        // 固定 UserAgent 追加
        const auto pDeviceContext = m_DeviceContext;
        if (pDeviceContext)
        {
            auto uaHeader = (pHandle->isHttpsUrl)
                ? pDeviceContext->GetPrivateUserAgentHeader()
                : pDeviceContext->GetPublicUserAgentHeader();
            pSlist = curl_slist_append(pSlist, uaHeader);
            NN_RESULT_THROW_UNLESS(pSlist, ResultCurlSlistAppendFailure());
            pHandle->pHeaders = pSlist;
        }
        NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_HTTPHEADER, pSlist));

        HeaderCallbackHolder headerCallbackHolder(headerCallback, this);
        NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_HEADERDATA, headerCallback ? &headerCallbackHolder : nullptr));
        NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_HEADERFUNCTION, CurlHeaderCallback));

        WriteCallbackHolder writeCallbackHolder(writeCallback, this);
        NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_WRITEDATA, writeCallback ? &writeCallbackHolder : nullptr));
        NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_WRITEFUNCTION, writeCallback ? CurlWriteCallback : nullptr));
        NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_NOBODY, writeCallback ? 0 : true));

        auto curlResult = curl_easy_perform(pCurl);

        // ポストデータ解放
        NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_POSTFIELDSIZE_LARGE, static_cast<curl_off_t>(-1LL)));
        NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_CUSTOMREQUEST, nullptr));
        NN_NIM_DO_CURL(curl_easy_setopt(pCurl, CURLOPT_POSTFIELDS, nullptr));

        // 結果チェック
        NN_RESULT_THROW_UNLESS(!m_IsCanceled, ResultHttpConnectionCanceled());
        NN_RESULT_THROW_UNLESS(!m_IsTimeout, ResultHttpConnectionTimeout());
        NN_RESULT_DO(m_LastResult);
        NN_RESULT_DO(GetDetailIfSslCurlError(pCurl, curlResult));
        NN_NIM_DO_CURL(curlResult);

        int responseCode;
        NN_NIM_DO_CURL(curl_easy_getinfo(pCurl, CURLINFO_RESPONSE_CODE, &responseCode));

        m_LastStatusCode = responseCode;
        NN_RESULT_DO(ToResultFromHttpStatusCode(responseCode));

        NN_RESULT_SUCCESS;
    }

    void HttpConnection::CleanTransaction(TransactionHandle* pHandle) NN_NOEXCEPT
    {
        // 本関数は以下目的で利用します。
        //  BeginTransaction ～ EndTransaction 間が分散要求が中断された場合、
        //  Curl が内部確保しているヒープメモリリソースのうち、解放可能なリソースを解放する。
        CURL* const pCurl = m_Impl;
        if (pCurl)
        {
            curl_easy_setopt(pCurl, CURLOPT_POSTFIELDSIZE_LARGE, static_cast<curl_off_t>(-1LL));
            curl_easy_setopt(pCurl, CURLOPT_CUSTOMREQUEST, nullptr);
            curl_easy_setopt(pCurl, CURLOPT_POSTFIELDS, nullptr);
        }
        if (pHandle)
        {
            pHandle->Reset();
        }
    }
    //! @}
    //!----------------------------------------------------------------------------

}}}
