﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_FormatString.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/factory/settings_DeviceCertificate.h>

#include "nim_DynamicRightsDownloadETickets.h"
#include "nim_StringUtil.h"

namespace nn { namespace nim { namespace srv {
namespace DynamicRights {

//-----------------------------------------------------------------------------
namespace {
//-----------------------------------------------------------------------------
// デバッグコード
#if !defined(NN_SDK_BUILD_RELEASE)
#define DEBUG_TRACE(...) NN_DETAIL_NIM_TRACE( "[DynamicRights::DownloadETickets] " __VA_ARGS__ )
#else
#define DEBUG_TRACE(...)    static_cast<void>(0)
#endif
//-----------------------------------------------------------------------------

//! @brief      レスポンスアダプタ。
//! @details    各要素の型は http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=325012215 に基づきます。
class ETicketTokenAdaptor : public HttpJson::Canceler
{
    typedef HttpJson::EntryLookup<4, 48>            EntryLookup;
    typedef HttpJson::ValueStore<bool, EntryLookup> StoreToken;

    struct StoreResponse
    {
        StoreToken  token;

        bool CreateLookupPath() NN_NOEXCEPT
        {
            return token.CreateLookupPath("$.eticket_token");
        }

        void Reset() NN_NOEXCEPT
        {
            token.Reset(false);
        }
    };

public:
    typedef EntryLookup::JsonPathType   JsonPathType;

    explicit ETicketTokenAdaptor() NN_NOEXCEPT
    {
        m_Store.Reset();
        NN_ABORT_UNLESS(m_Store.CreateLookupPath());
    }

    //! @brief      内部異常値の取得。
    //!
    //! @details    本クラスインスタンスをキャンセラ登録する事で内部異常が発生した場合には、@ref ::nn::http::json::ImportJsonByRapidJson() の返値に http::ResultCanceled が返ります。@n
    //!             その場合、本メソッドで異常値を取得し適切に処理してください。@n
    //!             異常状態が発生していない場合は、ResultSuccess が返されます。
    Result GetFailure() const NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }

    //! @brief      動的チケットトークン要素の発見結果を返します。
    bool IsFound() const NN_NOEXCEPT
    {
        return m_Store.token.GetValue();
    }

    // パース中に文字列値をもつJSONパスが見つかった時に呼ばれます。
    void Update(const JsonPathType& jsonPath, const char* pValue, int valueLength) NN_NOEXCEPT
    {
        // 見つかったら一応 true 書き込み。
        NN_ABORT_UNLESS_RESULT_SUCCESS(HttpJson::EntryStoreUtil::FilterIgnorable(m_Store.token.Store(jsonPath, true)));
        NN_UNUSED(valueLength);
        NN_UNUSED(pValue);
    }

    // 以下検出対象外
    void NotifyObjectBegin(const JsonPathType&) NN_NOEXCEPT {}
    void NotifyObjectEnd(const JsonPathType&) NN_NOEXCEPT {}
    void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT {}
    void Update(const JsonPathType&, bool) NN_NOEXCEPT {}
    void Update(const JsonPathType&, double) NN_NOEXCEPT {}
    void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT {}
    void Update(const JsonPathType&, int64_t ) NN_NOEXCEPT {}

    // 内部処理で失敗があったらパースをキャンセルする。
    virtual bool IsCancelled() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

private:
    StoreResponse   m_Store;
};


//-----------------------------------------------------------------------------
}   // ~unnamed
//-----------------------------------------------------------------------------

//!============================================================================
//! @name ETicketsTokenProfile 実装
//-----------------------------------------------------------------------------
ETicketsTokenProfile::ETicketsTokenProfile(char* pOutTokenBuffer, size_t tokenBufferCapacity, const ::nn::account::Uid& uid, TemporaryFileStore* pELicenseIdsStore) NN_NOEXCEPT
    : m_pTokenBuffer(pOutTokenBuffer)
    , m_pValueStore(pELicenseIdsStore)
    , m_User(uid)
    , m_StartIndex(0u)
    , m_RangeCount(0u)
    , m_TokenBufferCapacity(tokenBufferCapacity)
{
    NN_SDK_ASSERT_NOT_NULL(pOutTokenBuffer);
    NN_SDK_ASSERT_NOT_NULL(pELicenseIdsStore);
    NN_SDK_ASSERT_GREATER(tokenBufferCapacity, 0u);
    m_RangeCount = static_cast<size_t>(pELicenseIdsStore->GetWrittenSize() / sizeof(::nn::es::ELicenseId));
}

//-----------------------------------------------------------------------------
Result ETicketsTokenProfile::RequestRange(size_t index, size_t count) NN_NOEXCEPT
{
    // eLicense データの有効性確認。
    const auto& valueStore = *m_pValueStore;
    const auto nLicenses = static_cast<size_t>(valueStore.GetWrittenSize() / sizeof(::nn::es::ELicenseId));
    NN_RESULT_THROW_UNLESS(nLicenses > 0U, ResultContentNotFound());

    // 要求範囲チェック。
    NN_RESULT_THROW_UNLESS((index + count) <= nLicenses, ResultContentNotFound());

    m_StartIndex = index;
    m_RangeCount = count;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ETicketsTokenProfile::OnQueryAccessProfile(AccessProfile* pOut, char* pOutPathUrl, size_t availablePathUrlCapacity) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);

    constexpr char base[] = "/v1/elicenses/eticket_token";
    NN_ABORT_UNLESS(sizeof(base) <= availablePathUrlCapacity && availablePathUrlCapacity <= static_cast<size_t>(std::numeric_limits<int>::max()));

    const auto capacity = static_cast<int>(availablePathUrlCapacity);
    NN_ABORT_UNLESS(::nn::util::Strlcpy(pOutPathUrl, base, capacity) < capacity);

    pOut->uid = m_User;
    pOut->naId = ::nn::account::InvalidNintendoAccountId;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result ETicketsTokenProfile::OnSetupRequestBody(ConnectionType* pConnection) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConnection);

    // eLicense データの有効性確認。
    const auto& valueStore = *m_pValueStore;
    const auto startIndex = m_StartIndex;
    const auto rangeCount = m_RangeCount;

    constexpr char PostBasePrefix[] = "{\"elicense_ids\": [";
    constexpr char PostBaseSuffix[] = "]}";

    // ポストデータバッファ確保。( null終端含めてざっくり確保、35 は "\"32桁\"," の文字数 )
    const size_t requiredPostSize = sizeof(PostBasePrefix) + sizeof(PostBaseSuffix) + static_cast<size_t>(35 * rangeCount);
    NN_RESULT_DO(m_Post.Acquire(sizeof(Bit64), requiredPostSize));
    const auto pOutPost = static_cast<char*>(m_Post.pTop);

    size_t postLength = 0;
    NN_RESULT_DO(AppendRequestBodyAsELicenses(&postLength, pOutPost, requiredPostSize, valueStore, startIndex, rangeCount, PostBasePrefix, PostBaseSuffix));
    DEBUG_TRACE("Post filed generate complete, (heap: %zu, length: %zu)=>`%s`\n", requiredPostSize, postLength, pOutPost);
    NN_SDK_ASSERT(postLength == std::strlen(pOutPost)); // NOTE: デバッグ用の保険検証
    NN_RESULT_DO(pConnection->SetTransactionPostFields(pOutPost));

    // "Content-Length", "Content-Type" ヘッダ。
    char data[96];
    NN_ABORT_UNLESS(::nn::util::SNPrintf(data, sizeof(data), "Content-Length:%zu", postLength) < sizeof(data));
    NN_RESULT_DO(pConnection->SetTransactionHeader(data));
    NN_RESULT_DO(pConnection->SetTransactionHeader("Content-Type:application/json"));
    NN_RESULT_SUCCESS;
}
//-----------------------------------------------------------------------------
Result ETicketsTokenProfile::OnResolveResponse(InputStream* pInputStream) NN_NOEXCEPT
{
    // 内部確保はダミーで 4byte。
    // 改めて StringBuffer を出力先のトークンバッファ領域に再設定。
    HttpJson::Stream::ProxyInputStream<InputStream, 4> stream(pInputStream);
    stream.SetStringBuffer(m_pTokenBuffer, m_TokenBufferCapacity);   // 上書き

    ETicketTokenAdaptor adaptor;
    NN_RESULT_TRY(::nn::http::json::ImportJsonByRapidJson<::nn::http::json::DefaultJsonErrorMap>(adaptor, stream, static_cast<HttpJson::Canceler*>(&adaptor)))
        NN_RESULT_CATCH_CONVERT(::nn::http::ResultCanceled, adaptor.GetFailure())
    NN_RESULT_END_TRY;

    // レスポンスに "eticket_token" 要素がない or null だった場合。
    //  TODO: 適切な Result 用意すべき？
    NN_RESULT_THROW_UNLESS(adaptor.IsFound(), ResultContentNotFound());
    NN_RESULT_SUCCESS;
}

//!============================================================================
//! @name AsyncDownloadETicketsImpl 実装
//
// NOTE:
//  Shimレイヤの nim::AsyncResult の Get() アクセスにおいて Wait() される事を前提想定した実装です。
//  非同期に Get() を呼び出す要件になった場合は排他が必要です。
//-----------------------------------------------------------------------------
AsyncDownloadETicketsImpl::AsyncDownloadETicketsImpl() NN_NOEXCEPT
    : AsyncBase(this, GetSharedThreadAllocator()), Executor()
    , m_pDeviceAccountStore(nullptr)
    , m_User(::nn::account::InvalidUid)
{
}

//-----------------------------------------------------------------------------
AsyncDownloadETicketsImpl::~AsyncDownloadETicketsImpl() NN_NOEXCEPT
{
    Join();
}

//-----------------------------------------------------------------------------
Result AsyncDownloadETicketsImpl::Initialize(DeviceContext* pDeviceContext
    , DeviceAccountStore* pDeviceAccountStore
    , const ::nn::account::Uid& uid
    , const ::nn::sf::InArray<::nn::es::ELicenseId>& elicenseIds
    , const char* pTemporaryFilePath) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pDeviceAccountStore);
    NN_ABORT_UNLESS_NOT_NULL(pDeviceContext);

    // 不正なパラメタチェック
    const auto nLicenses = elicenseIds.GetLength();
    NN_RESULT_THROW_UNLESS(nLicenses > 0, ResultContentNotFound());
    NN_RESULT_DO(Executor::Initialize(pDeviceContext));

    m_User = uid;
    m_ValueStore.Initialize(pTemporaryFilePath);
    m_pDeviceAccountStore = pDeviceAccountStore;
    m_pDeviceContext = pDeviceContext;

    // 入力 elicenseIds をファイルへ
    m_ValueStore.SetInitialFileSize(nLicenses * sizeof(::nn::es::ELicenseId));
    NN_RESULT_DO(m_ValueStore.Append(elicenseIds.GetData(), nLicenses * sizeof(::nn::es::ELicenseId)));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncDownloadETicketsImpl::Execute() NN_NOEXCEPT
{
    return DragonsAsyncAccessTaskBase::RequestTaskFinished(OnExecute());
}

//-----------------------------------------------------------------------------
Result AsyncDownloadETicketsImpl::OnExecute() NN_NOEXCEPT
{
    const auto nLicenses = static_cast<size_t>(m_ValueStore.GetWrittenSize() / sizeof(::nn::es::ELicenseId));
    NN_RESULT_THROW_UNLESS(nLicenses > 0u, ResultContentNotFound());
    DEBUG_TRACE("Download request eLicense count ( %zu ).\n", nLicenses);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

    HeapUtil::OnetimeHeapSession work;
    NN_RESULT_DO(work.Acquire(sizeof(Bit32), 32 * 1024));    // とりあえず 32 KiB.

    EciAccessor::DynamicETicketsContext ticketContext;
    const size_t requestCount = ticketContext.Initialize(static_cast<char*>(work.pTop), work.size);
    NN_RESULT_THROW_UNLESS(requestCount > 0u, ResultAllocationMemoryFailed());
    DEBUG_TRACE("Expect available count for the eTicket download ( %zu ), worksize( %zu byte ).\n", requestCount, work.size);

    // ETicketsTokenProfile に eTicketToken 受け取りバッファ設定。
    //  NOTE:
    //      内部で JSONパース時のバッファとしても利用します。
    //      "eticket_token" 要素のみのレスポンス前提の運用です。
    //      ロジックとしては以下の順で設定バッファに書き込みが発生する。
    //          1. rapidjson.Key("eticket_token") が書き込まれる。
    //          2. rapidjon.String("$token") が書き込まれる。
    //          3. 他要素ないのでバッファの中身は String() の状態になる。
    //
    //      この手法採用できないと SyncELicenses の用に OutputStream 作って、
    //      Put() で分割インポートのような仕組みが必要。
    const auto pTokenBuffer = ticketContext.GetBufferForETicketToken();
    const auto tokenBufferSize = ticketContext.GetCapacityForETicketToken();
    ETicketsTokenProfile token(pTokenBuffer, tokenBufferSize, m_User, &m_ValueStore);
    EciAccessor eci(m_pDeviceContext, GetConnection()->GetImpl());

    DeviceAccountInfo info;
    NN_RESULT_DO(m_pDeviceAccountStore->Get(&info));

    for (size_t index = 0; index < nLicenses; index += requestCount)
    {
        // eTicket token 要求
        const size_t count = ((index + requestCount) <= nLicenses) ? requestCount : nLicenses - index;
        NN_RESULT_DO(token.RequestRange(index, count));
        NN_RESULT_DO(Request(&token));

        // ECI へ eTicket 要求
        // eTicket のダウンロード & インポート。
        // EciAccessor 中のキャンセルは HttpConnection::Cancel() の呼び出しにより行われる。
        NN_RESULT_DO(eci.RequestDynamicETickets(ticketContext, info.id, info.token, pTokenBuffer));
    }

#endif  // defined(NN_BUILD_CONFIG_OS_HORIZON)

    NN_RESULT_SUCCESS;
}


}   // ~DynamicRights
}}} // ~nn::nim::srv
