﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nim/nim_DynamicRightsApi.h>
#include <nn/nim/srv/nim_DeviceContext.h>
#include <nn/nim/srv/nim_HttpConnection.h>
#include <nn/nim/srv/nim_ThreadAllocator.h>
#include <nn/result/result_HandlingUtility.h>

#include "nim_AsyncImpl.h"
#include "nim_HeapUtil.h"
#include "nim_TokenStore.h"
#include "nim_HttpJsonUtil.h"

namespace nn { namespace nim { namespace srv {

namespace DynamicRights {

//----------------------------------------------------------------------------
namespace Dragons {

//! @brief  Dragons サーバー $reason 定数。
enum class Reason : uint8_t
{
    Unknown,            //!< 無効化された理由が不明です
    LimitExceeded,      //!< "limit_exceeded":      発行可能ライセンス数を超過している
    NoRights,           //!< "no_rights":           権利を持っていない
    NotReleased,        //!< "not_released":        権利を持っているがリリース前
    Expired,            //!< "expired":             期限が切れました
    UsedByAnother,      //!< "used_by_another":     他デバイスでライセンス利用中
    NotDeviceLinked,    //!< "not_device_linked":   権利を持っているが機器認証デバイスではない
};

//! @brief  Dragons サーバー $status 定数。
enum class Status : uint8_t
{
    Active,             //!< "active":              ライセンス利用可能
    InactiveOnServer,   //!< "inactive_on_server":  Dragons上は無効化されているが NX に同期されていない状態
    Inactive,           //!< "inactive":            ライセンス利用不可
};

//! @brief  Dragons サーバー $elicense_type 定数。
enum class ELicenseType : uint8_t
{
    Unknown,
    Temporary,              //!< "temporary":               有効期限が設けられている一時的なライセンス
    Permanent,              //!< "permanent":               永続的に利用可能なライセンス
    DeviceLinkedPermanent,  //!< "device_linked_permanent": 機器認証デバイスに付与される永続的に利用可能なライセンス
    Unavailable,            //!< "unavailable":             ライセンス利用不可(利用可否問い合わせ API でのみ使用)
};

//----------------------------------------------------------------------------
// @note    レスポンス文字列から Dragons 定義の列挙子に変換する関数名は以下のルールで作成します。
//          "Parse" + 変換先列挙子型名 + "From" (文字列引数)

//! @brief      ライセンスタイプ変換。
//! @param[in]  pSource         サーバー解釈可能なライセンスタイプ文字列を指定します。
//! @param[in]  sourceLength    @a pSource に指定した文字列の null終端を除く文字列長を指定します。
ELicenseType ParseELicenseTypeFrom(const char* const pSource, const int sourceLength) NN_NOEXCEPT;

//! @brief      ライセンス剥奪理由変換。
//! @param[in]  pSource         サーバー解釈可能なライセンスタイプ文字列を指定します。
//! @param[in]  sourceLength    @a pSource に指定した文字列の null終端を除く文字列長を指定します。
Reason ParseReasonFrom(const char* const pSource, const int sourceLength) NN_NOEXCEPT;

}   // ~Dragons

//----------------------------------------------------------------------------
// @note    特定引数構成から列挙子へ変換する関数名は以下のルールで作成します。
//          "Get" + 変換先列挙子型名 + "From" (変換元引数構成)

//! @brief      nim API 用ライセンスタイプ変換。
//! @return     Dragonsサーバー値を nim API 層の値に変換します。
nim::ELicenseType GetELicenseTypeFrom(Dragons::ELicenseType type) NN_NOEXCEPT;

//! @brief      nim API 用発行拒否理由変換。
//! @return     Dragonsサーバー値を nim API 層の値に変換します。
nim::ELicenseStatus GetELicenseStatusFrom(bool isAvailable, Dragons::ELicenseType type, Dragons::Reason reason) NN_NOEXCEPT;

//! @brief      nim API 用剥奪理由変換。
//! @return     Dragonsサーバー値を nim API 層の値に変換します。
nim::RevokeReason GetRevokeReasonFrom(Dragons::Reason reason) NN_NOEXCEPT;

//----------------------------------------------------------------------------
// @note    列挙子・定数から文字列へ変換する関数名は以下のルールで作成します。
//          "GetStringFrom" (オーバーロード可能な変換元定数型)

//! @brief      nim API 用ライセンスタイプ文字列変換。
//! @return     null終端付き、サーバー解釈可能なライセンスタイプ文字列を返します。
const char* GetStringFrom(const ELicenseType& type) NN_NOEXCEPT;

//----------------------------------------------------------------------------
//! @brief  スレッドアロケータ。
ThreadAllocator* GetSharedThreadAllocator() NN_NOEXCEPT;

/**
 * @brief   トークン取得処理キャンセルハンドル付き通信コンテキスト実装。
 */
class HttpConnectionContext : private HttpConnection
{
    NN_DISALLOW_COPY(HttpConnectionContext);
    NN_DISALLOW_MOVE(HttpConnectionContext);

public:
    typedef HttpConnection::HeaderCallback  HeaderCallback;
    typedef HttpConnection::WriteCallback   WriteCallback;
    typedef TokenStore::CancelerSet         TokenCanceler;

    //! @brief  コンストラクタ。
    HttpConnectionContext() NN_NOEXCEPT;

    //! @brief  デストラクタ。
    ~HttpConnectionContext() NN_NOEXCEPT;

    //! @brief  初期化。
    Result Initialize(DeviceContext* pDeviceContext) NN_NOEXCEPT;

    //! @brief  中断確認。
    NN_FORCEINLINE bool IsCanceled() const NN_NOEXCEPT
    {
        return HttpConnection::IsCanceled();
    }

    //! @brief  中断要求。
    void Cancel() NN_NOEXCEPT;

    //! @brief  エラーコンテキスト取得。
    Result GetErrorContext(::nn::err::ErrorContext* pOutValue) NN_NOEXCEPT;

    //!----------------------------------------------------------------------------
    //! @name パラメータ分散設定型インタフェースラッパー。
    //! @{
    Result BeginTransaction() NN_NOEXCEPT;
    Result SetTransactionUrl(const char* pUrl) NN_NOEXCEPT;
    Result SetTransactionHeader(const char* pHeader) NN_NOEXCEPT;
    Result SetTransactionPostFields(const char* pFields, const int64_t size, bool isClone) NN_NOEXCEPT;
    NN_FORCEINLINE Result SetTransactionPostFields(const char* pFields) NN_NOEXCEPT
    {
        return SetTransactionPostFields(pFields, -1, false);
    }
    Result SetTransactionCustomMethod(const char* pMethod) NN_NOEXCEPT;
    Result EndTransaction(::nn::util::optional<HeaderCallback> headerCallback, ::nn::util::optional<WriteCallback> writeCallback, int timeoutSecond) NN_NOEXCEPT;
    void CleanTransaction() NN_NOEXCEPT;
    //! @}
    //!----------------------------------------------------------------------------

    //! @brief  トークン取得キャンセラ受領コンテナ参照の取得。
    NN_FORCEINLINE TokenCanceler& GetTokenCanceler() NN_NOEXCEPT
    {
        return m_TokenCanceler;
    }

    //! @brief  @ref nim::srv::HttpConnection の本体実装を取得します。
    NN_FORCEINLINE HttpConnection* GetImpl() NN_NOEXCEPT
    {
        return static_cast<HttpConnection*>(this);
    }

private:
    TokenCanceler       m_TokenCanceler;    //!< 認証トークン取得キャンセル用。
    TransactionHandle   m_Transaction;      //!< 分散通信要素設定用。
};

/**
 * @brief    Dragons 非同期アクセスタスクベース。
 */
class DragonsAsyncAccessTaskBase
{
    NN_DISALLOW_COPY(DragonsAsyncAccessTaskBase);
    NN_DISALLOW_MOVE(DragonsAsyncAccessTaskBase);

public:
    typedef HttpJson::Stream::Base::MemoryInput<>   InputStream;
    typedef HttpConnectionContext                   ConnectionType;

    //! @brief  デフォルトタイムアウト ( 秒 )
    //! @note   constexpr による定数化を行っているため、参照渡し引数などで利用する場合は一時変数へ格納してください。
    constexpr static int DefaultTimeOutSeconds = 60;

    struct AccessProfile
    {
        ::nn::account::Uid                  uid;        //!< 通信要求 account::Uid ( Invalid の場合は使用しません )
        ::nn::account::NintendoAccountId    naId;       //!< 通信要求 NintendoAccountId ( Invalid の場合は使用しません )
        int                                 timeout;    //!< 通信タイムアウト( 秒 ), デフォルト値は DefaultTimeOutSeconds 推奨。

        explicit AccessProfile() NN_NOEXCEPT
            : uid(::nn::account::InvalidUid)
            , naId(::nn::account::InvalidNintendoAccountId)
            , timeout(DefaultTimeOutSeconds)
        {}
    };

    class RequestProfileHolder
    {
    public:
        virtual Result OnQueryAccessProfile(AccessProfile* pOut, char* pOutPathUrl, size_t availablePathUrlCapacity) NN_NOEXCEPT = 0;

        virtual Result OnSetupRequestBody(ConnectionType* pConnection) NN_NOEXCEPT = 0;

        virtual Result OnResolveResponse(InputStream* pInputStream) NN_NOEXCEPT = 0;
    };

    Result Initialize(DeviceContext* pDeviceContext) NN_NOEXCEPT;

    Result Cancel() NN_NOEXCEPT;

protected:
    DragonsAsyncAccessTaskBase() NN_NOEXCEPT;

    //! @brief      プロファイルに基づいた同期通信実施を要求します。
    Result Request(RequestProfileHolder* pRequestProfile) NN_NOEXCEPT;

    //! @brief      タスクが持つ通信コネクションのエラーコンテキストを取得します。
    //! @details    @ref Request(RequestProfileHolder*) の要求に伴って上書きされます。
    Result GetErrorContext(::nn::err::ErrorContext* pOutValue) NN_NOEXCEPT;

    //! @brief      タスク終了状態への移行を要求します。
    //! @note       単一通信の結果に依存するタスクであれば、@ref Request(RequestProfileHolder*) の返値をそのまま設定できます。
    Result RequestTaskFinished(Result resultOfTask) NN_NOEXCEPT;

    //! @brief      タスクの処理結果状態を取得します。
    //! @retval     nn::nim::ResultTaskStillRunning     タスク処理中です。
    Result GetResult() const NN_NOEXCEPT;

    //! @brief      タスクの処理継続確認を行います。
    //! @return     nn::ResultSuccess 以外の場合は、該当 Result の理由により処理継続不可です。@n
    //!             但し、タスクの全処理が期待通り正常終了し、処理結果が nn::ResultSuccess の場合も nn::ResultSuccess が返されます。@n
    //!             タスクが処理を終了しているか確認する場合は @ref GetResult() が @ref nn::nim::ResultTaskStillRunning でない事を確認してください。
    //! @note       本メソッドはタスク処理中の呼び出しを想定しています。
    Result CanContinue() NN_NOEXCEPT;

    //! @brief      タスクが持つ HTTP通信コンテキストハンドルを取得します。
    NN_FORCEINLINE ConnectionType* GetConnection() NN_NOEXCEPT
    {
        return &m_Connection;
    }

private:
    Result RegisterSystemRequestHeader(ConnectionType* pConnection, const AccessProfile& profile) NN_NOEXCEPT;

    Result Connect(size_t* const pOutReceivedSize, HeapUtil::OnetimeHeapSession* const pOutResponse, RequestProfileHolder* pRequestProfile, const char* pUrl, const AccessProfile& profile) NN_NOEXCEPT;

    //! @brief  非同期アクセスタスクステータスコンテナ
    struct ResponseValue
    {
        Result  result; //!< 処理結果保存コンテナ

        NN_IMPLICIT ResponseValue() NN_NOEXCEPT;
        NN_IMPLICIT ResponseValue(const Result result_) NN_NOEXCEPT;
    };
    NN_STATIC_ASSERT(sizeof(ResponseValue) == 4);
    typedef detail::MutexAtomic<ResponseValue> ResponseValueAtomic;

    ConnectionType      m_Connection;
    ResponseValueAtomic m_ResponseValue;
};

/**
 * @brief   Dragons 単一 REST API アクセスタスクベース。
 */
class DragonsAccessAsyncImpl : public DragonsAsyncAccessTaskBase, private DragonsAsyncAccessTaskBase::RequestProfileHolder
{
    NN_DISALLOW_COPY(DragonsAccessAsyncImpl);
    NN_DISALLOW_MOVE(DragonsAccessAsyncImpl);

public:
    DragonsAccessAsyncImpl() NN_NOEXCEPT {}

    Result Execute() NN_NOEXCEPT;
};



/**
 * @brief   追記型テンポラリファイルストア。
 */
class TemporaryFileStore
{
    typedef ::nn::util::optional<::nn::fs::FileHandle>  FileHandle;
    typedef ::nn::kvdb::BoundedString<64>               Path;

public:
    typedef int64_t FileSizeType;

    explicit TemporaryFileStore() NN_NOEXCEPT;
    ~TemporaryFileStore() NN_NOEXCEPT;

    void Initialize(const char* pPath) NN_NOEXCEPT;

    //! @brief      書き込み位置を先頭に戻します。
    void Rewind() NN_NOEXCEPT;

    //! @brief      ファイル生成に適用される初期ファイルサイズを指定します。
    //! @details    ファイルは初回 @ref Append() 時に生成されますので、@ref Append() が呼び出されるまでの設定のみが有効になります。@n
    //!             初期設定値はゼロです。@n
    //!             追記モードでのファイルアクセスのため、ファイル上限ではない点に注意してください。
    void SetInitialFileSize(const FileSizeType fileSize) NN_NOEXCEPT;

    Result Append(const void* pSource, size_t writeSize) NN_NOEXCEPT;
    Result Read(size_t* pOutReadedSize, int64_t offset, void* pOutData, size_t readSize) const NN_NOEXCEPT;
    Result Read(int64_t offset, void* pOutData, size_t readSize) const NN_NOEXCEPT;

    //! @brief      書き込み済サイズを取得します。
    //! @details    ファイルサイズ値ではない点に注意してください。
    NN_FORCEINLINE uint64_t GetWrittenSize() const NN_NOEXCEPT
    {
        return m_Written;
    }

private:
    FileHandle  m_File;
    Path        m_Path;
    uint64_t    m_Written;
    int64_t     m_InitialFileSize;
};

/**
 * @brief       Dragons エラーレスポンスアダプタ。
 * @details     JSON要素は基本的には RFC7807 Problem Details for HTTP APIs に基づきますが、
 *              実質的な要素は Dragons の仕様に依存して "type" 要素のみが対象となります。
 */
class DragonsErrorResponseAdaptor
{
    typedef HttpJson::EntryLookup<6, 48>                EntryLookup;
    typedef HttpJson::ValueStore<Result, EntryLookup>   StoreResult;

public:
    typedef EntryLookup::JsonPathType   JsonPathType;

    explicit DragonsErrorResponseAdaptor() NN_NOEXCEPT;

    Result GetResult(const Result& invalidCase) const NN_NOEXCEPT;

    // パース中に文字列値をもつJSONパスが見つかった時に呼ばれます。
    void Update(const JsonPathType& jsonPath, const char* pValue, int valueLength) NN_NOEXCEPT;

    // 以下検出対象外
    void Update(const JsonPathType&, int64_t) NN_NOEXCEPT {}
    void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT {}
    void NotifyObjectEnd(const JsonPathType&) NN_NOEXCEPT {}
    void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT {}
    void Update(const JsonPathType&, bool) NN_NOEXCEPT {}
    void Update(const JsonPathType&, double) NN_NOEXCEPT {}
    void NotifyObjectBegin(const JsonPathType&) NN_NOEXCEPT {}

private:
    //! @brief      "RFC7807 Problem Details for HTTP APIs" エラーレスポンスの type 要素 → エラーリザルト変換。
    //! @param[in]  pValue      `type` 要素が持つ値文字列。
    //! @param[in]  valueLength `type` 要素が持つ値文字列長( null 終端を含めない )。
    static Result ToResult(const char* pValue, int valueLength) NN_NOEXCEPT;

private:
    StoreResult m_Store;
};

//----------------------------------------------------------------------------
//! @brief      64bit値16進数16桁表記 要素配列リクエストボディ配列要素生成用ユーティリティ。
//!
//! @param[out] pOutWriteLength         出力された文字列数( null終端文字を含まない )を返します。
//! @param[out] pOutBuffer              生成された配列要素文字列が出力されるバッファを指定します。( null終端されます )
//! @param[in]  availableBufferCapacity @a pOutBuffer が利用可能な領域サイズ( バイトサイズ )を指定します。
//! @param[in]  source                  配列要素を保持するストアコンテナを指定します。
//! @param[in]  startIndex              ストアコンテナの読み取り開始位置を指定します。( 64bit値 単位 )
//! @param[in]  sourceCount             ストアコンテナの読み取り総数を指定します。( 64bit値 単位 )
//! @param[in]  pPrefix                 配列要素文字列出力の先頭に付与する文字列を指定します。( nullptr 指定時は付与されません )
//! @param[in]  pSuffix                 配列要素文字列出力の末尾に付与する文字列を指定します。( nullptr 指定時は付与されません )
//!
//! @retval     nn::ResultSuccess               処理に成功しました。
//! @retval     nn::nim::ResultBufferNotEnough  出力バッファが不足しています。
//! @retval     上記以外                        ソースストアからの読み取りに失敗しました。
//!
//! @details    ストアコンテナからの読み込みバッファはスタックから確保し、単位は 32 です。( 32 * sizeof(nn::Bit64) = 256 byte )
Result AppendRequestBodyAsBit64(size_t* pOutWriteLength, char* pOutBuffer, const size_t availableBufferCapacity, const TemporaryFileStore& source, size_t startIndex, size_t sourceCount, const char* pPrefix, const char* pSuffix) NN_NOEXCEPT;

//----------------------------------------------------------------------------
//! @brief      @ref nn::es::ELicenseId (16進数32桁表記) 要素配列リクエストボディ配列要素生成用ユーティリティ。
//!
//! @param[out] pOutWriteLength         出力された文字列数( null終端文字を含まない )を返します。
//! @param[out] pOutBuffer              生成された配列要素文字列が出力されるバッファを指定します。( null終端されます )
//! @param[in]  availableBufferCapacity @a pOutBuffer が利用可能な領域サイズ( バイトサイズ )を指定します。
//! @param[in]  source                  配列要素を保持するストアコンテナを指定します。
//! @param[in]  startIndex              ストアコンテナの読み取り開始位置を指定します。( @ref nn::es::ELicenseId 単位 )
//! @param[in]  sourceCount             ストアコンテナの読み取り総数を指定します。( @ref nn::es::ELicenseId 単位 )
//! @param[in]  pPrefix                 配列要素文字列出力の先頭に付与する文字列を指定します。( nullptr 指定時は付与されません )
//! @param[in]  pSuffix                 配列要素文字列出力の末尾に付与する文字列を指定します。( nullptr 指定時は付与されません )
//!
//! @retval     nn::ResultSuccess               処理に成功しました。
//! @retval     nn::nim::ResultBufferNotEnough  出力バッファが不足しています。
//! @retval     上記以外                        ソースストアからの読み取りに失敗しました。
//!
//! @details    ストアコンテナからの読み込みバッファはスタックから確保し、単位は 32 です。( 32 * sizeof(nn::es::ELicenseId) = 512 byte )
Result AppendRequestBodyAsELicenses(size_t* pOutWriteLength, char* pOutBuffer, const size_t availableBufferCapacity, const TemporaryFileStore& source, size_t startIndex, size_t sourceCount, const char* pPrefix, const char* pSuffix) NN_NOEXCEPT;

}   // ~DynamicRights

}}}
