﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/util/util_FormatString.h>
#include <nn/result/result_HandlingUtility.h>

#include "nim_DynamicRightsAvailableELicenses.h"
#include "nim_StringUtil.h"

namespace nn { namespace nim { namespace srv {
namespace DynamicRights {

//-----------------------------------------------------------------------------
namespace {
//-----------------------------------------------------------------------------
// デバッグコード
#if !defined(NN_SDK_BUILD_RELEASE)
#define DEBUG_TRACE(...) NN_DETAIL_NIM_TRACE( "[DynamicRights::AvailableELicenses] " __VA_ARGS__ )
#else
#define DEBUG_TRACE(...)    static_cast<void>(0)
#endif
//-----------------------------------------------------------------------------

//! @brief      結果格納要素値型。
typedef ::nn::nim::AvailableELicense    ValueType;

//! @brief      レスポンスアダプタ。
//! @details    各要素の型は http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=325012215 に基づきます。
class AvailableELicensesAdaptor : public HttpJson::Canceler
{
    typedef HttpJson::EntryLookup<6, 64>                                EntryLookup;
    typedef HttpJson::ValueStore<es::RightsId, EntryLookup>             StoreRightsId;
    typedef HttpJson::ValueStore<Bit64, EntryLookup>                    StoreAccountId;
    typedef HttpJson::ValueStore<Dragons::ELicenseType, EntryLookup>    StoreLicenseType;
    typedef HttpJson::ValueStore<Dragons::Reason, EntryLookup>          StoreReason;
    typedef HttpJson::ValueStore<bool, EntryLookup>                     StoreIsAvailable;

    struct StoreResponse
    {
        StoreAccountId      accountId;
        StoreRightsId       rightsId;
        StoreReason         reason;
        StoreLicenseType    type;
        StoreIsAvailable    isAvailable;

        bool CreateLookupPath(int index) NN_NOEXCEPT
        {
            return (
                type.CreateLookupPath("$.available_elicenses[%u].elicense_type", index) &&
                accountId.CreateLookupPath("$.available_elicenses[%u].account_id", index) &&
                rightsId.CreateLookupPath("$.available_elicenses[%u].rights_id", index) &&
                reason.CreateLookupPath("$.available_elicenses[%u].reason", index) &&
                isAvailable.CreateLookupPath("$.available_elicenses[%u].is_available", index)
            );
        }

        void Reset() NN_NOEXCEPT
        {
            accountId.Reset(account::InvalidNintendoAccountId.id);
            rightsId.Reset();
            reason.Reset(Dragons::Reason::Unknown);
            type.Reset(Dragons::ELicenseType::Unknown);
            isAvailable.Reset(false);
        }
    };

public:
    typedef unsigned int                CountType;
    typedef EntryLookup::JsonPathType   JsonPathType;

    explicit AvailableELicensesAdaptor(TemporaryFileStore* pFileStore) NN_NOEXCEPT
        : m_pFileStore(pFileStore)
        , m_Index(0)
        , m_Result(::nn::ResultSuccess())
    {
        NN_ABORT_UNLESS_NOT_NULL(pFileStore);
        pFileStore->Rewind();
        m_Store.Reset();
    }

    CountType GetCount() const NN_NOEXCEPT
    {
        return m_Index;
    }

    //! @brief      内部異常値の取得。
    //!
    //! @details    本クラスインスタンスをキャンセラ登録する事で内部異常が発生した場合には、@ref ::nn::http::json::ImportJsonByRapidJson() の返値に http::ResultCanceled が返ります。@n
    //!             その場合、本メソッドで異常値を取得し適切に処理してください。@n
    //!             異常状態が発生していない場合は、ResultSuccess が返されます。
    Result GetFailure() const NN_NOEXCEPT
    {
        return m_Result;
    }

    // パース中に文字列値をもつJSONパスが見つかった時に呼ばれます。
    void Update(const JsonPathType& jsonPath, const char* pValue, int valueLength) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(HttpJson::EntryStoreUtil::FilterIgnorable(
            m_Store.accountId.Store(jsonPath, pValue, valueLength,
            [](StoreAccountId::ValueType* pOut, const char* pValue, int) NN_NOEXCEPT->Result{
                *pOut = static_cast<Bit64>(NN_NIM_STR_TO_ULL(pValue, nullptr, 16));  // 不正ならゼロ
                NN_RESULT_SUCCESS;
            })
        ));
        NN_ABORT_UNLESS_RESULT_SUCCESS(HttpJson::EntryStoreUtil::FilterIgnorable(
            m_Store.rightsId.Store(jsonPath, pValue, valueLength,
            [](StoreRightsId::ValueType* pOut, const char* pValue, int) NN_NOEXCEPT->Result {
                *pOut = static_cast<es::RightsId>(NN_NIM_STR_TO_ULL(pValue, nullptr, 16));  // 不正ならゼロ
                NN_RESULT_SUCCESS;
            })
        ));
        NN_ABORT_UNLESS_RESULT_SUCCESS(HttpJson::EntryStoreUtil::FilterIgnorable(
            m_Store.type.Store(jsonPath, pValue, valueLength,
            [](StoreLicenseType::ValueType* pOut, const char* pValue, int length) NN_NOEXCEPT->Result{
                *pOut = Dragons::ParseELicenseTypeFrom(pValue, length);
                NN_RESULT_SUCCESS;
            })
        ));
        NN_ABORT_UNLESS_RESULT_SUCCESS(HttpJson::EntryStoreUtil::FilterIgnorable(
            m_Store.reason.Store(jsonPath, pValue, valueLength,
            [](StoreReason::ValueType* pOut, const char* pValue, int length) NN_NOEXCEPT->Result{
                *pOut = Dragons::ParseReasonFrom(pValue, length);
                NN_RESULT_SUCCESS;
            })
        ));
    }

    // パース中に bool 値をもつJSONパスが見つかった時に呼ばれます。
    void Update(const JsonPathType& jsonPath, bool value) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(HttpJson::EntryStoreUtil::FilterIgnorable(m_Store.isAvailable.Store(jsonPath, value)));
    }

    void NotifyObjectBegin(const JsonPathType& jsonPath) NN_NOEXCEPT
    {
        char path[64];
        const auto index = m_Index;
        if (::nn::util::SNPrintf(path, sizeof(path), "$.available_elicenses[%u]", index) < sizeof(path) && jsonPath.Match(path))
        {
            NN_ABORT_UNLESS(m_Store.CreateLookupPath(index));
        }
    }

    void NotifyObjectEnd(const JsonPathType& jsonPath) NN_NOEXCEPT
    {
        char path[64];
        const auto index = m_Index;
        if (::nn::util::SNPrintf(path, sizeof(path), "$.available_elicenses[%u]", index) < sizeof(path) && jsonPath.Match(path))
        {
            // ファイルへ保存。
            if (m_Result.IsSuccess())
            {
                // FAILの場合は、パースがキャンセルされる。
                m_Result = Write();
            }

            // 更新
            m_Index = index + 1;

            // ストアを再利用可能に。
            m_Store.Reset();
        }
    }

    // 以下検出対象外
    void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT {}
    void Update(const JsonPathType&, double) NN_NOEXCEPT {}

    // パース中に std::numeric_limit<int64_t>::max 以下の整数値をもつJSONパスが見つかった時に呼ばれます。
    void Update(const JsonPathType&, int64_t) NN_NOEXCEPT {}
    // パース中に std::numeric_limit<int64_t>::max より大きい整数値をもつJSONパスが見つかった時に呼ばれます。
    void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT {}

    // 内部処理で失敗があったらパースをキャンセルする。
    virtual bool IsCancelled() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Result.IsFailure();
    }

private:
    Result Write() NN_NOEXCEPT
    {
        ValueType value;
        value.naId.id = m_Store.accountId.GetValue();
        value.rightsId = m_Store.rightsId.GetValue();
        const auto elicenseType = m_Store.type.GetValue();
        value.licenseType = GetELicenseTypeFrom(elicenseType);
        // NOTE: "is_available" 要素がない場合は、$license_type が unknown, unavailable 以外なら true とする。
        const auto isAvailable = (false == static_cast<bool>(m_Store.isAvailable))
            ? (Dragons::ELicenseType::Unavailable != elicenseType && Dragons::ELicenseType::Unknown != elicenseType)
            : m_Store.isAvailable.GetValue();
        value.licenseStatus = GetELicenseStatusFrom(isAvailable, elicenseType, m_Store.reason.GetValue());
        NN_RESULT_DO(m_pFileStore->Append(&value, sizeof(value)));
        NN_RESULT_SUCCESS;
    }

private:
    TemporaryFileStore* const   m_pFileStore;
    StoreResponse               m_Store;
    CountType                   m_Index;
    Result                      m_Result;
};

//-----------------------------------------------------------------------------
}   //  ~unnamed
//-----------------------------------------------------------------------------

//!============================================================================
//! @name AsyncAvailableElicensesImpl 実装
//
// NOTE:
//  処理結果リソース( mValueStore, m_Count )は非同期処理部との排他は行っていません。
//  これは、Shimレイヤの nim::AsyncResult の Get() アクセスにおいて Wait() される事を前提想定した実装です。
//  非同期に Get() を呼び出す要件になった場合は排他が必要です。
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
AsyncAvailableELicensesImpl::AsyncAvailableELicensesImpl() NN_NOEXCEPT
    : AsyncBase(this, GetSharedThreadAllocator()), Executor()
    , m_User(::nn::account::InvalidUid)
    , m_Count(0)
{
}

//-----------------------------------------------------------------------------
AsyncAvailableELicensesImpl::~AsyncAvailableELicensesImpl() NN_NOEXCEPT
{
    Join();
};

//-----------------------------------------------------------------------------
Result AsyncAvailableELicensesImpl::Initialize(DeviceContext* pDeviceContext
    , const ::nn::account::Uid& uid
    , const ::nn::sf::InArray<::nn::es::RightsId>& rightsIds
    , const char* pTemporaryFilePath) NN_NOEXCEPT
{
    // 不正なパラメタチェック
    const auto nRights = rightsIds.GetLength();
    NN_RESULT_THROW_UNLESS(nRights > 0, ResultContentNotFound());
    NN_RESULT_DO(Executor::Initialize(pDeviceContext));

    m_User = uid;
    m_ValueStore.Initialize(pTemporaryFilePath);

    // 入力 rigths をファイルへ
    m_ValueStore.SetInitialFileSize(nRights * sizeof(::nn::es::RightsId));
    NN_RESULT_DO(m_ValueStore.Append(rightsIds.GetData(), nRights * sizeof(::nn::es::RightsId)));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncAvailableELicensesImpl::GetSize(::nn::sf::Out<std::int64_t> outValue) NN_NOEXCEPT
{
    NN_RESULT_DO(Executor::GetResult());

    *outValue = static_cast<int64_t>(sizeof(ValueType) * m_Count);
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncAvailableELicensesImpl::Read(::nn::sf::Out<std::uint64_t> outValue, int64_t offset, const ::nn::sf::OutBuffer& buffer) NN_NOEXCEPT
{
    NN_RESULT_DO(Executor::GetResult());

    *outValue = 0;

    // NOTE: 負値のインデクスオフセットは不正引数扱い。
    NN_RESULT_THROW_UNLESS(offset >= 0, ResultInvalidOffsetArgument());

    // NOTE: ValueType アライメント違反、及び 出力バッファの未指定は不正引数指定の扱い。
    auto pOutBuffer = buffer.GetPointerUnsafe();
    const uint64_t outCapacity = buffer.GetSize();
    NN_RESULT_THROW_UNLESS(nullptr != pOutBuffer && outCapacity > 0 && 0 == (offset % sizeof(ValueType)) && 0 == (outCapacity % sizeof(ValueType)), ResultInvalidOffsetArgument());

    // NOTE: オーバーランオフセット指定の場合は要素数ゼロを返却して成功扱い。
    const uint64_t requestOffset = static_cast<uint64_t>(offset);
    const uint64_t totalValueSize = static_cast<uint64_t>(sizeof(ValueType) * m_Count);
    NN_RESULT_THROW_UNLESS(requestOffset < totalValueSize, ResultSuccess());

    size_t readedSize;
    const uint64_t writeSize = std::min(outCapacity, static_cast<uint64_t>(totalValueSize - requestOffset));
    NN_RESULT_DO(m_ValueStore.Read(&readedSize, static_cast<int64_t>(requestOffset), pOutBuffer, static_cast<size_t>(writeSize)));
    NN_RESULT_THROW_UNLESS(readedSize == static_cast<size_t>(writeSize), ResultContentNotFound());

    *outValue = writeSize;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncAvailableELicensesImpl::OnQueryAccessProfile(AccessProfile* pOut, char* pOutPathUrl, size_t availablePathUrlCapacity) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);
    constexpr char base[] = "/v1/rights/available_elicenses";
    NN_ABORT_UNLESS(sizeof(base) <= availablePathUrlCapacity && availablePathUrlCapacity <= static_cast<size_t>(std::numeric_limits<int>::max()));

    const auto capacity = static_cast<int>(availablePathUrlCapacity);
    NN_ABORT_UNLESS(::nn::util::Strlcpy(pOutPathUrl, base, capacity) < capacity);

    pOut->uid = m_User;
    pOut->naId = ::nn::account::InvalidNintendoAccountId;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncAvailableELicensesImpl::OnSetupRequestBody(ConnectionType* pConnection) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConnection);

    // Rights データの有効性確認。
    auto nRights = static_cast<size_t>(m_ValueStore.GetWrittenSize() / sizeof(::nn::es::RightsId));
    NN_RESULT_THROW_UNLESS(nRights > 0U, ResultContentNotFound());

    constexpr char PostBasePrefix[] = "{\"rights_ids\": [";
    constexpr char PostBaseSuffix[] = "]}";

    // ポストデータバッファ確保。( null終端含めてざっくり確保、19 は "\"%016llx\"," の文字数 )
    const size_t requiredPostSize = sizeof(PostBasePrefix) + sizeof(PostBaseSuffix) + static_cast<size_t>(19 * nRights);
    NN_RESULT_DO(m_Post.Acquire(sizeof(Bit64), requiredPostSize));
    const auto pOutPost = static_cast<char*>(m_Post.pTop);

    size_t postLength = 0;
    NN_RESULT_DO(AppendRequestBodyAsBit64(&postLength, pOutPost, requiredPostSize, m_ValueStore, 0, nRights, PostBasePrefix, PostBaseSuffix));
    DEBUG_TRACE("Post filed generate complete, (heap: %zu, length: %zu)=>`%s`\n", requiredPostSize, postLength, pOutPost);
    NN_SDK_ASSERT(postLength == std::strlen(pOutPost)); // NOTE: デバッグ用の保険検証
    NN_RESULT_DO(pConnection->SetTransactionPostFields(pOutPost));

    // "Content-Length", "Content-Type" ヘッダ。
    char data[96];
    NN_ABORT_UNLESS(::nn::util::SNPrintf(data, sizeof(data), "Content-Length:%zu", postLength) < sizeof(data));
    NN_RESULT_DO(pConnection->SetTransactionHeader(data));
    NN_RESULT_DO(pConnection->SetTransactionHeader("Content-Type:application/json"));
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncAvailableELicensesImpl::OnResolveResponse(InputStream* pInputStream) NN_NOEXCEPT
{
    // { "key": "value" } 要素は、1024 byte に収まる想定の Insitu パース用ストリーム。
    HttpJson::Stream::ProxyInputStream<InputStream, 1024> stream(pInputStream);

    AvailableELicensesAdaptor adaptor(&m_ValueStore);
    NN_RESULT_TRY(::nn::http::json::ImportJsonByRapidJson<::nn::http::json::DefaultJsonErrorMap>(adaptor, stream, static_cast<HttpJson::Canceler*>(&adaptor)))
    NN_RESULT_CATCH_CONVERT(::nn::http::ResultCanceled, adaptor.GetFailure())
    NN_RESULT_END_TRY;

    m_Count = static_cast<CountType>(adaptor.GetCount());
    NN_RESULT_SUCCESS;
}

}   // ~DynamicRights
}}} // ~nn::nim::srv
