﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/util/nifm_IpConfiguration.horizon.h>

#include <nn/nifm/detail/core/profile/nifm_IpSetting.h>
#include <nn/nifm/detail/util/nifm_CancelFlagHolder.h>
#include <nn/nifm/detail/util/nifm_FwdbgSettings.h>

#include <nn/tsc/tsc_Types.h>
#include <nn/tsc/tsc_Result.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nn/bsdsocket/cfg/cfg_Types.h>
#include <nn/bsdsocket/cfg/cfg_ClientApi.h>

#include <mutex>


namespace nn
{
namespace nifm
{
namespace detail
{

namespace
{
uint32_t IpV4AddressToTscStorage(const IpV4Address& ipV4Address)
{
    return static_cast<uint32_t>(ipV4Address.data[0])       |
           static_cast<uint32_t>(ipV4Address.data[1]) <<  8 |
           static_cast<uint32_t>(ipV4Address.data[2]) << 16 |
           static_cast<uint32_t>(ipV4Address.data[3]) << 24;
}

void TscStorageToIpV4Address(IpV4Address* ipV4Address, uint32_t storage)
{
    ipV4Address->data[0] = storage       & 0xFF;
    ipV4Address->data[1] = storage >> 8  & 0xFF;
    ipV4Address->data[2] = storage >> 16 & 0xFF;
    ipV4Address->data[3] = storage >> 24 & 0xFF;
}

nn::Result ConvertTscApplyConfigResultToNifmResult(nn::Result result)
{
    if (result.IsSuccess())
    {
        NN_RESULT_SUCCESS;
    }
    else if (result <= nn::tsc::ResultDuplicateIp())  // TCP / IP stack detected a duplicate Ip on the network
    {
        NN_RESULT_THROW(ResultIpAddressConflicted());
    }
    else if (result <= nn::tsc::ResultProcessAborted())  // @ref ApplyConfig call is canceled by @ref CancelApplyIpConfig
    {
        NN_RESULT_THROW(ResultApplyIpConfigTimeout());
    }
    else if (result <= nn::tsc::ResultProcessTimeout())  // Timeout occurred in TCP / IP stack
    {
        NN_RESULT_THROW(ResultApplyIpConfigTimeout());
    }
    else if (result <= nn::tsc::ResultIpv4AddressInvalid())  // Stored interface address is invalid
    {
        NN_RESULT_THROW(ResultIpv4AddressInvalid());
    }
    else if (result <= nn::tsc::ResultIpv4SubnetMaskInvalid())  // Stored subnet mask is invalid
    {
        NN_RESULT_THROW(ResultIpv4SubnetMaskInvalid());
    }
    else if (result <= nn::tsc::ResultIpv4DefaultGatewayInvalid())  // Stored default gateway address is invalid
    {
        NN_RESULT_THROW(ResultIpv4DefaultGatewayInvalid());
    }
    else if (result <= nn::tsc::ResultIpv4PreferredDnsInvalid())  // Stored preferred DNS address is invalid
    {
        NN_RESULT_THROW(ResultIpv4PreferredDnsInvalid());
    }
    else if (result <= nn::tsc::ResultIpv4AlternativeDnsInvalid())  // Stored alternative DNS address is invalid
    {
        NN_RESULT_THROW(ResultIpv4AlternativeDnsInvalid());
    }
    else if (result <= nn::tsc::ResultMtuInvalid())  // Stored MTU value is invalid
    {
        NN_RESULT_THROW(ResultMtuInvalid());
    }
    else if (result <= nn::tsc::ResultInvalidReissuedLease())
    {
        NN_RESULT_THROW(ResultDhcpRenewFailed());
    }
    else if (result <= nn::tsc::ResultInvalidResponse())
    {
        NN_RESULT_THROW(ResultDhcpFailedForInvalidResponse());
    }
#if 0
    else if (result <= nn::tsc::ResultInterfaceNameInvalid())  // Interface name is not specified yet
    {
    }
    else if (result <= nn::tsc::ResultLibraryNotInitialized())  // TSC library is not initialized yet
    {
    }
    else if (result <= nn::tsc::ResultIpv4ConfigMethodInvalid())  // Stored IP configuration method is invalid
    {
    }
    else if (result <= nn::tsc::ResultInternalModuleFailed())  // Failed to process necessary modules
    {
    }
    else if (result <= nn::tsc::ResultInvalidInternalLogic())  // Failed to process request due to internal problem
    {
    }
    else if (result <= nn::tsc::ResultMaxInterfaces())  // The maximum number of network interface is already in use
    {
    }
    else if (result <= nn::tsc::ResultResourceBusy())  // Failed to process the request due to insufficient internal resource
    {
    }
#endif
    else
    {
        NN_RESULT_THROW(ResultApplyIpConfigFailed());
    }
}

}

IpConfiguration::IpConfiguration() NN_NOEXCEPT
    : m_Ipv4ConfigContext(nn::util::nullopt),
      m_ActiveConfigContext(nn::util::nullopt)
{
}

IpConfiguration::~IpConfiguration() NN_NOEXCEPT
{
    if (IsActive())
    {
        NotifyInterfaceDown();
    }
}

nn::Result IpConfiguration::Initialize(const char* pInterfaceName, NetworkInterfaceType networkInterfaceType) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!IsInitialized(), ResultInternalError());    // TODO: [TORIAEZU]

    m_NetworkInterfaceType = networkInterfaceType;
    m_Ipv4ConfigContext.emplace(pInterfaceName, nn::os::EventClearMode_AutoClear);

    NN_RESULT_SUCCESS;
}

bool IpConfiguration::IsInitialized() const NN_NOEXCEPT
{
    return m_Ipv4ConfigContext;
}

nn::Result IpConfiguration::ApplyIpSetting(const IpSetting& ipSetting) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsInitialized(), ResultInternalError());     // TODO: [TORIAEZU]
    NN_RESULT_THROW_UNLESS(!IsActive(), ResultInternalError());         // TODO: [TORIAEZU]

    {
        nn::tsc::Ipv4Config ipv4Config;

        if( ipSetting.IsIpAddressSettingAuto() )
        {
            ipv4Config.interfaceAddress.storage = 0;
            ipv4Config.subnetMask.storage       = 0;
            ipv4Config.defaultGateway.storage   = 0;

            if( ipSetting.IsDnsSettingAuto() )
            {
                ipv4Config.method = nn::tsc::Ipv4ConfigMethod_Dhcp;

                ipv4Config.preferredDns.storage     = 0;
                ipv4Config.alternativeDns.storage   = 0;
            }
            else
            {
                ipv4Config.method = nn::tsc::Ipv4ConfigMethod_DhcpHybrid;

                IpV4Address ipv4Address;
                ipv4Config.preferredDns.storage     = IpV4AddressToTscStorage(*ipSetting.GetPreferredDns(&ipv4Address));
                ipv4Config.alternativeDns.storage   = IpV4AddressToTscStorage(*ipSetting.GetAlternateDns(&ipv4Address));
            }
        }
        else
        {
            NN_SDK_ASSERT(!ipSetting.IsDnsSettingAuto());

            ipv4Config.method = nn::tsc::Ipv4ConfigMethod_Static;

            IpV4Address ipv4Address;
            ipv4Config.interfaceAddress.storage = IpV4AddressToTscStorage(*ipSetting.GetIpAddress(&ipv4Address));
            ipv4Config.subnetMask.storage       = IpV4AddressToTscStorage(*ipSetting.GetSubnetMask(&ipv4Address));
            ipv4Config.defaultGateway.storage   = IpV4AddressToTscStorage(*ipSetting.GetDefaultGateway(&ipv4Address));
            ipv4Config.preferredDns.storage     = IpV4AddressToTscStorage(*ipSetting.GetPreferredDns(&ipv4Address));
            ipv4Config.alternativeDns.storage   = IpV4AddressToTscStorage(*ipSetting.GetAlternateDns(&ipv4Address));
        }

        m_Ipv4ConfigContext->SetConfig(&ipv4Config);
    }

    m_Ipv4ConfigContext->SetMtu(ipSetting.GetMtu());

    m_Ipv4ConfigContext->GetSystemEventPointer();    // ApplyConfig() の前に呼んでおくことで SystemEvent が使われるようになる

    NN_DETAIL_NIFM_INFO("nn::tsc::Ipv4ConfigContext::ApplyConfig started.\n");
    NN_RESULT_DO(
        ConvertTscApplyConfigResultToNifmResult(m_Ipv4ConfigContext->ApplyConfig(nn::tsc::IpApplyModeMask_DuplicateIpCheck))
    );

    auto pEvent = m_Ipv4ConfigContext->GetEventPointer();
    NN_SDK_ASSERT_NOT_NULL(pEvent);

    nn::Result cancelReasonResult = nn::ResultSuccess();

    nn::TimeSpan timeout = nn::os::GetSystemTick().ToTimeSpan() +  FwdbgSettings::GetSingleton().GetApplyConfigTimeout();
    while (!pEvent->TimedWait(nn::TimeSpan::FromMilliSeconds(100)))
    {
        cancelReasonResult = CancelFlagHolder::GetSingleton().ConfirmInternetAdmitted(m_NetworkInterfaceType);

        if (cancelReasonResult.IsFailure() || (nn::os::GetSystemTick().ToTimeSpan() > timeout))
        {
            NN_DETAIL_NIFM_WARN("Socket config is cancelled (%d-%d).\n", cancelReasonResult.GetModule(), cancelReasonResult.GetDescription());
            m_Ipv4ConfigContext->CancelApplyConfig();
            // キャンセルに成功した場合は GetApplyResult() が失敗になり、
            // キャンセルが間に合わなかった場合は GetApplyResult() が成功になるので
            // キャンセル自身の結果はハンドリングしない
        }
    }

    auto result = ConvertTscApplyConfigResultToNifmResult(m_Ipv4ConfigContext->GetApplyResult());

    if (result.IsSuccess())
    {
        // 成功を失敗に捻じ曲げると後処理が必要になるので、
        // キャンセル要求があったとしても次のキャンセルポイントへ進める
    }
    else
    {
        if (cancelReasonResult.IsFailure())
        {
            result = cancelReasonResult;
        }
        else if (result <= ResultApplyIpConfigTimeout() && ipSetting.IsIpAddressSettingAuto())
        {
            // DHCP が有効な場合、タイムアウトは DHCP の失敗として扱う
            result = ResultDhcpFailed();
        }
    }

    NN_RESULT_DO(result);

    char interfaceName[nn::tsc::g_MaxInterfaceNameLength];
    NN_RESULT_DO(m_Ipv4ConfigContext->GetInterfaceName(interfaceName, NN_ARRAY_SIZE(interfaceName)));

    {
        NN_UTIL_LOCK_GUARD(m_Mutex);

        m_ActiveConfigContext.emplace(interfaceName);
    }

    NN_RESULT_SUCCESS;
}

bool IpConfiguration::IsActive() const NN_NOEXCEPT
{
    return m_ActiveConfigContext;
}

nn::Result IpConfiguration::NotifyInterfaceDown() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsActive(), ResultInternalError());          // TODO: [TORIAEZU]

    NN_SDK_ASSERT(m_Ipv4ConfigContext);
    NN_SDK_ASSERT(m_ActiveConfigContext);

    NN_RESULT_DO(m_Ipv4ConfigContext->NotifyInterfaceDown());

    int socketValue = nn::socket::ShutdownAllSockets(false);    // SIGLO-61083 の議論により、リンク切断時も一部のソケットは残す
    NN_UNUSED(socketValue);
    if (socketValue == -1)
    {
        NN_DETAIL_NIFM_WARN("nn::socket::ShutdownAllSockets returned %d\n", socketValue);
    }

    {
        NN_UTIL_LOCK_GUARD(m_Mutex);

        m_ActiveConfigContext = nn::util::nullopt;
    }

    m_DefaultGatewayMacAddressCache = nn::util::nullopt;

    // SIGLONTD-8766 のワークアラウンド
    char interfaceName[nn::tsc::g_MaxInterfaceNameLength];
    NN_RESULT_DO(m_Ipv4ConfigContext->GetInterfaceName(interfaceName, NN_ARRAY_SIZE(interfaceName)));
    m_Ipv4ConfigContext = nn::util::nullopt;
    m_Ipv4ConfigContext.emplace(interfaceName, nn::os::EventClearMode_AutoClear);

    NN_RESULT_SUCCESS;
}

nn::Result IpConfiguration::GetIpAddress(IpV4Address* pOutIpAddress) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(IsActive(), ResultInternalError());          // TODO: [TORIAEZU]

    NN_SDK_ASSERT(m_ActiveConfigContext);

    nn::tsc::Ipv4AddrStorage ipv4Address;
    NN_RESULT_DO(m_ActiveConfigContext->GetInterfaceAddress(&ipv4Address));

    NN_RESULT_THROW_UNLESS(ipv4Address.storage, ResultInternalError());     // TODO: [TORIAEZU]

    TscStorageToIpV4Address(pOutIpAddress, ipv4Address.storage);

    NN_RESULT_SUCCESS;
}

nn::Result IpConfiguration::GetIpConfigInfo(IpAddressSetting* pOutIpAddressSetting, DnsSetting* pDnsSetting) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(IsActive(), ResultInternalError());          // TODO: [TORIAEZU]

    NN_SDK_ASSERT(m_ActiveConfigContext);

    nn::tsc::Ipv4AddrStorage ipv4Address = {};
    NN_RESULT_DO(m_ActiveConfigContext->GetInterfaceAddress(&ipv4Address));
    NN_RESULT_THROW_UNLESS(ipv4Address.storage != 0, nn::nifm::ResultIpv4AddressInvalid());
    TscStorageToIpV4Address(&pOutIpAddressSetting->ipAddress, ipv4Address.storage);

    nn::tsc::Ipv4AddrStorage subnetMask = {};
    NN_RESULT_DO(m_ActiveConfigContext->GetSubnetMask(&subnetMask));
    NN_RESULT_THROW_UNLESS(subnetMask.storage != 0, nn::nifm::ResultIpv4SubnetMaskInvalid());
    TscStorageToIpV4Address(&pOutIpAddressSetting->subnetMask, subnetMask.storage);

    nn::tsc::Ipv4AddrStorage defaultGateway = {};
    NN_RESULT_DO(m_ActiveConfigContext->GetDefaultGateway(&defaultGateway));
    NN_RESULT_THROW_UNLESS(defaultGateway.storage != 0, nn::nifm::ResultIpv4DefaultGatewayInvalid());
    TscStorageToIpV4Address(&pOutIpAddressSetting->defaultGateway, defaultGateway.storage);

    nn::tsc::Ipv4AddrStorage preferredDns = {};
    if (m_ActiveConfigContext->GetPreferredDns(&preferredDns).IsFailure())
    {
        preferredDns.storage = 0;
    }
    TscStorageToIpV4Address(&pDnsSetting->preferredDns, preferredDns.storage);

    nn::tsc::Ipv4AddrStorage alternativeDns = {};
    if (m_ActiveConfigContext->GetAlternativeDns(&alternativeDns).IsFailure())
    {
        alternativeDns.storage = 0;
    }
    TscStorageToIpV4Address(&pDnsSetting->alternateDns, alternativeDns.storage);

    // 優先・代替 DNS 共に 0 の場合はエラー
    NN_RESULT_THROW_UNLESS(preferredDns.storage != 0 || alternativeDns.storage != 0, nn::nifm::ResultIpv4PreferredDnsInvalid());

    NN_RESULT_SUCCESS;
}

nn::os::SystemEvent* IpConfiguration::GetSystemEventPointer() NN_NOEXCEPT
{
    if (!IsInitialized() || !IsActive())
    {
        return nullptr;
    }

    NN_SDK_ASSERT(m_Ipv4ConfigContext);

    NN_SDK_ASSERT(m_Ipv4ConfigContext->IsUseSystemEvent());

    return m_Ipv4ConfigContext->GetSystemEventPointer();
}

nn::Result IpConfiguration::GetInterfaceStatus() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsInitialized(), ResultInternalError());     // TODO: [TORIAEZU]
    NN_RESULT_THROW_UNLESS(IsActive(), ResultInternalError());          // TODO: [TORIAEZU]

    NN_SDK_ASSERT(m_ActiveConfigContext);

    nn::tsc::ActiveConfigContext::InterfaceStatistics interfaceStatistics;
    m_ActiveConfigContext->GetInterfaceStatistics(&interfaceStatistics);

    NN_RESULT_THROW(interfaceStatistics.interfaceStatus);
}

nn::Result IpConfiguration::GetDefaultGatewayMacAddress(MacAddress* pOutMacAddress) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(IsActive(), ResultInternalError());          // TODO: [TORIAEZU]

    NN_SDK_ASSERT(m_ActiveConfigContext);

    nn::tsc::Ipv4AddrStorage defaultGateway = {};
    NN_RESULT_DO(m_ActiveConfigContext->GetDefaultGateway(&defaultGateway));
    NN_RESULT_THROW_UNLESS(defaultGateway.storage != 0, nn::nifm::ResultIpv4DefaultGatewayInvalid());

    nn::socket::InAddr inAddr = { defaultGateway.storage };

    auto result = nn::bsdsocket::cfg::LookupArpEntry(pOutMacAddress->data, MacAddress::Size, inAddr);

    if (result.IsSuccess())
    {
        m_DefaultGatewayMacAddressCache = *pOutMacAddress;
    }
    else
    {
        NN_DETAIL_NIFM_WARN(">LookupArpEntry failed (result 0x%08x)\n", result.GetInnerValueForDebug());

        NN_RESULT_THROW_UNLESS(m_DefaultGatewayMacAddressCache, result);

        *pOutMacAddress = *m_DefaultGatewayMacAddressCache;
    }

    NN_RESULT_SUCCESS;
}

}
}
}
