﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/nifm_RequestClient.h>
#include <nn/nifm/detail/nifm_TypesServiceObjectList.h>
#include <nn/nifm/detail/service/nifm_INifmService.sfdl.h>

#include <nn/applet/applet_LibraryApplet.h>
#include <nn/oe/oe_OperationStateControlSystem.h>

#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_LockGuard.h>

#include <mutex>


namespace nn
{
namespace nifm
{
namespace detail
{

namespace
{
    inline void ForwardOutBuffer(nn::sf::OutBuffer* pOutBuffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutBuffer);
        NN_SDK_ASSERT_LESS_EQUAL(size, pOutBuffer->GetSize());

        *pOutBuffer = pOutBuffer->MakePart(static_cast<ptrdiff_t>(size), pOutBuffer->GetSize() - size);
    }

    inline nn::Result ReadAndForwardOutBuffer(nn::sf::OutBuffer* pOutBuffer, void* pBuffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutBuffer);
        NN_SDK_ASSERT_NOT_NULL(pBuffer);

        NN_RESULT_THROW_UNLESS(pOutBuffer->GetSize() >= size, ResultInvalidArgument());
        std::memcpy(pBuffer, pOutBuffer->GetPointerUnsafe(), size);
        ForwardOutBuffer(pOutBuffer, size);

        NN_RESULT_SUCCESS;
    }
}

nn::Result GetGeneralServicePointer(nn::sf::SharedPointer<detail::IGeneralService>* ppGeneralService) NN_NOEXCEPT;

RequestClient::RequestClient(RequirementPreset requirementPreset, nn::os::EventClearMode eventClearMode) NN_NOEXCEPT
    : m_pIRequest(nullptr),
      m_RequestStateCache(RequestState_Invalid),
      m_ResultCache(nn::nifm::ResultNotInitialized())
{
    nn::Result result;

    nn::sf::SharedPointer<detail::IGeneralService> pGeneralService;
    result = GetGeneralServicePointer(&pGeneralService);

    NN_SDK_ASSERT(result.IsSuccess());
    if (result.IsFailure())
    {
        return;
    }

    nn::sf::SharedPointer<detail::IRequest> pIRequest;
    NN_ABORT_UNLESS_RESULT_SUCCESS(pGeneralService->CreateRequest(&pIRequest, static_cast<int32_t>(requirementPreset)));

    nn::sf::NativeHandle systemEventInternalReadableHandle, systemEventExposedReadableHandle;
    result = pIRequest->GetSystemEventReadableHandles(&systemEventInternalReadableHandle, &systemEventExposedReadableHandle);  // キャッシュ更新用の SystemEvent が先に更新されなければならない

    NN_SDK_ASSERT(result.IsSuccess());
    if (result.IsFailure())
    {
        return;
    }

    m_SystemEventInternal.AttachReadableHandle(systemEventInternalReadableHandle.GetOsHandle(), systemEventInternalReadableHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    systemEventInternalReadableHandle.Detach();
    m_SystemEventExposed.AttachReadableHandle(systemEventExposedReadableHandle.GetOsHandle(), systemEventExposedReadableHandle.IsManaged(), eventClearMode);
    systemEventExposedReadableHandle.Detach();

    m_pIRequest = pIRequest;
    m_RequestStateCache = RequestState_Free;
}

RequestClient::~RequestClient() NN_NOEXCEPT
{
}

void RequestClient::UpdateInternalState() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread(), "m_Mutex must be locked before UpdateInternalState()\n" );

    if (m_pIRequest != nullptr)
    {
        detail::IntRequestState requestState;

        nn::Result result = m_pIRequest->GetRequestState(&requestState);

        m_RequestStateCache = result.IsSuccess() ? static_cast<RequestState>(requestState) : RequestState_Invalid;
        m_ResultCache = m_pIRequest->GetResult();
    }
    else
    {
        m_RequestStateCache = RequestState_Invalid;
        m_ResultCache = nn::nifm::ResultNotInitialized();   // TODO
    }
}

void RequestClient::Submit() NN_NOEXCEPT
{
    nn::Result result;

    NN_UTIL_LOCK_GUARD(m_Mutex);

    auto requestState = GetRequestState();
    switch(requestState)
    {
    case RequestState_Free:
    case RequestState_OnHold:
    case RequestState_Accepted:
    case RequestState_Blocking:
        m_pIRequest->Submit();
        UpdateInternalState();
        break;

    case RequestState_Invalid:
        break;

    default:
        NN_SDK_ASSERT(false, "RequestClient: invalid state (%d)\n", requestState);
        break;
    }
}

void RequestClient::SubmitAndWait() NN_NOEXCEPT
{
    Submit();

    while( GetRequestState() == RequestState_OnHold )
    {
        // IsRequestOnHold/IsAvailable によって Event の Signal が落ちてしまってもデッドロックに入らないように時限待ち
        // 待ち時間に深い意味はない
        if( m_SystemEventInternal.TimedWait(nn::TimeSpan::FromSeconds(10)) )
        {
            NN_UTIL_LOCK_GUARD(m_Mutex);

            m_SystemEventInternal.Clear();
            UpdateInternalState();

            break;
        }
    }
}

void RequestClient::Cancel() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_pIRequest->Cancel();
}

nn::Result RequestClient::GetResult() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    if( m_SystemEventInternal.TryWait() )
    {
        m_SystemEventInternal.Clear();
        UpdateInternalState();
    }

    return m_ResultCache;
}

RequestState RequestClient::GetRequestState() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    if( m_SystemEventInternal.TryWait() )
    {
        m_SystemEventInternal.Clear();
        UpdateInternalState();
    }

    return m_RequestStateCache;
}

nn::os::SystemEvent& RequestClient::GetSystemEvent() NN_NOEXCEPT
{
    return m_SystemEventExposed;
}

nn::Result RequestClient::GetAdditionalInfo(AdditionalInfo* pOutAdditionalInfo, uint32_t* pOutRevision) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pIRequest->GetAdditionalInfo(pOutAdditionalInfo, pOutRevision));
}

nn::Result RequestClient::SetRequirement(const Requirement& requirement) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pIRequest->SetRequirement(requirement));
}

nn::Result RequestClient::GetRequirement(Requirement* pOutRequirement) const NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pIRequest->GetRequirement(pOutRequirement));
}

nn::Result RequestClient::GetSubmitId(uint32_t *pOutSubmitId) const NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pIRequest->GetRevision(pOutSubmitId));
}

nn::Result RequestClient::SetRequirementPreset(RequirementPreset requirementPreset) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pIRequest->SetRequirementPreset(static_cast<int32_t>(requirementPreset)));
}

nn::Result RequestClient::SetRequirementBySubmitId(uint32_t submitId) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pIRequest->SetRequirementByRevision(submitId));
}

nn::Result RequestClient::SetPriority(uint8_t priority) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pIRequest->SetPriority(priority));
}

nn::Result RequestClient::SetRawPriority(uint8_t rawPriority) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pIRequest->SetRawPriority(rawPriority));
}

nn::Result RequestClient::SetNetworkProfileId(nn::util::Uuid id) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pIRequest->SetNetworkProfileId(id));
}

nn::Result RequestClient::SetRejectable(bool isRejectable) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pIRequest->SetRejectable(isRejectable));
}

nn::Result RequestClient::SetConnectionConfirmationOption(ConnectionConfirmationOption connectionConfirmationOption) NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS(connectionConfirmationOption, ConnectionConfirmationOption_Count);
    NN_RESULT_THROW_UNLESS(connectionConfirmationOption < ConnectionConfirmationOption_Count, ResultOutOfRange());

    NN_RESULT_THROW(m_pIRequest->SetConnectionConfirmationOption(static_cast<int8_t>(connectionConfirmationOption)));
}

nn::Result RequestClient::SetPersistent(bool isPersistent) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pIRequest->SetPersistent(isPersistent));
}

nn::Result RequestClient::SetInstant(bool isInstant) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pIRequest->SetInstant(isInstant));
}

nn::Result RequestClient::SetSustainable(bool isSustainable) NN_NOEXCEPT
{
    // TODO: ここで渡される priority は現在は未使用
    NN_RESULT_THROW(m_pIRequest->SetSustainable(isSustainable, 0));
}

nn::Result RequestClient::SetGreedy(bool isGreedy) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pIRequest->SetGreedy(isGreedy));

}

nn::Result RequestClient::SetSharable(bool isSharable) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pIRequest->SetSharable(isSharable));
}

nn::Result RequestClient::SetKeptInSleep(bool isKeptInSleep) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_pIRequest->SetKeptInSleep(isKeptInSleep));
}

nn::Result RequestClient::RegisterSocketDescriptor(int socketDescriptor) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(int) == sizeof(int32_t));

    NN_RESULT_THROW(m_pIRequest->RegisterSocketDescriptor(socketDescriptor));
}

nn::Result RequestClient::UnregisterSocketDescriptor(int socketDescriptor) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(int) == sizeof(int32_t));

    NN_RESULT_THROW(m_pIRequest->UnregisterSocketDescriptor(socketDescriptor));
}

nn::Result RequestClient::PrepareLibraryApplet(nn::applet::LibraryAppletHandle* pOutLibraryAppletHandle) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutLibraryAppletHandle);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

    Bit32 appletId;
    Bit32 libraryAppletMode;
    uint32_t size;
    char argumentBuffer[LibraryAppletArgumentBufferSize];
    nn::sf::OutBuffer outBuffer(argumentBuffer, sizeof(argumentBuffer));
    auto colorIndex = nn::oe::GetExpectedThemeColor();

    NN_RESULT_DO(
        m_pIRequest->GetAppletInfo(
            &appletId,
            &libraryAppletMode,
            &size,
            outBuffer,
            static_cast<int32_t>(colorIndex)
        )
    );

    NN_DETAIL_NIFM_INFO("AppletId=%d\n", appletId);

    nn::applet::StorageHandle storageHandle = nn::applet::InvalidStorageHandle;

    NN_UTIL_SCOPE_EXIT
    {
        if (storageHandle != nn::applet::InvalidStorageHandle)
        {
            nn::applet::ReleaseStorage(storageHandle);

            if (*pOutLibraryAppletHandle != nn::applet::InvalidLibraryAppletHandle)
            {
                nn::applet::CloseLibraryApplet(*pOutLibraryAppletHandle);
                *pOutLibraryAppletHandle = nn::applet::InvalidLibraryAppletHandle;
            }
        }
    };

    NN_RESULT_DO(
        nn::applet::CreateLibraryApplet(
            pOutLibraryAppletHandle,
            static_cast<nn::applet::AppletId>(appletId),
            static_cast<nn::applet::LibraryAppletMode>(libraryAppletMode)
        )
    );

    outBuffer = outBuffer.MakePart(0, size);

    for (;;)
    {
        uint32_t size = 0;

        NN_RESULT_DO(ReadAndForwardOutBuffer(&outBuffer, &size, sizeof(size)));

        if (size == LibraryAppletArgumentBufferSentinel)
        {
            break;
        }

        NN_RESULT_THROW_UNLESS(outBuffer.GetSize() > size, ResultInternalError()); // TODO

        NN_RESULT_DO(nn::applet::CreateStorage(&storageHandle, size));
        NN_RESULT_DO(nn::applet::WriteToStorage(storageHandle, 0, outBuffer.GetPointerUnsafe(), size));
        nn::applet::PushToInChannel(*pOutLibraryAppletHandle, storageHandle);

        ForwardOutBuffer(&outBuffer, size);
        storageHandle = nn::applet::InvalidStorageHandle;
    }

    NN_RESULT_SUCCESS;

#else
    NN_UNUSED(pOutLibraryAppletHandle);
    NN_RESULT_THROW(ResultNotImplemented());
#endif
}

nn::Result RequestClient::GetLibaryAppletArgument(nn::applet::AppletId* pOutAppletId, nn::applet::LibraryAppletMode* pOutLibrayAppletMode, size_t* pOutSize, void* pOutBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutAppletId);
    NN_SDK_ASSERT_NOT_NULL(pOutLibrayAppletMode);
    NN_SDK_ASSERT_NOT_NULL(pOutSize);
    NN_SDK_ASSERT_NOT_NULL(pOutBuffer);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    Bit32 appletId;
    Bit32 libraryAppletMode;
    uint32_t outSize;
    nn::sf::OutBuffer outBuffer(reinterpret_cast<char*>(pOutBuffer), size);
    auto colorIndex = nn::oe::GetExpectedThemeColor();

    NN_RESULT_DO(
        m_pIRequest->GetAppletInfo(
            &appletId,
            &libraryAppletMode,
            &outSize,
            outBuffer,
            static_cast<int32_t>(colorIndex)
        )
    );

    *pOutAppletId = static_cast<nn::applet::AppletId>(appletId);
    *pOutLibrayAppletMode = static_cast<nn::applet::LibraryAppletMode>(libraryAppletMode);
    *pOutSize = outSize;

    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(pOutAppletId);
    NN_UNUSED(pOutLibrayAppletMode);
    NN_UNUSED(pOutSize);
    NN_UNUSED(pOutBuffer);
    NN_UNUSED(size);

    NN_RESULT_THROW(ResultNotImplemented());
#endif
}

}
}
}
