﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/profile/nifm_NetworkProfileManager.h>
#include <nn/nifm/detail/core/profile/nifm_IpSetting.h>

#include <nn/nifm/detail/util/nifm_DebugUtility.h>
#include <nn/nifm/detail/util/nifm_SsidListUtility.h>

#include <nn/util/util_StringUtil.h>
#include <nn/util/util_LockGuard.h>

namespace nn
{
namespace nifm
{
namespace detail
{

namespace
{

bool UpdateUserNetworkProfileIfTrivial(UserNetworkProfile* pOutUserNetworkProfile, const NetworkProfileData& networkProfileData)
{
    NetworkProfileData networkProfileDataOrigin;
    pOutUserNetworkProfile->Export(&networkProfileDataOrigin);
    NetworkProfileData networkProfileDataMod(networkProfileData);

    // 接続に影響しない項目だけを塗りつぶした後に比較
    networkProfileDataOrigin.name[0] = '\0';
    networkProfileDataMod.name[0] = '\0';
    networkProfileDataOrigin.isAutoConnect = false;
    networkProfileDataMod.isAutoConnect = false;
    networkProfileDataOrigin.wirelessSetting.ssidConfig.nonBroadcast = false;
    networkProfileDataMod.wirelessSetting.ssidConfig.nonBroadcast = false;

    if(networkProfileDataOrigin == networkProfileDataMod)
    {
        pOutUserNetworkProfile->SetName(networkProfileData.name);
        pOutUserNetworkProfile->SetAutoConnect(networkProfileData.isAutoConnect);
        pOutUserNetworkProfile->GetWirelessSettingPointer()->SetNonBroadcastEnabled(networkProfileData.wirelessSetting.ssidConfig.nonBroadcast);

        return true;
    }

    return false;
}

}

const DummyNetworkProfile NetworkProfileManager::s_DummyNetworkProfile;

NetworkProfileManager::NetworkProfileManager() NN_NOEXCEPT
    : m_NetworkProfileEvent(nn::os::EventClearMode_AutoClear),
      m_NetworkProfileEventHandler(m_NetworkProfileEvent),
      m_NetworkProfileEventCallback(),
      m_SsidListVersion()
{
    ImportUserNetworkProfile();
    ImportSsidListNetworkProfile();

    m_NetworkProfileEventHandler.Register(&m_NetworkProfileEventCallback);
}

NetworkProfileManager::~NetworkProfileManager() NN_NOEXCEPT
{
    ExportUserNetworkProfile();
}

bool NetworkProfileManager::ConfirmValidProfileId(nn::util::Uuid* id) NN_NOEXCEPT
{
    if (*id != nn::util::InvalidUuid)
    {
        bool foundId = false;
        for (auto pNp : m_UserNetworkProfileList)
        {
            // ID が同一の設定は削除される
            if (pNp->GetId() == *id)
            {
                m_UserNetworkProfileList.Free(pNp);
                foundId = true;
                *id = nn::util::GenerateUuid(); // ID を再生成
                break;
            }
        }

        // 存在しない ID 指定は失敗
        if (!foundId)
        {
            return false;
        }
    }
    else
    {
        // InvalidUuid 指定の場合は ID 生成
        *id = nn::util::GenerateUuid();
    }

    return true;
}

nn::Result NetworkProfileManager::UpdateUserNetworkProfile(const NetworkProfileData& networkProfile) NN_NOEXCEPT
{
    for (auto&& pNp : m_UserNetworkProfileList)
    {
        if (UpdateUserNetworkProfileIfTrivial(pNp, networkProfile))
        {
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultUserNetworkProfileNotFound());
}

nn::Result NetworkProfileManager::AddUserNetworkProfile(nn::util::Uuid *pOutId, const NetworkProfileData& networkProfile) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    if (UpdateUserNetworkProfile(networkProfile).IsSuccess())
    {
        // 削除せずに更新可能な場合
        *pOutId = networkProfile.id;
        m_NetworkProfileEvent.Signal();
        NN_RESULT_SUCCESS;
    }

    nn::util::Uuid id = networkProfile.id;
    NN_RESULT_THROW_UNLESS(ConfirmValidProfileId(&id), ResultUserNetworkProfileNotFound()); // 存在しない ID が指定された

    WirelessUserSetting wirelessSetting(networkProfile.wirelessSetting);
    IpSetting ipSetting(networkProfile.ipSetting);

    auto pNp = m_UserNetworkProfileList.Create(id, networkProfile.name, networkProfile.isAutoConnect, networkProfile.isLargeCapacity, wirelessSetting, ipSetting);
    NN_RESULT_THROW_UNLESS(pNp != nullptr, ResultOutOfUserNetworkProfileBuffer());

    pNp->SetNetworkInterfaceType(networkProfile.networkInterfaceType);
    *pOutId = id;

    m_NetworkProfileEvent.Signal();

    NN_RESULT_SUCCESS;
}

nn::Result NetworkProfileManager::AddSsidListNetworkProfile(nn::util::Uuid *pOutId, const NetworkProfileData& networkProfile) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    nn::util::Uuid id = networkProfile.id;

    NN_RESULT_THROW_UNLESS(ConfirmValidProfileId(&id), ResultUserNetworkProfileNotFound()); // 存在しない ID が指定された

    WirelessServiceSetting wirelessSetting(networkProfile.wirelessSetting);

    auto pNp = m_SsidListNetworkProfileList.Create(id, wirelessSetting);
    NN_RESULT_THROW_UNLESS(pNp != nullptr, ResultOutOfSsidListNetworkProfileBuffer());

    pNp->SetNetworkInterfaceType(networkProfile.networkInterfaceType);
    *pOutId = id;

    NN_RESULT_SUCCESS;
}

nn::Result NetworkProfileManager::AddTemporaryNetworkProfile(nn::util::Uuid *pOutId, const NetworkProfileData& networkProfile) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    nn::util::Uuid id = networkProfile.id;
    NN_RESULT_THROW_UNLESS(ConfirmValidProfileId(&id), ResultUserNetworkProfileNotFound()); // 存在しない ID が指定された

    WirelessUserSetting wirelessSetting(networkProfile.wirelessSetting);
    IpSetting ipSetting(networkProfile.ipSetting);

    auto pNp = m_TemporaryNetworkProfileList.Create(id, networkProfile.name, networkProfile.isAutoConnect, networkProfile.isLargeCapacity, wirelessSetting, ipSetting);
    NN_RESULT_THROW_UNLESS(pNp != nullptr, ResultOutOfTemporaryNetworkProfileBuffer());

    pNp->SetNetworkInterfaceType(networkProfile.networkInterfaceType);
    *pOutId = id;

    NN_RESULT_SUCCESS;
}

nn::Result NetworkProfileManager::PerpetuateTemporaryNetworkProfile(nn::util::Uuid* pOutId, const nn::util::Uuid& id) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutId);
    NN_SDK_ASSERT(nn::util::InvalidUuid != id);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    for (auto pNp : m_TemporaryNetworkProfileList)
    {
        if (id == pNp->GetId())
        {
            NetworkProfileData networkProfileData;
            pNp->Export(&networkProfileData);

            WirelessUserSetting wirelessSetting(networkProfileData.wirelessSetting);
            IpSetting ipSetting(networkProfileData.ipSetting);

            auto pUNp = m_UserNetworkProfileList.Create(id, networkProfileData.name, networkProfileData.isAutoConnect, networkProfileData.isLargeCapacity, wirelessSetting, ipSetting);

            NN_RESULT_THROW_UNLESS(pUNp != nullptr, ResultOutOfUserNetworkProfileBuffer());

            pUNp->SetNetworkInterfaceType(networkProfileData.networkInterfaceType);

            // ID の更新
            *pOutId = nn::util::GenerateUuid();
            pNp->SetId(*pOutId);

            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultTemporaryUserNetworkProfileNotFound());
}

nn::Result NetworkProfileManager::RemoveNetworkProfileImpl(NetworkProfileBase::ProfileType type, const nn::util::Uuid& id) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread(), "m_Mutex must be locked before RemoveNetworkProfile()\n");

    switch (type)
    {
    case NetworkProfileBase::ProfileType::UserProfile:
        for (auto pNp : m_UserNetworkProfileList)
        {
            if (pNp->GetId() == id && m_UserNetworkProfileList.Free(pNp))
            {
                m_NetworkProfileEvent.Signal();
                NN_RESULT_SUCCESS;
            }
        }
        NN_RESULT_THROW(ResultUserNetworkProfileNotFound());
        break;
    case NetworkProfileBase::ProfileType::SsidListProfile:
        for (auto pNp : m_SsidListNetworkProfileList)
        {
            if (pNp->GetId() == id && m_SsidListNetworkProfileList.Free(pNp))
            {
                m_NetworkProfileEvent.Signal();
                NN_RESULT_SUCCESS;
            }
        }
        NN_RESULT_THROW(ResultSsidListNetworkProfileNotFound());
        break;
    case NetworkProfileBase::ProfileType::TemporaryProfile:
        for (auto pNp : m_TemporaryNetworkProfileList)
        {
            if (pNp->GetId() == id && m_TemporaryNetworkProfileList.Free(pNp))
            {
                m_NetworkProfileEvent.Signal();
                NN_RESULT_SUCCESS;
            }
        }
        NN_RESULT_THROW(ResultTemporaryUserNetworkProfileNotFound());
        break;
    case NetworkProfileBase::ProfileType::Any:
        NN_FALL_THROUGH;
    case NetworkProfileBase::ProfileType::AnyIncludingDummy:
        NN_FALL_THROUGH;
    case NetworkProfileBase::ProfileType::DummyProfile:
        NN_FALL_THROUGH;
    default:
        break;
    }

    NN_RESULT_THROW(ResultNetworkProfileNotFound());
}

nn::Result NetworkProfileManager::RemoveNetworkProfile(NetworkProfileBase::ProfileType type, const nn::util::Uuid& id) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    if (type == NetworkProfileBase::ProfileType::Any || type == NetworkProfileBase::ProfileType::AnyIncludingDummy)
    {
        // いずれかの type で削除に成功すれば即座に返る
        if (false
            || RemoveNetworkProfileImpl(NetworkProfileBase::ProfileType::UserProfile, id).IsSuccess()
            || RemoveNetworkProfileImpl(NetworkProfileBase::ProfileType::SsidListProfile, id).IsSuccess()
            || RemoveNetworkProfileImpl(NetworkProfileBase::ProfileType::TemporaryProfile, id).IsSuccess())
        {
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_THROW(ResultNetworkProfileNotFound());
    }

    NN_RESULT_THROW(RemoveNetworkProfileImpl(type, id));
}

void NetworkProfileManager::RemoveAllNetworkProfile( NetworkProfileBase::ProfileType type ) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    switch( type )
    {
    case NetworkProfileBase::ProfileType::UserProfile:
        m_UserNetworkProfileList.FreeAll();
        break;
    case NetworkProfileBase::ProfileType::SsidListProfile:
        m_SsidListNetworkProfileList.FreeAll();
        break;
    case NetworkProfileBase::ProfileType::TemporaryProfile:
        m_TemporaryNetworkProfileList.FreeAll();
        break;
    case NetworkProfileBase::ProfileType::Any:
    case NetworkProfileBase::ProfileType::AnyIncludingDummy:
        m_UserNetworkProfileList.FreeAll();
        m_SsidListNetworkProfileList.FreeAll();
        m_TemporaryNetworkProfileList.FreeAll();
        break;
    case NetworkProfileBase::ProfileType::DummyProfile:
    default:
        break;
    }
}

int NetworkProfileManager::GetNetworkProfileCount( NetworkProfileBase::ProfileType type ) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    switch( type )
    {
    case NetworkProfileBase::ProfileType::UserProfile:
        return m_UserNetworkProfileList.GetCount();
    case NetworkProfileBase::ProfileType::SsidListProfile:
        return m_SsidListNetworkProfileList.GetCount();
    case NetworkProfileBase::ProfileType::TemporaryProfile:
        return m_TemporaryNetworkProfileList.GetCount();
    case NetworkProfileBase::ProfileType::DummyProfile:
        return 1;
    case NetworkProfileBase::ProfileType::Any:
        return m_UserNetworkProfileList.GetCount() + m_SsidListNetworkProfileList.GetCount() + m_TemporaryNetworkProfileList.GetCount();
    case NetworkProfileBase::ProfileType::AnyIncludingDummy:
        return m_UserNetworkProfileList.GetCount() + m_SsidListNetworkProfileList.GetCount() + m_TemporaryNetworkProfileList.GetCount() + 1;
    default:
        return 0;
    }
}

nn::Result NetworkProfileManager::GetNetworkProfileBasicInfo(NetworkProfileBasicInfo* pOutNetworkProfileBasicInfo, NetworkProfileBase::ProfileType type, int index) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutNetworkProfileBasicInfo);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    const NetworkProfileBase* pNetworkProfile = nullptr;

    switch (type)
    {
    case NetworkProfileBase::ProfileType::UserProfile:
        pNetworkProfile = &m_UserNetworkProfileList[index];
        break;
    case NetworkProfileBase::ProfileType::SsidListProfile:
        pNetworkProfile = &m_SsidListNetworkProfileList[index];
        break;
    case NetworkProfileBase::ProfileType::TemporaryProfile:
        pNetworkProfile = &m_TemporaryNetworkProfileList[index];
        break;
    case NetworkProfileBase::ProfileType::DummyProfile:
        pNetworkProfile = &s_DummyNetworkProfile;
        break;
    case NetworkProfileBase::ProfileType::Any:
        NN_FALL_THROUGH;
    case NetworkProfileBase::ProfileType::AnyIncludingDummy:
        if (index < m_UserNetworkProfileList.GetCount())
        {
            pNetworkProfile = &m_UserNetworkProfileList[index];
            break;
        }
        index -= m_UserNetworkProfileList.GetCount();
        if (index < m_SsidListNetworkProfileList.GetCount())
        {
            pNetworkProfile = &m_SsidListNetworkProfileList[index];
            break;
        }
        index -= m_SsidListNetworkProfileList.GetCount();
        if (index < m_TemporaryNetworkProfileList.GetCount())
        {
            pNetworkProfile = &m_TemporaryNetworkProfileList[index];
            break;
        }
        if (type == NetworkProfileBase::ProfileType::AnyIncludingDummy)
        {
            index -= m_TemporaryNetworkProfileList.GetCount();
            if (index < 1)
            {
                pNetworkProfile = &s_DummyNetworkProfile;
                break;
            }
        }
        break;
    default:
        break;
    }

    NN_RESULT_THROW_UNLESS(pNetworkProfile != nullptr, ResultNetworkProfileNotFound());
    NN_RESULT_THROW(ConvertToNetworkProfileBasicInfoFromNetworkProfile(pOutNetworkProfileBasicInfo, *pNetworkProfile));
}

nn::Result NetworkProfileManager::GetNetworkProfileData(NetworkProfileData* pOutNetworkProfileData, const nn::util::Uuid& id) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutNetworkProfileData);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    for (auto pNp : m_UserNetworkProfileList)
    {
        if (pNp->GetId() == id)
        {
            pNp->Export(pOutNetworkProfileData);
            NN_RESULT_SUCCESS;
        }
    }
    for (auto pNp : m_SsidListNetworkProfileList)
    {
        if (pNp->GetId() == id)
        {
            pNp->Export(pOutNetworkProfileData);
            NN_RESULT_SUCCESS;
        }
    }
    for (auto pNp : m_TemporaryNetworkProfileList)
    {
        if (pNp->GetId() == id)
        {
            pNp->Export(pOutNetworkProfileData);
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultNetworkProfileNotFound());
}

bool NetworkProfileManager::IsAnyEthernetNetworkProfileAvailable() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    for (auto pNp : m_UserNetworkProfileList)
    {
        if (pNp->GetNetworkInterfaceType() == NetworkInterfaceType_Ethernet && pNp->GetAutoConnect())
        {
            return true;
        }
    }
#if 0
    // 実運用上 BG 動作が利用する有線設定を SSID リストやテンポラリ設定から与えることはないはずなので省略しておく
    for (auto pNp : m_SsidListNetworkProfileList)
    {
        if (pNp->GetNetworkInterfaceType() == NetworkInterfaceType_Ethernet && pNp->GetAutoConnect())
        {
            return true;
        }
    }
    for (auto pNp : m_TemporaryNetworkProfileList)
    {
        if (pNp->GetNetworkInterfaceType() == NetworkInterfaceType_Ethernet && pNp->GetAutoConnect())
        {
            return true;
        }
    }
#endif
    return false;
}

bool NetworkProfileManager::UpdateUserNetworkProfileOrder( const nn::util::Uuid& id ) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    bool ret = false;

    int count = m_UserNetworkProfileList.GetCount();
    for( int i = 0; i < count; ++i )
    {
        if( id == m_UserNetworkProfileList[i].GetId() )
        {
            // 末尾（新しい）に移動
            ret = m_UserNetworkProfileList.Move( i, count - 1 );
            break;
        }
    }

    return ret;
}

nn::Result NetworkProfileManager::ImportUserNetworkProfile() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    std::memset(m_NetworkSettings, 0, sizeof(m_NetworkSettings));

    int outCount = nn::settings::system::GetNetworkSettings(m_NetworkSettings, UserNetworkProfileCountMax);

    for( int i = 0; i < outCount; ++i )
    {
        auto pNp = m_UserNetworkProfileList.Create();
        const auto& s = m_NetworkSettings[i];

        pNp->SetName( s.name );
        nn::util::Uuid id;
        std::memcpy( id.data, s.id, nn::util::Uuid::Size );
        pNp->SetId( id );

        pNp->SetAutoConnect(s.flags.Test(nn::settings::system::ProfileFlag::AutoConnectable::Index));
        pNp->SetLargeCapacity(s.flags.Test(nn::settings::system::ProfileFlag::LargeCapacity::Index));

        switch( s.nicType )
        {
        case nn::settings::system::NicType_Ieee80211:
            pNp->SetNetworkInterfaceType( NetworkInterfaceType_Ieee80211 );
            break;
        case nn::settings::system::NicType_Ethernet:
            pNp->SetNetworkInterfaceType( NetworkInterfaceType_Ethernet );
            break;
        default:
            break;
        }

        pNp->GetWirelessSettingPointer()->SetNonBroadcastEnabled( s.ssidSettings.flags[nn::settings::system::SsidFlag::IsCloaked::Index] );
        pNp->GetWirelessSettingPointer()->SetSsidHex( s.ssidSettings.ssid, s.ssidSettings.ssidLength );

        switch( s.securitySettings.authenticationMode )
        {
        case nn::settings::system::AuthenticationType_Open:
            pNp->GetWirelessSettingPointer()->SetAuthentication( Authentication_Open );
            break;
        case nn::settings::system::AuthenticationType_Shared:
            pNp->GetWirelessSettingPointer()->SetAuthentication( Authentication_Shared );
            break;
        case nn::settings::system::AuthenticationType_WpaPsk:
            pNp->GetWirelessSettingPointer()->SetAuthentication( Authentication_WpaPsk );
            break;
        case nn::settings::system::AuthenticationType_Wpa2Psk:
            pNp->GetWirelessSettingPointer()->SetAuthentication( Authentication_Wpa2Psk );
            break;
        default:
            break;
        }

        switch( s.securitySettings.encryptionMode )
        {
        case nn::settings::system::EncryptionMode_None:
            pNp->GetWirelessSettingPointer()->SetEncryption( Encryption_None );
            break;
        case nn::settings::system::EncryptionMode_Wep:
            pNp->GetWirelessSettingPointer()->SetEncryption( Encryption_Wep );
            break;
        case nn::settings::system::EncryptionMode_Aes:
            pNp->GetWirelessSettingPointer()->SetEncryption( Encryption_Aes );
            break;
        default:
            break;
        }

        pNp->GetWirelessSettingPointer()->SetKeyMaterial( s.securitySettings.keyMaterial, s.securitySettings.keyMaterialLength );

        pNp->GetIpSettingPointer()->SetIpAddressSettingAuto( s.ipSettings.flags.Test( nn::settings::system::IpFlag::EnablesAutoIp::Index ) );
        pNp->GetIpSettingPointer()->SetDnsSettingAuto( s.ipSettings.flags.Test( nn::settings::system::IpFlag::EnablesAutoDns::Index ) );

        pNp->GetIpSettingPointer()->SetIpAddress( s.ipSettings.ipAddress );
        pNp->GetIpSettingPointer()->SetSubnetMask( s.ipSettings.subnetMask );
        pNp->GetIpSettingPointer()->SetDefaultGateway( s.ipSettings.defaultGateway );
        pNp->GetIpSettingPointer()->SetPreferredDns( s.ipSettings.preferredDns );
        pNp->GetIpSettingPointer()->SetAlternateDns( s.ipSettings.alternateDns );

        pNp->GetIpSettingPointer()->SetProxyEnabled( s.proxySettings.flags.Test( nn::settings::system::ProxyFlag::Enables::Index ) );
        pNp->GetIpSettingPointer()->SetProxyAuthEnabled( s.proxySettings.flags.Test( nn::settings::system::ProxyFlag::Authenticates::Index ) );

        pNp->GetIpSettingPointer()->SetProxyHostname( s.proxySettings.hostName );
        pNp->GetIpSettingPointer()->SetProxyPort( s.proxySettings.portNumber );
        pNp->GetIpSettingPointer()->SetProxyUsername( s.proxySettings.userName );
        pNp->GetIpSettingPointer()->SetProxyPassword( s.proxySettings.password );

        pNp->GetIpSettingPointer()->SetMtu( s.mtu );
    }

    NN_RESULT_SUCCESS;
}

nn::Result NetworkProfileManager::ExportUserNetworkProfile() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    std::memset(m_NetworkSettings, 0, sizeof(m_NetworkSettings));

    int userNetworkProfileCount = 0;
    for( const auto& p : m_UserNetworkProfileList )
    {
        auto s = &m_NetworkSettings[userNetworkProfileCount++];

        p->GetName( s->name );
        const nn::util::Uuid id = p->GetId();
        std::memcpy( s->id, id.data, nn::util::Uuid::Size );

        s->flags[nn::settings::system::ProfileFlag::AutoConnectable::Index] = p->GetAutoConnect();
        s->flags[nn::settings::system::ProfileFlag::LargeCapacity::Index] = p->GetLargeCapacity();

        switch( p->GetNetworkInterfaceType() )
        {
        case NetworkInterfaceType_Ieee80211:
            s->nicType = nn::settings::system::NicType_Ieee80211;
            break;
        case NetworkInterfaceType_Ethernet:
            s->nicType = nn::settings::system::NicType_Ethernet;
            break;
        default:
            break;
        }

        s->ssidSettings.flags[nn::settings::system::SsidFlag::IsCloaked::Index] = p->GetWirelessSettingPointer()->IsNonBroadcastEnabled();
        p->GetWirelessSettingPointer()->GetSsidHex( s->ssidSettings.ssid, &s->ssidSettings.ssidLength );

        switch ( p->GetWirelessSettingPointer()->GetAuthentication() )
        {
        case Authentication_Open:
            s->securitySettings.authenticationMode = nn::settings::system::AuthenticationType_Open;
            break;
        case Authentication_Shared:
            s->securitySettings.authenticationMode = nn::settings::system::AuthenticationType_Shared;
            break;
        case Authentication_WpaPsk:
            s->securitySettings.authenticationMode = nn::settings::system::AuthenticationType_WpaPsk;
            break;
        case Authentication_Wpa2Psk:
            s->securitySettings.authenticationMode = nn::settings::system::AuthenticationType_Wpa2Psk;
            break;
        default:
            break;
        }
        switch( p->GetWirelessSettingPointer()->GetEncryption() )
        {
        case Encryption_None:
            s->securitySettings.encryptionMode = nn::settings::system::EncryptionMode_None;
            break;
        case Encryption_Wep:
            s->securitySettings.encryptionMode = nn::settings::system::EncryptionMode_Wep;
            break;
        case Encryption_Aes:
            s->securitySettings.encryptionMode = nn::settings::system::EncryptionMode_Aes;
            break;
        default:
            break;
        }
        p->GetWirelessSettingPointer()->GetKeyMaterial( s->securitySettings.keyMaterial, &(s->securitySettings.keyMaterialLength), nn::settings::system::KeyMaterialLengthMax );

        s->ipSettings.flags[nn::settings::system::IpFlag::EnablesAutoIp::Index] = p->GetIpSettingPointer()->IsIpAddressSettingAuto();
        s->ipSettings.flags[nn::settings::system::IpFlag::EnablesAutoDns::Index] = p->GetIpSettingPointer()->IsDnsSettingAuto();

        p->GetIpSettingPointer()->GetIpAddress( s->ipSettings.ipAddress );
        p->GetIpSettingPointer()->GetSubnetMask( s->ipSettings.subnetMask );
        p->GetIpSettingPointer()->GetDefaultGateway( s->ipSettings.defaultGateway );
        p->GetIpSettingPointer()->GetPreferredDns( s->ipSettings.preferredDns );
        p->GetIpSettingPointer()->GetAlternateDns( s->ipSettings.alternateDns );

        s->proxySettings.flags[nn::settings::system::ProxyFlag::Enables::Index] = p->GetIpSettingPointer()->IsProxyEnabled();
        s->proxySettings.flags[nn::settings::system::ProxyFlag::Authenticates::Index] = p->GetIpSettingPointer()->IsProxyAuthEnabled();

        p->GetIpSettingPointer()->GetProxyHostname( s->proxySettings.hostName );
        s->proxySettings.portNumber = p->GetIpSettingPointer()->GetProxyPort();
        p->GetIpSettingPointer()->GetProxyUsername( s->proxySettings.userName );
        p->GetIpSettingPointer()->GetProxyPassword( s->proxySettings.password );

        s->mtu = p->GetIpSettingPointer()->GetMtu();
    }

    nn::settings::system::SetNetworkSettings(m_NetworkSettings, userNetworkProfileCount);

    NN_RESULT_SUCCESS;
}

nn::Result NetworkProfileManager::ConvertToNetworkProfileBasicInfoFromNetworkProfile(NetworkProfileBasicInfo* pOutNetworkProfileBasicInfo, const NetworkProfileBase& networkProfile) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutNetworkProfileBasicInfo);
    NN_DETAIL_NIFM_ASSERT_NOT_MEMORY_OVERLAP(pOutNetworkProfileBasicInfo, sizeof(*pOutNetworkProfileBasicInfo), &networkProfile, sizeof(networkProfile));

    pOutNetworkProfileBasicInfo->id = networkProfile.GetId();
    networkProfile.GetName(pOutNetworkProfileBasicInfo->name);
    switch(networkProfile.GetType())
    {
    case NetworkProfileBase::ProfileType::UserProfile:
        pOutNetworkProfileBasicInfo->networkProfileType = NetworkProfileType_User;
        break;
    case NetworkProfileBase::ProfileType::SsidListProfile:
        pOutNetworkProfileBasicInfo->networkProfileType = NetworkProfileType_SsidList;
        break;
    case NetworkProfileBase::ProfileType::TemporaryProfile:
        pOutNetworkProfileBasicInfo->networkProfileType = NetworkProfileType_Temporary;
        break;
    default:
        break;
    }
    pOutNetworkProfileBasicInfo->networkInterfaceType = networkProfile.GetNetworkInterfaceType();
    if (pOutNetworkProfileBasicInfo->networkInterfaceType == NetworkInterfaceType_Ieee80211)
    {
        networkProfile.GetWirelessSettingPointer()->GetSsid(&pOutNetworkProfileBasicInfo->ssid);
        pOutNetworkProfileBasicInfo->authentication = networkProfile.GetWirelessSettingPointer()->GetAuthentication();
        pOutNetworkProfileBasicInfo->encryption = networkProfile.GetWirelessSettingPointer()->GetEncryption();
    }
    else
    {
        std::memset(&pOutNetworkProfileBasicInfo->ssid, 0, sizeof(Ssid));
        pOutNetworkProfileBasicInfo->authentication = Authentication_Invalid;
        pOutNetworkProfileBasicInfo->encryption = Encryption_Invalid;
    }

    NN_RESULT_SUCCESS;
}

nn::Result NetworkProfileManager::ImportSsidListNetworkProfile() NN_NOEXCEPT
{
    SsidListAccessor ssidListAccessor;
#if defined(NN_BUILD_CONFIG_OS_WIN)
    NN_RESULT_DO(ssidListAccessor.Load("160913\n541CA4C7-B5EC-42C7-B13D-27BA88F38FD9,SE9UU1BPVF9TU0lE,,0,0,SE9UU1BPVA==,0"));
#else
    NN_RESULT_DO(ssidListAccessor.Load());
#endif

    char versionStr[SsidListVersion::Size];
    NN_RESULT_DO(ssidListAccessor.GetVersion(versionStr));

    if (nn::util::Strncmp(m_SsidListVersion.data, versionStr, SsidListVersion::Size) >= 0)
    {
        NN_DETAIL_NIFM_INFO("This is the latest version of SSID list: %s\n", m_SsidListVersion.data);
        NN_RESULT_SUCCESS;
    }
    else
    {
        nn::util::Strlcpy(m_SsidListVersion.data, versionStr, SsidListVersion::Size);
        m_SsidListNetworkProfileList.FreeAll();
    }

    nn::util::Uuid uuid;
    Ssid ssid;
    SharedKey sharedKey;
    Authentication authentication;
    Encryption encryption;
    char name[NetworkProfileBasicInfo::NameSize];
    bool isAutoConnect;

    while (NN_STATIC_CONDITION(true))
    {
        Result result = ssidListAccessor.GetNextEntry(&uuid, &ssid, &sharedKey, &authentication, &encryption, name, &isAutoConnect);

        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(ResultEndOfStream)
            {
                NN_RESULT_SUCCESS;
            }
            // 不正なエントリ以降はすべて読み込めない
        NN_RESULT_END_TRY;

        auto pNp = m_SsidListNetworkProfileList.Create(uuid);
        NN_SDK_ASSERT_NOT_NULL(pNp);

        auto pWs = pNp->GetWirelessSettingPointer();
        NN_SDK_ASSERT_NOT_NULL(pWs);

        pWs->SetNonBroadcastEnabled(false);
        pWs->SetSsidHex(ssid.hex, ssid.length);
        pWs->SetAuthentication(authentication);
        pWs->SetEncryption(encryption);
        pWs->SetKeyMaterial(sharedKey.keyMaterial, sharedKey.length);

        pNp->SetName(name);
        pNp->SetAutoConnect(isAutoConnect);
    }

    NN_SDK_ASSERT(false); //  ここにくることはない

    NN_RESULT_SUCCESS;
}

nn::Result NetworkProfileManager::GetSsidListVersion(SsidListVersion* pOutSsidListVersion) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutSsidListVersion);

    *pOutSsidListVersion = m_SsidListVersion;

    NN_RESULT_SUCCESS;
}

nn::Result NetworkProfileManager::CopyAllNetworkProfiles() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_CopiedUserNetworkProfileList.FreeAll();
    m_CopiedSsidListNetworkProfileList.FreeAll();
    m_CopiedTemporaryNetworkProfileList.FreeAll();

    for (const auto pNetworkProfile : m_UserNetworkProfileList)
    {
        auto pNp = m_CopiedUserNetworkProfileList.Create(*pNetworkProfile);
        pNp->SetNetworkInterfaceType(pNetworkProfile->GetNetworkInterfaceType());
    }

    for (const auto pNetworkProfile : m_SsidListNetworkProfileList)
    {
        auto pNp = m_CopiedSsidListNetworkProfileList.Create(*pNetworkProfile);
        // SSID リストは， NetworkInterfaceType_Ieee80211 固定
        NN_UNUSED(pNp);
    }

    for (const auto pNetworkProfile : m_TemporaryNetworkProfileList)
    {
        auto pNp = m_CopiedTemporaryNetworkProfileList.Create(*pNetworkProfile);
        pNp->SetNetworkInterfaceType(pNetworkProfile->GetNetworkInterfaceType());
    }

    NN_RESULT_SUCCESS;
}

const NetworkProfileBase* NetworkProfileManager::GetCopiedNetworkProfile( NetworkProfileBase::ProfileType type, int index ) const NN_NOEXCEPT
{
    switch( type )
    {
    case NetworkProfileBase::ProfileType::UserProfile:
        return &m_CopiedUserNetworkProfileList[index];
    case NetworkProfileBase::ProfileType::SsidListProfile:
        return &m_CopiedSsidListNetworkProfileList[index];
    case NetworkProfileBase::ProfileType::TemporaryProfile:
        return &m_CopiedTemporaryNetworkProfileList[index];
    case NetworkProfileBase::ProfileType::DummyProfile:
        return &s_DummyNetworkProfile;
    case NetworkProfileBase::ProfileType::Any:
    case NetworkProfileBase::ProfileType::AnyIncludingDummy:
        if( index < m_CopiedUserNetworkProfileList.GetCount() )
        {
            return &m_CopiedUserNetworkProfileList[index];
        }
        index -= m_CopiedUserNetworkProfileList.GetCount();
        if( index < m_CopiedSsidListNetworkProfileList.GetCount() )
        {
            return &m_CopiedSsidListNetworkProfileList[index];
        }
        index -= m_CopiedSsidListNetworkProfileList.GetCount();
        if( index < m_CopiedTemporaryNetworkProfileList.GetCount() )
        {
            return &m_CopiedTemporaryNetworkProfileList[index];
        }
        if (type == NetworkProfileBase::ProfileType::AnyIncludingDummy)
        {
            index -= m_CopiedTemporaryNetworkProfileList.GetCount();
            if (index < 1)
            {
                return &s_DummyNetworkProfile;
            }
        }
        return nullptr;
    default:
        return nullptr;
    }
}

int NetworkProfileManager::GetCopiedNetworkProfileCount( NetworkProfileBase::ProfileType type ) const NN_NOEXCEPT
{
    switch( type )
    {
    case NetworkProfileBase::ProfileType::UserProfile:
        return m_CopiedUserNetworkProfileList.GetCount();
    case NetworkProfileBase::ProfileType::SsidListProfile:
        return m_CopiedSsidListNetworkProfileList.GetCount();
    case NetworkProfileBase::ProfileType::TemporaryProfile:
        return m_CopiedTemporaryNetworkProfileList.GetCount();
    case NetworkProfileBase::ProfileType::DummyProfile:
        return 1;
    case NetworkProfileBase::ProfileType::Any:
        return m_CopiedUserNetworkProfileList.GetCount() + m_CopiedSsidListNetworkProfileList.GetCount() + m_CopiedTemporaryNetworkProfileList.GetCount();
    case NetworkProfileBase::ProfileType::AnyIncludingDummy:
        return m_CopiedUserNetworkProfileList.GetCount() + m_CopiedSsidListNetworkProfileList.GetCount() + m_CopiedTemporaryNetworkProfileList.GetCount() + 1;
    default:
        return 0;
    }
}

void NetworkProfileManager::Dump() const NN_NOEXCEPT
{
    int profListCount = GetCopiedNetworkProfileCount(NetworkProfileBase::ProfileType::AnyIncludingDummy);

    for (int i = 0; i < profListCount; ++i)
    {
        auto pNetworkProfile = GetCopiedNetworkProfile(NetworkProfileBase::ProfileType::AnyIncludingDummy, i);
        NetworkProfileData networkProfileData;
        pNetworkProfile->Export(&networkProfileData);
#if !defined(NN_DETAIL_NIFM_LOG_SSIDLIST_ENABLED)
        if (networkProfileData.networkProfileType != NetworkProfileType::NetworkProfileType_SsidList)
#endif
        {
            detail::Dump(networkProfileData);
        }
    }
}

void NetworkProfileManager::GetDefaultIpSetting(IpSettingData* pOutIpSetting) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutIpSetting);

    NetworkProfileBase::GetDefaultIpSetting(pOutIpSetting);
}

void NetworkProfileManager::SetDefaultIpSetting(const IpSettingData& ipSetting) NN_NOEXCEPT
{
    NetworkProfileBase::SetDefaultIpSetting(ipSetting);
}

}
}
}
