﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nifm/detail/core/profile/nifm_NetworkProfileBase.h>

#include <nn/util/util_Uuid.h>
#include <nn/util/util_StringUtil.h>

namespace nn
{
namespace nifm
{
namespace detail
{

WirelessUserSetting NetworkProfileBase::s_WirelessSettingDefault;  // TODO: TORIAEZU
IpSetting NetworkProfileBase::s_IpSettingDefault;

NN_IMPLICIT NetworkProfileBase::NetworkProfileBase( const nn::util::Uuid& id ) NN_NOEXCEPT
    : m_Id(id),
      m_IsAutoConnect(true),
      m_IsLargeCapacity(true)
{
    std::memset( m_Name, 0, NetworkProfileBasicInfo::NameSize );
}

NetworkProfileBase::NetworkProfileBase(const nn::util::Uuid& id, const char* pName, bool autoConnect, bool largeCapacity ) NN_NOEXCEPT
    : m_Id(id),
      m_IsAutoConnect(autoConnect),
      m_IsLargeCapacity(largeCapacity)
{
    util::Strlcpy( m_Name, pName, NetworkProfileBasicInfo::NameSize );
}

void NetworkProfileBase::GetName( char* pOutStr ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutStr);
    util::Strlcpy( pOutStr, m_Name, NetworkProfileBasicInfo::NameSize );
}

void NetworkProfileBase::SetName( const char* pStr ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pStr);
    util::Strlcpy( m_Name, pStr, NetworkProfileBasicInfo::NameSize );
}

bool NetworkProfileBase::GetAutoConnect() const NN_NOEXCEPT
{
    return m_IsAutoConnect;
}

void NetworkProfileBase::SetAutoConnect( bool autoConnect ) NN_NOEXCEPT
{
    m_IsAutoConnect = autoConnect;
}

bool NetworkProfileBase::GetLargeCapacity() const NN_NOEXCEPT
{
    return m_IsLargeCapacity;
}

void NetworkProfileBase::SetLargeCapacity(bool isLargeCapacity) NN_NOEXCEPT
{
    m_IsLargeCapacity = isLargeCapacity;
}

void NetworkProfileBase::Export(NetworkProfileData* pOutNetworkProfile) const NN_NOEXCEPT
{
    std::memset(pOutNetworkProfile, 0, sizeof(*pOutNetworkProfile));

    char profileName[NetworkProfileBasicInfo::NameSize];
    GetName(profileName);
    util::Strlcpy( pOutNetworkProfile->name, profileName, NetworkProfileBasicInfo::NameSize );
    pOutNetworkProfile->id = GetId();
    switch (GetType())
    {
    case ProfileType::UserProfile:
        pOutNetworkProfile->networkProfileType = NetworkProfileType_User;
        break;
    case ProfileType::SsidListProfile:
        pOutNetworkProfile->networkProfileType = NetworkProfileType_SsidList;
        break;
    case ProfileType::TemporaryProfile:
        pOutNetworkProfile->networkProfileType = NetworkProfileType_Temporary;
        break;
    case ProfileType::DummyProfile:
        break;
    default:
        NN_DETAIL_NIFM_INFO("Invalid profile type.\n");
        break;
    }
    pOutNetworkProfile->networkInterfaceType = GetNetworkInterfaceType();
    pOutNetworkProfile->isAutoConnect = GetAutoConnect();
    pOutNetworkProfile->isLargeCapacity = GetLargeCapacity();
    switch( pOutNetworkProfile->networkInterfaceType )
    {
    case NetworkInterfaceType_Ieee80211:
        GetWirelessSetting().Export(&pOutNetworkProfile->wirelessSetting);
        break;

    case NetworkInterfaceType_Ethernet:
        // TODO
        {
            WirelessUserSetting wirelessSettings(InvalidWirelessSettingData);
            wirelessSettings.Export(&pOutNetworkProfile->wirelessSetting);
        }
        break;

    default:
        // TODO
        break;
    }
    GetIpSetting().Export(&pOutNetworkProfile->ipSetting);
}

// For Test
void NetworkProfileBase::GetDefaultIpSetting(IpSettingData* pOutIpSettingData) NN_NOEXCEPT
{
    s_IpSettingDefault.Export(pOutIpSettingData);
}

void NetworkProfileBase::SetDefaultIpSetting(const IpSettingData& ipSetting) NN_NOEXCEPT
{
    s_IpSettingDefault = ipSetting;
}

}
}
}
