﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nifm/detail/core/accessPoint/nifm_AccessPointList.h>
#include <nn/nifm/detail/core/accessPoint/nifm_EthernetAccessPoint.h>
#include <nn/nifm/detail/core/networkInterface/nifm_EthernetInterface.horizon.h>
#include <nn/nifm/detail/core/profile/nifm_NetworkProfileBase.h>
#include <nn/nifm/detail/util/nifm_CancelFlagHolder.h>

namespace nn { namespace eth {
#include <nn/eth/eth_Result.public.h>
}}

#include <new>


namespace nn
{
namespace nifm
{
namespace detail
{

void EthernetInterface::InterfaceEventCallback::ExecuteImpl() NN_NOEXCEPT
{
    m_pEthernetInterface->InterfaceEventCallback();
}

EthernetInterface::EthernetInterface() NN_NOEXCEPT
    : m_InterfaceHandler(),
      m_pInterfaceEvent(nullptr),
      m_InterfaceEventCallback(this),
      m_InterfaceEventHandler(nn::util::nullopt),
      m_MediaCurrent(nn::eth::MediaType_Unknown),
      m_ReadyTime(nn::os::GetSystemTick().ToTimeSpan() + nn::TimeSpan::FromSeconds(WarmingTimeInSeconds)),
      m_IsInitialized(false)
{
}

EthernetInterface::~EthernetInterface() NN_NOEXCEPT
{
    if (IsInitialized())
    {
        NetworkInterfaceBase::Disconnect(ResultEthernetLinkDown());     // NIC 消失通知とリンクダウン通知の処理順は保証できないのでリンクダウン通知にまとめる
        NetworkInterfaceBase::Release();

        NN_SDK_ASSERT(m_InterfaceEventHandler);
        RemoveEventHandler(*m_InterfaceEventHandler);

        NN_SDK_ASSERT_NOT_NULL(m_pInterfaceEvent);
    }
}

nn::Result EthernetInterface::Initialize(nn::eth::client::InterfaceGroupHandler* pInterfaceGroupHandler, nn::eth::InterfaceInfo* pInterfaceInfo) NN_NOEXCEPT
{
    NN_RESULT_DO(m_InterfaceHandler.Initialize(pInterfaceGroupHandler, pInterfaceInfo, nn::os::EventClearMode_AutoClear));

    NN_RESULT_DO(m_IpConfiguration.Initialize(m_InterfaceHandler.GetInterfaceName(), NetworkInterfaceType_Ethernet));

    m_pInterfaceEvent = m_InterfaceHandler.GetSystemEventPointer();
    m_InterfaceEventHandler.emplace(*m_pInterfaceEvent);
    m_InterfaceEventHandler->Register(&m_InterfaceEventCallback);
    AddEventHandler(*m_InterfaceEventHandler);

    nn::eth::MediaType mediaRequested;
    uint32_t eventCounter;
    m_InterfaceHandler.GetMediaType(&mediaRequested, &m_MediaCurrent, &eventCounter);

    m_InterfaceHandler.SetMediaType(nn::eth::MediaType_AUTO);

    m_IsInitialized = true;

    NN_RESULT_SUCCESS;
}

bool EthernetInterface::IsInitialized() const NN_NOEXCEPT
{
    return m_IsInitialized;
}

bool EthernetInterface::IsAvailable() const NN_NOEXCEPT
{
    return IsInitialized();
}

nn::Result EthernetInterface::GetLatestAccessPointList(AccessPointListBase* pAccessPointList, nn::TimeSpan timeSpan) NN_NOEXCEPT
{
    NN_UNUSED(timeSpan);

    nn::Result result = ResultSuccess();

    // TODO: 更新との排他を図る
    NN_RESULT_THROW_UNLESS(IsAvailable(), nn::nifm::ResultNetworkInterfaceNotAvailable());

    if (pAccessPointList->CreateAndAdd<EthernetAccessPoint>(this) == nullptr)
    {
        NN_DETAIL_NIFM_WARN("Accesspoint list is full. (Eth)\n");
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW(result);
}

MacAddress* EthernetInterface::GetMacAddress( MacAddress* pOutMacAddress ) const NN_NOEXCEPT
{
    std::memcpy( pOutMacAddress->data, m_InterfaceHandler.GetInterfaceInfo()->interfaceMacAddress, MacAddress::Size );
    return pOutMacAddress;
}

nn::Result EthernetInterface::ConnectImpl(
    const EthernetAccessPoint& ethernetAccessPoint,
    const NetworkProfileBase& networkProfile,
    const AggregatedRequestType& aggregatedRequest) NN_NOEXCEPT
{
    NN_UNUSED(networkProfile);
    NN_UNUSED(aggregatedRequest);

    NN_DETAIL_NIFM_TRACE("EthernetInterface::ConnectImpl(const EthernetAccessPoint&, ...)\n");

    NN_RESULT_THROW_UNLESS(IsAvailable(), ResultNetworkInterfaceNotAvailable());

    // リンク層の接続は、コンストラクタで指定された eth ライブラリの
    // 自動ネゴシエーションにゆだねられているので
    // ここではすでにリンクアップされているかの確認のみをおこなう

    nn::eth::MediaType mediaRequested;
    nn::eth::MediaType mediaCurrent;
    uint32_t eventCounter;

    m_InterfaceHandler.GetMediaType(&mediaRequested, &mediaCurrent, &eventCounter);

    NN_DETAIL_NIFM_TRACE(
        "%s: #%u (requested, current) = (%08x, %08x)\n",
        m_InterfaceHandler.GetInterfaceName(), eventCounter, mediaRequested, mediaCurrent
    );

    auto currentTime = nn::os::GetSystemTick().ToTimeSpan();

    // TODO: [TORIAEZU] 起動直後にリンクアップするのに十分な時間を待つためのワークアラウンド
    while (mediaCurrent == nn::eth::MediaType_ConfigInProgress && currentTime <= m_ReadyTime)
    {
        if (CancelFlagHolder::GetSingleton().ConfirmInternetAdmitted(NetworkInterfaceType_Ethernet).IsFailure())
        {
            // USB Ethernet アダプター検出直後以外ではおこなわない待ちのキャンセルなので
            // 特別な失敗としては扱わない
            break;
        }

        auto wait = m_ReadyTime - currentTime;
        if (wait > nn::TimeSpan::FromMilliSeconds(100))
        {
            wait = nn::TimeSpan::FromMilliSeconds(100);
        }

        // m_pInterfaceEvent をほかのスレッドが同時待ちしない前提
        // 1本の DispatchLoop スレッドで接続と切断をハンドリングしているうちは大丈夫
        if (m_pInterfaceEvent->TimedWait(wait))
        {
            m_InterfaceHandler.GetMediaType(&mediaRequested, &mediaCurrent, &eventCounter);

            NN_DETAIL_NIFM_TRACE(
                "%s: #%u (requested, current) = (%08x, %08x)\n",
                m_InterfaceHandler.GetInterfaceName(), eventCounter, mediaRequested, mediaCurrent
            );

            // m_pInterfaceEvent を待つので InterfaceEventCallback 内の処理に
            // 不整合が出ないように m_MediaCurrent を更新する
            m_MediaCurrent = mediaCurrent;
        }

        currentTime = nn::os::GetSystemTick().ToTimeSpan();
    }

    NN_RESULT_THROW_UNLESS(EthLinkUp(mediaCurrent), ResultNoEthernetLink());

    nn::Result result = InitializeInternetConnection(networkProfile, aggregatedRequest);

    if (result.IsFailure())
    {
        DisconnectImpl(ethernetAccessPoint);
    }

    NN_RESULT_THROW(result);
}

nn::Result EthernetInterface::DisconnectImpl(const EthernetAccessPoint& ethernetAccessPoint) NN_NOEXCEPT
{
    NN_UNUSED(ethernetAccessPoint);

    NN_DETAIL_NIFM_INFO("EthernetInterface::DisconnectImpl(const EthernetAccessPoint&)\n");

    NN_RESULT_SUCCESS;
}

nn::Result EthernetInterface::ReleaseImpl(const EthernetAccessPoint& ethernetAccessPoint) NN_NOEXCEPT
{
    NN_UNUSED(ethernetAccessPoint);

    NN_DETAIL_NIFM_INFO("EthernetInterface::ReleaseImpl(const EthernetAccessPoint&)\n");

    NN_RESULT_SUCCESS;
}

void EthernetInterface::UpdateAccessPoint(EthernetAccessPoint* pInOutEthernetAccessPoint, const NetworkProfileBase& networkProfile) NN_NOEXCEPT
{
    NN_UNUSED(pInOutEthernetAccessPoint);
    NN_UNUSED(networkProfile);
}

const char* EthernetInterface::GetInterfaceName() const NN_NOEXCEPT
{
    return m_InterfaceHandler.GetInterfaceName();
}

void EthernetInterface::InterfaceEventCallback() NN_NOEXCEPT
{
    nn::eth::MediaType mediaRequested;
    nn::eth::MediaType mediaCurrent;
    uint32_t eventCounter;

    m_InterfaceHandler.GetMediaType(&mediaRequested, &mediaCurrent, &eventCounter);

    NN_DETAIL_NIFM_TRACE(
        "%s: #%u (requested, current) = (%08x, %08x)\n",
        m_InterfaceHandler.GetInterfaceName(), eventCounter, mediaRequested, mediaCurrent
    );

    // Up → Up で遷移した場合も、途中の Down への遷移通知を取り逃したものと見なして切断扱いにする
    if (!EthLinkUp(m_MediaCurrent) && EthLinkUp(mediaCurrent))
    {
    }
    else
    {
        NotifyLinkDisconnected(ResultEthernetLinkDown());
    }

    m_MediaCurrent = mediaCurrent;
}

bool EthernetInterface::IsLinkUp() const NN_NOEXCEPT
{
    return EthLinkUp(m_MediaCurrent);
}

bool EthernetInterface::IsPresent() const NN_NOEXCEPT
{
    // USB Ethernet アダプターが抜去されていたとしても、ネットワークリソースがまだ解放されていないなら
    // リンクダウンが適切に処理されていないため USB Ethernet アダプターがまだ存在するものとして扱う
    if (GetNetworkResourceState() != NetworkResourceState::Free)
    {
        return true;
    }

    return !(m_InterfaceHandler.GetResult() <= nn::eth::ResultInterfaceRemoved());
}

EthernetInterface* EthernetInterface::Find(const nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    if (!m_InterfaceEventHandler)
    {
        return nullptr;
    }

    if (m_InterfaceEventHandler->GetMultiWaitHolderPointer() != pMultiWaitHolder)
    {
        return nullptr;
    }

    return this;
}

nn::Result EthernetInterface::PutToSleepImpl() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

nn::Result EthernetInterface::WakeUpImpl() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

}
}
}

