﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "../detail/ngc_WorkBufAllocator.h"
#include "./ngc_ErrnoT.h"

namespace nn { namespace ngc { namespace detail {

// code snippets:
//   OutputStream& stream = 'stream that inherits OutputStream';
//   if (!stream.Write(dataptr, nbyte)) { error = stream.GetErrorValue(); .... }
//   for (.....) { if (!stream.Write(byte)) { error = stream.GetErrorValue(); .... } }
//   # you should Flush() and Close() explicitly
//   if (!stream.Flush()) { error = stream.GetErrorValue(); .... } }
//   if (!stream.Close()) { error = stream.GetErrorValue(); .... } }
class  OutputStream {
public:
    enum BufferingMode
    {
        BufferingMode_BlockBuffered = 0,  // default
        BufferingMode_LineBuffered,       // if console output
        BufferingMode_UnBuffered
    };

public:
    OutputStream() NN_NOEXCEPT : m_Pos(0),
                                 m_OsBufCuridx(0),
                                 m_pOsBuf(NULL),
                                 m_OsBufSize(0),
                                 m_Errno(0),
                                 m_BufferingMode(BufferingMode_BlockBuffered) {}
    virtual ~OutputStream() NN_NOEXCEPT
    {
        // NOTE:
        // cannot call Close_() from this destructor.
        // the destructor of the derived class has to call Close_() if needed.
    }
    // use Pos64() if you need a 64bit value on 32bit arch
    size_t Pos() const NN_NOEXCEPT
    {
        return static_cast<size_t>(Pos64());
    }
    uint64_t Pos64() const NN_NOEXCEPT
    {
        return m_Pos + m_OsBufCuridx;
    }
    bool Write(int b) NN_NOEXCEPT
    {
        // NOTE:
        // if the stream is closed,
        // Write() will succeed before the internal buffer becomes full.
        if (m_OsBufCuridx == m_OsBufSize)
        {
            if (!this->Flush_(false))
            {
                return false;
            }
            if (m_OsBufSize == 0)
            {
                this->SetError(EIO);
                return false;
            }
        }
        m_pOsBuf[m_OsBufCuridx++] = static_cast<unsigned char>(b & 0xff);
        return true;
    }
    bool Write(const void* p, size_t n) NN_NOEXCEPT
    {
        if (n > RSIZE_MAX)
        {
            this->SetError(EINVAL);
            return false;
        }
        if (m_OsBufCuridx + n <= m_OsBufSize)
        {
            MemCpy(&m_pOsBuf[m_OsBufCuridx], n, p, n);
            m_OsBufCuridx += n;
            return true;
        }
        return this->Write_(p, n);
    }
    bool WriteGather(const FdIoVec* iov, int iovcnt) NN_NOEXCEPT
    {
        if (iovcnt < 0)
        {
            this->SetError(EINVAL);
            return false;
        }
        return this->WriteGather_(iov, iovcnt);
    }
    bool Flush() NN_NOEXCEPT
    {
        return Flush_(true);
    }
    bool Close() NN_NOEXCEPT;
    errno_t GetErrorValue() const NN_NOEXCEPT
    {
        return m_Errno;
    }
    // returns true if there has been no errors
    // you can also write the codes such that 'if (!!stream) { ... }'
    BufferingMode GetBufferingMode() const NN_NOEXCEPT
    {
        return m_BufferingMode;
    }

public:
    inline NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return (m_Errno == 0);
    }

protected:
    void ResetBuffer(void* p, size_t nbytes) NN_NOEXCEPT
    {
        m_pOsBuf = reinterpret_cast<unsigned char*>(p);
        m_OsBufSize = static_cast<int>(nbytes);
    }
    void SetError(errno_t e) const NN_NOEXCEPT
    {
        if (m_Errno == 0)
        {
            m_Errno = e;
        }
    }

private:
    virtual bool PushBuffer_(const void* p, size_t nbytes, bool doFlush) NN_NOEXCEPT = 0;
    virtual bool Close_() NN_NOEXCEPT = 0;
    virtual void* GetWorkBuffer_(size_t* nbytes) NN_NOEXCEPT;
    virtual bool WriteGather_(const FdIoVec* iov, int iovcnt) NN_NOEXCEPT;
    bool Write_(const void* p, size_t n) NN_NOEXCEPT;

private:
    bool Flush_(bool flushDevice) NN_NOEXCEPT;
    bool GetWorkBuffer_() NN_NOEXCEPT;

private:
    uint64_t m_Pos;
    size_t m_OsBufCuridx;
    unsigned char* m_pOsBuf;
    size_t m_OsBufSize;
    mutable ErrnoT m_Errno;

protected:
    // NOTE:
    // BufferingMode is for the codes which use OutputStream.
    // OutputStream only shows the buffering mode which the user specified.
    // for example, TextWriter (the user of OutputStream) has to decide
    // when to call Flush() looking at GetBufferingMode().
    BufferingMode m_BufferingMode;

private:
    OutputStream(const OutputStream&) = delete;
    void operator=(const OutputStream&) = delete;
};

class  NullOutputStream final : public OutputStream
{
public:
    NullOutputStream() NN_NOEXCEPT;
    virtual ~NullOutputStream() NN_NOEXCEPT override {}

private:
    unsigned char m_pDummyBuf[256];
    virtual bool PushBuffer_(const void* p, size_t, bool doFlush) NN_NOEXCEPT override;
    virtual bool Close_() NN_NOEXCEPT override
    {
        return true;
    }
};

template<size_t N>
class MiniBufOut
{
public:
    static const size_t BufSize = N;
    inline MiniBufOut() NN_NOEXCEPT
    {
        m_pCur = &m_pBuf[0];
    }
    template<class OS>
    inline bool Flush(OS* stream) NN_NOEXCEPT
    {
        size_t n = m_pCur - &m_pBuf[0];
        m_pCur = &m_pBuf[0];
        return stream->Write(&m_pBuf[0], n);
    }
    template<class OS>
    inline bool CheckAndFlush(OS* stream, size_t n) NN_NOEXCEPT
    {
        if (m_pCur + n > &m_pBuf[0] + N)
        {
            NN_SDK_ASSERT(n <= N);
            return Flush(stream);
        }
        return true;
    }
    inline void Advance(size_t n) NN_NOEXCEPT
    {
        // CheckAndFlush(stream, n) must be successful beforehand.
        m_pCur += n;
    }
    inline uint8_t& operator[](size_t n) NN_NOEXCEPT
    {
        // CheckAndFlush(stream, n + 1) must be successful beforehand.
        return m_pCur[n];
    }
    template<class OS>
    inline bool WriteAndAdvance(OS* stream, uint8_t data) NN_NOEXCEPT
    {
        if (CheckAndFlush(stream, 1))
        {
            *m_pCur = data;
            ++m_pCur;
            return true;
        }
        return false;
    }
    template<class OS>
    inline bool WriteAndAdvance(OS* stream, const void* data, size_t n) NN_NOEXCEPT
    {
        if (CheckAndFlush(stream, n))
        {
            (void)MemCpy(m_pCur, n, data, n);
            m_pCur += n;
            return true;
        }
        return false;
    }

private:
    uint8_t* m_pCur;
    uint8_t m_pBuf[N];

    MiniBufOut(const MiniBufOut&) = delete;
    void operator=(const MiniBufOut&) = delete;
};

template<class T>
class MiniReallocOut
{
public:
    inline MiniReallocOut(T** mem, uint32_t* cur, uint32_t* memsize) NN_NOEXCEPT
        : m_ppMem(mem), m_pCur(cur), m_pMemSize(memsize), m_pAllocator(NULL) {}
    inline MiniReallocOut(T** mem, uint32_t* cur, uint32_t* memsize, nn::ngc::detail::WorkBufAllocator* pAllocator) NN_NOEXCEPT
        : m_ppMem(mem), m_pCur(cur), m_pMemSize(memsize), m_pAllocator(pAllocator) {}
    inline ~MiniReallocOut() NN_NOEXCEPT {}
    inline bool SetAllocator(nn::ngc::detail::WorkBufAllocator* pAllocator) NN_NOEXCEPT
    {
        if (!pAllocator)
        {
            return false;
        }
        m_pAllocator = pAllocator;
        return true;
    }
    inline bool Resize(uint32_t n) NN_NOEXCEPT
    {
        if (n <= *m_pMemSize)
        {
            *m_pCur = n;
        }
        else
        {
            void* p;
            if (m_pAllocator)
            {
                p = m_pAllocator->Reallocate(*m_ppMem, n * sizeof(**m_ppMem));
            }
            else
            {
                p = ReallocateMemoryNgc(*m_ppMem, n * sizeof(**m_ppMem));
            }
            if (!p) return false;
            *m_ppMem = reinterpret_cast<T*>(p);
            *m_pMemSize = n;
            *m_pCur = n;
        }
        return true;
    }
    inline bool Prepare(uint32_t n) NN_NOEXCEPT
    {
        // T must be int, uint32_t, ... etc. for example
        if (*m_pCur + n > *m_pMemSize)
        {
            uint32_t newsize;
            const uint32_t t = 4096 / sizeof(**m_ppMem);
            if (*m_pCur + n >= t)
            {
                newsize = ((*m_pCur + n) + t - 1) & ~(t - 1);
            }
            else
            {
                newsize = 1 << (32 - CountLeadingZero32(*m_pCur + n - 1));
            }
            void* p;
            if (m_pAllocator)
            {
                p = m_pAllocator->Reallocate(*m_ppMem, newsize * sizeof(**m_ppMem));
            }
            else
            {
                p = ReallocateMemoryNgc(*m_ppMem, newsize * sizeof(**m_ppMem));
            }
            if (!p)
            {
                return false;
            }
            *m_ppMem = reinterpret_cast<T*>(p);
            *m_pMemSize = newsize;
        }
        return true;
    }
    inline T& operator[](uint32_t n) NN_NOEXCEPT
    {
        // Prepare(n + 1) must be successful beforehand.
        return (*m_ppMem)[*m_pCur + n];
    }
    inline void Append(const T& v) NN_NOEXCEPT
    {
        // Prepare(1) must be successful beforehand.
        (*m_ppMem)[*m_pCur] = v;
        ++(*m_pCur);
    }
    inline void Advance(uint32_t n) NN_NOEXCEPT
    {
        // Prepare(n) must be successful beforehand.
        *m_pCur += n;
    }

private:
    T** m_ppMem;
    uint32_t* m_pCur;
    uint32_t* m_pMemSize;  // sizeof(*mem) * m_pMemSize bytes
    nn::ngc::detail::WorkBufAllocator* m_pAllocator;

    MiniReallocOut(const MiniReallocOut&) = delete;
    void operator=(const MiniReallocOut&) = delete;
};

}}} // nn::ngc::detail
