﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "../detail/ngc_WorkBufAllocator.h"
#include "./ngc_Sbv.h"

namespace nn { namespace ngc { namespace detail {

// AC の有限オートマトンについて、 goto() 部分のみを考え
// 簡潔データ構造で表すためのクラス
// AC の有限オートマトンの各ノードは以下の法則で id が割り振られます
//   - 接頭辞の昇順
//     - エッジのラベルが小さい順、かつ、ノードの深さが浅い順
//
// 例） { 11, 21, 22 } のキーワードを持つ AC オートマトンは以下のようになる
//      () をノード、ノードの中の数字をノード id 、
//      ----> をエッジ、エッジの中の数字をラベルとする
//  (0) -- 1 --> (1) -- 1 --> (2)
//   |---- 2 --> (4) -- 1 --> (3)
//                |---- 2 --> (5)
//
// これを GotoArcHolder では以下の簡潔データ構造で表す
// 0bit 目 : エッジのラベルが 0x00 のエッジをノード 0 が持っているなら 1
// 1bit 目 : エッジのラベルが 0x00 のエッジをノード 1 が持っているなら 1
// 2bit 目 : エッジのラベルが 0x00 のエッジをノード 2 が持っているなら 1
// ...
// 6bit 目 : エッジのラベルが 0x01 のエッジをノード 0 が持っているなら 1
// 7bit 目 : エッジのラベルが 0x01 のエッジをノード 1 が持っているなら 1
// ...
// したがってビットの数は 256 * ノードの数 になる
// 上の例だとビットベクトルは 000000 110010 100010 ...(後は 0 が 252 * 6 回続く)となる
class GotoArcHolder final
{
public:
    /**
     * @brief       ビットベクトル内でのノードを表すビットを返します
     * @param[in]   nodeid   知りたいノードの id
     * @param[in]   c        エッジのラベル
     */
    uint64_t GetPos(uint32_t nodeid, unsigned char c) NN_NOEXCEPT
    {
        uint64_t rval = m_NumNodeId;
        rval = rval * c + nodeid;
        return rval;
    }

public:
    GotoArcHolder() NN_NOEXCEPT ;
    ~GotoArcHolder() NN_NOEXCEPT;
    bool Init(uint32_t num_nodeid) NN_NOEXCEPT;
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator に置き換えます。
     */
    bool SetAllocator(nn::ngc::detail::WorkBufAllocator* pAllocator) NN_NOEXCEPT;
    /**
     * @brief   内部で行われる動的確保を WorkBufAllocator から元に戻します。
     */
    void ReleaseAllocator() NN_NOEXCEPT;
    /**
     * @brief       指定位置のビットを立てます
     * @param[in]   nodeid   ノード id
     * @param[in]   c        エッジのラベル(8bit)
     */
    bool TurnOn(uint32_t nodeid, unsigned char c) NN_NOEXCEPT
    {
        uint64_t idx = this->GetPos(nodeid, c);
        return m_Set.TurnOn(idx);
    }
    /**
     * @brief   SparseSet を Build します
     */
    bool Build() NN_NOEXCEPT
    {
        return m_Set.Build();
    }
    int32_t Goto(uint32_t nodeid, unsigned char c) NN_NOEXCEPT;
    void Reset() NN_NOEXCEPT;
    size_t MemSize() const NN_NOEXCEPT
    {
        return sizeof(m_NumNodeId) + m_Set.MemSize();
    }
    bool Export(BinaryWriter* w) const NN_NOEXCEPT;
    bool Import(BinaryReader* r) NN_NOEXCEPT;
    void Swap(GotoArcHolder& rhs) NN_NOEXCEPT
    {  // NOLINT
        using std::swap;
        swap(m_Set, rhs.m_Set);
        swap(m_NumNodeId, rhs.m_NumNodeId);
        nn::ngc::detail::WorkBufAllocator* pAllocatorTmp = rhs.m_pAllocator;
        rhs.m_pAllocator = m_pAllocator;
        m_pAllocator = pAllocatorTmp;
    }

private:
    SparseSet m_Set;
    uint32_t m_NumNodeId;   // ノード数
    nn::ngc::detail::WorkBufAllocator* m_pAllocator;

    GotoArcHolder(const GotoArcHolder&) = delete;
    void operator=(const GotoArcHolder&) = delete;
};

// (c, nodeId) -> nodeId' となる辞書を作成
// (c, nodeId)を1つの数値c##nodeIdにしてビットマップを作成
//   ここで、(c, nodeId)の並び順はnodeId'の並び順と同じになる。
//   nodeIdがsuffix lexicographic orderで整列されている場合、
//   c##nodeIdの順番もsuffix lexicographic orderになっている。
// したがってnodeId'は(c, nodeId)のrankを取れば取得できる。
inline GotoArcHolder::GotoArcHolder() NN_NOEXCEPT : m_Set(), m_NumNodeId(0), m_pAllocator(NULL) {}

inline GotoArcHolder::~GotoArcHolder() NN_NOEXCEPT
{
    if (m_pAllocator)
    {
        NN_SDK_LOG("ERROR: Please call GotoArcHolder::ReleaseAllocator() before calling GotoArcHolder::~GotoArcHolder()\n");
        NN_ABORT();
    }
}

/**
 * @brief       初期化します。
 * @param[in]   numNodeId   ノードの数
 */
inline bool GotoArcHolder::Init(uint32_t numNodeId) NN_NOEXCEPT
{
    if (numNodeId == 0)
    {
        return false;
    }
    m_NumNodeId = numNodeId;
    const unsigned char cmax = (1 << AhoCorasick::LabelWidth) - 1;
    return m_Set.Init(this->GetPos(numNodeId, cmax));
}

inline bool GotoArcHolder::SetAllocator(nn::ngc::detail::WorkBufAllocator* pAllocator) NN_NOEXCEPT
{
    if (!pAllocator)
    {
        return false;
    }
    m_pAllocator = pAllocator;

    m_Set.SetAllocator(m_pAllocator);
    return true;
}

inline void GotoArcHolder::ReleaseAllocator() NN_NOEXCEPT
{
    if (!m_pAllocator)
    {
        return;
    }
    m_Set.ReleaseAllocator();
    m_pAllocator = NULL;
}

/**
 * @brief   goto を実行します
 * @return  goto 先のノードの id を返します
 * @details ノード id が接頭辞の昇順でつけられているので、 rank1 操作をすることで
 *          行先を見つけることができる
 */
inline int32_t GotoArcHolder::Goto(uint32_t nodeId, unsigned char c) NN_NOEXCEPT
{
    uint64_t idx = this->GetPos(nodeId, c);
    uint32_t rank;
    bool has = m_Set.Has(&rank, idx);
    if (!has)
    {
        return -1;
    }
    return static_cast<int32_t>(rank);
}

/**
 * @brief   オブジェクトをコンストラクタが呼ばれた直後の状態にします。
 * @details SetAllocator() を呼び出している場合アロケータ情報は初期化されません。
 *          アロケータ情報の初期化には Reset() の後に ReleaseAllocator() を呼び出してください。
 */
inline void GotoArcHolder::Reset() NN_NOEXCEPT
{
    m_NumNodeId = 0;
    m_Set.Reset();
}

inline bool GotoArcHolder::Export(BinaryWriter* w) const NN_NOEXCEPT
{
    if (!w->Write(m_NumNodeId))
    {
        return false;
    }
    if (!m_Set.Export(w))
    {
        return false;
    }
    return true;
}

inline bool GotoArcHolder::Import(BinaryReader* r) NN_NOEXCEPT
{
    if (!r->Read(&m_NumNodeId) || !m_Set.Import(r))
    {
        this->Reset();
        return false;
    }
    return true;
}

// AC の実装クラス
// 簡潔データ構造の Map と Bp でできている
// GotoArcHolder は SparseSet
struct AhoCorasick::AhoCorasickPrivate
{
    Map lenHolder;         // holds len       // 葉ノードの接頭辞の長さを保持する Map
    Map reporttreeHolder;  // holds nodeid    // ノードが failure を使って単語の終端ノードにたどり着くかを集めた Map
    GotoArcHolder gotoarcHolder;              // AC オートマージの goto を簡潔データ構造で表したもの
    Bp failuretreeHolder;                     // failure で構成される木構造
};

}}} // nn::ngc::detail
