﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_StaticAssert.h>
#include <nn/settings/settings_Language.h>
#include <nn/ngc/ngc_ProfanityFilterBase.h>

#include "ngc_Private.h"
#include "detail/ngc_Config.h"

namespace nn
{
namespace ngc
{

namespace
{
    //! カタカナの小さいものと大きいもののマッピングルールです。
    const int BigsmallMapNum = 12;

    //! 後続に濁音も半濁音も伴わない半角カタカナと全角カタカナのマッピングルールの数です。
    const int HalfwidthMapNum = 58;

    //! 後続に濁音を伴う半角カタカナと全角カタカナのマッピングルールの数です。
    const int VoicedMapNum = 21;

    //! 後続に半濁音を伴う半角カタカナと全角カタカナのマッピングルールの数です。
    const int SemivoicedMapNum = 5;

    /**
     * @brief カタカナの大きい文字(ア)と小さい文字(ァ)のマッピングです。
     */
    struct SmallMap
    {
        //! 小さい文字(ァ)
        char16_t small;

        //! 大きい文字(ア)
        char16_t big;
    };
    // UTF-8 版
    struct SmallMapUtf8
    {
        char small[3];
        char big[3];
    };

    /*!
     * @brief カタカナの半角文字と全角文字のマッピングです。
     */
    struct HalfwidthMap
    {
        //! 半角文字
        char16_t half;

        //! 全角文字
        char16_t full;
    };
    // UTF-8 版
    struct HalfwidthMapUtf8
    {
        char half[3];
        char full[3];
    };

    /*!
      @brief 捨て仮名における、半角カタカナと全角カタカナのマッピングです。
     */
    const SmallMap g_SmallMapping[ BigsmallMapNum ] =
    {
        { 0x30A1, 0x30A2 },     // ア
        { 0x30A3, 0x30A4 },     // イ
        { 0x30A5, 0x30A6 },     // ウ
        { 0x30A7, 0x30A8 },     // エ
        { 0x30A9, 0x30AA },     // オ
        { 0x30C3, 0x30C4 },     // ツ
        { 0x30E3, 0x30E4 },     // ヤ
        { 0x30E5, 0x30E6 },     // ユ
        { 0x30E7, 0x30E8 },     // ヨ
        { 0x30EE, 0x30EF },     // ワ
        { 0x30F5, 0x30AB },     // カ
        { 0x30F6, 0x30B1 },     // ケ
    };
    // Utf-8 版
    const SmallMapUtf8 g_SmallMappingUtf8[BigsmallMapNum] =
    {
        { { '\xE3', '\x82', '\xA1' }, { '\xE3', '\x82', '\xA2' } },     // ァ ア
        { { '\xE3', '\x82', '\xA3' }, { '\xE3', '\x82', '\xA4' } },     // ィ イ
        { { '\xE3', '\x82', '\xA5' }, { '\xE3', '\x82', '\xA6' } },     // ゥ ウ
        { { '\xE3', '\x82', '\xA7' }, { '\xE3', '\x82', '\xA8' } },     // ェ エ
        { { '\xE3', '\x82', '\xA9' }, { '\xE3', '\x82', '\xAA' } },     // ォ オ
        { { '\xE3', '\x83', '\x83' }, { '\xE3', '\x83', '\x84' } },     // ッ ツ
        { { '\xE3', '\x83', '\xA3' }, { '\xE3', '\x83', '\xA4' } },     // ャ ヤ
        { { '\xE3', '\x83', '\xA5' }, { '\xE3', '\x83', '\xA6' } },     // ュ ユ
        { { '\xE3', '\x83', '\xA7' }, { '\xE3', '\x83', '\xA8' } },     // ョ ヨ
        { { '\xE3', '\x83', '\xAE' }, { '\xE3', '\x83', '\xAF' } },     // ヮ ワ
        { { '\xE3', '\x83', '\xB5' }, { '\xE3', '\x82', '\xAB' } },     // ヵ カ
        { { '\xE3', '\x83', '\xB6' }, { '\xE3', '\x82', '\xB1' } },     // ヶ ケ
    };

    /*!
      @brief 後ろに濁音も半濁音も伴わないときにおける、半角カタカナと全角カタカナのマッピングです。
     */
    const HalfwidthMap g_HalfMapping[ HalfwidthMapNum ] =
    {
        { 0xFF66, 0x30F2 },     // ヲ
        { 0xFF67, 0x30A1 },     // ァ
        { 0xFF68, 0x30A3 },     // ィ
        { 0xFF69, 0x30A5 },     // ゥ
        { 0xFF6A, 0x30A7 },     // ェ
        { 0xFF6B, 0x30A9 },     // ォ
        { 0xFF6C, 0x30E3 },     // ャ
        { 0xFF6D, 0x30E5 },     // ュ
        { 0xFF6E, 0x30E7 },     // ョ
        { 0xFF6F, 0x30C3 },     // ッ
        { 0xFF70, 0x30FC },     // ー
        { 0xFF71, 0x30A2 },     // ア
        { 0xFF72, 0x30A4 },     // イ
        { 0xFF73, 0x30A6 },     // ウ
        { 0xFF74, 0x30A8 },     // エ
        { 0xFF75, 0x30AA },     // オ
        { 0xFF76, 0x30AB },     // カ
        { 0xFF77, 0x30AD },     // キ
        { 0xFF78, 0x30AF },     // ク
        { 0xFF79, 0x30B1 },     // ケ
        { 0xFF7A, 0x30B3 },     // コ
        { 0xFF7B, 0x30B5 },     // サ
        { 0xFF7C, 0x30B7 },     // シ
        { 0xFF7D, 0x30B9 },     // ス
        { 0xFF7E, 0x30BB },     // セ
        { 0xFF7F, 0x30BD },     // ソ
        { 0xFF80, 0x30BF },     // タ
        { 0xFF81, 0x30C1 },     // チ
        { 0xFF82, 0x30C4 },     // ツ
        { 0xFF83, 0x30C6 },     // テ
        { 0xFF84, 0x30C8 },     // ト
        { 0xFF85, 0x30CA },     // ナ
        { 0xFF86, 0x30CB },     // ニ
        { 0xFF87, 0x30CC },     // ヌ
        { 0xFF88, 0x30CD },     // ネ
        { 0xFF89, 0x30CE },     // ノ
        { 0xFF8A, 0x30CF },     // ハ
        { 0xFF8B, 0x30D2 },     // ヒ
        { 0xFF8C, 0x30D5 },     // フ
        { 0xFF8D, 0x30D8 },     // ヘ
        { 0xFF8E, 0x30DB },     // ホ
        { 0xFF8F, 0x30DE },     // マ
        { 0xFF90, 0x30DF },     // ミ
        { 0xFF91, 0x30E0 },     // ム
        { 0xFF92, 0x30E1 },     // メ
        { 0xFF93, 0x30E2 },     // モ
        { 0xFF94, 0x30E4 },     // ヤ
        { 0xFF95, 0x30E6 },     // ユ
        { 0xFF96, 0x30E8 },     // ヨ
        { 0xFF97, 0x30E9 },     // ラ
        { 0xFF98, 0x30EA },     // リ
        { 0xFF99, 0x30EB },     // ル
        { 0xFF9A, 0x30EC },     // レ
        { 0xFF9B, 0x30ED },     // ロ
        { 0xFF9C, 0x30EF },     // ワ
        { 0xFF9D, 0x30F3 },     // ン
        { 0xFF9E, 0x3099 },     // 濁点
        { 0xFF9F, 0x309A },     // 半濁点
    };
    // UTF-8 版
    const HalfwidthMapUtf8 g_HalfMappingUtf8[HalfwidthMapNum] =
    {
        { { '\xEF', '\xBD', '\xA0' }, { '\xE3', '\x83', '\xB2' }  },     // ｦ ヲ
        { { '\xEF', '\xBD', '\xA7' }, { '\xE3', '\x82', '\xA1' } },     // ァ
        { { '\xEF', '\xBD', '\xA8' }, { '\xE3', '\x82', '\xA3' } },     // ィ
        { { '\xEF', '\xBD', '\xA9' }, { '\xE3', '\x82', '\xA5' } },     // ゥ
        { { '\xEF', '\xBD', '\xAA' }, { '\xE3', '\x82', '\xA7' } },     // ェ
        { { '\xEF', '\xBD', '\xAB' }, { '\xE3', '\x82', '\xA9' } },     // ォ
        { { '\xEF', '\xBD', '\xAC' }, { '\xE3', '\x83', '\xA3' } },     // ャ
        { { '\xEF', '\xBD', '\xAD' }, { '\xE3', '\x83', '\xA5' } },     // ュ
        { { '\xEF', '\xBD', '\xAE' }, { '\xE3', '\x83', '\xA7' } },     // ョ
        { { '\xEF', '\xBD', '\xAF' }, { '\xE3', '\x83', '\x83' } },     // ッ
        { { '\xEF', '\xBD', '\xB0' }, { '\xE3', '\x83', '\xBC' } },     // ー
        { { '\xEF', '\xBD', '\xB1' }, { '\xE3', '\x82', '\xA2' } },     // ア
        { { '\xEF', '\xBD', '\xB2' }, { '\xE3', '\x82', '\xA4' } },     // イ
        { { '\xEF', '\xBD', '\xB3' }, { '\xE3', '\x82', '\xA6' } },     // ウ
        { { '\xEF', '\xBD', '\xB4' }, { '\xE3', '\x82', '\xA8' } },     // エ
        { { '\xEF', '\xBD', '\xB5' }, { '\xE3', '\x82', '\xAA' } },     // オ
        { { '\xEF', '\xBD', '\xB6' }, { '\xE3', '\x82', '\xAB' } },     // カ
        { { '\xEF', '\xBD', '\xB7' }, { '\xE3', '\x82', '\xAD' } },     // キ
        { { '\xEF', '\xBD', '\xB8' }, { '\xE3', '\x82', '\xAF' } },     // ク
        { { '\xEF', '\xBD', '\xB9' }, { '\xE3', '\x82', '\xB1' } },     // ケ
        { { '\xEF', '\xBD', '\xBA' }, { '\xE3', '\x82', '\xB3' } },     // コ
        { { '\xEF', '\xBD', '\xBB' }, { '\xE3', '\x82', '\xB5' } },     // サ
        { { '\xEF', '\xBD', '\xBC' }, { '\xE3', '\x82', '\xB7' } },     // シ
        { { '\xEF', '\xBD', '\xBD' }, { '\xE3', '\x82', '\xB9' } },     // ス
        { { '\xEF', '\xBD', '\xBE' }, { '\xE3', '\x82', '\xBB' } },     // セ
        { { '\xEF', '\xBD', '\xBF' }, { '\xE3', '\x82', '\xBD' } },     // ソ
        { { '\xEF', '\xBE', '\x80' }, { '\xE3', '\x82', '\xBF' } },     // タ
        { { '\xEF', '\xBE', '\x81' }, { '\xE3', '\x83', '\x81' } },     // チ
        { { '\xEF', '\xBE', '\x82' }, { '\xE3', '\x83', '\x84' } },     // ツ
        { { '\xEF', '\xBE', '\x83' }, { '\xE3', '\x83', '\x86' } },     // テ
        { { '\xEF', '\xBE', '\x84' }, { '\xE3', '\x83', '\x88' } },     // ト
        { { '\xEF', '\xBE', '\x85' }, { '\xE3', '\x83', '\x8A' } },     // ナ
        { { '\xEF', '\xBE', '\x86' }, { '\xE3', '\x83', '\x8B' } },     // ニ
        { { '\xEF', '\xBE', '\x87' }, { '\xE3', '\x83', '\x8C' } },     // ヌ
        { { '\xEF', '\xBE', '\x88' }, { '\xE3', '\x83', '\x8D' } },     // ネ
        { { '\xEF', '\xBE', '\x89' }, { '\xE3', '\x83', '\x8E' } },     // ノ
        { { '\xEF', '\xBE', '\x8A' }, { '\xE3', '\x83', '\x8F' } },     // ハ
        { { '\xEF', '\xBE', '\x8B' }, { '\xE3', '\x83', '\x92' } },     // ヒ
        { { '\xEF', '\xBE', '\x8C' }, { '\xE3', '\x83', '\x95' } },     // フ
        { { '\xEF', '\xBE', '\x8D' }, { '\xE3', '\x83', '\x98' } },     // ヘ
        { { '\xEF', '\xBE', '\x8E' }, { '\xE3', '\x83', '\x9B' } },     // ホ
        { { '\xEF', '\xBE', '\x8F' }, { '\xE3', '\x83', '\x9E' } },     // マ
        { { '\xEF', '\xBE', '\x90' }, { '\xE3', '\x83', '\x9F' } },     // ミ
        { { '\xEF', '\xBE', '\x91' }, { '\xE3', '\x83', '\xA0' } },     // ム
        { { '\xEF', '\xBE', '\x92' }, { '\xE3', '\x83', '\xA1' } },     // メ
        { { '\xEF', '\xBE', '\x93' }, { '\xE3', '\x83', '\xA2' } },     // モ
        { { '\xEF', '\xBE', '\x94' }, { '\xE3', '\x83', '\xA4' } },     // ヤ
        { { '\xEF', '\xBE', '\x95' }, { '\xE3', '\x83', '\xA6' } },     // ユ
        { { '\xEF', '\xBE', '\x96' }, { '\xE3', '\x83', '\xA8' } },     // ヨ
        { { '\xEF', '\xBE', '\x97' }, { '\xE3', '\x83', '\xA9' } },     // ラ
        { { '\xEF', '\xBE', '\x98' }, { '\xE3', '\x83', '\xAA' } },     // リ
        { { '\xEF', '\xBE', '\x99' }, { '\xE3', '\x83', '\xAB' } },     // ル
        { { '\xEF', '\xBE', '\x9A' }, { '\xE3', '\x83', '\xAC' } },     // レ
        { { '\xEF', '\xBE', '\x9B' }, { '\xE3', '\x83', '\xAD' } },     // ロ
        { { '\xEF', '\xBE', '\x9C' }, { '\xE3', '\x83', '\xAF' } },     // ワ
        { { '\xEF', '\xBE', '\x9D' }, { '\xE3', '\x83', '\xB3' } },     // ン
        { { '\xEF', '\xBE', '\x9E' }, { '\xE3', '\x82', '\x99' } },     // 濁点
        { { '\xEF', '\xBE', '\x9F' }, { '\xE3', '\x82', '\x9A' } },     // 半濁点
    };

    /*!
      @brief 濁音が後ろについた場合の、半角カタカナと全角カタカナのマッピングです。
     */
    const HalfwidthMap g_VoicedMapping[ VoicedMapNum ] =
    {
        { 0xFF73, 0x30F4 },     // ウ
        { 0xFF76, 0x30AC },     // カ
        { 0xFF77, 0x30AE },     // キ
        { 0xFF78, 0x30B0 },     // ク
        { 0xFF79, 0x30B2 },     // ケ
        { 0xFF7A, 0x30B4 },     // コ
        { 0xFF7B, 0x30B6 },     // サ
        { 0xFF7C, 0x30B8 },     // シ
        { 0xFF7D, 0x30BA },     // ス
        { 0xFF7E, 0x30BC },     // セ
        { 0xFF7F, 0x30BE },     // ソ
        { 0xFF80, 0x30C0 },     // タ
        { 0xFF81, 0x30C2 },     // チ
        { 0xFF82, 0x30C5 },     // ツ
        { 0xFF83, 0x30C7 },     // テ
        { 0xFF84, 0x30C9 },     // ト
        { 0xFF8A, 0x30D0 },     // ハ
        { 0xFF8B, 0x30D3 },     // ヒ
        { 0xFF8C, 0x30D6 },     // フ
        { 0xFF8D, 0x30D9 },     // ヘ
        { 0xFF8E, 0x30DC },     // ホ
    };
    // UTF-8 版
    const HalfwidthMapUtf8 g_VoicedMappingUtf8[ VoicedMapNum ] =
    {
        { { '\xEF', '\xBD', '\xB3' }, { '\xE3', '\x83', '\xB4' } },    // ｳ ヴ
        { { '\xEF', '\xBD', '\xB6' }, { '\xE3', '\x82', '\xAC' } },    // ｶ ガ
        { { '\xEF', '\xBD', '\xB7' }, { '\xE3', '\x82', '\xAE' } },    // ｷ ギ
        { { '\xEF', '\xBD', '\xB8' }, { '\xE3', '\x82', '\xB0' } },    // ｸ グ
        { { '\xEF', '\xBD', '\xB9' }, { '\xE3', '\x82', '\xB2' } },    // ｹ ゲ
        { { '\xEF', '\xBD', '\xBA' }, { '\xE3', '\x82', '\xB4' } },    // コ
        { { '\xEF', '\xBD', '\xBB' }, { '\xE3', '\x82', '\xB6' } },    // サ
        { { '\xEF', '\xBD', '\xBC' }, { '\xE3', '\x82', '\xB8' } },    // シ
        { { '\xEF', '\xBD', '\xBD' }, { '\xE3', '\x82', '\xBA' } },    // ス
        { { '\xEF', '\xBD', '\xBE' }, { '\xE3', '\x82', '\xBC' } },    // セ
        { { '\xEF', '\xBD', '\xBF' }, { '\xE3', '\x82', '\xBD' } },    // ソ
        { { '\xEF', '\xBE', '\x80' }, { '\xE3', '\x83', '\x80' } },    // タ
        { { '\xEF', '\xBE', '\x81' }, { '\xE3', '\x83', '\x82' } },    // チ
        { { '\xEF', '\xBE', '\x82' }, { '\xE3', '\x83', '\x85' } },    // ツ
        { { '\xEF', '\xBE', '\x83' }, { '\xE3', '\x83', '\x87' } },    // テ
        { { '\xEF', '\xBE', '\x84' }, { '\xE3', '\x83', '\x89' } },    // ト
        { { '\xEF', '\xBE', '\x8A' }, { '\xE3', '\x83', '\x90' } },    // ハ
        { { '\xEF', '\xBE', '\x8B' }, { '\xE3', '\x83', '\x93' } },    // ヒ
        { { '\xEF', '\xBE', '\x8C' }, { '\xE3', '\x83', '\x96' } },    // フ
        { { '\xEF', '\xBE', '\x8D' }, { '\xE3', '\x83', '\x99' } },    // ヘ
        { { '\xEF', '\xBE', '\x8E' }, { '\xE3', '\x83', '\x9C' } },    // ホ
    };

    /*!
      @brief 半濁音が後ろについた場合の、半角カタカナと全角カタカナのマッピングです。
     */
    const HalfwidthMap g_SemiVoicedMapping[ SemivoicedMapNum ] =
    {
        { 0xFF8A, 0x30D1 },     // ハ
        { 0xFF8B, 0x30D4 },     // ヒ
        { 0xFF8C, 0x30D7 },     // フ
        { 0xFF8D, 0x30DA },     // ヘ
        { 0xFF8E, 0x30DD },     // ホ
    };
    // UTF-8 版
    const HalfwidthMapUtf8 g_SemiVoicedMappingUtf8[ SemivoicedMapNum ] =
    {
        { { '\xEF', '\xBE', '\x8A' }, { '\xE3', '\x83', '\x91' } },     // ﾊ パ
        { { '\xEF', '\xBE', '\x8B' }, { '\xE3', '\x83', '\x94' } },     // ﾋ ピ
        { { '\xEF', '\xBE', '\x8C' }, { '\xE3', '\x83', '\x97' } },     // ﾌ プ
        { { '\xEF', '\xBE', '\x8D' }, { '\xE3', '\x83', '\x9A' } },     // ﾍ ペ
        { { '\xEF', '\xBE', '\x8E' }, { '\xE3', '\x83', '\x9D' } },     // ﾎ ポ
    };



    /*!--------------------------------------------------------------------------*
      Name:         ConvertKatakanaToBig

      @param pKana  [in/out] 変換するカタカナ

      @brief        小さなカタカナを大きなカタカナへ変換します。
     *---------------------------------------------------------------------------*/
    void ConvertKatakanaToBig( char16_t *pKana ) NN_NOEXCEPT
    {
        for ( int i = 0; i < BigsmallMapNum; i++ )
        {
            if ( g_SmallMapping[ i ].small == *pKana )
            {
                *pKana = g_SmallMapping[ i ].big;
                return;
            }
        }
    }
    // UTF-8 版
    void ConvertKatakanaToBig(char *pOutKana) NN_NOEXCEPT
    {
        for (int i = 0; i < BigsmallMapNum; i++)
        {
            if (g_SmallMappingUtf8[i].small[0] == pOutKana[0] &&
                g_SmallMappingUtf8[i].small[1] == pOutKana[1] &&
                g_SmallMappingUtf8[i].small[2] == pOutKana[2])
            {
                pOutKana[0] = g_SmallMappingUtf8[i].big[0];
                pOutKana[1] = g_SmallMappingUtf8[i].big[1];
                pOutKana[2] = g_SmallMappingUtf8[i].big[2];
                return;
            }
        }
    }

    /*!--------------------------------------------------------------------------*
      Name:         ConvertHalfwidthKatakanaToFullwidth

      @param [out] pFullwidth   出力先へのポインタ
      @param [in]  halfwidth    変換元のカタカナ
      @param [in]  pMapping     変換マッピングルールの先頭へのポインタ
      @param [in]  mapNum       マッピングルールのサイズ

      @brief        指定されたマッピングルールに従って、半角カタカナを全角カタカナに
                    変換する処理を行います。
     *---------------------------------------------------------------------------*/
    bool ConvertHalfwidthKatakanaToFullwidth( char16_t *pFullwidth,
                                              char16_t halfwidth,
                                              const HalfwidthMap* pMapping,
                                              int mapNum ) NN_NOEXCEPT
    {
        for ( int i = 0; i < mapNum; i++ )
        {
            if ( pMapping[ i ].half == halfwidth )
            {
                *pFullwidth = pMapping[ i ].full;
                return true;
            }
        }
        return false;
    }

    // UTF8 版
    bool ConvertHalfwidthKatakanaToFullwidth(char *pOutFullwidth,
                                             const char *pHalfwidth,
                                             const HalfwidthMapUtf8* pMapping,
                                             int mapNum) NN_NOEXCEPT
    {
        for (int i = 0; i < mapNum; i++)
        {
            if (pMapping[i].half[0] == pHalfwidth[0] &&
                pMapping[i].half[1] == pHalfwidth[1] &&
                pMapping[i].half[2] == pHalfwidth[2])
            {
                pOutFullwidth[0] = pMapping[i].full[0];
                pOutFullwidth[1] = pMapping[i].full[1];
                pOutFullwidth[2] = pMapping[i].full[2];
                return true;
            }
        }
        return false;
    }


}   // namespace

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::ConvertUserInputForWord

  @param [out] pConvertedWord       書き出し先へのポインタを指定します。
  @param [in]  length               変換後のバッファサイズを指定します。
  @param [in]  pWord                ユーザーが指定した文字列へのポインタを指定します。

  @brief        ユーザーが入力した単語を、マッチング用文字列へと変換します。
                この変換ルールは任天堂の認証サーバで行われている処理と同様です。
 *---------------------------------------------------------------------------*/
void ProfanityFilterBase::ConvertUserInputForWord( char16_t* pConvertedWord,
                                                   int length,
                                                   const char16_t* pWord ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pConvertedWord );
    NN_SDK_ASSERT_NOT_NULL( pWord );

    const SmallMap pTable[] = { { 0x00FF, 0x0178 },     // ÿ Ÿ (フランス語)
                                { 0x0153, 0x0152 },     // œ Œ (フランス語)
                                { 0x0133, 0x0132 },     // ij IJ (オランダ語)
                                { 0x00DF, 0x1E9E },     // ß ẞ (ドイツ語)
                                { 0x0451, 0x0401 }      // ё Ё (ロシア語)
                              };

    int destIndex = 0;
    int srcIndex  = 0;

    for ( ;; )
    {
        // 変換後は小さくなっているはずです。
        NN_SDK_ASSERT( destIndex <= srcIndex );

        // バッファ最大量を超えそうなときはnullptrで埋めて抜けます。
        if ( destIndex == length - 1 )
        {
            pConvertedWord[ destIndex ] = L'\0';
            break;
        }
        // スペースは削除します
        if (pWord[srcIndex] == L' ' || pWord[srcIndex] == 0x3000 || pWord[srcIndex] == 0x00A0)
        {
            srcIndex++;
            continue;
        }

        if (pWord[srcIndex] < 0xFF00)
        {
            if (pWord[srcIndex] < 0x3041)
            {
                // アルファベット大文字を小文字へ
                if ((0x0041 <= pWord[srcIndex] && pWord[srcIndex] <= 0x005A) ||
                    (0x0410 <= pWord[srcIndex] && pWord[srcIndex] <= 0x042F) ||
                    (0x00C0 <= pWord[srcIndex] && pWord[srcIndex] <= 0x00DE && pWord[srcIndex] != 0x00D7)
                    )
                {
                    pConvertedWord[destIndex] = pWord[srcIndex] + 0x20;
                    srcIndex++;
                    destIndex++;
                    continue;
                }
            }
            else
            {
                // ひらがなをカタカナへ
                if (0x3041 <= pWord[srcIndex] && pWord[srcIndex] <= 0x3096)
                {
                    pConvertedWord[destIndex] = pWord[srcIndex] - 0x3041 + 0x30A1;
                    ConvertKatakanaToBig(pConvertedWord + destIndex);
                    srcIndex++;
                    destIndex++;
                    continue;
                }
                // カタカナの小さい文字を大きく
                else if (0x30A1 <= pWord[srcIndex] && pWord[srcIndex] <= 0x30FA)
                {
                    pConvertedWord[destIndex] = pWord[srcIndex];
                    ConvertKatakanaToBig(pConvertedWord + destIndex);
                    srcIndex++;
                    destIndex++;
                    continue;
                }
            }
        }
        else
        {
            if (pWord[srcIndex] < 0xFF41)
            {
                // 全角数字は半角へ
                // ０[0xFF10] - ９[0xFF19]
                if (0xFF10 <= pWord[srcIndex] && pWord[srcIndex] <= 0xFF19)
                {
                    pConvertedWord[destIndex] = pWord[srcIndex] - 0xFF10 + L'0';
                    srcIndex++;
                    destIndex++;
                    continue;
                }
                // 全角アルファベットは半角へ
                // Ａ[0xFF21] - Ｚ[0xFF3A]
                else if (0xFF21 <= pWord[srcIndex] && pWord[srcIndex] <= 0xFF3A)
                {
                    // 大文字は小文字にする
                    pConvertedWord[destIndex] = pWord[srcIndex] - 0xFF21 + L'a';
                    srcIndex++;
                    destIndex++;
                    continue;
                }
            }
            else
            {
                // 全角アルファベットは半角へ
                // ａ[0xFF41] - ｚ[0xFF5A]
                if (0xFF41 <= pWord[srcIndex] && pWord[srcIndex] <= 0xFF5A)
                {
                    pConvertedWord[destIndex] = pWord[srcIndex] - 0xFF41 + L'a';
                    srcIndex++;
                    destIndex++;
                    continue;
                }
                // 半角カタカナを全角カタカナへ
                else if (0xFF66 <= pWord[srcIndex] && pWord[srcIndex] <= 0xFF9F)
                {
                    // 濁点を伴う
                    if (pWord[srcIndex + 1] == 0xFF9E)
                    {
                        if (ConvertHalfwidthKatakanaToFullwidth(pConvertedWord + destIndex,
                            pWord[srcIndex], g_VoicedMapping, VoicedMapNum))
                        {
                            srcIndex += 2;
                            destIndex++;
                            continue;
                        }
                    }
                    // 半濁点を伴う
                    else if (pWord[srcIndex + 1] == 0xFF9F)
                    {
                        if (ConvertHalfwidthKatakanaToFullwidth(pConvertedWord + destIndex,
                            pWord[srcIndex], g_SemiVoicedMapping, SemivoicedMapNum))
                        {
                            srcIndex += 2;
                            destIndex++;
                            continue;
                        }
                    }

                    // 何も伴わない
                    ConvertHalfwidthKatakanaToFullwidth(pConvertedWord + destIndex,
                                                        pWord[srcIndex],
                                                        g_HalfMapping, HalfwidthMapNum);
                    ConvertKatakanaToBig(pConvertedWord + destIndex);
                    srcIndex++;
                    destIndex++;
                    continue;
                }
            }
        }
        // その他小文字に変換しなければならない文字との比較
        bool isFind = false;
        for (int i = 0; i < sizeof(pTable) / sizeof(pTable[0]); ++i)
        {
            if (pWord[srcIndex] == pTable[i].big)
            {
                pConvertedWord[destIndex] = pTable[i].small;
                isFind = true;
                break;
            }
        }
        if (!isFind)
        {
            pConvertedWord[destIndex] = pWord[srcIndex];
            if (pConvertedWord[destIndex] == L'\0')
            {
                break;
            }
        }
        srcIndex++;
        destIndex++;
    }
}   // NOLINT(impl/function_size)

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::ConvertUserInputForText

  @param [out] pConvertedWord       書き出し先へのポインタを指定します。
  @param [out] pSourceLengthMapping 変換前と後のマッピング
  @param [in]  length               変換後のバッファサイズを指定します。
  @param [in]  pWord                ユーザーが指定した文字列へのポインタを指定します。

  @brief        ユーザーの入力を、長文チェック用に変換します。
                この処理は任天堂の認証サーバの挙動と似ていますが、半角スペースを
                削除しない点が異なります。また、文章中に現れた不正文字に対して
                マスク処理を行うために、元の文字列に対するマッピング情報を格納します。
  @details      pSourceLengthMapping : 変換前と後のマッピング
                - インデックス   pConvertedWord のインデックスと同じ
                - 値             変換前の文字のバイト数を 1/n にした際の n (絶対値)が入る
                    - 正の値の場合 … 文字数的に 1 対 1 対応している
                    - 負の値の場合 … 文字数的に 1 対 n 対応している
                        - 負の値は半角カタカナの濁点、半濁点が使われた際に利用される
                        (例えば「ｿﾞ」は変換後は「ゾ」となり 2 文字が 1 文字になるため)
                        (* 埋めするときは「ｿﾞ」は「**」となる。「*」ではない)
 *---------------------------------------------------------------------------*/
void ProfanityFilterBase::ConvertUserInputForText( char* pConvertedWord,
                                                   int8_t* pSourceLengthMapping,
                                                   size_t length,
                                                   const char* pWord ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConvertedWord);
    NN_SDK_ASSERT_NOT_NULL(pSourceLengthMapping);
    NN_SDK_ASSERT_NOT_NULL(pWord);

    const SmallMapUtf8 pTable[] = { { "\xc3\xbf", "\xc5\xb8" },                 // ÿ Ÿ (フランス語)
                                    { "\xc5\x93", "\xc5\x92" },                 // œ Œ (フランス語)
                                    { "\xc4\xb3", "\xc4\xb2" },                 // ij IJ (オランダ語)
                                    { "\xd1\x91", "\xd0\x81" }                  // ё Ё (ロシア語)
                                  };

    int destIndex = 0;
    int srcIndex = 0;

    for (;;)
    {
        // 変換後は小さくなっているはずです。
        NN_SDK_ASSERT(destIndex <= srcIndex);

        // バッファ最大量を超えそうなときは 0 埋めで抜けます。
        if (static_cast<size_t>(destIndex) == length - 1)
        {
            pConvertedWord[destIndex] = L'\0';
            pSourceLengthMapping[destIndex] = 0;
            break;
        }
        if (pWord[srcIndex] == '\xEF')
        {
            // 全角アルファベット小文字は半角アルファベット小文字へ
            // ａ 0xEF 0xBD 0x81 (U+FF41) - ｚ 0xEF 0xBD 0x9A (U+FF5A)
            // 必ず正しい UTF-8 文字列が来るので UTF-8 の正当性チェックの必要はない
            if (pWord[srcIndex + 1] == '\xBD' && '\x81' <= pWord[srcIndex + 2] && pWord[srcIndex + 2] <= '\x9A')
            {
                pConvertedWord[destIndex] = pWord[srcIndex + 2] - '\x81' + 'a';
                pSourceLengthMapping[destIndex] = 3;
                srcIndex += 3;
                destIndex++;
            }
            else if (pWord[srcIndex + 1] == '\xBC')
            {
                // 全角アルファベット大文字は半角アルファベット小文字へ
                // Ａ 0xEF 0xBC 0xA1 (U+FF21) - Ｚ 0xEF 0xBC 0xBA (U+FF3A)
                if ('\xA1' <= pWord[srcIndex + 2] && pWord[srcIndex + 2] <= '\xBA')
                {
                    pConvertedWord[destIndex] = pWord[srcIndex + 2] - '\xA1' + 'a';
                    pSourceLengthMapping[destIndex] = 3;
                    srcIndex += 3;
                    destIndex++;
                }
                // 全角数字は半角へ
                // ０ 0xEF 0xBC 0x90 (U+FF10) - ９ 0xEF 0xBC 0x99 (U+FF19)
                else if ('\x90' <= pWord[srcIndex + 2] && pWord[srcIndex + 2] <= '\x99')
                {
                    pConvertedWord[destIndex] = pWord[srcIndex + 2] - '\x90' + '0';
                    pSourceLengthMapping[destIndex] = 3;
                    srcIndex += 3;
                    destIndex++;
                }
                else
                {
                    pConvertedWord[destIndex] = pWord[srcIndex];
                    if (pConvertedWord[destIndex] == '\0')
                    {
                        break;
                    }
                    pSourceLengthMapping[destIndex] = 1;
                    srcIndex++;
                    destIndex++;
                }
            }
            // 半角カタカナを全角カタカナへ
            else if ( static_cast<size_t>(destIndex) + 2 < length - 1 &&
                      ( (pWord[srcIndex + 1] == '\xBD' && pWord[srcIndex + 2] >= '\xA6' && pWord[srcIndex + 2] <= '\xBF') ||
                        (pWord[srcIndex + 1] == '\xBE' && pWord[srcIndex + 2] >= '\x80' && pWord[srcIndex + 2] <= '\x9F') ) )
            {
                // 濁点を伴う
                if (pWord[srcIndex + 3] == '\xEF' && pWord[srcIndex + 4] == '\xBE' && pWord[srcIndex + 5] == '\x9E')
                {
                    if (ConvertHalfwidthKatakanaToFullwidth(pConvertedWord + destIndex,
                        &pWord[srcIndex], g_VoicedMappingUtf8, VoicedMapNum))
                    {
                        // バイト数だけでなく文字数にも差が出ることを負数にして伝える
                        pSourceLengthMapping[destIndex] = -2;
                        pSourceLengthMapping[destIndex + 1] = 2;
                        pSourceLengthMapping[destIndex + 2] = 2;
                        srcIndex += 6;
                        destIndex += 3;
                        continue;
                    }
                }
                // 半濁点を伴う
                else if (pWord[srcIndex + 3] == '\xEF' && pWord[srcIndex + 4] == '\xBE' && pWord[srcIndex + 5] == '\x9F')
                {
                    if (ConvertHalfwidthKatakanaToFullwidth(pConvertedWord + destIndex,
                        &pWord[srcIndex], g_SemiVoicedMappingUtf8, SemivoicedMapNum))
                    {
                        pSourceLengthMapping[destIndex] = -2;
                        pSourceLengthMapping[destIndex + 1] = 2;
                        pSourceLengthMapping[destIndex + 2] = 2;
                        srcIndex += 6;
                        destIndex += 3;
                        continue;
                    }
                }

                // 何も伴わない
                ConvertHalfwidthKatakanaToFullwidth(pConvertedWord + destIndex,
                    &pWord[srcIndex], g_HalfMappingUtf8, HalfwidthMapNum);
                ConvertKatakanaToBig(pConvertedWord + destIndex);
                pSourceLengthMapping[destIndex] = 1;
                pSourceLengthMapping[destIndex + 1] = 1;
                pSourceLengthMapping[destIndex + 2] = 1;
                srcIndex += 3;
                destIndex += 3;
            }
            else
            {
                pConvertedWord[destIndex] = pWord[srcIndex];
                if (pConvertedWord[destIndex] == '\0')
                {
                    break;
                }
                pSourceLengthMapping[destIndex] = 1;
                srcIndex++;
                destIndex++;
            }
        }
        else if (pWord[srcIndex] == '\xE3' && static_cast<size_t>(destIndex) + 2 < length - 1)
        {
            // ひらがなをカタカナへ
            if ( (pWord[srcIndex + 1] == '\x81' && pWord[srcIndex + 2] >= '\x81' && pWord[srcIndex + 2] <= '\xBF') ||
                 (pWord[srcIndex + 1] == '\x82' && pWord[srcIndex + 2] >= '\x80' && pWord[srcIndex + 2] <= '\x93') )
            {
                pConvertedWord[destIndex] = pWord[srcIndex];
                if (pWord[srcIndex + 1] == '\x81' && pWord[srcIndex + 2] >= '\x81' && pWord[srcIndex + 2] <= '\x9F')
                {
                    pConvertedWord[destIndex + 1] = '\x82';
                    pConvertedWord[destIndex + 2] = pWord[srcIndex + 2] + '\x20';
                }
                else if (pWord[srcIndex + 1] == '\x81' && pWord[srcIndex + 2] >= '\xA0' && pWord[srcIndex + 2] <= '\xBF')
                {
                    pConvertedWord[destIndex + 1] = '\x83';
                    pConvertedWord[destIndex + 2] = pWord[srcIndex + 2] - '\x20';
                }
                else if (pWord[srcIndex + 1] == '\x82' && pWord[srcIndex + 2] >= '\x80' && pWord[srcIndex + 2] <= '\x93')
                {
                    pConvertedWord[destIndex + 1] = '\x83';
                    pConvertedWord[destIndex + 2] = pWord[srcIndex + 2] + '\x20';
                }
                ConvertKatakanaToBig(pConvertedWord + destIndex);
                pSourceLengthMapping[destIndex] = 1;
                pSourceLengthMapping[destIndex + 1] = 1;
                pSourceLengthMapping[destIndex + 2] = 1;
                srcIndex += 3;
                destIndex += 3;
            }
            // カタカナの小さい文字（捨て仮名）を大きく
            else if ( (pWord[srcIndex + 1] == '\x82' && pWord[srcIndex + 2] >= '\xA1' && pWord[srcIndex + 2] <= '\xBF') ||
                        (pWord[srcIndex + 1] == '\x83' && pWord[srcIndex + 2] >= '\x80' && pWord[srcIndex + 2] <= '\xB6') )
            {
                pConvertedWord[destIndex] = pWord[srcIndex];
                pConvertedWord[destIndex + 1] = pWord[srcIndex + 1];
                pConvertedWord[destIndex + 2] = pWord[srcIndex + 2];
                ConvertKatakanaToBig(pConvertedWord + destIndex);
                pSourceLengthMapping[destIndex] = 1;
                pSourceLengthMapping[destIndex + 1] = 1;
                pSourceLengthMapping[destIndex + 2] = 1;
                srcIndex += 3;
                destIndex += 3;
            }
            else
            {
                pConvertedWord[destIndex] = pWord[srcIndex];
                if (pConvertedWord[destIndex] == '\0')
                {
                    break;
                }
                pSourceLengthMapping[destIndex] = 1;
                srcIndex++;
                destIndex++;
            }
        }
        // アルファベット大文字を小文字へ
        else if ( ('A' <= pWord[srcIndex] && pWord[srcIndex] <= 'Z') )
        {
            pConvertedWord[destIndex] = pWord[srcIndex] + '\x20';
            pSourceLengthMapping[destIndex] = 1;
            srcIndex++;
            destIndex++;
        }
        // ラテン 1 補助大文字を小文字へ(英語以外のアルファベット)
        else if ( (pWord[srcIndex] == '\xd0' && '\x90' <= pWord[srcIndex + 1] &&
                   pWord[srcIndex + 1] <= '\xaf') ||
                  (pWord[srcIndex] == '\xc3' && '\x80' <= pWord[srcIndex + 1] &&
                   pWord[srcIndex + 1] <= '\x9e' &&  pWord[srcIndex + 1] != '\x97')
                )
        {
            unsigned char upper = static_cast<unsigned char>(pWord[srcIndex]);
            unsigned char lower = static_cast<unsigned char>(pWord[srcIndex + 1]) + '\x20';
            if (lower >= 0xC0)
            {
                // UTF-8 として有効な次の値へ繰り上がり
                upper++;
                lower = 0x80 + (lower - 0xC0);
            }
            pConvertedWord[destIndex] = static_cast<char>(upper);
            pConvertedWord[destIndex + 1] = static_cast<char>(lower);
            pSourceLengthMapping[destIndex] = 1;
            pSourceLengthMapping[destIndex + 1] = 1;
            srcIndex += 2;
            destIndex += 2;
        }
        // ẞ → ß (ドイツ語)
        else if (pWord[srcIndex] == '\xe1' && pWord[srcIndex + 1] == '\xba' && pWord[srcIndex + 2] == '\x9e')
        {
            pConvertedWord[destIndex] = '\xc3';
            pConvertedWord[destIndex + 1] = '\x9f';
            pSourceLengthMapping[destIndex] = 1;
            pSourceLengthMapping[destIndex + 1] = 2;
            srcIndex += 3;
            destIndex += 2;
        }
        else
        {
            // その他小文字に変換しなければならない文字との比較
            bool isFind = false;
            for (int i = 0; i < sizeof(pTable) / sizeof(pTable[0]); ++i)
            {
                isFind = true;
                for (int j = 0; j < 3 && pTable[i].big[j] != '\0'; ++j)
                {
                    if (pWord[srcIndex + j] != pTable[i].big[j])
                    {
                        isFind = false;
                        break;
                    }
                }
                if (isFind)
                {
                    for (int j = 0; j < 3 && pTable[i].small[j] != '\0'; ++j)
                    {
                        pConvertedWord[destIndex] = pTable[i].small[j];
                        pSourceLengthMapping[destIndex] = 1;
                        srcIndex++;
                        destIndex++;
                    }
                    break;
                }
            }
            if (!isFind)
            {
                pConvertedWord[destIndex] = pWord[srcIndex];
                if (pConvertedWord[destIndex] == '\0')
                {
                    break;
                }
                pSourceLengthMapping[destIndex] = 1;
                srcIndex++;
                destIndex++;
            }
        }
    }
}   // NOLINT(impl/function_size)

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::IsIncludesAtSign

  @param [in]   pWord    調査対象文字列を指定します。
  @param [in]   length   調査対象文字列が取りうる最大の長さを指定します。
                         この長さに達するか、nullptr文字が見つかった場合に調査は終了します。

  @brief        アットマーク記号が含まれているか含まれていないかを調査します。
 *---------------------------------------------------------------------------*/
bool ProfanityFilterBase::IsIncludesAtSign( const char16_t* pWord, int length ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pWord );
    for ( int i = 0; i < length; i++ )
    {
        if ( pWord[ i ] == L'\0' )
        {
            break;
        }

        // ＠[0xFF20]
        if ( pWord[ i ] == L'@' || pWord[ i ] == 0xFF20 )
        {
            return true;
        }
    }
    return false;
}

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::GetPatternBitsFromRegion

  @param [in]  bCommunicateWithOtherRegions     このアプリケーションが他のリージョンとの通信を
                                                行うかどうかを指定します。
  @param [in]  checkDesiredLanguage             アプリケーションが対応する最も優先度の高い言語に対してチェックを行うかを指定します。

  @brief        cfgから現在のリージョンコードと現在の言語設定を取得し、
                その情報とUGCガイドラインの情報を元に、チェックすべきパターンリストを
                取得する処理です。
 *---------------------------------------------------------------------------*/
Bit32 ProfanityFilterBase::GetPatternBitsFromRegion(
                                                  bool bCommunicateWithOtherRegions,
                                                  bool checkDesiredLanguage) NN_NOEXCEPT
{
    // ビットフラグで結果を返している都合上、
    // 32bit値までにリージョン数が収まっている必要があります。
    NN_STATIC_ASSERT( ProfanityFilterPatternList_Max <= 32 );
    // 他リージョンのとの通信でチェックしたい単語は、リストの内容で対応するので、
    // リストの組み合わせを変える必要はない。
    NN_UNUSED( bCommunicateWithOtherRegions );

    // 基本的には1個以上
    Bit32 patternBits = 0;

    // リージョン、言語の取得
    Bit32 language = nn::ngc::detail::GetLanguageSetting(checkDesiredLanguage);
    patternBits |= language;

    return patternBits;
}

}   // namespace ngc
}   // namespace nn
