﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_Endian.h>
#include <nn/ngc/ngc_Result.h>
#include <nn/ngc/ngc_ProfanityFilterBase.h>
#include <nn/ngc/ngc_ProfanityFilter.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_Decompression.h>

#include "ngc_Private.h"
#include "ngc_String.h"
#include "detail/ngc_Contents.h"
#include "detail/ngc_Config.h"
#include "detail/ngc_WorkBufAllocator.h"

#if defined( NN_BUILD_CONFIG_OS_WIN32 )
#include <nn/nn_Windows.h>
#include "detail/ngc_AcAutomaton-os.win32.h"
#define GET_CHAR16LITERAL( string ) \
            reinterpret_cast< const char16_t* >( NN_CHAR16LITERAL( string ) )
#else
#define GET_CHAR16LITERAL( string ) NN_CHAR16LITERAL( string )
#endif

#include "./succinct/ngc_AhoCorasick.h"
#include "./succinct/ngc_MemoryInputStream.h"

using nn::ngc::detail::AhoCorasick;
using nn::ngc::detail::Sbv;
using nn::ngc::detail::BinaryWriter;
using nn::ngc::detail::BinaryReader;
using nn::ngc::detail::MemoryInputStream;

namespace {

    // NG キーワードの '*' 借り埋め処理で利用する UTF-8 ではありえない文字
    const char g_DummyChar = '\xC1';
    const char g_DummyForOneChar = '\xC0';

    // Initialize の参照カウント
    int g_InitializeCount = 0;
    // 参照カウントを守る Mutex
    struct StaticMutex
    {
        nn::os::MutexType mutex;
        void lock() NN_NOEXCEPT
        {
            nn::os::LockMutex(&mutex);
        }
        void unlock() NN_NOEXCEPT
        {
            nn::os::UnlockMutex(&mutex);
        }
    } g_InitializeCountMutex = { NN_OS_MUTEX_INITIALIZER(false) };

    /**
     * @brief   CheckProfanityWords の AC 法のコールバックで利用する構造体です。
     */
    struct WordMatchSet
    {
        nn::Bit32*   checkResult;
        nn::ngc::ProfanityFilterPatternList patternList;
        bool checkDesiredLanguage;
    };

    /**
     * @brief   AhoCorasick::Match() 時にユーザオブジェクトとして渡す構造体
     */
    struct ScrapedTextMatchSet
    {
        char* pTextUtf8;                            // 元のテキスト
        char* pConvertedTextUtf8;                   // pTextUtf8 の変換後のテキスト
        char* pScrapedText;                         // pConvertedTextUtf8 から区切り文字をなくしたもの
        int8_t* pMapping;                           // pConvertedTextUtf8 から pTextUtf8 の位置を割り出すためのマッピング
        int* pProfanityWordCount;                   // 不正文字のカウンタ
        nn::ngc::ProfanityFilter::MaskMode mode;    // マスクモード
        Sbv* pBoundPos;                             // pScrapedText から pConvertedTextUtf8 の位置を割り出すための簡潔ビットベクトル
        AhoCorasick* pAcB2;                         // NG ワードチェック用 AC ツリー（区切り文字あり Ver.）
    };

    /**
     * @brief   UTF-8 文字とそのサイズの組です
     */
    struct SizeAndChar
    {
        int size;
        char word[8];
    };

    const int g_BoundCount = 80;            // 単語境界のパターンの数

    // 単語境界の種類
    const SizeAndChar g_pWordBound[g_BoundCount] = {
        { 1,{ '\x0D' } },                   // キャリッジリターン
        { 1,{ '\x0A' } },                   // ラインフィールド
        { 2,{ '\xC2', '\x85' } },           // 次行
        { 3,{ '\xE2', '\x80', '\xA8' } },   // 行区切り
        { 3,{ '\xE2', '\x80', '\xA9' } },   // 段落区切り
        { 1,{ '\x09' } },                   // タブ
        { 1,{ '\x0B' } },                   // 垂直タブ
        { 1,{ '\x0C' } },                   // フォームフィード（用紙送り）
        { 1,{ '\x20' } },                   // 半角スペース
        { 3,{ '\xEF', '\xBD', '\xA1' } },   // ｡ 句点（半角）
        { 3,{ '\xEF', '\xBD', '\xA4' } },   // ､ 読点（半角）
        { 1,{ '\x2E' } },                   // . ピリオド（半角）
        { 1,{ '\x2C' } },                   // , カンマ（半角）
        { 1,{ '\x5B' } },                   // [
        { 1,{ '\x21' } },                   // !
        { 1,{ '\x22' } },                   // "
        { 1,{ '\x23' } },                   // #
        { 1,{ '\x24' } },                   // $
        { 1,{ '\x25' } },                   // %
        { 1,{ '\x26' } },                   // &
        { 1,{ '\x27' } },                   // '
        { 1,{ '\x28' } },                   // (
        { 1,{ '\x29' } },                   // )
        { 1,{ '\x2A' } },                   // *
        { 1,{ '\x2B' } },                   // +
        { 1,{ '\x2F' } },                   // /
        { 1,{ '\x3A' } },                   // :
        { 1,{ '\x3B' } },                   // ;
        { 1,{ '\x3C' } },                   // <
        { 1,{ '\x3D' } },                   // =
        { 1,{ '\x3E' } },                   // >
        { 1,{ '\x3F' } },                   // ?
        { 1,{ '\x5C' } },                   // \ (エンマークもしくはバックスラッシュ)
        { 1,{ '\x40' } },                   // @
        { 1,{ '\x5E' } },                   // ^
        { 1,{ '\x5F' } },                   // _
        { 1,{ '\x60' } },                   // `
        { 1,{ '\x7B' } },                   // {
        { 1,{ '\x7C' } },                   // |
        { 1,{ '\x7D' } },                   // }
        { 1,{ '\x7E' } },                   // ~
        { 1,{ '\x2D' } },                   // -
        { 1,{ '\x5D' } },                   // ]
        { 3,{ '\xE3', '\x80', '\x80' } },   // 全角スペース
        { 3,{ '\xE3', '\x80', '\x82' } },   // 。 句点（全角）
        { 3,{ '\xE3', '\x80', '\x81' } },   // 、 読点（全角）
        { 3,{ '\xEF', '\xBC', '\x8E' } },   // ． ピリオド（全角）
        { 3,{ '\xEF', '\xBC', '\x8C' } },   // ， カンマ（全角）
        { 3,{ '\xEF', '\xBC', '\xBB' } },   // ［
        { 3,{ '\xEF', '\xBC', '\x81' } },   // ！
        { 3,{ '\xE2', '\x80', '\x9C' } },   // “
        { 3,{ '\xE2', '\x80', '\x9D' } },   // ”
        { 3,{ '\xEF', '\xBC', '\x83' } },   // ＃
        { 3,{ '\xEF', '\xBC', '\x84' } },   // ＄
        { 3,{ '\xEF', '\xBC', '\x85' } },   // ％
        { 3,{ '\xEF', '\xBC', '\x86' } },   // ＆
        { 3,{ '\xE2', '\x80', '\x98' } },   // ‘
        { 3,{ '\xE2', '\x80', '\x99' } },   // ’
        { 3,{ '\xEF', '\xBC', '\x88' } },   // （
        { 3,{ '\xEF', '\xBC', '\x89' } },   // ）
        { 3,{ '\xEF', '\xBC', '\x8A' } },   // ＊
        { 3,{ '\xEF', '\xBC', '\x8B' } },   // ＋
        { 3,{ '\xEF', '\xBC', '\x8F' } },   // ／
        { 3,{ '\xEF', '\xBC', '\x9A' } },   // ：
        { 3,{ '\xEF', '\xBC', '\x9B' } },   // ；
        { 3,{ '\xEF', '\xBC', '\x9C' } },   // ＜
        { 3,{ '\xEF', '\xBC', '\x9D' } },   // ＝
        { 3,{ '\xEF', '\xBC', '\x9E' } },   // ＞
        { 3,{ '\xEF', '\xBC', '\x9F' } },   // ？
        { 3,{ '\xEF', '\xBC', '\xA0' } },   // ＠
        { 3,{ '\xEF', '\xBF', '\xA5' } },   // ￥
        { 3,{ '\xEF', '\xBC', '\xBE' } },   // ＾
        { 3,{ '\xEF', '\xBC', '\xBF' } },   // ＿
        { 3,{ '\xEF', '\xBD', '\x80' } },   // ｀
        { 3,{ '\xEF', '\xBD', '\x9B' } },   // ｛
        { 3,{ '\xEF', '\xBD', '\x9C' } },   // ｜
        { 3,{ '\xEF', '\xBD', '\x9D' } },   // ｝
        { 3,{ '\xEF', '\xBD', '\x9E' } },   // ～
        { 3,{ '\xEF', '\xBC', '\x8D' } },   // －
        { 3,{ '\xEF', '\xBC', '\xBD' } }    // ］
    };

    // システムデータマウント用のキャッシュ領域
    static char g_NgcSystemDataCacheBuffer[NgcSystemDataCacheSize] = {};

    // システムデータ呼び出しを管理するクラスインスタンス
    nn::ngc::detail::ContentsReader g_ContentsReader;

    /**
     * @brief   nn::fs::MountSystemData 対応した InputStream 基底のファイル読み込みクラス
     * @details AhoCorasick クラスの Import() で MountRom を使いたいため独自定義します
     */
    class AcInputStream : public nn::ngc::detail::InputStream {
    public:
        AcInputStream() NN_NOEXCEPT : m_WorkBufSize(4096), m_IsOpen(false), m_BufSize(0),
            m_DecompressDstSize(0), m_FileSize(0), m_Offset(0), m_pBuf(NULL),
            m_pReadData(NULL), m_pDecompressDstBuf(NULL), m_pDecompressWorkBuf(NULL),
            m_pAllocator(NULL) {}
        ~AcInputStream() NN_NOEXCEPT
        {
            Finalize();
        }

        /**
         * @brief       初期化します
         * @param[in]   pBuf    ワークバッファ。 GetWorkBufSize() 以上必要
         * @param[in]   bufSize pBuf のサイズ
         */
        nn::ngc::detail::errno_t Init(void* pBuf, size_t bufSize, nn::ngc::detail::WorkBufAllocator* pAllocator) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(pBuf);

            NN_SDK_ASSERT(bufSize >= m_WorkBufSize);
            if (m_pBuf)
            {
                return EALREADY;
            }
            if (!pAllocator)
            {
                return EINVAL;
            }
            m_pBuf = reinterpret_cast<unsigned char*>(pBuf);
            m_BufSize = bufSize;
            m_Offset = 0;
            m_pAllocator = pAllocator;

            return 0;
        }

        /**
         * @brief   終了処理です
         */
        void Finalize() NN_NOEXCEPT
        {
            if (m_pBuf)
            {
                if (m_IsOpen)
                {
                    nn::fs::CloseFile(m_File);
                }
                m_pBuf = NULL;
                if (m_pReadData)
                {
                    m_pAllocator->Free(m_pReadData);
                    m_pReadData = NULL;
                }
                if (m_pDecompressDstBuf)
                {
                    m_pAllocator->Free(m_pDecompressDstBuf);
                    m_pDecompressDstBuf = NULL;
                }
                if (m_pDecompressWorkBuf)
                {
                    m_pAllocator->Free(m_pDecompressWorkBuf);
                    m_pDecompressWorkBuf = NULL;
                }
                m_pAllocator = NULL;
            }
        }

        /**
         * @brief       ファイルをオープンします
         * @param[in]   filename    ファイル名。g_ContentsReader でマウントされたパスである必要がある
         * @pre         Init() 済みである必要があります
         */
        nn::ngc::detail::errno_t Open(const char* filename)
        {
            NN_SDK_ASSERT_NOT_NULL(m_pBuf);
            NN_SDK_ASSERT(m_BufSize > 0);
            if (m_IsOpen)
            {
                return 0;
            }
            if (!g_ContentsReader.CheckMountPrefix(filename))
            {
                return EINVAL;
            }

            // Initialize() 時にマウントされているのでマウントは不要
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&m_File, filename, nn::fs::OpenMode_Read));
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&m_FileSize, m_File));

            // ファイルを読んで解凍
            NN_SDK_ASSERT(m_FileSize > 0);
            m_pReadData = m_pAllocator->Allocate(static_cast<size_t>(m_FileSize));
            m_pDecompressWorkBuf = m_pAllocator->Allocate(nn::util::DecompressGzipWorkBufferSize);
            NN_SDK_ASSERT(m_pReadData && m_pDecompressWorkBuf);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(m_File, 0, m_pReadData, static_cast<size_t>(m_FileSize)));

            m_DecompressDstSize = nn::util::GetGzipDecompressedSize(m_pReadData, static_cast<size_t>(m_FileSize));
            m_pDecompressDstBuf = reinterpret_cast<char*>(m_pAllocator->Allocate(m_DecompressDstSize));
            NN_SDK_ASSERT(m_pDecompressDstBuf);
            NN_ABORT_UNLESS(nn::util::DecompressGzip(m_pDecompressDstBuf, m_DecompressDstSize, m_pReadData, static_cast<size_t>(m_FileSize), m_pDecompressWorkBuf, nn::util::DecompressGzipWorkBufferSize));

            m_IsOpen = true;

            return 0;
        }

        /**
         * @brief   必要なワークバッファのサイズを返します。
         */
        size_t GetWorkBufSize()
        {
            return m_WorkBufSize;
        }
    private:
        // InputStream の派生クラスとしてオーバーライドしておかなければならない関数 3 つ

        /**
         * @brief   ファイルを nbytes 分読み込み p に格納します
         * @return  実際に読み込まれたデータのサイズ
         */
        virtual size_t FillBuffer_(void* p, size_t nbytes) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(p);
            size_t n;

            if (static_cast<int64_t>(nbytes) + m_Offset > static_cast<int64_t>(m_DecompressDstSize))
            {
                // オフセットに不正な値が入らないよう調整
                NN_SDK_ASSERT(m_Offset >= 0);
                nbytes = m_DecompressDstSize - static_cast<size_t>(m_Offset);
            }
            if (nbytes == 0)
            {
                return 0;
            }
            memcpy(p, &(m_pDecompressDstBuf[static_cast<size_t>(m_Offset)]), nbytes);
            n = nbytes;
            m_Offset += nbytes;

            return n;
        }

        /**
         * @brief   内部のハンドル等をクローズします
         */
        virtual bool Close_() NN_NOEXCEPT
        {
            if (m_IsOpen)
            {
                nn::fs::CloseFile(m_File);
                m_IsOpen = false;
            }
            return true;
        }

        /**
         * @brief   InputStream がバッファを獲得します。
         */
        virtual void* GetWorkBuffer_(size_t* nbytes) NN_NOEXCEPT
        {
            if (!m_pBuf)
            {
                return NULL;
            }
            *nbytes = m_BufSize;
            return m_pBuf;
        }

    private:
        const size_t m_WorkBufSize;                         // ワークバッファとして必要なサイズ (=4096)
        bool m_IsOpen;                                      // ファイルはオープン状態か
        size_t m_BufSize;                                   // m_pBuf のサイズ
        size_t m_DecompressDstSize;                         // m_pDecompressDstBuf のサイズ
        int64_t m_FileSize;                                 // 読み込むファイルのサイズ
        int64_t m_Offset;                                   // ファイル読み込み位置
        unsigned char* m_pBuf;                              // InputStream が利用するワークバッファ
        void* m_pReadData;                                  // 解凍前のファイルから読み込んだデータ
        char* m_pDecompressDstBuf;                          // 解凍結果のデータを指すポインタ
        void* m_pDecompressWorkBuf;                         // 解凍に使うワークバッファ
        nn::ngc::detail::WorkBufAllocator* m_pAllocator;    // ワークバッファのアロケータ
        nn::fs::FileHandle m_File;                          // ファイルハンドル
    };

    /**
        @brief 切り上げ計算
     */
    template <typename T, typename U>
    NN_FORCEINLINE T RoundUp( T value, U alignment ) NN_NOEXCEPT
    {
        return ( ( ( value ) + ( alignment - 1 ) ) & ~( alignment - 1 ) );
    }

    /**
     * @brief   nn::ngc::ProfanityFilter::m_WorkingHead から、ワークバッファ用アロケータのインスタンスを返します
     */
    inline nn::ngc::detail::WorkBufAllocator* GetAllocator(uintptr_t ptr)
    {
        return reinterpret_cast<nn::ngc::detail::WorkBufAllocator*>(ptr);
    }

    /**
     * @brief       UTF-8 部分文字列の文字数(≠バイト数)を返します。
     * @param[in]   pStr    UTF-8 文字列
     * @param[in]   first   UTF-8 文字列の文字数をカウントしたい部分の先頭インデックス
     * @param[in]   last    UTF-8 文字列の文字数をカウントしたい部分の直後インデックス(末尾ではない)
     * @details     終端文字が来ても処理を打ち切りません
     */
    size_t GetLengthOfUtf8(const char* pStr, size_t first, size_t last)
    {
        size_t count = 0;
        for (size_t i = first; i < last;)
        {
            if (pStr[i] >= '\x00' && pStr[i] <= '\x7F')
            {
                i++;
            }
            else if (pStr[i] >= '\xC2' && pStr[i] <= '\xDF')
            {
                i += 2;
            }
            else if (pStr[i] >= '\xE0' && pStr[i] <= '\xEF')
            {
                i += 3;
            }
            else if (pStr[i] >= '\xF0' && pStr[i] <= '\xF7')
            {
                i += 4;
            }
            else
            {
                return 0;
            }
            count++;
        }
        return count;
    }

    /**
     * @brief       UTF-8 文字列の指定した文字数のインデックスを返します
     * @param[out]  pOutIndex   返却するインデックス
     * @param[in]   pStr        対象 UTF-8 文字列
     * @param[in]   length      pStr の長さ
     * @return      成功したら true を返します
     * @details     length より先に終端文字が来た場合は *pOutIndex に 0 を代入します
     *              指定した文字数の直後のインデックスではないので注意してください
     */
    bool GetUtf8Ptr(size_t* pOutIndex, const char* pStr, size_t length)
    {
        size_t i = 0;
        size_t retVal = 0;
        size_t count = 0;
        for (count = 0; count < length; ++count)
        {
            if (pStr[i] > '\x00' && pStr[i] <= '\x7F')
            {
                retVal = i;
                i++;
            }
            else if (pStr[i] >= '\xC2' && pStr[i] <= '\xDF')
            {
                retVal = i;
                i += 2;
            }
            else if (pStr[i] >= '\xE0' && pStr[i] <= '\xEF')
            {
                retVal = i;
                i += 3;
            }
            else if (pStr[i] >= '\xF0' && pStr[i] <= '\xF7')
            {
                retVal = i;
                i += 4;
            }
            else if (pStr[i] == '\0')
            {
                *pOutIndex = 0;
                return true;
            }
            else
            {
                NN_SDK_LOG("ERROR: Invalid UTF-8 string\n");
                return false;
            }
        }
        *pOutIndex = retVal;    // 超えた直後ではなく超える前の末尾インデックスを返す

        return true;
    }

    nn::Bit32 GetCheckResultPattern(const nn::ngc::ProfanityFilterPatternList pattern,
        bool checkDesiredLanguage)  NN_NOEXCEPT
    {
        nn::Bit32 resultPattern = pattern;
        if (pattern == NgcProfanityFilterPatternListCommon)
        {
            resultPattern |= nn::ngc::detail::GetLanguageSetting(checkDesiredLanguage);
        }
        return resultPattern;
    }

    /**
     * @brief       マッチした NG キーワードが現れるテキスト中の位置の直前ビットに
     *              区切り文字が含まれているかどうかを判定します
     * @param[in]   pStr    チェック対象文字列
     * @param[in]   pos     マッチした NG キーワードの先頭が現れる pTextUtf8 のインデックス
     * @returns     含まれているなら true を返します
     */
    bool CheckFirstIsBoundary(const char* pStr, size_t pos)
    {
        if (pos == 0)
        {
            // 先頭の場合は区切り文字であったとみなす（ngc の仕様に合わせる）
            return true;
        }

        for (int i = 0; i < g_BoundCount; ++i)
        {
            if (pos < static_cast<size_t>(g_pWordBound[i].size))
            {
                continue;
            }
            bool isMatch = true;
            for (int j = 0; j < g_pWordBound[i].size; ++j)
            {
                if (pStr[pos - 1 - j] != g_pWordBound[i].word[(g_pWordBound[i].size - 1 - j)])
                {
                    isMatch = false;
                    break;
                }
            }
            if (isMatch)
            {
                return true;
            }
        }
        return false;
    }

    /**
     * @brief       マッチした NG キーワードが現れるテキスト中の位置の直後ビットに
     *              区切り文字が含まれているかどうかを判定します
     * @param[in]   pStr    チェック対象文字列
     * @param[in]   strSize pStr のサイズ
     * @param[in]   pos     マッチした NG キーワードの直後が現れる pTextUtf8 のインデックス
     * @returns     含まれているなら true を返します
     */
    bool CheckLastIsBoundary(const char* pStr, size_t strSize, size_t pos)
    {
        if (pos == strSize || pStr[pos] == '\0')
        {
            // last が末尾である場合は、区切り文字であったとみなす（ngc に合わせる）
            return true;
        }

        for (int i = 0; i < g_BoundCount; ++i)
        {
            if (pos + g_pWordBound[i].size - 1 >= strSize)
            {
                continue;
            }
            bool isMatch = true;
            for (int j = 0; j < g_pWordBound[i].size; ++j)
            {
                if (pStr[pos + j] != g_pWordBound[i].word[j])
                {
                    isMatch = false;
                    break;
                }
            }
            if (isMatch)
            {
                return true;
            }
        }
        return false;
    }

    /**
     * @brief       pStr から単語境界を取り除いた文字列を pOut に入れます
     * @param[out]  pOut            pStr から単語境界を抜いたもの
     * @param[out]  pOutBoundPos    単語境界がある位置を記憶しておく簡潔データ構造
     * @param[in]   pStr            単語境界を抜く対象 UTF-8 文字列
     * @details     pOutBoundPos に単語境界がある位置を設定します
     *              単語境界がある位置が 1 になる Sbv ができます
     */
    size_t CreateScrapedText(char* pOut, Sbv* pOutBoundPos, const char* pStr)
    {
        size_t count = 0;
        for (size_t i = 0; i < pOutBoundPos->GetBitVectorSize(); ++i)
        {
            bool isMatch = true;
            for (int j = 0; j < g_BoundCount; ++j)
            {
                isMatch = true;
                const char* pTmp = &pStr[i];
                for (int k = 0; k < g_pWordBound[j].size; ++k)
                {
                    if (*pTmp != g_pWordBound[j].word[k])
                    {
                        isMatch = false;
                        break;
                    }
                    pTmp++;
                }
                if (isMatch)
                {
                    // 単語境界が見つかった
                    // 単語境界の位置をメモ
                    for (int k = 0; k < g_pWordBound[j].size; ++k)
                    {
                        pOutBoundPos->TurnOn(static_cast<uint32_t>(i + k));
                    }
                    // 走査位置を変える
                    i += g_pWordBound[j].size - 1;
                    break;
                }
            }
            if (!isMatch)
            {
                // 単語境界でないなら pScrapedText へ代入
                pOut[count] = pStr[i];
                count++;
            }
        }
        return count;
    }

    /**
     * @brief   AhoCorasick 木を読み込みます
     */
    bool ImportAcTree(AhoCorasick* pAcNotB, AhoCorasick* pAcB1, AhoCorasick* pAcB2,
        nn::ngc::ProfanityFilterPatternList patternList,
        nn::ngc::detail::WorkBufAllocator* pAllocator,
        int pattern)
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        // pattern から必要なパスを生成
        char pAcListForNxNotB[64];  // \b を持たないキーワードの AC
        char pAcListForNxB1[64];    // \b を持つキーワードの AC 1 (\b を抜いてキーワードを格納)
        char pAcListForNxB2[64];    // \b を持つキーワードの AC 2 (\b を抜かずにキーワードを格納)
        NN_ABORT_UNLESS(g_ContentsReader.MakeMountPath(pAcListForNxNotB, 64, nn::ngc::detail::ContentsReader::AcType_NotBoundary, pattern));
        NN_ABORT_UNLESS(g_ContentsReader.MakeMountPath(pAcListForNxB1, 64, nn::ngc::detail::ContentsReader::AcType_BoundaryScraped, pattern));
        NN_ABORT_UNLESS(g_ContentsReader.MakeMountPath(pAcListForNxB2, 64, nn::ngc::detail::ContentsReader::AcType_BoundaryNotScraped, pattern));
        const size_t workBufSize = 4096;
        void* pWorkBuf = pAllocator->Allocate(workBufSize);
#else
        NN_UNUSED(pAllocator);
        NN_UNUSED(pattern);
#endif
        for (int i = 0; i < 3; ++i)
        {
            if ((i == 0 && !pAcNotB) || (i == 1 && !pAcB1) || (i == 2 && !pAcB2))
            {
                continue;
            }
            BinaryReader reader;

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
            AcInputStream stream;
            NN_SDK_ASSERT(workBufSize >= stream.GetWorkBufSize());
            // NX なら独自定義 InputStream を利用してシステムデータから NG リストを読み込み
            nn::ngc::detail::errno_t e = stream.Init(pWorkBuf, stream.GetWorkBufSize(), pAllocator);
            if (e != 0)
            {
                NN_SDK_LOG("AcInputStream Init error!: %d\n", e);
                NN_ABORT();
            }
            switch (i)
            {
            case 0:
                e = stream.Open(pAcListForNxNotB);
                break;
            case 1:
                e = stream.Open(pAcListForNxB1);
                break;
            case 2:
                e = stream.Open(pAcListForNxB2);
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
                break;
            }
            if (e != 0)
            {
                NN_SDK_LOG("AcInputStream Open error!: %d\n", e);
                NN_ABORT();
            }
#else
            auto SwitchInitArray = [&](int i, MemoryInputStream* pStream,
                unsigned char* pNotB, size_t notBSize,
                unsigned char* pB1, size_t b1Size,
                unsigned char* pB2, size_t b2Size)
            {
                switch (i)
                {
                case 0:
                    pStream->Init(pNotB, notBSize);
                    break;
                case 1:
                    pStream->Init(pB1, b1Size);
                    break;
                case 2:
                    pStream->Init(pB2, b2Size);
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                    break;
                }
            };

            MemoryInputStream stream;
            if (patternList == NgcProfanityFilterPatternListCommon)
            {
                //stream.Init(nn::ngc::detail::pAcCommon);
                SwitchInitArray(i, &stream,
                    nn::ngc::detail::pAcCommonNotB, sizeof(nn::ngc::detail::pAcCommonNotB),
                    nn::ngc::detail::pAcCommonB1, sizeof(nn::ngc::detail::pAcCommonB1),
                    nn::ngc::detail::pAcCommonB2, sizeof(nn::ngc::detail::pAcCommonB2));
            }
            else
            {
                switch (patternList)
                {
                case nn::ngc::ProfanityFilterPatternList_Japanese:
                    SwitchInitArray(i, &stream,
                        nn::ngc::detail::pAcJapaneseNotB, sizeof(nn::ngc::detail::pAcJapaneseNotB),
                        nn::ngc::detail::pAcJapaneseB1, sizeof(nn::ngc::detail::pAcJapaneseB1),
                        nn::ngc::detail::pAcJapaneseB2, sizeof(nn::ngc::detail::pAcJapaneseB2));
                    break;
                case nn::ngc::ProfanityFilterPatternList_AmericanEnglish:
                    SwitchInitArray(i, &stream,
                        nn::ngc::detail::pAcAmericanEnglishNotB, sizeof(nn::ngc::detail::pAcAmericanEnglishNotB),
                        nn::ngc::detail::pAcAmericanEnglishB1, sizeof(nn::ngc::detail::pAcAmericanEnglishB1),
                        nn::ngc::detail::pAcAmericanEnglishB2, sizeof(nn::ngc::detail::pAcAmericanEnglishB2));
                    break;
                case nn::ngc::ProfanityFilterPatternList_CanadianFrench:
                    SwitchInitArray(i, &stream,
                        nn::ngc::detail::pAcCanadianFrenchNotB, sizeof(nn::ngc::detail::pAcCanadianFrenchNotB),
                        nn::ngc::detail::pAcCanadianFrenchB1, sizeof(nn::ngc::detail::pAcCanadianFrenchB1),
                        nn::ngc::detail::pAcCanadianFrenchB2, sizeof(nn::ngc::detail::pAcCanadianFrenchB2));
                    break;
                case nn::ngc::ProfanityFilterPatternList_LatinAmericanSpanish:
                    SwitchInitArray(i, &stream,
                        nn::ngc::detail::pAcLatinAmericanSpanishNotB, sizeof(nn::ngc::detail::pAcLatinAmericanSpanishNotB),
                        nn::ngc::detail::pAcLatinAmericanSpanishB1, sizeof(nn::ngc::detail::pAcLatinAmericanSpanishB1),
                        nn::ngc::detail::pAcLatinAmericanSpanishB2, sizeof(nn::ngc::detail::pAcLatinAmericanSpanishB2));
                    break;
                case nn::ngc::ProfanityFilterPatternList_BritishEnglish:
                    SwitchInitArray(i, &stream,
                        nn::ngc::detail::pAcBritishEnglishNotB, sizeof(nn::ngc::detail::pAcBritishEnglishNotB),
                        nn::ngc::detail::pAcBritishEnglishB1, sizeof(nn::ngc::detail::pAcBritishEnglishB1),
                        nn::ngc::detail::pAcBritishEnglishB2, sizeof(nn::ngc::detail::pAcBritishEnglishB2));
                    break;
                case nn::ngc::ProfanityFilterPatternList_French:
                    SwitchInitArray(i, &stream,
                        nn::ngc::detail::pAcFrenchNotB, sizeof(nn::ngc::detail::pAcFrenchNotB),
                        nn::ngc::detail::pAcFrenchB1, sizeof(nn::ngc::detail::pAcFrenchB1),
                        nn::ngc::detail::pAcFrenchB2, sizeof(nn::ngc::detail::pAcFrenchB2));
                    break;
                case nn::ngc::ProfanityFilterPatternList_German:
                    SwitchInitArray(i, &stream,
                        nn::ngc::detail::pAcGermanNotB, sizeof(nn::ngc::detail::pAcGermanNotB),
                        nn::ngc::detail::pAcGermanB1, sizeof(nn::ngc::detail::pAcGermanB1),
                        nn::ngc::detail::pAcGermanB2, sizeof(nn::ngc::detail::pAcGermanB2));
                    break;
                case nn::ngc::ProfanityFilterPatternList_Italian:
                    SwitchInitArray(i, &stream,
                        nn::ngc::detail::pAcItalianNotB, sizeof(nn::ngc::detail::pAcItalianNotB),
                        nn::ngc::detail::pAcItalianB1, sizeof(nn::ngc::detail::pAcItalianB1),
                        nn::ngc::detail::pAcItalianB2, sizeof(nn::ngc::detail::pAcItalianB2));
                    break;
                case nn::ngc::ProfanityFilterPatternList_Spanish:
                    SwitchInitArray(i, &stream,
                        nn::ngc::detail::pAcSpanishNotB, sizeof(nn::ngc::detail::pAcSpanishNotB),
                        nn::ngc::detail::pAcSpanishB1, sizeof(nn::ngc::detail::pAcSpanishB1),
                        nn::ngc::detail::pAcSpanishB2, sizeof(nn::ngc::detail::pAcSpanishB2));
                    break;
                case nn::ngc::ProfanityFilterPatternList_Dutch:
                    SwitchInitArray(i, &stream,
                        nn::ngc::detail::pAcDutchNotB, sizeof(nn::ngc::detail::pAcDutchNotB),
                        nn::ngc::detail::pAcDutchB1, sizeof(nn::ngc::detail::pAcDutchB1),
                        nn::ngc::detail::pAcDutchB2, sizeof(nn::ngc::detail::pAcDutchB2));
                    break;
                case nn::ngc::ProfanityFilterPatternList_Korean:
                    SwitchInitArray(i, &stream,
                        nn::ngc::detail::pAcKoreanNotB, sizeof(nn::ngc::detail::pAcKoreanNotB),
                        nn::ngc::detail::pAcKoreanB1, sizeof(nn::ngc::detail::pAcKoreanB1),
                        nn::ngc::detail::pAcKoreanB2, sizeof(nn::ngc::detail::pAcKoreanB2));
                    break;
                case nn::ngc::ProfanityFilterPatternList_SimplifiedChinese:
                    SwitchInitArray(i, &stream,
                        nn::ngc::detail::pAcChineseNotB, sizeof(nn::ngc::detail::pAcChineseNotB),
                        nn::ngc::detail::pAcChineseB1, sizeof(nn::ngc::detail::pAcChineseB1),
                        nn::ngc::detail::pAcChineseB2, sizeof(nn::ngc::detail::pAcChineseB2));
                    break;
                case nn::ngc::ProfanityFilterPatternList_Portuguese:
                    SwitchInitArray(i, &stream,
                        nn::ngc::detail::pAcPortugueseNotB, sizeof(nn::ngc::detail::pAcPortugueseNotB),
                        nn::ngc::detail::pAcPortugueseB1, sizeof(nn::ngc::detail::pAcPortugueseB1),
                        nn::ngc::detail::pAcPortugueseB2, sizeof(nn::ngc::detail::pAcPortugueseB2));
                    break;
                case nn::ngc::ProfanityFilterPatternList_Russian:
                    SwitchInitArray(i, &stream,
                        nn::ngc::detail::pAcRussianNotB, sizeof(nn::ngc::detail::pAcRussianNotB),
                        nn::ngc::detail::pAcRussianB1, sizeof(nn::ngc::detail::pAcRussianB1),
                        nn::ngc::detail::pAcRussianB2, sizeof(nn::ngc::detail::pAcRussianB2));
                    break;
                case nn::ngc::ProfanityFilterPatternList_SouthAmericanPortuguese:
                    SwitchInitArray(i, &stream,
                        nn::ngc::detail::pAcSouthAmericanPortugueseNotB, sizeof(nn::ngc::detail::pAcSouthAmericanPortugueseNotB),
                        nn::ngc::detail::pAcSouthAmericanPortugueseB1, sizeof(nn::ngc::detail::pAcSouthAmericanPortugueseB1),
                        nn::ngc::detail::pAcSouthAmericanPortugueseB2, sizeof(nn::ngc::detail::pAcSouthAmericanPortugueseB2));
                    break;
                case nn::ngc::ProfanityFilterPatternList_TraditionalChinese:
                    SwitchInitArray(i, &stream,
                        nn::ngc::detail::pAcTaiwaneseNotB, sizeof(nn::ngc::detail::pAcTaiwaneseNotB),
                        nn::ngc::detail::pAcTaiwaneseB1, sizeof(nn::ngc::detail::pAcTaiwaneseB1),
                        nn::ngc::detail::pAcTaiwaneseB2, sizeof(nn::ngc::detail::pAcTaiwaneseB2));
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
#endif

            reader.Init(reader.EndianSetting_EndianLittle);
            reader.Open(&stream);
            switch (i)
            {
            case 0:
                NN_ABORT_UNLESS(pAcNotB->Import(&reader));
                break;
            case 1:
                NN_ABORT_UNLESS(pAcB1->Import(&reader));
                break;
            case 2:
                NN_ABORT_UNLESS(pAcB2->Import(&reader));
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
                break;
            }
            reader.Close();
            stream.Close();
        }
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        pAllocator->Free(pWorkBuf);
#endif

        return true;
    }   // NOLINT(impl/function_size)

    /**
     * @brief           pOutStr を法則にしたがって * 埋めします
     * @param[in/out]   pOutStr     * 埋めする文字列
     * @param[in/out]   pOutSize    pOut のサイズ。処理後は * 埋め後のサイズを返す
     * @details         '*' で文字埋めするときの法則
     *                  マルチバイト文字も 1 つの * で埋めるための対応
     *                  UTF-8 的にありえない 0x00 が出てきたら * 埋めであると判定
     *                      - 1バイト目: 0x00
     *                      - 2バイト目: * 埋めする前の NG キーワードのバイト数
     *                      - 3バイト目: * 埋めした後のバイト数(=文字数)
     *                  2 バイト目と 3 バイト目は、 0xFF(255)までで足りない場合、
     *                  次のバイトに残りの差分を入れる。その分 3 バイト目は後ろにずれる。
     *                  e.g. * 埋めする前の NG キーワードのバイト数が 510 で、
     *                       * 埋めした後のバイト数が 1 だった場合
     *                      -> 0x00 0xFF 0xFF 0x00 0x01
     */
    void FillWithAsterisk(char* pOutStr, size_t* pOutSize)
    {
        for (size_t i = 0; i < *pOutSize; ++i)
        {
            if (pOutStr[i] == 0x00 && i + 1 < *pOutSize)
            {
                size_t beforeByte = 0;  // * 埋めする前の NG キーワードのバイト数
                size_t afterByte = 0;   // * 埋めした後のバイト数(=文字数)
                size_t index = i + 1;
                while (index < *pOutSize)
                {
                    beforeByte += static_cast<unsigned char>(pOutStr[index]);
                    if (static_cast<unsigned char>(pOutStr[index]) != 0xFFu)
                    {
                        index++;
                        break;
                    }
                    index++;
                }
                while (index < *pOutSize)
                {
                    afterByte += static_cast<unsigned char>(pOutStr[index]);
                    if (static_cast<unsigned char>(pOutStr[index]) != 0xFFu)
                    {
                        index++;
                        break;
                    }
                    index++;

                }
                if (index > *pOutSize)
                {
                    // * 埋めは発生しない
                    return;
                }
                NN_SDK_ASSERT(beforeByte > 0 && afterByte <= beforeByte);

                for (size_t j = i; j < i + beforeByte; ++j)
                {
                    if (j < i + afterByte)
                    {
                        pOutStr[j] = '*';
                    }
                    else
                    {
                        // 後で切り詰める部分には 0 を入れておく
                        pOutStr[j] = '\x0';
                    }
                }
                i += (beforeByte - 1);
            }
            else if (pOutStr[i] == g_DummyForOneChar &&
                     i + 1 < *pOutSize && pOutStr[i + 1] == g_DummyForOneChar)
            {
                // 2 バイトの NG キーワードを '*' 1 つに置き換え
                pOutStr[i] = '*';
                pOutStr[i + 1] = '\x0';
                i++;
            }
            else if (pOutStr[i] == g_DummyChar)
            {
                // 借り埋め部分を * に
                pOutStr[i] = '*';
            }
        }
        // 0 埋めした部分を詰める
        for (size_t i = 0; i < *pOutSize; ++i)
        {
            if (pOutStr[i] == '\x0')
            {
                size_t j;
                for (j = i; pOutStr[j] == '\x0' && j < *pOutSize; ++j) {}
                if (j >= *pOutSize)
                {
                    break;
                }
                size_t k = i;
                while (k + (j - i) < *pOutSize)
                {
                    for (size_t l = 0; l < j - i && k + (j - i) < *pOutSize; ++l)
                    {
                        std::swap(pOutStr[k], pOutStr[k + (j - i)]);
                        ++k;
                    }
                }
                *pOutSize = k;
            }
        }
        *pOutSize = strlen(pOutStr);
    }

    /**
     * @brief   pConvertedTextUtf8 を見ながら pMapping を更新します。
     * @details pConvertedTextUtf8 は
     *              - 0x00
     *              - [* 埋めする前の NG キーワードのバイト数] (0xFF で収まらない場合は)次バイトにつづく)
     *              - [* 埋めした後のバイト数(=文字数)] (0xFF で収まらない場合は)次バイトにつづく)
     *          の法則が出てくるところが * 埋め予定地。
     *          そこと一致するインデックスの pMapping は更新する必要がある
     */
    void UpdateMapping(int8_t* pMapping, const char* pConvertedTextUtf8, size_t mappingSize)
    {
        // マッピングの更新
        for (size_t i = 0; i < mappingSize; ++i)
        {
            if (pConvertedTextUtf8[i] == '\0')
            {
                size_t beforeByte = 0;  // * 埋めする前の NG キーワードのバイト数
                size_t afterByte = 0;   // * 埋めした後のバイト数(=文字数)
                size_t index = i + 1;
                for (;;)
                {
                    beforeByte += static_cast<unsigned char>(pConvertedTextUtf8[index]);
                    if (static_cast<unsigned char>(pConvertedTextUtf8[index]) != 0xFFu)
                    {
                        index++;
                        break;
                    }
                    index++;
                }
                for (;;)
                {
                    afterByte += static_cast<unsigned char>(pConvertedTextUtf8[index]);
                    if (static_cast<unsigned char>(pConvertedTextUtf8[index]) != 0xFFu)
                    {
                        index++;
                        break;
                    }
                    index++;
                }
                for (size_t j = 0; j < beforeByte; ++j)
                {
                    pMapping[i] = (j < afterByte) ? 1 : 0;
                }
            }
        }

        // pMapping[i] = 0 のところは pConvertedTextUtf8 からも切り詰められるため、
        // pMapping からも削除する(≒後ろに追いやる)
        for (size_t i = 0; i < mappingSize; ++i)
        {
            if (pMapping[i] == 0)
            {
                for (size_t j = i + 1; j < mappingSize; ++j)
                {
                    if (pMapping[j] != 0)
                    {
                        pMapping[i] = pMapping[j];
                        pMapping[j] = 0;
                        break;
                    }
                }
            }
        }
    }

    /**
     * @brief       NG ワードチェックのために変換した UTF-8 文字列の
     *              NG キーワードの一致インデックスから、
     *              変換前の UTF-8 文字列での NG キーワードのインデックスと
     *              その文字数（≠バイト数）を計算します
     * @param[out]  pOutFirst       変換前の UTF-8 文字列での NG キーワードの先頭インデックス
     * @param[out]  pOutLast        変換前の UTF-8 文字列での NG キーワードの直後インデックス
     * @param[out]  pOutLengthDiff  変換前の UTF-8 文字列での NG キーワードの長さの差
     * @param[in]   pMapping        変換後 から 変換前 のキーワードの位置を割り出すためのマッピング
     * @param[in]   first           変換後の UTF-8 文字悦での NG キーワードの先頭インデックス
     * @param[in]   last            変換後の UTF-8 文字列での NG キーワードの直後インデックス
     */
    void CalcUtf8Info(size_t* pOutFirst, size_t* pOutLast, size_t *pOutLengthDiff,
                      const int8_t* pMapping, size_t first, size_t last)
    {
        *pOutFirst = 0;
        *pOutLast = 0;
        for (size_t i = 0; i < last; ++i)
        {
            int absMapVal = (pMapping[i] < 0) ? -pMapping[i] : pMapping[i];
            if (i < first)
            {
                *pOutFirst += absMapVal;
            }
            *pOutLast += absMapVal;
        }

        // pTextUtf8 の埋める範囲を測定

        *pOutLengthDiff = 0;    // pTextUtf8 と pConvertedTextUtf8 の文字数の差
                                // pConvertedTextUtf8 と pTextUtf8 の文字数の差を求める
                                // pTextUtf8 に 半角濁音or半濁音があった時に活用される
        for (size_t i = first; i < last; ++i)
        {
            if (pMapping[i] < 0)
            {
                *pOutLengthDiff += -pMapping[i] - 1;
            }
        }
    }

    /**
     * @brief       FillWithAsterisk() の法則に則って不正文字の部分を借り埋めしておく部分です
     * @param[out]  pStr        借り埋めする文字列
     * @param[in]   firstPos    借り埋めする場所の最初のインデックス
     * @param[in]   lastPos     借り埋めする場所の直後のインデックス
     * @param[in]   mode        マスクモード
     * @param[in]   length      借り埋めする文字の文字数（≠バイト数）
     * @details     実際に * 埋めすると UTF-8 なのでバイト数が変わり、後のチェックで見つかった不正文字をうまく * 埋めできなくなる
     */
    void WriteFillRule(char* pStr, size_t firstPos, size_t lastPos, nn::ngc::ProfanityFilter::MaskMode mode, size_t length)
    {
        NN_SDK_ASSERT(firstPos <= lastPos);
        if (lastPos - firstPos == 1)
        {
            // UTF-8 的にあり得ないバイトで借り埋め
            pStr[firstPos] = g_DummyChar;
        }
        else if (lastPos - firstPos == 2)
        {
            if (mode == nn::ngc::ProfanityFilter::MaskMode::MaskMode_ReplaceByOneCharacter)
            {
                // UTF-8 的にあり得ない表現 0xC0 0xC0 を入れておく
                // これを MaskMode_ReplaceByOneCharacter のサインにする
                pStr[firstPos] = g_DummyForOneChar;
                pStr[firstPos + 1] = g_DummyForOneChar;
            }
            else if (mode == nn::ngc::ProfanityFilter::MaskMode::MaskMode_OverWrite)
            {
                // 2 バイトで 1 文字を表すか、 2 文字を表すか確認
                if (GetLengthOfUtf8(pStr, firstPos, firstPos + 2) == 1)
                {
                    // 1 文字なら 1 文字分の * にする（MaskMode_ReplaceByOneCharacter のときと同じ）
                    pStr[firstPos] = g_DummyForOneChar;
                    pStr[firstPos + 1] = g_DummyForOneChar;
                }
                else
                {
                    // UTF-8 的にあり得ないバイトで借り埋め
                    pStr[firstPos] = g_DummyChar;
                    pStr[firstPos + 1] = g_DummyChar;
                }
            }
        }
        else
        {
            pStr[firstPos] = 0x00;  // prefix
            size_t beforeByte = lastPos - firstPos; // * 埋めする前の NG キーワードのバイト数。 0xFF 区切りで格納していく
            size_t index = firstPos + 1;
            for (;;)
            {
                if (beforeByte >= 0xFFu)
                {
                    pStr[index] = '\xFF';
                    beforeByte -= 0xFFu;
                    index++;
                }
                else
                {
                    pStr[index] = static_cast<char>(beforeByte);
                    index++;
                    break;
                }
            }
            if (mode == nn::ngc::ProfanityFilter::MaskMode::MaskMode_ReplaceByOneCharacter)
            {
                pStr[index] = 1;
                index++;
            }
            else if (mode == nn::ngc::ProfanityFilter::MaskMode::MaskMode_OverWrite)
            {
                // * 埋めした後のバイト数( * なので =文字数)
                for (;;)
                {
                    if (length >= 0xFFu)
                    {
                        pStr[index] = '\xFF';
                        length -= 0xFFu;
                        index++;
                    }
                    else
                    {
                        pStr[index] = static_cast<char>(length);
                        index++;
                        break;
                    }
                }
            }
            // それ以降を今後マッチしないように 0xC1 で仮埋め
            for (size_t i = index; i < lastPos; ++i)
            {
                pStr[i] = g_DummyChar;
            }
        }
    }

   /**
    * @brief       メールアドレスのパターンチェックを行います。
    * @param[out]  pTextUtf8   元のテキスト。マッチしたアドレスは * で埋められる
    * @param[in]   length      pTextUtf8 の長さ
    * @return      メールアドレスにマッチした回数を返します。
    * @details     MaskText() に以下の正規表現を与えたときと同じ動作になるようにします。
    *      [a-zA-Z0-9][a-zA-Z0-9_\.\-]+@[a-zA-Z0-9][a-zA-Z0-9_\.\-]+\.[a-zA-Z0-9]+
    *      a : 0x61    z : 0x7A    A : 0x41    Z : 0x5A
    *      0 : 0x30    9 : 0x39    _ : 0x5F    . : 0x2E
    *      - : 0x2D    @ : 0x40    * : 0x2A
    */
    int CheckMailAddressPattern(char* pTextUtf8, size_t length, nn::ngc::ProfanityFilter::MaskMode mode)
    {
        int flag = 0;
        int retVal = 0;     // マッチした回数
        size_t begin = 0;   // マッチした先頭位置
        size_t end = 0;     // マッチした末尾位置
        for (size_t i = 0; i < length; ++i)
        {
            size_t checkStartPos = i;
            size_t conditionStartPos = 0;
            size_t j = 0;
            unsigned char c = static_cast<unsigned char>(pTextUtf8[i]);
            switch (flag)
            {
            case 0:
                // 第1条件 : 1文字目が a-zA-Z0-9 である
                if ((c >= 0x61 && c <= 0x7A) || (c >= 0x41 && c <= 0x5A) ||
                    (c >= 0x30 && c <= 0x39))
                {
                    flag = 1;     // 第1条件クリア
                    begin = i;
                }
                else
                {
                    flag = 0;
                }
                break;
            case 1:
                // 第2条件 : a-zA-Z0-9_\.\- が連続し @ がくる
                while (i < length)
                {
                    c = static_cast<unsigned char>(pTextUtf8[i]);
                    if ((c >= 0x61 && c <= 0x7A) || (c >= 0x41 && c <= 0x5A) ||
                        (c >= 0x30 && c <= 0x39) || c == 0x5F || c == 0x2E || c == 0x2D)
                    {
                        flag = 2;
                    }
                    else
                    {
                        if (flag != 2 || c != 0x40)
                        {
                            flag = 0;
                        }
                        break;
                    }
                    i++;
                }
                break;
            case 2:
                // 第3条件 : a-zA-Z0-9 がくる
                if ((c >= 0x61 && c <= 0x7A) || (c >= 0x41 && c <= 0x5A) ||
                    (c >= 0x30 && c <= 0x39))
                {
                    flag = 3;
                }
                else
                {
                    flag = 0;
                }
                break;
            case 3:
                // 第4条件 : a-zA-Z0-9_\.\- が連続し最後は .[a-zA-Z0-9]+ で終わる
                c = static_cast<unsigned char>(pTextUtf8[i]);
                conditionStartPos = i;
                while (i < length && ((c >= 0x61 && c <= 0x7A) || (c >= 0x41 && c <= 0x5A) ||
                    (c >= 0x30 && c <= 0x39) || c == 0x5F || c == 0x2E || c == 0x2D))
                {
                    i++;
                    c = static_cast<unsigned char>(pTextUtf8[i]);
                }
                // 最後の文字からさかのぼり a-zA-Z0-9 以外で出てくる最初の記号は . である
                // かつ、その . より前にも第4条件が 1 文字以上存在する
                {
                    bool isFirstAlphaNumeralFound = false;
                    for (j = i - 1; j > begin; --j)
                    {
                        c = static_cast<unsigned char>(pTextUtf8[j]);
                        if (((c >= 0x61 && c <= 0x7A) || (c >= 0x41 && c <= 0x5A) ||
                            (c >= 0x30 && c <= 0x39)) && !isFirstAlphaNumeralFound)
                        {
                            // 末尾が記号だといけないのではじく
                            isFirstAlphaNumeralFound = true;
                            end = j;
                        }
                        else if (((c < 0x61 || c > 0x7A) && (c < 0x41 || c > 0x5A) &&
                            (c < 0x30 || c > 0x39)) && isFirstAlphaNumeralFound)
                        {
                            break;
                        }
                    }
                }
                if (c == 0x2E && j > conditionStartPos)
                {
                    // パターンに一致した
                    // FillWithAsterisk で * 埋めできるパターンを作成する
                    WriteFillRule(pTextUtf8, begin, end + 1, mode, end - begin + 1);
                    retVal++;
                    flag = 0;
                    i = end;
                }
                else
                {
                    // パターンに一致しなかったが第4条件の比較開始位置からは
                    // パターンに一致する可能性があるのでチェック位置をそこにする
                    flag = 0;
                    i = checkStartPos - 1;
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
                break;
            }
        }
        return retVal;
    }   // NOLINT(impl/function_size)

    /**
     * @brief       発見した NG ワードから変換前の UTF-8 文字列での位置を割り出し
     *              * 埋めするための準備をします
     * @param[out]  pOutMatchSet    マッチした UTF-8 文字列に関する構造体
     * @param[in]   firstPos        マッチした変換後のキーワードの先頭インデックス
     * @param[in]   lastPos         マッチした変換後のキーワードの直後インデックス
     * @details     実際の * 埋めは AC 内のすべてのマッチング確認が終わった後
     */
    void FillNgKeyword(ScrapedTextMatchSet* pOutMatchSet, size_t firstPos, size_t lastPos)
    {
        // NG キーワードの文字数
        size_t length = GetLengthOfUtf8(pOutMatchSet->pConvertedTextUtf8, firstPos, lastPos);
        if (length == 0 ||
            (pOutMatchSet->pConvertedTextUtf8[firstPos] == '\x00') ||
            (firstPos >= 1 && pOutMatchSet->pConvertedTextUtf8[firstPos - 1] == '\x00') ||
            (firstPos >= 2 && pOutMatchSet->pConvertedTextUtf8[firstPos - 2] == '\x00'))
        {
            // 既に仮埋めされている
            return;
        }

        if (pOutMatchSet->pProfanityWordCount)
        {
            (*(pOutMatchSet->pProfanityWordCount))++;    // 不正文字列カウンタの増加
        }

        size_t utf8FirstPos = 0;
        size_t utf8LastPos = 0;
        size_t lengthDiff = 0;
        CalcUtf8Info(&utf8FirstPos, &utf8LastPos, &lengthDiff, pOutMatchSet->pMapping,
                     firstPos, lastPos);

        // ビットを仮パターンで埋めておき、その法則に従って後で * 埋めしてもらう
        WriteFillRule(pOutMatchSet->pConvertedTextUtf8, firstPos, lastPos,
                      pOutMatchSet->mode, length);
        WriteFillRule(pOutMatchSet->pTextUtf8, utf8FirstPos, utf8LastPos,
                      pOutMatchSet->mode, length + lengthDiff);
    }

    /**
     * @brief       変換後の UTF-8 文字列からマッチした NG キーワードのインデックスを取得します
     * @param[out]  pOutFirst   マッチした変換後のキーワードの先頭インデックス
     * @param[out]  pOutLast    マッチした変換後のキーワードの直後インデックス
     * @param[in]   first       検索語のマッチ位置の先頭
     * @param[in]   last        検索語のマッチ位置の直後
     * @param[in]   pMatchSet   マッチした UTF-8 文字列に関する構造体
     */
    void GetKeywordPos(int32_t* pOutFirst, int32_t* pOutLast,
                       const char* first, const char*last, ScrapedTextMatchSet* pMatchSet)
    {
        // pConvertedTextUtf8 における * にするべき位置を特定
        // first の pScrapedText におけるインデックス
        int index = static_cast<int>((reinterpret_cast<uintptr_t>(first) - reinterpret_cast<uintptr_t>(pMatchSet->pScrapedText)) / sizeof(char));

        // ここから pConvertedTextUtf8 における first の文字の位置を割り出す
        *pOutFirst = pMatchSet->pBoundPos->Select0(static_cast<uint32_t>(index));
        NN_SDK_ASSERT(*pOutFirst >= 0);

        // last も同様に処理
        // この時点の lastPos には、マッチしたキーワードの直後の「区切り文字を除くの1文字目の位置」が入るので注意
        index = static_cast<int>((reinterpret_cast<uintptr_t>(last) - reinterpret_cast<uintptr_t>(pMatchSet->pScrapedText)) / sizeof(char));
        *pOutLast = pMatchSet->pBoundPos->Select0(static_cast<uint32_t>(index));
        if (*pOutLast < 0)
        {
            // ここで lastPos が -1 の時、 last は末尾である
            *pOutLast = static_cast<int32_t>(pMatchSet->pBoundPos->GetBitVectorSize());
        }
        // lastPos をアジャスト
        for (int i = 0; i < g_BoundCount; ++i)
        {
            bool isMatch = true;
            for (int j = 0; j < g_pWordBound[i].size; ++j)
            {
                if (pMatchSet->pConvertedTextUtf8[*pOutLast - 1 - j] != g_pWordBound[i].word[(g_pWordBound[i].size - 1 - j)])
                {
                    isMatch = false;
                    break;
                }
            }
            if (isMatch)
            {
                // 区切り文字が見つかったらそれを削り再度全チェック
                *pOutLast -= g_pWordBound[i].size;
                i = -1;
            }
        }
        NN_SDK_ASSERT(*pOutLast >= 0);
    }

    // ------------------------------------------------------------------
    // AhoCorasick::Match 用コールバック
    // ------------------------------------------------------------------

    /**
     * @brief           AhoCorasick でマッチしたときのコールバック
     * @param[in/out]   pFirst       検索語のマッチ位置の先頭
     * @param[in/out]   pLast        検索語のマッチ位置の直後
     * @param[in]       nodeid      検出した語の id
     * @param[in/out]   pUserObj    ユーザ指定のポインタ（ここでは WordMatchSet）
     * @details         ProfanityFilter::CheckProfanityWords() で利用されます
     */
    bool WordMatchCallBack(const char* pFirst, const char *pLast, uint32_t nodeid, void *pUserObj)
    {
        NN_UNUSED(pFirst);
        NN_UNUSED(pLast);
        NN_UNUSED(nodeid);
        WordMatchSet* pMatchSet = reinterpret_cast<WordMatchSet*>(pUserObj);

        *(pMatchSet->checkResult) |= GetCheckResultPattern(pMatchSet->patternList, pMatchSet->checkDesiredLanguage);

        return true;
    }

    /**
     * @brief           AhoCorasick でマッチしたときのコールバック
     * @param[in]       pFirst       検索語のマッチ位置の先頭
     * @param[in]       pLast        検索語のマッチ位置の直後
     * @param[in]       nodeid      検出した語の id
     * @param[in/out]   pUserObj    ユーザ指定のポインタ（ここでは ScrapedTextMatchSet）
     * @details         不正文字列のカウントと * への置き換えを行います
     *                  ProfanityFilter::MaskProfanityWordsInText() で利用されます
     */
    bool TextMatchCallBack(const char* pFirst, const char *pLast, uint32_t nodeid, void *pUserObj)
    {
        NN_UNUSED(nodeid);
        ScrapedTextMatchSet* pMatchSet = reinterpret_cast<ScrapedTextMatchSet*>(pUserObj);

        // マッチした NG キーワードのインデックスを取得
        size_t firstPos = reinterpret_cast<uintptr_t>(pFirst) - reinterpret_cast<uintptr_t>(pMatchSet->pConvertedTextUtf8) / sizeof(char);
        size_t lastPos = reinterpret_cast<uintptr_t>(pLast) - reinterpret_cast<uintptr_t>(pMatchSet->pConvertedTextUtf8) / sizeof(char);

        // * 埋めする前の借り埋め
        FillNgKeyword(pMatchSet, firstPos, lastPos);

        return true;
    }

    /**
     * @brief           AhoCorasick でマッチしたときのコールバック
     * @param[in]       pFirst       検索語のマッチ位置の先頭
     * @param[in]       pLast        検索語のマッチ位置の直後
     * @param[in]       nodeid      検出した語の id
     * @param[in/out]   pUserObj    ユーザ指定のポインタ（ここでは ScrapedTextMatchSet）
     * @details         不正文字列のカウントと * への置き換えを行います
     *                  ProfanityFilter::MaskProfanityWordsInText() で利用されます
     */
    bool ScrapedTextMatchCallBack(const char* pFirst, const char *pLast, uint32_t nodeid, void *pUserObj)
    {
        NN_UNUSED(nodeid);

        ScrapedTextMatchSet* pMatchSet = reinterpret_cast<ScrapedTextMatchSet*>(pUserObj);

        // マッチした NG キーワードのインデックスを取得
        int32_t firstPos = 0;
        int32_t lastPos = 0;
        GetKeywordPos(&firstPos, &lastPos, pFirst, pLast, pMatchSet);

        // * 埋めする前の借り埋め
        FillNgKeyword(pMatchSet, firstPos, lastPos);

        return true;
    }

    /**
     * @brief           AhoCorasick でマッチしたときのコールバック(\b を含むキーワード)
     * @param[in/out]   pFirst       検索語のマッチ位置の先頭
     * @param[in/out]   pLast        検索語のマッチ位置の直後
     * @param[in]       nodeid      検出した語の id
     * @param[in/out]   pUserObj    ユーザ指定のポインタ（ここでは ScrapedTextMatchSet）
     * @details         マッチしていたら pUserObj に入る予定の bool を true にして処理を打ち切る
     *                  ProfanityFilter::MaskProfanityWordsInText() で利用されます
     */
    bool SimpleMatchCallBack(const char* pFirst, const char *pLast, uint32_t nodeid, void *pUserObj)
    {
        NN_UNUSED(pFirst);
        NN_UNUSED(pLast);
        NN_UNUSED(nodeid);
        bool* pIsMatch = reinterpret_cast<bool*>(pUserObj);
        *pIsMatch = true;
        return false;   // マッチしたら処理を打ち切って良い
    }

    /**
     * @brief           AhoCorasick でマッチしたときのコールバック(\b を含むキーワード)
     * @param[in]       pFirst       検索語のマッチ位置の先頭
     * @param[in]       pLast        検索語のマッチ位置の直後
     * @param[in]       nodeid      検出した語の id
     * @param[in/out]   pUserObj    ユーザ指定のポインタ（ここでは ScrapedTextMatchSet）
     * @details         不正文字列のカウントと * への置き換えを行います
     *                  \b を含むキーワードを登録してある AC のコールバックは、これに設定します
     *                  ProfanityFilter::MaskProfanityWordsInText() で利用されます
     */
    bool ScrapedBoundaryTextMatchCallBack(const char* pFirst, const char *pLast, uint32_t nodeid, void *pUserObj)
    {
        NN_UNUSED(nodeid);

        ScrapedTextMatchSet* pMatchSet = reinterpret_cast<ScrapedTextMatchSet*>(pUserObj);

        // マッチした NG キーワードのインデックスを取得
        int32_t firstPos = 0;
        int32_t lastPos = 0;
        GetKeywordPos(&firstPos, &lastPos, pFirst, pLast, pMatchSet);

        // マッチしたキーワードに \b を含む文字列を作成
        char pKeyword[64] = { 0 };
        int currentPos = 0;
        // pFirst の 1 つ前の文字を見る
        if (CheckFirstIsBoundary(pMatchSet->pConvertedTextUtf8, firstPos))
        {
            // 1 つ前の文字が区切り文字ならば pKeyword に "\b" を入れる
            pKeyword[0] = '\\';
            pKeyword[1] = 'b';
            currentPos = 2;
        }
        else
        {
            // \b 以外の文字を入れておく
            pKeyword[0] = 'a';
            currentPos = 1;
        }
        size_t matchKeywordSize = reinterpret_cast<uintptr_t>(pLast) - reinterpret_cast<uintptr_t>(pFirst);
        memcpy(&pKeyword[currentPos], pFirst, matchKeywordSize);
        currentPos += static_cast<int>(matchKeywordSize);
        // pLast の 2 つ後までの文字を見る
        if (CheckLastIsBoundary(pMatchSet->pConvertedTextUtf8, pMatchSet->pBoundPos->GetBitVectorSize(), lastPos))
        {
            pKeyword[currentPos] = '\\';
            pKeyword[currentPos + 1] = 'b';
            currentPos += 2;
        }
        else
        {
            // \b 以外の文字を入れておく
            pKeyword[currentPos] = 'a';
            currentPos++;
        }

        pKeyword[currentPos] = '\0';

        // \b を入れたキーワードに対しても、マッチするなら、正真正銘マッチしたことになる
        bool isMatch = false;
        pMatchSet->pAcB2->Match(pKeyword, static_cast<size_t>(currentPos), SimpleMatchCallBack, &isMatch);

        if (isMatch)
        {
            // * 埋めする前の借り埋め
            FillNgKeyword(pMatchSet, firstPos, lastPos);
        }

        return true;
    }

}   // namespace

namespace nn { namespace ngc {

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::ProfanityFilter

  @brief        初期化処理を一切行わないコンストラクタです。
 *---------------------------------------------------------------------------*/
ProfanityFilter::ProfanityFilter() NN_NOEXCEPT:m_Initialized(false)
{
    CleanUpVariables();
    return;
}

ProfanityFilter::ProfanityFilter( void* pWorkMemory, size_t workMemorySize, bool checkDesiredLanguage ) NN_NOEXCEPT:m_Initialized(false)
{
    Initialize( pWorkMemory, workMemorySize, checkDesiredLanguage );
    return;
}

ProfanityFilter::ProfanityFilter( void* pWorkMemory, size_t workMemorySize ) NN_NOEXCEPT:m_Initialized(false)
{
    Initialize( pWorkMemory, workMemorySize, true );
    return;
}

ProfanityFilter::~ProfanityFilter() NN_NOEXCEPT
{
    Finalize();

    return;
}

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::CleanUpVariables

  @brief        内部変数を初期化します。
 *---------------------------------------------------------------------------*/
void ProfanityFilter::CleanUpVariables() NN_NOEXCEPT
{
    m_SkipAtSignCheck          = SkipMode_NotSkip;
    m_MountSystemDataCacheHead = 0;
    m_MaskMode                 = MaskMode_OverWrite;
}

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::Initialize

  @param [in]   pWorkMemory ワーキングメモリ領域の先頭へのポインタを指定します。
  @param [in]   workMemorySize        ワーキングメモリ領域のサイズを指定します。
  @param [in]   checkDesiredLanguage  アプリケーションが対応する最も優先度の高い言語に対してチェックを行うかを指定します。

  @brief        NGワードフィルタリングクラスを初期化します。
  @return       初期化処理に対する戻り値が返ります。
 *---------------------------------------------------------------------------*/
nn::Result ProfanityFilter::Initialize( void* pWorkMemory, size_t workMemorySize, bool checkDesiredLanguage ) NN_NOEXCEPT
{
    if (!pWorkMemory)
    {
        return ResultInvalidPointer();
    }
    if (workMemorySize < WorkMemorySize)
    {
        return ResultInvalidSize();
    }

    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    // 初期化状態を確認します。
    if( m_Initialized == true ){
        return ResultAlreadyInitialized();
    }

    // 内部変数を初期化します。
    CleanUpVariables();

    // ワークバッファ用アロケータを初期化します。
    m_WorkingHead = reinterpret_cast<uintptr_t>(pWorkMemory);

    detail::WorkBufAllocator* pAllocator = new (pWorkMemory) detail::WorkBufAllocator;
    size_t allocatorSize = workMemorySize - sizeof(detail::WorkBufAllocator);
    uintptr_t pAllocatorHead = m_WorkingHead + sizeof(detail::WorkBufAllocator);
    // 最低限のアライメントをとる
    if (pAllocatorHead % NN_ALIGNOF(std::max_align_t) != 0)
    {
        pAllocatorHead += NN_ALIGNOF(std::max_align_t) - (pAllocatorHead % NN_ALIGNOF(std::max_align_t));
    }

    bool isSuccess = pAllocator->Initialize(reinterpret_cast<void*>(pAllocatorHead), allocatorSize);
    NN_UNUSED(isSuccess);
    NN_SDK_ASSERT(isSuccess);
    m_MountSystemDataCacheHead = reinterpret_cast< uintptr_t >(g_NgcSystemDataCacheBuffer);

    if(g_InitializeCount == 0)
    {
        // 共有コンテンツをマウントします
        nn::Result result = g_ContentsReader.MountContents(
             reinterpret_cast<void*>(m_MountSystemDataCacheHead),NgcSystemDataCacheSize);
        if ( result.IsFailure() )
        {
            return result;
        }
    }
    m_CheckDesiredLanguage = checkDesiredLanguage;

    g_InitializeCount++;
    m_Initialized = true;

    return nn::ResultSuccess();
}

nn::Result ProfanityFilter::Initialize( void* pWorkMemory ,size_t workMemorySize ) NN_NOEXCEPT
{
    return Initialize(pWorkMemory ,workMemorySize, true);
}

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::SkipAtSignCheck

  @param [in]   アットマーク記号のチェックをスキップするかどうかを指定します。

  @brief        アットマーク記号のチェックをスキップするかどうかを設定します。
 *---------------------------------------------------------------------------*/
void ProfanityFilter::SkipAtSignCheck( SkipMode bSkipChecking ) NN_NOEXCEPT
{
    m_SkipAtSignCheck = bSkipChecking;
}

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::SetMaskMode

  @param [in]   bOverWrite  問題ある語句が見つかった時、すべての文字をアスタリスク記号で
                            上書きするときに true を指定します(デフォルト状態)。
                            問題ある語句が見つかった時、その語句を1文字のアスタリスク記号で
                            置換するとき false を指定します。

  @brief        文章をチェックし、問題のある語句を発見した時の挙動を設定します。
 *---------------------------------------------------------------------------*/
void ProfanityFilter::SetMaskMode( ProfanityFilter::MaskMode mask ) NN_NOEXCEPT
{
    m_MaskMode = mask;
}

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::Finalize

  @brief        NGワードフィルタリングクラスの使用を終了します。
 *---------------------------------------------------------------------------*/
void ProfanityFilter::Finalize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    // 初期化状態を確認します。
    if( m_Initialized == false ){
        return;
    }

    if(g_InitializeCount > 0){
        --g_InitializeCount;
        if(g_InitializeCount == 0)
        {
            g_ContentsReader.UnmountContents();
        }
        m_Initialized = false;
    }

    GetAllocator(m_WorkingHead)->Finalize();

    return;
}

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::GetContentVersion

  @brief        現在本体にインストールされているNGワードパターンファイルのバージョン番号を取得
                します。
  @return       インストールされているバージョン番号が返ります。バージョンは1から始まり、
                数が大きいほど新しいことを示します。取得に失敗した場合0が返ります。
 *---------------------------------------------------------------------------*/
uint32_t ProfanityFilter::GetContentVersion() NN_NOEXCEPT
{
    nn::Result result;

    // バージョン情報ファイルを取得します。
    uint32_t size;
    result = g_ContentsReader.GetVersionDataSize( &size );

    // ファイルが正しく4バイトであることを確認します。
    if ( result.IsFailure() && size != sizeof( uint32_t ) )
    {
        return 0;
    }

    // ファイルからバージョン情報を読み込みます。
    uint32_t version;
    result = g_ContentsReader.GetVersionData( &version, sizeof( uint32_t ) );
    if ( result.IsFailure() )
    {
        return 0;
    }

    nn::util::SwapEndian(&version);

    return version;
}

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::CheckProfanityWords

  @param [out] pCheckResults    問題があったかどうかの判定を格納するためのバッファを指定します。
                                nWordCountで指定した個数の配列が必要です。
  @param [in]  nPatternCode     どのパターンリストに対してチェックを行うのかを指定します。
  @param [in]  ppWords          調査対象となるnullptr終端文字列の配列を指定します。
                                文字コードはUTF8にしてください。
  @param [in]  nWordCount       調査対象の文字列の数を指定します。

  @brief        指定された複数の文字列が、問題のある文字列であるかを確認します。
  @return       終了処理に対する戻り値が返ります。
 *---------------------------------------------------------------------------*/
nn::Result ProfanityFilter::CheckProfanityWords( Bit32* pCheckResults, Bit32 nPatterns,
                                                 const char** ppWordsUtf8,
                                                 size_t nWordCount ) NN_NOEXCEPT
{
    nn::Result result;

    const char16_t** ppWords = reinterpret_cast< const char16_t** >( ppWordsUtf8 );

    // 状態をチェックします。
    result = CheckArgumentsWord( pCheckResults, ppWords, nWordCount );
    if ( result.IsFailure() )
    {
        return result;
    }

    // 出力用の配列を初期化します。
    for ( size_t i = 0; i < nWordCount; i++ )
    {
        pCheckResults[ i ] = 0;
    }

    // UGC ガイドラインでチェックが必要なリストを取得します
    Bit32 ugcGuidelineTargets = GetPatternBitsFromRegion( true, m_CheckDesiredLanguage );

    // ユーザが追加で指定したリストを OR します
    Bit32 checkTargets = ugcGuidelineTargets | nPatterns;

    // 共通パターンリストでテストします。
    m_PatternList =
            static_cast< ProfanityFilterPatternList >( NgcProfanityFilterPatternListCommon );
    result = CheckProfanityWordsImpl( pCheckResults, ppWords, nWordCount, true );
    if ( result.IsFailure() )
    {
        return result;
    }
    for( size_t i = 0 ; i < nWordCount ; i++ )
    {
        if(pCheckResults[i] != 0)
        {
            //共通パターンリストで不正文字が検出されている場合は入力された全てのフラグの内、
            //ProfanityFilterPatternListとして有効なパターンをセットする。
            pCheckResults[i] =
                    checkTargets & ((1 << nn::ngc::ProfanityFilterPatternList_Max) - 1);
        }
    }

    // チェック対象のリストでそれぞれテストします。
    for ( int i = 0; i < ProfanityFilterPatternList_Max; i++ )
    {
        if( (checkTargets >> i) & 0x00000001 )
        {
            m_PatternList = static_cast< ProfanityFilterPatternList >( 1 << i );
            result = CheckProfanityWordsImpl( pCheckResults, ppWords, nWordCount, true );
            if ( result.IsFailure() )
            {
                return result;
            }
        }
    }

    return nn::ResultSuccess();
}

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::CheckProfanityWords

  @param [out] pCheckResults    問題があったかどうかの判定を格納するためのバッファを指定します。
                                nWordCountで指定した個数の配列が必要です。
  @param [in]  nPatternCode     どのパターンリストに対してチェックを行うのかを指定します。
  @param [in]  ppWords          調査対象となるnullptr終端文字列の配列を指定します。
                                文字コードはUTF16ビッグエンディアンにしてください。
  @param [in]  nWordCount       調査対象の文字列の数を指定します。

  @brief        指定された複数の文字列が、問題のある文字列であるかを確認します。
  @return       終了処理に対する戻り値が返ります。
 *---------------------------------------------------------------------------*/
nn::Result ProfanityFilter::CheckProfanityWords(Bit32* pCheckResults, Bit32 nPatterns,
                                                const uint16_t** ppWords,
                                                size_t nWordCount) NN_NOEXCEPT
{
    nn::Result result;

    const char16_t** ppWordsChar16 = reinterpret_cast< const char16_t** >(ppWords);

    // 状態をチェックします。
    result = CheckArgumentsWord(pCheckResults, ppWordsChar16, nWordCount);
    if (result.IsFailure())
    {
        return result;
    }

    // 出力用の配列を初期化します。
    for (size_t i = 0; i < nWordCount; i++)
    {
        pCheckResults[i] = 0;
    }

    // UGC ガイドラインでチェックが必要なリストを取得します
    Bit32 ugcGuidelineTargets = GetPatternBitsFromRegion(true, m_CheckDesiredLanguage);

    // ユーザが追加で指定したリストを OR します
    Bit32 checkTargets = ugcGuidelineTargets | nPatterns;

    // 共通パターンリストでテストします。
    m_PatternList =
        static_cast< ProfanityFilterPatternList >(NgcProfanityFilterPatternListCommon);
    result = CheckProfanityWordsImpl(pCheckResults, ppWordsChar16, nWordCount, false);
    if (result.IsFailure())
    {
        return result;
    }
    for (size_t i = 0; i < nWordCount; i++)
    {
        if (pCheckResults[i] != 0)
        {
            //共通パターンリストで不正文字が検出されている場合は入力された全てのフラグの内、
            //ProfanityFilterPatternListとして有効なパターンをセットする。
            pCheckResults[i] =
                checkTargets & ((1 << nn::ngc::ProfanityFilterPatternList_Max) - 1);
        }
    }

    // チェック対象のリストでそれぞれテストします。
    for (int i = 0; i < ProfanityFilterPatternList_Max; i++)
    {
        if ((checkTargets >> i) & 0x00000001)
        {
            m_PatternList = static_cast< ProfanityFilterPatternList >(1 << i);
            result = CheckProfanityWordsImpl(pCheckResults, ppWordsChar16, nWordCount, false);
            if (result.IsFailure())
            {
                return result;
            }
        }
    }

    return nn::ResultSuccess();
}

// char16_t オーバーロード
nn::Result ProfanityFilter::CheckProfanityWords(Bit32* pCheckResults, Bit32 patterns,
    const char16_t** ppWords,
    size_t wordCount) NN_NOEXCEPT
{
    return CheckProfanityWords(pCheckResults, patterns, reinterpret_cast<const uint16_t**>(ppWords), wordCount);
}

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::CheckArguments_Word

  @param [in]  pCheckResults    問題があったかどうかの判定を格納するためのバッファを指定します。
                                nWordCountで指定した個数の配列が必要です。
  @param [in]  ppWords          調査対象となるnullptr終端文字列の配列を指定します。
                                文字コードはUTF16ビッグエンディアンにしてください。
  @param [in]  nWordCount       調査対象の文字列の数を指定します。

  @brief        単語チェック用のAPIが呼び出されたときに、アプリ側から渡された引数が正しいかどうか、
                及び未初期化でないかのチェックを行います。
  @return       チェックした結果が返ります。
 *---------------------------------------------------------------------------*/
nn::Result ProfanityFilter::CheckArgumentsWord( const Bit32* pCheckResults,
                                                const char16_t** ppWords,
                                                size_t nWordCount ) const NN_NOEXCEPT
{
    // 未初期化エラーチェック
    if(!m_Initialized)
    {
        return ResultNotInitialized();
    }

    // 引数エラーチェック
    if ( !pCheckResults || !ppWords )
    {
        return ResultInvalidPointer();
    }
    if ( WordCountMax < nWordCount )
    {
        return ResultInvalidSize();
    }

    return nn::ResultSuccess();
}

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::CheckProfanityWordsImpl

  @param [out] pCheckResults    問題があったかどうかの判定を格納するためのバッファを指定します。
  @param [in]  pFileStream      入力ファイルストリーム
  @param [in]  ppWords          調査対象となるnullptr終端文字列の配列を指定します。
                                文字コードはUTF16ビッグエンディアンにしてください。
  @param [in]  nWordCount       調査対象の文字列の数を指定します。
                                nn::ngc::NgcMaxWordNum以下である必要があります。
  @param [in]  isConvert        UTF-16 文字への変換を行うかどうかを指定します。

  @brief        指定された複数の文字列が、問題のある文字列であるかを確認します。
  @return       終了処理に対する戻り値が返ります。
 *---------------------------------------------------------------------------*/
nn::Result ProfanityFilter::CheckProfanityWordsImpl( Bit32* pCheckResults,
                                                     const char16_t** ppWords,
                                                     size_t nWordCount,
                                                     bool isConvert) NN_NOEXCEPT
{
    nn::Result result;

    // 状態のチェック
    NN_SDK_ASSERT_NOT_NULL( pCheckResults );
    NN_SDK_ASSERT_NOT_NULL( ppWords );
    NN_SDK_ASSERT( nWordCount <= WordCountMax );

    // メモリ作業領域の中に、ユーザーが入力した文字列をチェックするために変換するための
    // バッファを用意します。
    char16_t* ppConvertedWords[ WordCountMax ];
    for ( int i = 0; i < WordCountMax; i++ )
    {
        ppConvertedWords[i] = reinterpret_cast<char16_t*>(GetAllocator(m_WorkingHead)->Allocate(sizeof(char16_t) * WordLengthMax));
        NN_ABORT_UNLESS(ppConvertedWords[i]);
    }
    const char** ppWordsUtf8 = reinterpret_cast< const char** >( ppWords );
    // ユーザーが入力したチェック対象文字列を変換します。
    for ( size_t i = 0; i < nWordCount; i++ )
    {
        if ( !ppWords[ i ] )
        {
            return nn::ngc::ResultInvalidPointer();
        }

        uint16_t* pString = nullptr;
        if (isConvert)
        {
            int lengthForUtf16;
            auto encodingResult = nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native(
                &lengthForUtf16, ppWordsUtf8[i],
                static_cast<int>(std::strlen(ppWordsUtf8[i])));
            NN_ABORT_UNLESS(nn::util::CharacterEncodingResult_Success == encodingResult);
            lengthForUtf16 += 1;

            pString = reinterpret_cast<uint16_t*>(GetAllocator(m_WorkingHead)->Allocate(lengthForUtf16 * sizeof(uint16_t)));
            NN_ABORT_UNLESS(pString);
            std::memset(pString, 0x00, lengthForUtf16 * sizeof(uint16_t));

            encodingResult = nn::util::ConvertStringUtf8ToUtf16Native(
                pString, lengthForUtf16, ppWordsUtf8[i],
                static_cast<int>(std::strlen(ppWordsUtf8[i])));
            NN_ABORT_UNLESS(nn::util::CharacterEncodingResult_Success == encodingResult);
        }
        else
        {
            pString = reinterpret_cast<uint16_t*>(const_cast<char16_t*>(ppWords[i]));
        }
        ConvertUserInputForWord( ppConvertedWords[ i ], WordLengthMax,
                                 reinterpret_cast<char16_t*>(pString) );
        if (isConvert)
        {
            GetAllocator(m_WorkingHead)->Free(pString);
        }
    }

    // アットマーク記号のチェックをスキップしない場合 ...
    if ( m_SkipAtSignCheck == SkipMode_NotSkip )
    {
        // アットマーク記号が含まれている場合は不正文字列とみなします。
        for ( size_t i = 0; i < nWordCount; i++ )
        {
            if ( IsIncludesAtSign( ppConvertedWords[ i ], WordLengthMax ) )
            {
                pCheckResults[ i ] |= GetCheckResultPattern(m_PatternList,m_CheckDesiredLanguage);
            }
        }
    }

    // UTF-16 の長さを測る
    auto Strlen16 = [](char16_t* pStr) -> int
    {
        int len = 0;
        while (pStr[len] != '\0')
        {
            if (pStr[len + 1] == '\0')
            {
                return len + 1;
            }
            if (pStr[len + 2] == '\0')
            {
                return len + 2;
            }
            if (pStr[len + 3] == '\0')
            {
                return len + 3;
            }
            len += 4;
        }
        return len;
    };

    // UTF-8 に戻す
    char* ppUtf8[WordCountMax];
    for (size_t i = 0; i < nWordCount; ++i)
    {
        int lengthForUtf8;
        auto encodingResult = nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8(
            &lengthForUtf8, reinterpret_cast<uint16_t*>(ppConvertedWords[i]), Strlen16(ppConvertedWords[i]) );
        NN_SDK_ASSERT(lengthForUtf8 > 0);
        size_t dstLen = lengthForUtf8 + 5;
        ppUtf8[i] = reinterpret_cast<char*>(GetAllocator(m_WorkingHead)->Allocate(dstLen));
        encodingResult = nn::util::ConvertStringUtf16NativeToUtf8(
            &(ppUtf8[i][2]), static_cast<int>(lengthForUtf8 + 1), reinterpret_cast<uint16_t*>(ppConvertedWords[i])
        );
        if (nn::util::CharacterEncodingResult_Success != encodingResult)
        {
            NN_SDK_LOG("encodingResult: %d\n", encodingResult);
            NN_ABORT_UNLESS(nn::util::CharacterEncodingResult_Success == encodingResult);
        }
        // 単語の前後に \b を付加
        ppUtf8[i][0] = '\\';
        ppUtf8[i][1] = 'b';
        ppUtf8[i][dstLen - 3] = '\\';
        ppUtf8[i][dstLen - 2] = 'b';
        ppUtf8[i][dstLen - 1] = '\0';
    }

    // ファイルの解析
    AhoCorasick acNotB(GetAllocator(m_WorkingHead));     // \b を持たないキーワードの AC
    AhoCorasick acB(GetAllocator(m_WorkingHead));        // \b を持つキーワードの AC (\b を抜かずにキーワードを格納)
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    int pattern = -1;
    if (m_PatternList != 0)
    {
        for (int i = 0; i < ProfanityFilterPatternList_Max; i++)
        {
            if (1 << i == m_PatternList)
            {
                pattern = i;
                break;
            }
        }
    }
    NN_ABORT_UNLESS(ImportAcTree(&acNotB, NULL, &acB, m_PatternList, GetAllocator(m_WorkingHead), pattern));
#else
    NN_ABORT_UNLESS(ImportAcTree(&acNotB, NULL, &acB, m_PatternList, GetAllocator(m_WorkingHead), 0));
#endif

    // マッチング処理
    for (size_t i = 0; i < nWordCount; ++i)
    {
        if ((pCheckResults[i] & m_PatternList) == 0)
        {
            WordMatchSet matchSet;
            matchSet.checkResult = &(pCheckResults[i]);
            matchSet.patternList = m_PatternList;
            matchSet.checkDesiredLanguage = m_CheckDesiredLanguage;

            acNotB.Match(ppUtf8[i], WordMatchCallBack, &matchSet);
            acB.Match(ppUtf8[i], WordMatchCallBack, &matchSet);
        }
    }

    for (size_t i = 0; i < nWordCount; ++i)
    {
        GetAllocator(m_WorkingHead)->Free(ppUtf8[i]);
    }

    for (int i = 0; i < WordCountMax; i++)
    {
        GetAllocator(m_WorkingHead)->Free(ppConvertedWords[i]);
    }

    acNotB.ReleaseAllocator();
    acB.ReleaseAllocator();
    return nn::ResultSuccess();
}   // NOLINT(impl/function_size)

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::MaskProfanityWordsInText

  @param [out]      pProfanityWordCount    文章中に現れた不正単語の個数を取得するためのポインタを
                                           指定します。情報が不要な場合nullptrを指定します。
  @param [in]       nPatternCode           NGワードパターンリストを指定します。
  @param [in/out]   pText                  対象の文字列(文章)を指定します。

  @brief        指定された文章に対してNGワードチェックを行います。
                NGワードが検出された場合、マスク文字 '*' で指定された文字列が上書きされます。
  @return       チェックした結果が返ります。
 *---------------------------------------------------------------------------*/
nn::Result ProfanityFilter::MaskProfanityWordsInText( int* pProfanityWordCount,
                                                      char* pTextUtf8,
                                                      Bit32 nPatterns ) NN_NOEXCEPT
{
    nn::Result result;

    // 状態をチェックします。
    result = CheckArgumentsText( reinterpret_cast<char16_t*>(pTextUtf8) );
    if ( result.IsFailure() )
    {
        return result;
    }

    // メイン処理
    result = MaskProfanityWordsInTextCommon(pProfanityWordCount, pTextUtf8, nPatterns);
    if (result.IsFailure())
    {
        return result;
    }

    return nn::ResultSuccess();
}

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::MaskProfanityWordsInText

  @param [out]      pProfanityWordCount    文章中に現れた不正単語の個数を取得するためのポインタを
                                           指定します。情報が不要な場合nullptrを指定します。
  @param [in/out]   pText                  対象の文字列(文章)を指定します。
  @param [in]       nPatternCode           NGワードパターンリストを指定します。

  @brief        指定された文章に対してNGワードチェックを行います。
                NGワードが検出された場合、マスク文字 '*' で指定された文字列が上書きされます。
  @return       チェックした結果が返ります。
 *---------------------------------------------------------------------------*/
nn::Result ProfanityFilter::MaskProfanityWordsInText(int* pProfanityWordCount,
                                                     uint16_t* pText,
                                                     Bit32 nPatterns) NN_NOEXCEPT
{
    nn::Result result;

    // 状態をチェックします。
    result = CheckArgumentsText(reinterpret_cast<char16_t*>(pText));
    if (result.IsFailure())
    {
        return result;
    }

    // UTF-8 で処理
    // 変換前より大きくなることはないのでバッファオーバーにはならない
    int lengthForUtf8;

    auto encodingResult = nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8(
        &lengthForUtf8, reinterpret_cast< uint16_t* >(pText),
        static_cast< int >(nn::ngc::wcslen(reinterpret_cast<char16_t*>(pText))));
    NN_ABORT_UNLESS(nn::util::CharacterEncodingResult_Success == encodingResult);
    lengthForUtf8 += 1;

    char* pTextUtf8 = reinterpret_cast< char* >(GetAllocator(m_WorkingHead)->Allocate(lengthForUtf8 * sizeof(char)));
    NN_ABORT_UNLESS(pTextUtf8);
    std::memset(pTextUtf8, 0x00, lengthForUtf8 * sizeof(char));

    encodingResult = nn::util::ConvertStringUtf16NativeToUtf8(
        pTextUtf8, lengthForUtf8, reinterpret_cast< uint16_t* >(pText),
        static_cast< int >(nn::ngc::wcslen(reinterpret_cast<char16_t*>(pText))));
    NN_ABORT_UNLESS(nn::util::CharacterEncodingResult_Success == encodingResult);

    // メイン処理
    result = MaskProfanityWordsInTextCommon(pProfanityWordCount, pTextUtf8, nPatterns);
    if (result.IsFailure())
    {
        GetAllocator(m_WorkingHead)->Free(pTextUtf8);
        return result;
    }

    //マスキング後の文字列を再度UTF-16に変換してバッファに格納
    //返還前より大きくなることはないのでバッファオーバーにはならない
    std::memset(pText, 0x00, nn::ngc::wcslen(reinterpret_cast< char16_t* >(pText)) * sizeof(uint16_t));

    int lengthForUtf16;
    encodingResult = nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native(
        &lengthForUtf16, pTextUtf8, static_cast<int>(strlen(pTextUtf8)));
    NN_ABORT_UNLESS(nn::util::CharacterEncodingResult_Success == encodingResult);
    lengthForUtf16 += 1;

    encodingResult = nn::util::ConvertStringUtf8ToUtf16Native(
        pText, lengthForUtf16, pTextUtf8, static_cast<int>(strlen(pTextUtf8)));
    NN_ABORT_UNLESS(nn::util::CharacterEncodingResult_Success == encodingResult);

    GetAllocator(m_WorkingHead)->Free(pTextUtf8);

    return nn::ResultSuccess();
}

// char16_t オーバーロード
nn::Result ProfanityFilter::MaskProfanityWordsInText(int* pProfanityWordCount,
    char16_t* pText,
    Bit32 patterns) NN_NOEXCEPT
{
    return MaskProfanityWordsInText(pProfanityWordCount, reinterpret_cast<uint16_t*>(pText), patterns);
}

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::MaskProfanityWordsInTextCommon

  @param [in]  pText    アプリケーションから渡された pText の値を指定します。

  @brief        指定された文章に対してNGワードチェックを行います。
  @return       チェックした結果が返ります。
 *---------------------------------------------------------------------------*/
nn::Result ProfanityFilter::MaskProfanityWordsInTextCommon(int* pProfanityWordCount,
                                                           char* pTextUtf8,
                                                           Bit32 nPatterns) NN_NOEXCEPT
{
    nn::Result result;

    // 単語の数を初期化します。
    if (pProfanityWordCount)
    {
        *pProfanityWordCount = 0;
    }

    // 変換後の UTF-8 文字列の長さ。この段階では仕様上の UTF-8 文字列最大サイズを入れておく
    size_t convertedTextUtf8Length = TextLengthMax * 4 * sizeof(char);
    char* pConvertedText = reinterpret_cast<char*>(GetAllocator(m_WorkingHead)->Allocate(convertedTextUtf8Length));
    NN_ABORT_UNLESS(pConvertedText);
    int8_t* pSourceLengthMapping = reinterpret_cast<int8_t*>(GetAllocator(m_WorkingHead)->Allocate(convertedTextUtf8Length));

    // UGC ガイドラインでチェックが必要なリストを取得します
    Bit32 ugcGuidelineTargets = GetPatternBitsFromRegion(true, m_CheckDesiredLanguage);

    // ユーザが追加で指定したリストを OR します
    Bit32 checkTargets = ugcGuidelineTargets | nPatterns;

    // UTF-8 としての文字数をカウント
    // この時点で nn::ngc::TextLengthMax を超える部分は処理しないようにする
    char overVal = '\0';
    size_t overIndex = 0;   // nn::ngc::TextLengthMax を超える直前の末尾インデックス
    size_t textUtf8OriginLength = 0;
    bool retVal = GetUtf8Ptr(&overIndex, pTextUtf8, TextLengthMax);
    NN_SDK_ASSERT(retVal);
    NN_UNUSED(retVal);
    if (overIndex != 0)
    {
        textUtf8OriginLength = strlen(pTextUtf8);
        overVal = pTextUtf8[overIndex];
        pTextUtf8[overIndex] = '\0';
    }

    size_t textUtf8Length = strlen(pTextUtf8);

    // メールアドレスのパターンによるチェックを行います
    if (pProfanityWordCount && m_SkipAtSignCheck == SkipMode_NotSkip)
    {
        *pProfanityWordCount += CheckMailAddressPattern(pTextUtf8, textUtf8Length, m_MaskMode);
        FillWithAsterisk(pTextUtf8, &textUtf8Length);
    }

    // ユーザーが入力したチェック対象文字列を変換します。
    ConvertUserInputForText(pConvertedText, pSourceLengthMapping, convertedTextUtf8Length, pTextUtf8);
    convertedTextUtf8Length = strlen(pConvertedText);


    // 共通パターンリストでテストします。
    // * 埋めされた場合は lengthForUtf8 の値が更新される
    m_PatternList =
        static_cast< ProfanityFilterPatternList >(NgcProfanityFilterPatternListCommon);
    result = MaskProfanityWordsInTextImpl(pProfanityWordCount,
        pTextUtf8,
        pConvertedText,
        pSourceLengthMapping,
        -1, //"ac_common",
        &textUtf8Length,
        &convertedTextUtf8Length);
    if (result.IsFailure())
    {
        GetAllocator(m_WorkingHead)->Free(pConvertedText);
        GetAllocator(m_WorkingHead)->Free(pSourceLengthMapping);
        return result;
    }

    // チェック対象のリストでそれぞれテストします。
    for (int i = 0; i < ProfanityFilterPatternList_Max; i++)
    {
        if ((checkTargets >> i) & 0x00000001)
        {
            m_PatternList = static_cast< ProfanityFilterPatternList >(1 << i);
            result = MaskProfanityWordsInTextImpl(pProfanityWordCount,
                pTextUtf8,
                pConvertedText,
                pSourceLengthMapping,
                i,
                &textUtf8Length,
                &convertedTextUtf8Length);
            if (result.IsFailure())
            {
                GetAllocator(m_WorkingHead)->Free(pConvertedText);
                GetAllocator(m_WorkingHead)->Free(pSourceLengthMapping);
                return result;
            }
        }
    }

    // nn::ngc::TextLengthMax を超えたため処理しなかった部分を末尾に追加します。
    if (overVal != '\0')
    {
        pTextUtf8[textUtf8Length] = overVal;    // \0 が入っているはず
                                                // 空いた空間を詰める
        if (overIndex >= textUtf8Length)
        {
            size_t indexSub = overIndex - textUtf8Length;
            for (size_t i = textUtf8Length + 1; i <= textUtf8OriginLength - indexSub; ++i)
            {
                pTextUtf8[i] = pTextUtf8[i + indexSub];
            }
            NN_SDK_ASSERT(pTextUtf8[textUtf8OriginLength] == '\0');
        }
    }

    GetAllocator(m_WorkingHead)->Free(pConvertedText);
    GetAllocator(m_WorkingHead)->Free(pSourceLengthMapping);

    return nn::ResultSuccess();
}

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::CheckArguments_Text

  @param [in]  pText    アプリケーションから渡された pText の値を指定します。

  @brief        長文チェックのAPIを呼び出されたときに、アプリ側から渡された引数が正しいかどうか、
                及び未初期化でないかのチェックを行います。
  @return       チェックした結果が返ります。
 *---------------------------------------------------------------------------*/
nn::Result ProfanityFilter::CheckArgumentsText( const char16_t* pText ) const NN_NOEXCEPT
{
    // 未初期化エラーチェック
    if (!m_Initialized)
    {
        return ResultNotInitialized();
    }

    // 引数エラーチェック
    if ( !pText )
    {
        return ResultInvalidPointer();
    }
    return nn::ResultSuccess();
}

/*!--------------------------------------------------------------------------*
  Name:         ProfanityFilter::MaskProfanityWordsInTextImpl

  @param[out]  pProfanityWordCount 不正単語の個数
  @param[out]  pTextUtf8           元のテキスト
  @param[in]   pConvertedTextUtf8  整形済みの Utf8 のテキスト(チェック対象)
  @param[in]   pMapping            元テキストと整形済みのマッピングテーブル( * 埋めで利用)
  @param[in]   pattern             読み込み NG ワードファイルを表す識別子
  @param[out]  textUtf8Length      pTextUtf8 の長さ（ * 埋め時に長さが変わる可能性がある）\0 は入っていない
  @param[out]  convertedTextUtf8Length pConvertedTextUtf8 の長さ ( * 埋め時に長さが変わる可能性がある) \0 は入っていない

  @brief        指定された文章に対してNGワードチェックを行います。
                NGワードが検出された場合、マスク文字 '*' で指定された文字列が上書きされます。
  @return       チェックした結果が返ります。
 *---------------------------------------------------------------------------*/
nn::Result ProfanityFilter::MaskProfanityWordsInTextImpl(int* pProfanityWordCount,
                                                         char* pTextUtf8,
                                                         char* pConvertedTextUtf8,
                                                         int8_t* pMapping,
                                                         int pattern,
                                                         size_t* pTextUtf8Length,
                                                         size_t* pConvertedTextUtf8Length) NN_NOEXCEPT
{
#if !defined( NN_BUILD_CONFIG_OS_HORIZON )
    NN_UNUSED(pattern);
#endif
    nn::Result result;

    // 状態のチェック
    NN_SDK_ASSERT_NOT_NULL( pMapping );

    // 3 種の AC 木を外部から読み込む
    AhoCorasick acNotB(GetAllocator(m_WorkingHead));     // \b を持たないキーワードの AC
    AhoCorasick acB1(GetAllocator(m_WorkingHead));       // \b を持つキーワードの AC 1 (\b を抜いてキーワードを格納)
    AhoCorasick acB2(GetAllocator(m_WorkingHead));       // \b を持つキーワードの AC 2 (\b を抜かずにキーワードを格納)

    NN_ABORT_UNLESS(ImportAcTree(&acNotB, &acB1, &acB2, m_PatternList, GetAllocator(m_WorkingHead), pattern));

    // マッチング処理

    // 1. 単語境界を抜かずにマッチング

    ScrapedTextMatchSet matchSet;
    matchSet.pTextUtf8 = pTextUtf8;
    matchSet.pConvertedTextUtf8 = pConvertedTextUtf8;
    matchSet.pMapping = pMapping;
    matchSet.pProfanityWordCount = pProfanityWordCount;
    matchSet.mode = m_MaskMode;

    acNotB.Match(pConvertedTextUtf8, *pConvertedTextUtf8Length, TextMatchCallBack, &matchSet);

    // * 埋め処理
    FillWithAsterisk(pTextUtf8, pTextUtf8Length);
    UpdateMapping(pMapping, pConvertedTextUtf8, *pConvertedTextUtf8Length);
    FillWithAsterisk(pConvertedTextUtf8, pConvertedTextUtf8Length);

    // 2. 入力文字列から単語境界を抜いてマッチング
    // nn::ngc::ProfanityFilter::MaskProfanityWordsInText が
    // 不正文字列内に単語境界が含まれている場合でも不正文字列として判定するため工夫する
    char* pScrapedText = reinterpret_cast<char*>(GetAllocator(m_WorkingHead)->Allocate(*pTextUtf8Length));  // 単語境界を抜いた後の pTextUtf8
    NN_ABORT_UNLESS(pScrapedText);

    // 単語境界がある位置を簡潔ビットベクトルで保持
    // 単語境界の場所を 1 、それ以外は 0 の長さ matchLength のビットベクトル
    Sbv boundPos;
    boundPos.SetAllocator(GetAllocator(m_WorkingHead));
    boundPos.Init(static_cast<uint32_t>(*pConvertedTextUtf8Length));
    size_t scrapedTextLength = CreateScrapedText(pScrapedText, &boundPos, pConvertedTextUtf8);
    boundPos.Build();

    matchSet.pTextUtf8 = pTextUtf8;
    matchSet.pConvertedTextUtf8 = pConvertedTextUtf8;
    matchSet.pMapping = pMapping;
    matchSet.pScrapedText = pScrapedText;
    matchSet.pProfanityWordCount = pProfanityWordCount;
    matchSet.pBoundPos = &boundPos;
    matchSet.pAcB2 = &acB2;

    // \b を含まないキーワードについて処理
    acNotB.Match(pScrapedText, scrapedTextLength, ScrapedTextMatchCallBack, &matchSet);
    FillWithAsterisk(pTextUtf8, pTextUtf8Length);
    UpdateMapping(pMapping, pConvertedTextUtf8, *pConvertedTextUtf8Length);
    FillWithAsterisk(pConvertedTextUtf8, pConvertedTextUtf8Length);
    boundPos.Init(static_cast<uint32_t>(*pConvertedTextUtf8Length));
    scrapedTextLength = CreateScrapedText(pScrapedText, &boundPos, pConvertedTextUtf8);
    boundPos.Build();
    matchSet.pTextUtf8 = pTextUtf8;
    matchSet.pConvertedTextUtf8 = pConvertedTextUtf8;
    matchSet.pMapping = pMapping;
    matchSet.pScrapedText = pScrapedText;
    matchSet.pBoundPos = &boundPos;

    // 続いて \b を含むキーワードについて処理
    acB1.Match(pScrapedText, scrapedTextLength, ScrapedBoundaryTextMatchCallBack, &matchSet);

    // * 埋め処理
    FillWithAsterisk(pTextUtf8, pTextUtf8Length);
    UpdateMapping(pMapping, pConvertedTextUtf8, *pConvertedTextUtf8Length);
    FillWithAsterisk(pConvertedTextUtf8, pConvertedTextUtf8Length);

    GetAllocator(m_WorkingHead)->Free(pScrapedText);

    boundPos.ReleaseAllocator();
    acNotB.ReleaseAllocator();
    acB1.ReleaseAllocator();
    acB2.ReleaseAllocator();

    return nn::ResultSuccess();
}

}   // namespace ngc
}   // namespace nn
