﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_SdkLog.h>
#include <nn/arp/arp_Api.h>
#include <nn/prepo.h>
#include <nn/prepo/prepo_SystemPlayReport.h>
#include <nn/hid/system/hid_NpadCommon.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nfc/server/util/nfc_UtilUtil.h>
#include <nn/nfp/server/nfp_PlayReport.h>
#include "nfp_Util.h"

namespace nn { namespace nfp {namespace server {

namespace
{
const int SavePeriodSeconds = 60;

void AddUid(nn::prepo::SystemPlayReport* report, const nn::nfc::TagId& tagId)
{
    int64_t int64Value;

    int64Value = 0;
    for(int i = 0; i < 7; ++i)
    {
        int64Value |= (static_cast<nn::Bit64>(tagId.uid[i]) << ((7 - 1 - i) * 8));
    }
    report->Add("Uid", int64Value);
    //NN_SDK_LOG("Uid 0x%016llx\n", int64Value);
}

void AddModelInfo(nn::prepo::SystemPlayReport* report, const nn::nfp::ModelInfo& modelInfo)
{
    nn::prepo::Any64BitId any64BitValue;

    // Character ID
    any64BitValue.id = 0;
    for(int i = 0; i < 3; ++i)
    {
        any64BitValue.id |= (static_cast<nn::Bit64>(modelInfo.characterId[i]) << ((3 - 1 - i) * 8));
    }
    report->Add("CharacterId", any64BitValue);
    //NN_SDK_LOG("CharacterId 0x%016llx\n", any64BitValue.id);

    // Numbering ID
    any64BitValue.id = modelInfo.numberingId;
    report->Add("NumberingId", any64BitValue);
    //NN_SDK_LOG("NumberingId %lld\n", any64BitValue.id);

    // 4. Series ID
    any64BitValue.id = modelInfo.seriesId;
    report->Add("SeriesId", any64BitValue);
    //NN_SDK_LOG("seriesId %lld\n", any64BitValue.id);

    // 5. NFP Type
    any64BitValue.id = modelInfo.nfpType;
    report->Add("NfpType", any64BitValue);
    //NN_SDK_LOG("NfpType %lld\n", any64BitValue.id);
}

void AddApplicationId(nn::prepo::SystemPlayReport* report, const nn::os::ProcessId& processId) NN_NOEXCEPT
{
    nn::prepo::Any64BitId any64BitValue;

    nn::arp::ApplicationLaunchProperty prop;
    if(nn::arp::GetApplicationLaunchProperty(&prop, processId).IsFailure())
    {
        prop.id.value = 0;
    }

    any64BitValue.id = prop.id.value;
    report->Add("ApplicationId", any64BitValue);
    //NN_SDK_LOG("ApplicationId 0x%016llx\n", any64BitValue.id);
}

void AddAccessInfo(nn::prepo::SystemPlayReport* report, PlayReport::AccessInfo accessInfo) NN_NOEXCEPT
{
    nn::prepo::Any64BitId any64BitValue;
    any64BitValue.id = accessInfo;
    report->Add("AccessInfo", any64BitValue);
    //NN_SDK_LOG("AccessInfo %lld\n", any64BitValue.id);
}

void AddDeviceType(nn::prepo::SystemPlayReport* report, nn::hid::NpadIdType npadId) NN_NOEXCEPT
{
    nn::prepo::Any64BitId any64BitValue;

    nn::hid::system::NpadDeviceTypeSet deviceTypeSet = nn::hid::system::GetNpadDeviceType(npadId);
    if(deviceTypeSet.Test<nn::hid::system::NpadDeviceType::JoyConRight>())
    {
        any64BitValue.id = PlayReport::DeviceType_JoyConRight;
    }
    else if(deviceTypeSet.Test<nn::hid::system::NpadDeviceType::Handheld>()
            || deviceTypeSet.Test<nn::hid::system::NpadDeviceType::HandheldJoyRight>())
    {
        any64BitValue.id = PlayReport::DeviceType_Handheld;
    }
    else
    {
        any64BitValue.id = PlayReport::DeviceType_FullKeyController;
    }
    report->Add("DeviceType", any64BitValue);
    //NN_SDK_LOG("DeviceType %lld\n", any64BitValue.id);
}

}

PlayReport::PlayReport() NN_NOEXCEPT
{
    nn::os::InitializeMutex(&m_Mutex, true, 0);
}

PlayReport::~PlayReport() NN_NOEXCEPT
{
    nn::os::FinalizeMutex(&m_Mutex);
}

void PlayReport::Lock() NN_NOEXCEPT
{
    nn::os::LockMutex(&m_Mutex);
}

void PlayReport::Unlock() NN_NOEXCEPT
{
    nn::os::UnlockMutex(&m_Mutex);
}

void PlayReport::SetTagId(const nn::nfc::TagId& tagId) NN_NOEXCEPT
{
    m_TagId = tagId;
}

void PlayReport::SetModelInfo(const nn::nfp::ModelInfo& modelInfo) NN_NOEXCEPT
{
    m_ModelInfo = modelInfo;
}

void PlayReport::SetProcessId(const nn::os::ProcessId& processId) NN_NOEXCEPT
{
    m_ProcessId = processId;
}

void PlayReport::SetAccessInfo(AccessInfo accessInfo) NN_NOEXCEPT
{
    m_AccessInfo = accessInfo;
}

void PlayReport::SetNpadId(nn::hid::NpadIdType npadId) NN_NOEXCEPT
{
    m_NpadId = npadId;
}

void PlayReport::Save() NN_NOEXCEPT
{
    if(!IsSaveEnabled())
    {
        return;
    }

    nn::prepo::SystemPlayReport report("nfp_tag_access");
    nn::ApplicationId appId = {0x0100000000000020};
    report.SetApplicationId(appId);

    // 1. UID
    // 2. Character ID
    // 3. Numbering ID
    // 4. Series ID
    // 5. NFP Type
    // 6. アプリケーション ID
    // 7. タグアクセスの種類(Mii・ニックネーム書き込み / ゲームデータ消去 / 初期化 / アプリケーションデータ書き込み / 読み込みのみ)
    // 8. 読み取りデバイスの情報（ Joy-Con 無線 / Joy-Con 有線 / フルキーコン ）
    const int keyValueCount = 8;

    size_t bufferSize = nn::prepo::SystemPlayReport::BufferSizeMin + nn::prepo::KeyValueSizeMax * keyValueCount;
    std::unique_ptr<nn::Bit8[]> buffer(new nn::Bit8[bufferSize]);
    report.SetBuffer(buffer.get(), bufferSize);

    // 1. UID
    AddUid(&report, m_TagId);

    // 2. Character ID
    // 3. Numbering ID
    // 4. Series ID
    // 5. NFP Type
    AddModelInfo(&report, m_ModelInfo);

    // 6. アプリケーション ID
    AddApplicationId(&report, m_ProcessId);

    // 7. タグアクセスの種類(Mii・ニックネーム書き込み / ゲームデータ消去 / 初期化 / アプリケーションデータ書き込み / 読み込みのみ)
    AddAccessInfo(&report, m_AccessInfo);

    // 8. 読み取りデバイスの情報（ Joy-Con 無線 / Joy-Con 有線 / フルキーコン ）
    AddDeviceType(&report, m_NpadId);

    report.Save();

    m_LastSavedTick = nn::os::GetSystemTick();
    m_LastSavedTagId = m_TagId;
}

bool PlayReport::IsSaveEnabled()
{
    if(nn::settings::fwdbg::IsDebugModeEnabled())
    {
        // 開発機の場合

        bool isEnabled;
        if(nn::settings::fwdbg::GetSettingsItemValue(&isEnabled, sizeof(isEnabled), "nfp", "play_report") != sizeof(isEnabled))
        {
            // Key が見つからなかった場合は安全な方法を選択する(レポートは保存しない)
            return false;
        }

        if(!isEnabled)
        {
            return false;
        }
    }

    bool isFirstTime = (m_LastSavedTick.GetInt64Value() == 0);

    // 以下いずれかの条件を満している場合に Save 可能
    // 初回
    // 前回から1分超過
    // タグが変わっている
    // 書き込みがおこなわれた
    return (isFirstTime
            || SavePeriodSeconds < nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - m_LastSavedTick).GetSeconds()
            || !nn::nfc::server::util::IsEqualTagId(m_TagId, m_LastSavedTagId)
            || m_AccessInfo != AccessInfo_Read);
}

}}}  // namespace nn::nfp::server
