﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/util/util_ScopeExit.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include "nfp_Date.h"

namespace nn { namespace nfp {namespace server {

namespace
{
}

Date::Date() NN_NOEXCEPT
{
    SetDefault();
};

Date::Date(int year, int month, int day) NN_NOEXCEPT
{
    m_Year = year;
    m_Month = month;
    m_Day = day;

    Validate();
}

Date::Date(nn::Bit16 tagDate) NN_NOEXCEPT
{
    m_Year  = (tagDate >> 9) + 2000;
    m_Month = (tagDate >> 5) & 0x0F;
    m_Day   = tagDate & 0x1F;

    Validate();
}

Date::Date(nn::Bit16 tagDate, bool withValidate) NN_NOEXCEPT
{
    m_Year  = (tagDate >> 9) + 2000;
    m_Month = (tagDate >> 5) & 0x0F;
    m_Day   = tagDate & 0x1F;

    if(withValidate)
    {
        Validate();
    }
}

Date::Date(const nn::nfp::Date& nfpDate) NN_NOEXCEPT
{
    m_Year  = nfpDate.year;
    m_Month = nfpDate.month;
    m_Day   = nfpDate.day;

    Validate();
}

Date::Date(const nn::nfp::Date& nfpDate, bool withValidate) NN_NOEXCEPT
{
    m_Year  = nfpDate.year;
    m_Month = nfpDate.month;
    m_Day   = nfpDate.day;

    if(withValidate)
    {
        Validate();
    }
}

nn::Bit16 Date::GetTagDate() NN_NOEXCEPT
{
    return static_cast<nn::Bit16>((static_cast<nn::Bit16>(m_Year - 2000) << 9)
                                  | (static_cast<nn::Bit16>(m_Month) << 5)
                                  | (static_cast<nn::Bit16>(m_Day)));
}

nn::nfp::Date Date::GetNfpDate() NN_NOEXCEPT
{
    nn::nfp::Date date;
    date.year  = static_cast<int16_t>(m_Year);
    date.month = static_cast<int8_t>(m_Month);
    date.day   = static_cast<int8_t>(m_Day);
    return date;
}

Date Date::GetNow() NN_NOEXCEPT
{
    nn::time::PosixTime posixTime;

    nn::Result result = nn::time::Initialize();
    if(result.IsFailure())
    {
        return Date();
    }
    NN_UTIL_SCOPE_EXIT
    {
        nn::time::Finalize();
    };

    result = nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime);
    if(result.IsFailure())
    {
        return Date();
    }

    // PosixTime を日時へ変換します。
    nn::time::CalendarTime calendarTime;
    nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
    result = nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime);
    if(result.IsFailure())
    {
        return Date();
    }

    return Date(calendarTime.year, calendarTime.month, calendarTime.day);
}

void Date::SetDefault() NN_NOEXCEPT
{
    m_Year = 2000;
    m_Month = 1;
    m_Day = 1;
}

void Date::Validate() NN_NOEXCEPT
{
    if(!nn::time::IsValidDate(m_Year, m_Month, m_Day))
    {
        SetDefault();
    }
}

}}}  // namespace nn::nfp::server
