﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/news_Database.h>
#include <nn/news/news_Result.h>
#include <nn/news/detail/news_ShimLibraryGlobal.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace news {

Database::Database() NN_NOEXCEPT :
    m_Handle(nullptr)
{
}

Database::~Database() NN_NOEXCEPT
{
    Close();
}

nn::Result Database::Open() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_Handle, ResultAlreadyOpened());

    detail::ipc::INewsDatabaseService* handle = nullptr;

    NN_RESULT_DO(detail::ShimLibraryGlobal::GetInstance().CreateNewsDatabaseService(&handle));

    m_Handle = handle;

    NN_RESULT_SUCCESS;
}

void Database::Close() NN_NOEXCEPT
{
    if (m_Handle)
    {
        nn::sf::ReleaseSharedObject(m_Handle);
        m_Handle = nullptr;
    }
}

nn::Result Database::GetList(int* outCount, NewsRecord* outRecords,
    const char* wherePhrase, const char* orderByPhrase, int offset, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outRecords);
    NN_SDK_REQUIRES_NOT_NULL(wherePhrase);
    NN_SDK_REQUIRES_NOT_NULL(orderByPhrase);
    NN_SDK_REQUIRES(offset >= 0);
    NN_SDK_REQUIRES(count > 0);

    size_t wherePhraseLength = static_cast<size_t>(nn::util::Strnlen(wherePhrase, WherePhraseLengthMax + 1));

    NN_RESULT_THROW_UNLESS(wherePhraseLength <= WherePhraseLengthMax, ResultInvalidArgument());

    size_t orderByPhraseLength = static_cast<size_t>(nn::util::Strnlen(orderByPhrase, OrderByPhraseLengthMax + 1));

    NN_RESULT_THROW_UNLESS(orderByPhraseLength <= OrderByPhraseLengthMax, ResultInvalidArgument());

    NN_RESULT_THROW_UNLESS(m_Handle, ResultNotOpened());

    int32_t actualCount = 0;

    NN_RESULT_DO(m_Handle->GetList(&actualCount, nn::sf::OutArray<NewsRecord>(outRecords, static_cast<size_t>(count)),
        nn::sf::InArray<char>(wherePhrase, wherePhraseLength + 1),
        nn::sf::InArray<char>(orderByPhrase, orderByPhraseLength + 1),
        offset));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result Database::Count(int* outCount, const char* wherePhrase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(wherePhrase);

    size_t length = static_cast<size_t>(nn::util::Strnlen(wherePhrase, WherePhraseLengthMax + 1));

    NN_RESULT_THROW_UNLESS(length <= WherePhraseLengthMax, ResultInvalidArgument());

    NN_RESULT_THROW_UNLESS(m_Handle, ResultNotOpened());

    int32_t actualCount = 0;

    NN_RESULT_DO(m_Handle->Count(&actualCount,
        nn::sf::InArray<char>(wherePhrase, length + 1)));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result Database::Count(int* outCount, bool distinct, const char* key, const char* wherePhrase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(key);
    NN_SDK_REQUIRES_NOT_NULL(wherePhrase);

    size_t keyLength = static_cast<size_t>(nn::util::Strnlen(key, KeyLengthMax + 1));

    NN_RESULT_THROW_UNLESS(keyLength > 0 && keyLength <= KeyLengthMax, ResultInvalidArgument());

    size_t wherePhraseLength = static_cast<size_t>(nn::util::Strnlen(wherePhrase, WherePhraseLengthMax + 1));

    NN_RESULT_THROW_UNLESS(wherePhraseLength <= WherePhraseLengthMax, ResultInvalidArgument());

    NN_RESULT_THROW_UNLESS(m_Handle, ResultNotOpened());

    int32_t actualCount = 0;

    NN_RESULT_DO(m_Handle->CountWithKey(&actualCount,
        distinct,
        nn::sf::InArray<char>(key, keyLength + 1),
        nn::sf::InArray<char>(wherePhrase, wherePhraseLength + 1)));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result Database::Update(const char* key, int32_t value, const char* wherePhrase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(key);
    NN_SDK_REQUIRES_NOT_NULL(wherePhrase);

    size_t keyLength = static_cast<size_t>(nn::util::Strnlen(key, KeyLengthMax + 1));

    NN_RESULT_THROW_UNLESS(keyLength > 0 && keyLength <= KeyLengthMax, ResultInvalidArgument());

    size_t wherePhraseLength = static_cast<size_t>(nn::util::Strnlen(wherePhrase, WherePhraseLengthMax + 1));

    NN_RESULT_THROW_UNLESS(wherePhraseLength <= WherePhraseLengthMax, ResultInvalidArgument());

    NN_RESULT_THROW_UNLESS(m_Handle, ResultNotOpened());

    NN_RESULT_DO(m_Handle->UpdateIntegerValue(nn::sf::InArray<char>(key, keyLength + 1),
        value,
        nn::sf::InArray<char>(wherePhrase, wherePhraseLength + 1)));

    NN_RESULT_SUCCESS;
}

nn::Result Database::Update(const char* key, const char* value, const char* wherePhrase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(key);
    NN_SDK_REQUIRES_NOT_NULL(value);
    NN_SDK_REQUIRES_NOT_NULL(wherePhrase);

    size_t keyLength = static_cast<size_t>(nn::util::Strnlen(key, KeyLengthMax + 1));

    NN_RESULT_THROW_UNLESS(keyLength > 0 && keyLength <= KeyLengthMax, ResultInvalidArgument());

    size_t valueLength = static_cast<size_t>(nn::util::Strnlen(value, StringValueLengthMax + 1));

    NN_RESULT_THROW_UNLESS(valueLength <= StringValueLengthMax, ResultInvalidArgument());

    size_t wherePhraseLength = static_cast<size_t>(nn::util::Strnlen(wherePhrase, WherePhraseLengthMax + 1));

    NN_RESULT_THROW_UNLESS(wherePhraseLength <= WherePhraseLengthMax, ResultInvalidArgument());

    NN_RESULT_THROW_UNLESS(m_Handle, ResultNotOpened());

    NN_RESULT_DO(m_Handle->UpdateStringValue(nn::sf::InArray<char>(key, keyLength + 1),
        nn::sf::InArray<char>(value, valueLength + 1),
        nn::sf::InArray<char>(wherePhrase, wherePhraseLength + 1)));

    NN_RESULT_SUCCESS;
}

nn::Result Database::UpdateWithAddition(const char* key, int32_t value, const char* wherePhrase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(key);
    NN_SDK_REQUIRES_NOT_NULL(wherePhrase);

    size_t keyLength = static_cast<size_t>(nn::util::Strnlen(key, KeyLengthMax + 1));

    NN_RESULT_THROW_UNLESS(keyLength > 0 && keyLength <= KeyLengthMax, ResultInvalidArgument());

    size_t wherePhraseLength = static_cast<size_t>(nn::util::Strnlen(wherePhrase, WherePhraseLengthMax + 1));

    NN_RESULT_THROW_UNLESS(wherePhraseLength <= WherePhraseLengthMax, ResultInvalidArgument());

    NN_RESULT_THROW_UNLESS(m_Handle, ResultNotOpened());

    NN_RESULT_DO(m_Handle->UpdateIntegerValueWithAddition(nn::sf::InArray<char>(key, keyLength + 1),
        value,
        nn::sf::InArray<char>(wherePhrase, wherePhraseLength + 1)));

    NN_RESULT_SUCCESS;
}

}}
