﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/news_ApiAdmin.h>
#include <nn/news/news_Result.h>
#include <nn/news/detail/news_ShimLibraryGlobal.h>
#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace news {

nn::Result SetPassphrase(nn::ApplicationId appId, const char* passphrase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(passphrase);

    size_t length = static_cast<size_t>(nn::util::Strnlen(passphrase, PassphraseLengthMax + 1));

    NN_RESULT_THROW_UNLESS(length > 0 && length <= PassphraseLengthMax, ResultInvalidArgument());

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    NN_RESULT_DO(session->SetPassphrase(appId, nn::sf::InArray<char>(passphrase, length + 1)));

    NN_RESULT_SUCCESS;
}

// NN_DEPRECATED
nn::Result SetSubscribeStatus(const char* topicId, SubscribeStatus status) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(topicId);

    size_t length = static_cast<size_t>(nn::util::Strnlen(topicId, TopicIdLengthMax + 1));

    NN_RESULT_THROW_UNLESS(length > 0 && length <= TopicIdLengthMax, ResultInvalidArgument());

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    NN_RESULT_DO(session->SetSubscriptionStatus(nn::sf::InArray<char>(topicId, length + 1), status));

    NN_RESULT_SUCCESS;
}

nn::Result SetSubscriptionStatus(const char* topicId, SubscriptionStatus status) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(topicId);

    size_t length = static_cast<size_t>(nn::util::Strnlen(topicId, TopicIdLengthMax + 1));

    NN_RESULT_THROW_UNLESS(length > 0 && length <= TopicIdLengthMax, ResultInvalidArgument());

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    NN_RESULT_DO(session->SetSubscriptionStatus(nn::sf::InArray<char>(topicId, length + 1), status));

    NN_RESULT_SUCCESS;
}

// NN_DEPRECATED
nn::Result GetSubscribeStatus(SubscribeStatus* outStatus, const char* topicId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outStatus);
    NN_SDK_REQUIRES_NOT_NULL(topicId);

    size_t length = static_cast<size_t>(nn::util::Strnlen(topicId, TopicIdLengthMax + 1));

    NN_RESULT_THROW_UNLESS(length > 0 && length <= TopicIdLengthMax, ResultInvalidArgument());

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    int32_t status = 0;

    NN_RESULT_DO(session->GetSubscriptionStatus(&status, nn::sf::InArray<char>(topicId, length + 1)));

    *outStatus = static_cast<SubscribeStatus>(status);

    NN_RESULT_SUCCESS;
}

nn::Result GetSubscriptionStatus(SubscriptionStatus* outStatus, const char* topicId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outStatus);
    NN_SDK_REQUIRES_NOT_NULL(topicId);

    size_t length = static_cast<size_t>(nn::util::Strnlen(topicId, TopicIdLengthMax + 1));

    NN_RESULT_THROW_UNLESS(length > 0 && length <= TopicIdLengthMax, ResultInvalidArgument());

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    int32_t status = 0;

    NN_RESULT_DO(session->GetSubscriptionStatus(&status, nn::sf::InArray<char>(topicId, length + 1)));

    *outStatus = static_cast<SubscriptionStatus>(status);

    NN_RESULT_SUCCESS;
}

nn::Result GetTopicList(int* outCount, TopicId* outTopicIds, int count, SubscriptionStatusFilter filter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outTopicIds);
    NN_SDK_REQUIRES_GREATER(count, 0);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    int32_t actualCount = 0;

    NN_RESULT_DO(session->GetTopicList(&actualCount, nn::sf::OutArray<TopicId>(outTopicIds, count), static_cast<int32_t>(filter)));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result RequestAutoSubscription(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    NN_RESULT_DO(session->RequestAutoSubscription(appId));

    NN_RESULT_SUCCESS;
}

nn::Result ClearStorage() NN_NOEXCEPT
{
    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    NN_RESULT_DO(session->ClearStorage());

    NN_RESULT_SUCCESS;
}

// NN_DEPRECATED
nn::Result ClearSubscribeStatusAll() NN_NOEXCEPT
{
    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    NN_RESULT_DO(session->ClearSubscriptionStatusAll());

    NN_RESULT_SUCCESS;
}

nn::Result ClearSubscriptionStatusAll() NN_NOEXCEPT
{
    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    NN_RESULT_DO(session->ClearSubscriptionStatusAll());

    NN_RESULT_SUCCESS;
}

nn::Result IsSystemUpdateRequired(bool* outIsRequired) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIsRequired);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    NN_RESULT_DO(session->IsSystemUpdateRequired(outIsRequired));

    NN_RESULT_SUCCESS;
}

nn::Result RequestImmediateReception(const char* topicId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(topicId);

    size_t length = static_cast<size_t>(nn::util::Strnlen(topicId, TopicIdLengthMax + 1));

    NN_RESULT_THROW_UNLESS(length > 0 && length <= TopicIdLengthMax, ResultInvalidArgument());

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    NN_RESULT_DO(session->RequestImmediateReception(nn::sf::InArray<char>(topicId, length + 1)));

    NN_RESULT_SUCCESS;
}

nn::Result PostLocalNews(const void* data, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(data);
    NN_SDK_REQUIRES(dataSize > 0);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    NN_RESULT_DO(session->PostLocalNews(nn::sf::InBuffer(static_cast<const char*>(data), dataSize)));

    NN_RESULT_SUCCESS;
}

nn::Result DecodeArchiveFile(size_t* outSize, void* decoded, size_t decodedSize,
    const void* encoded, size_t encodedSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);
    NN_SDK_REQUIRES_NOT_NULL(decoded);
    NN_SDK_REQUIRES_GREATER(decodedSize, 0u);
    NN_SDK_REQUIRES_NOT_NULL(encoded);
    NN_SDK_REQUIRES_GREATER(encodedSize, 0u);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    uint64_t actualSize = 0;

    NN_RESULT_DO(session->DecodeArchiveFile(&actualSize, nn::sf::OutBuffer(reinterpret_cast<char*>(decoded), decodedSize),
        nn::sf::InBuffer(reinterpret_cast<const char*>(encoded), encodedSize)));

    *outSize = static_cast<size_t>(actualSize);

    NN_RESULT_SUCCESS;
}

nn::Result RegisterNewlyArrivedEvent(nn::os::SystemEvent* outEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outEvent);

    NN_RESULT_DO(detail::ShimLibraryGlobal::GetInstance().RegisterNewlyArrivedEvent(outEvent));

    NN_RESULT_SUCCESS;
}

nn::Result RegisterOverwriteEvent(nn::os::SystemEvent* outEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outEvent);

    NN_RESULT_DO(detail::ShimLibraryGlobal::GetInstance().RegisterOverwriteEvent(outEvent));

    NN_RESULT_SUCCESS;
}

}}
