﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/news/detail/service/core/news_NewsDownloader.h>
#include <nn/news/detail/service/core/news_AccountCountryChecker.h>
#include <nn/news/detail/service/core/news_NewsDatabase.h>
#include <nn/news/detail/service/core/news_NewsImporter.h>
#include <nn/news/detail/service/core/news_NewsListReader.h>
#include <nn/news/detail/service/core/news_NewsTaskManager.h>
#include <nn/news/detail/service/core/news_PassphraseManager.h>
#include <nn/news/detail/service/core/news_ReceivedHistoryManager.h>
#include <nn/news/detail/service/msgpack/news_FileInputStream.h>
#include <nn/bcat/bcat_Result.h>
#include <nn/bcat/bcat_ResultPrivate.h>
#include <nn/bcat/service/bcat_ArchiveDownloader.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/settings/system/settings_SerialNumber.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

namespace
{
    const char* Fqdn = "bcat-list-%.cdn.nintendo.net";

    const char* ListPath = "news-dl:/list.msgpack";
    const char* ListMetaPath = "news-dl:/list.msgpack.meta";

    const char* DataPath = "news-dl:/data.msgpack";
    const char* DataMetaPath = "news-dl:/data.msgpack.meta";
}

namespace
{
    nn::bcat::service::ArchiveDownloader g_ArchiveDownloader;
    Bit8 g_FileWorkBuffer[128 * 1024];
}

namespace
{
    bool ProgressCallback(int64_t now, int64_t total, void* param) NN_NOEXCEPT
    {
        NN_UNUSED(now);
        NN_UNUSED(total);

        nn::nifm::NetworkConnection* connection = reinterpret_cast<nn::nifm::NetworkConnection*>(param);

        // NN_DETAIL_NEWS_INFO("[news] Progress(now=%lld, total=%lld, isAvailable=%d)\n", now, total, connection->IsAvailable());

        return connection->IsAvailable();
    }

    int8_t GetLanguage(const char* languageCode) NN_NOEXCEPT
    {
        nn::settings::LanguageCode code = {};
        nn::util::Strlcpy(code.string, languageCode, sizeof (code.string));

        for (int i = nn::settings::Language_Japanese; i <= nn::settings::Language_TraditionalChinese; i++)
        {
            if (code == static_cast<nn::settings::Language>(i))
            {
                return static_cast<int8_t>(i);
            }
        }

        // 未知の言語
        return -1;
    }

    nn::Bit16 GetSerialBit() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(nn::Bit16, s_SerialBit, = 0);

        if (s_SerialBit == 0)
        {
            nn::settings::system::SerialNumber serialNumber = {};
            nn::settings::system::GetSerialNumber(&serialNumber);

            // シリアルナンバーの 13 桁目（チェックディジットを除く末尾）の文字を取得する。
            char c = serialNumber.string[12];

            if (c >= '0' && c <= '9')
            {
                s_SerialBit = (1 << static_cast<nn::Bit16>(c - '0'));
            }
        }

        return s_SerialBit;
    }
}

namespace
{
    nn::Result DownloadList(ETag* outNewETag, const TopicId& topicId, const ETag& eTag,
        nn::nifm::NetworkConnection& connection, nn::os::Event* pCancelEvent) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(connection.IsAvailable(), ResultInternetRequestNotAccepted());

        nn::settings::LanguageCode language;
        nn::settings::GetLanguageCode(&language);

        AccountCountryChecker::QueryString queryString;
        AccountCountryChecker::GetInstance().GetQueryString(&queryString);

        Url url = {};

        nn::util::SNPrintf(url.value, sizeof (url.value), "https://%s/api/nx/v1/list/%s?l=%s%s%s",
            Fqdn, topicId.value, language.string, queryString.value[0] ? "&" : "", queryString.value);

        NN_DETAIL_NEWS_INFO("[news] DownloadList(%s) ...\n", url.value);

        nn::ApplicationId appId;
        char passpharase[PassphraseLengthMax + 1] = {};

        NN_RESULT_DO(PassphraseManager::GetInstance().Get(&appId, passpharase, sizeof (passpharase)));

        g_ArchiveDownloader.Reset();
        g_ArchiveDownloader.SetPath(ListPath, ListMetaPath);
        g_ArchiveDownloader.SetPassphrase(appId, passpharase);
        g_ArchiveDownloader.SetETagForIfNoneMatch(eTag.value);
        g_ArchiveDownloader.SetProgressCallback(ProgressCallback, &connection);
        g_ArchiveDownloader.SetJournalSize(DownloadStorageJournalSize);
        g_ArchiveDownloader.SetFileWorkBuffer(g_FileWorkBuffer, sizeof (g_FileWorkBuffer));

        NN_RESULT_DO(g_ArchiveDownloader.Download(url.value, pCancelEvent));

        nn::util::Strlcpy(outNewETag->value, g_ArchiveDownloader.GetETag().value, sizeof (outNewETag->value));

        NN_DETAIL_NEWS_INFO("[news] DownloadList done!\n");
        NN_DETAIL_NEWS_INFO("[news] ETag = %s\n", outNewETag->value);

        FileSystem::Commit(NN_DETAIL_NEWS_DOWNLOAD_MOUNT_NAME);

        NN_RESULT_SUCCESS;
    }

    nn::Result DownloadList(const Url& url,
        nn::nifm::NetworkConnection& connection, nn::os::Event* pCancelEvent) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(connection.IsAvailable(), ResultInternetRequestNotAccepted());

        nn::settings::LanguageCode language;
        nn::settings::GetLanguageCode(&language);

        AccountCountryChecker::QueryString queryString;
        AccountCountryChecker::GetInstance().GetQueryString(&queryString);

        Url processedUrl = {};

        nn::util::SNPrintf(processedUrl.value, sizeof (processedUrl.value), "%s?l=%s%s%s",
            url.value, language.string, queryString.value[0] ? "&" : "", queryString.value);

        NN_DETAIL_NEWS_INFO("[news] DownloadList(%s) ...\n", processedUrl.value);

        nn::ApplicationId appId;
        char passpharase[PassphraseLengthMax + 1] = {};

        NN_RESULT_DO(PassphraseManager::GetInstance().Get(&appId, passpharase, sizeof (passpharase)));

        g_ArchiveDownloader.Reset();
        g_ArchiveDownloader.SetPath(ListPath, ListMetaPath);
        g_ArchiveDownloader.SetPassphrase(appId, passpharase);
        g_ArchiveDownloader.SetProgressCallback(ProgressCallback, &connection);
        g_ArchiveDownloader.SetJournalSize(DownloadStorageJournalSize);
        g_ArchiveDownloader.SetFileWorkBuffer(g_FileWorkBuffer, sizeof (g_FileWorkBuffer));

        NN_RESULT_DO(g_ArchiveDownloader.Download(processedUrl.value, pCancelEvent));

        NN_DETAIL_NEWS_INFO("[news] DownloadList done!\n");

        FileSystem::Commit(NN_DETAIL_NEWS_DOWNLOAD_MOUNT_NAME);

        NN_RESULT_SUCCESS;
    }

    nn::Result DownloadEntryImpl(const TopicId& topicId, const Url& url, uint64_t newsId, int8_t language,
        uint64_t dataId, bool isOverwrite, bool isTestDistribution,
        nn::nifm::NetworkConnection& connection, nn::bgtc::Task& bgtcTask, nn::os::Event* pCancelEvent) NN_NOEXCEPT
    {
        if (isOverwrite && !isTestDistribution)
        {
            NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT();

            NewsRecord record = {};
            int count;

            char wherePhrase[64] = {};
            nn::util::SNPrintf(wherePhrase, sizeof (wherePhrase), "news_id = 'NS%020llu'", newsId);

            // DB にレコードがあるかどうかを確認する。
            NN_RESULT_DO(NewsDatabase::GetInstance().GetList(&count, &record, wherePhrase, "", 0, 1));

            if (count == 0)
            {
                NN_DETAIL_NEWS_INFO("[news] NS%020llu is already deleted.\n", newsId);
                NN_RESULT_SUCCESS;
            }
        }

        NN_DETAIL_NEWS_INFO("[news] DownloadEntry(%s) ...\n", url.value);

        nn::ApplicationId appId;
        char passpharase[PassphraseLengthMax + 1] = {};

        NN_RESULT_DO(PassphraseManager::GetInstance().Get(&appId, passpharase, sizeof (passpharase)));

        g_ArchiveDownloader.Reset();
        g_ArchiveDownloader.SetPath(DataPath, DataMetaPath);
        g_ArchiveDownloader.SetPassphrase(appId, passpharase);
        g_ArchiveDownloader.SetProgressCallback(ProgressCallback, &connection);
        g_ArchiveDownloader.SetJournalSize(DownloadStorageJournalSize);
        g_ArchiveDownloader.SetFileWorkBuffer(g_FileWorkBuffer, sizeof (g_FileWorkBuffer));

        NN_RESULT_DO(g_ArchiveDownloader.Download(url.value, pCancelEvent));

        NN_DETAIL_NEWS_INFO("[news] DownloadEntry done!\n");

        FileSystem::Commit(NN_DETAIL_NEWS_DOWNLOAD_MOUNT_NAME);

        NN_DETAIL_NEWS_INFO("[news] NewsImporter::ImportFromNetwork ...\n");

        // TORIAEZU:
        // Ocean がストレージがロックしていた場合、ここで無限待ちになってしまうためタスクの完了を告げる。
        bgtcTask.NotifyFinished();

        StorageManager::GetInstance().Lock();

        NN_UTIL_SCOPE_EXIT
        {
            StorageManager::GetInstance().Unlock();
        };

        // TORIAEZU:
        // Ocean がストレージをロックしていなかった場合、すぐにここに来てタスクを再開するため BG 処理は継続される。
        NN_RESULT_DO(bgtcTask.NotifyStarting());

        SubscriptionStatus status = NewsTaskManager::GetInstance().GetSubscriptionStatus(topicId);

        if (status == SubscriptionStatus_Unsubscribed || status == SubscriptionStatus_Unconfigured)
        {
            NN_DETAIL_NEWS_INFO("[news] '%s' is no longer subscribed.\n", topicId.value);
            NN_RESULT_THROW(ResultNoLongerSubscribed());
        }

        NN_RESULT_DO(NewsImporter::ImportFromNetwork(DataPath,
            false, 0, language, dataId, isOverwrite, isTestDistribution, g_FileWorkBuffer, sizeof (g_FileWorkBuffer)));

        // TODO: インポートしたら、データを削除する。

        NN_DETAIL_NEWS_INFO("[news] NewsImporter::ImportFromNetwork done!\n");

        NN_RESULT_SUCCESS;
    }

    nn::Result DownloadEntry(const TopicId& topicId, const NewsListReader::DataEntry& entry, bool isTestDistribution,
        nn::nifm::NetworkConnection& connection, nn::bgtc::Task& bgtcTask, nn::os::Event* pCancelEvent) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(connection.IsAvailable(), ResultInternetRequestNotAccepted());

        if (!Version::GetInstance().IsSupported(entry.version))
        {
            NN_DETAIL_NEWS_INFO("[news] System update is required. (v%d.%d)\n", entry.version.format, entry.version.semantics);
            NN_RESULT_DO(Version::GetInstance().RequireSystemUpdate());

            NN_RESULT_THROW(ResultSystemUpdateRequired());
        }

        ReceivedHistoryManager::SearchKey key = {false, 0, 'N', entry.newsId, GetLanguage(entry.language), entry.dataId};
        bool isImportable = false;
        bool isOverwritable = false;

        NN_DETAIL_NEWS_INFO("[news] The news language = %s(%d)\n", entry.language, key.language);

        NN_RESULT_DO(ReceivedHistoryManager::GetInstance().IsImportable(&isImportable, key));

        if (entry.isOverwrite)
        {
            NN_RESULT_DO(ReceivedHistoryManager::GetInstance().IsOverwritable(&isOverwritable, key));
        }

        if (isImportable || isOverwritable || isTestDistribution)
        {
            if (entry.conditions.isSerialBits)
            {
                nn::Bit16 serialBit = GetSerialBit();

                NN_DETAIL_NEWS_INFO("[news] <AB TEST> Serial bits : Device serial bit = 0x%04X : 0x%04X => %04X\n",
                    entry.conditions.serialBits, serialBit, (entry.conditions.serialBits & serialBit));

                if ((entry.conditions.serialBits & serialBit) == 0)
                {
                    NN_DETAIL_NEWS_INFO("[news] The serial pattern is mismatched.\n");
                    NN_RESULT_SUCCESS;
                }
            }

            NN_RESULT_DO(DownloadEntryImpl(topicId, entry.url, entry.newsId, key.language,
                entry.dataId, !isImportable, isTestDistribution, connection, bgtcTask, pCancelEvent));
        }
        else
        {
            NN_DETAIL_NEWS_INFO("[news] The news (id = %llu) is already downloaded.\n", entry.newsId);
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result DownloadImpl(nn::nifm::NetworkConnection& connection, nn::bgtc::Task& bgtcTask, nn::os::Event* pCancelEvent) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(connection.IsAvailable(), ResultInternetRequestNotAccepted());

        nn::settings::LanguageCode language;
        nn::settings::GetLanguageCode(&language);

        NN_DETAIL_NEWS_INFO("[news] Language = %s\n", language.string);

        int offset = 0;
        int count = 0;

        do
        {
            detail::service::msgpack::FileInputStream stream;

            char buffer[2048];
            stream.SetBuffer(buffer, sizeof (buffer));

            NN_RESULT_DO(stream.Open(ListPath));

            NewsListReader reader;

            reader.SetLanguage(language.string);

            NewsListReader::DataEntry entries[32];

            NN_RESULT_DO(reader.Read(&count, stream, entries, offset, 32));

            stream.Close();

            NewsListReader::ListInfo info = reader.GetListInfo();

            if (offset == 0)
            {
                NN_DETAIL_NEWS_INFO("[news] IsInService = %d\n", info.isInService);

                NN_RESULT_THROW_UNLESS(info.isInService, ResultServiceExpired());

                NN_DETAIL_NEWS_INFO("[news] IsNintendoAccountRequired = %d\n", info.isNintendoAccountRequired);

                if (info.isNintendoAccountRequired)
                {
                    NN_RESULT_THROW_UNLESS(Account::IsAvailableNintendoAccountLinked(), ResultNintendoAccountNotLinked());
                }

                NN_DETAIL_NEWS_INFO("[news] IsTestDistribution = %d\n", info.isTestDistribution);
            }

            NN_DETAIL_NEWS_INFO("[news] Entries [%d, %d]\n", offset, count);

            for (int i = 0; i < count; i++)
            {
                NN_RESULT_DO(DownloadEntry(info.topicId, entries[i], info.isTestDistribution,
                    connection, bgtcTask, pCancelEvent));
            }

            offset += count;
        }
        while (count > 0);

        NN_RESULT_SUCCESS;
    }
}

nn::Result NewsDownloader::Download(const TopicId& topicId, const ETag& eTag,
    nn::nifm::NetworkConnection& connection, nn::bgtc::Task& bgtcTask, nn::os::Event* pCancelEvent) NN_NOEXCEPT
{
    NN_DETAIL_NEWS_DOWNLOAD_STORAGE_SCOPED_MOUNT();

    ETag newETag = {};

    NN_RESULT_TRY(DownloadList(&newETag, topicId, eTag, connection, pCancelEvent))
        NN_RESULT_CATCH(nn::bcat::ResultServerError304)
        {
            NN_DETAIL_NEWS_INFO("[news] The news list is not modified. (status 304)\n");
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO(DownloadImpl(connection, bgtcTask, pCancelEvent));

    // 配信リストに含まれるニュースデータをすべて受信したので、ETag を更新する。
    NN_RESULT_DO(NewsTaskManager::GetInstance().UpdateETag(topicId, newETag));

    NN_RESULT_SUCCESS;
}

nn::Result NewsDownloader::Download(const Url& url,
    nn::nifm::NetworkConnection& connection, nn::bgtc::Task& bgtcTask, nn::os::Event* pCancelEvent) NN_NOEXCEPT
{
    NN_DETAIL_NEWS_DOWNLOAD_STORAGE_SCOPED_MOUNT();

    NN_RESULT_DO(DownloadList(url, connection, pCancelEvent));

    NN_RESULT_DO(DownloadImpl(connection, bgtcTask, pCancelEvent));

    NN_RESULT_SUCCESS;
}

}}}}}
