﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <curl/curl.h>

#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/netdiag/detail/netdiag_Log.h>
#include <nn/netdiag/netdiag_ResultPrivate.h>

namespace nn { namespace netdiag { namespace detail {

inline Result AcquireCurlHandle(CURL** pOut) NN_NOEXCEPT
{
    auto* curl = curl_easy_init();
    if (!curl)
    {
        NN_DETAIL_NETDIAG_INFO("[netdiag] curl_easy_init failed.\n");
        NN_RESULT_THROW(ResultFailedCurlInitialize());
    }
    *pOut = curl;
    NN_RESULT_SUCCESS;
}

inline Result AcquireCurlMultiHandle(CURLM** pOut) NN_NOEXCEPT
{
    auto* curlMultiHandle = curl_multi_init();
    if (!curlMultiHandle)
    {
        NN_DETAIL_NETDIAG_INFO("[netdiag] curl_multi_init failed.\n");
        NN_RESULT_THROW(ResultFailedCurlMultiInitialize());
    }
    *pOut = curlMultiHandle;
    NN_RESULT_SUCCESS;
}

inline Result HandleHttpStatus(long status) NN_NOEXCEPT
{
    if (status < 100 || status == 200)
    {
        // 1xx-2xx は成功扱いにする
        NN_RESULT_SUCCESS;
    }

    NN_DETAIL_NETDIAG_INFO("[netdiag] server error (%ld).\n", status);
    const auto Category = status / 100;
    const auto Description = status % 100;
    switch (Category)
    {
    case 4:
        switch (Description)
        {
        case 0:
            NN_RESULT_THROW(ResultHttpStatus400BadRequest());
        case 1:
            NN_RESULT_THROW(ResultHttpStatus401Unauthorized());
        case 3:
            NN_RESULT_THROW(ResultHttpStatus403Forbidden());
        case 4:
            NN_RESULT_THROW(ResultHttpStatus404NotFound());
        case 5:
            NN_RESULT_THROW(ResultHttpStatus405MethodNotAllowed());
        case 6:
            NN_RESULT_THROW(ResultHttpStatus406NotAcceptable());
        case 7:
            NN_RESULT_THROW(ResultHttpStatus407ProxyAuthenticationRequired());
        case 8:
            NN_RESULT_THROW(ResultHttpStatus408RequestTimeout());
        case 9:
            NN_RESULT_THROW(ResultHttpStatus409Conflict());
        case 10:
            NN_RESULT_THROW(ResultHttpStatus410Gone());
        case 11:
            NN_RESULT_THROW(ResultHttpStatus411LengthRequired());
        case 12:
            NN_RESULT_THROW(ResultHttpStatus412PreconditionFailed());
        case 13:
            NN_RESULT_THROW(ResultHttpStatus413PayloadTooLarge());
        case 14:
            NN_RESULT_THROW(ResultHttpStatus414UriTooLong());
        case 15:
            NN_RESULT_THROW(ResultHttpStatus415UnsupportedMediaType());
        case 16:
            NN_RESULT_THROW(ResultHttpStatus416RangeNotSatisfiable());
        case 17:
            NN_RESULT_THROW(ResultHttpStatus417ExpectationFailed());
        default:
            NN_RESULT_THROW(ResultHttpStatus4xx());
        }
    case 5:
        switch (Description)
        {
        case 0:
            NN_RESULT_THROW(ResultHttpStatus500InternalServerError());
        case 1:
            NN_RESULT_THROW(ResultHttpStatus501NotImplemented());
        case 2:
            NN_RESULT_THROW(ResultHttpStatus502BadGateway());
        case 3:
            NN_RESULT_THROW(ResultHttpStatus503ServiceUnavailable());
        case 4:
            NN_RESULT_THROW(ResultHttpStatus504GatewayTimeout());
        case 5:
            NN_RESULT_THROW(ResultHttpStatus505HttpVersionNotSupported());
        default:
            NN_RESULT_THROW(ResultHttpStatus5xx());
        }
    default:
        NN_RESULT_THROW(ResultFailedServerError());
    }
} // NOLINT(readability/fn_size)

inline Result HandleCurlError(CURLcode e) NN_NOEXCEPT
{
    if (e == CURLE_OK)
    {
        NN_RESULT_SUCCESS;
    }

    NN_DETAIL_NETDIAG_INFO("[netdiag] curl error (%d).\n", e);
    switch (e)
    {
    case CURLE_UNSUPPORTED_PROTOCOL:
        NN_RESULT_THROW(ResultCurlErrorUnsupportedProtocol());
    case CURLE_FAILED_INIT:
        NN_RESULT_THROW(ResultCurlErrorFailedInit());
    case CURLE_URL_MALFORMAT:
        NN_RESULT_THROW(ResultCurlErrorUrlMalformat());
    case CURLE_NOT_BUILT_IN:
        NN_RESULT_THROW(ResultCurlErrorNotBuiltIn());
    case CURLE_COULDNT_RESOLVE_PROXY:
        NN_RESULT_THROW(ResultCurlErrorCouldntResolveProxy());
    case CURLE_COULDNT_RESOLVE_HOST:
        NN_RESULT_THROW(ResultCurlErrorCouldntResolveHost());
    case CURLE_COULDNT_CONNECT:
        NN_RESULT_THROW(ResultCurlErrorCouldntConnect());
    case CURLE_REMOTE_ACCESS_DENIED:
        NN_RESULT_THROW(ResultCurlErrorRemoteAccessDenied());
    case CURLE_HTTP2:
        NN_RESULT_THROW(ResultCurlErrorHttp2());
    case CURLE_PARTIAL_FILE:
        NN_RESULT_THROW(ResultCurlErrorPartialFile());
    case CURLE_QUOTE_ERROR:
        NN_RESULT_THROW(ResultCurlErrorQuoteError());
    case CURLE_HTTP_RETURNED_ERROR:
        NN_RESULT_THROW(ResultCurlErrorHttpReturnedError());
    case CURLE_WRITE_ERROR:
        NN_RESULT_THROW(ResultCurlErrorWriteError());
    case CURLE_UPLOAD_FAILED:
        NN_RESULT_THROW(ResultCurlErrorUploadFailed());
    case CURLE_READ_ERROR:
        NN_RESULT_THROW(ResultCurlErrorReadError());
    case CURLE_OUT_OF_MEMORY:
        NN_RESULT_THROW(ResultCurlErrorOutOfMemory());
    case CURLE_OPERATION_TIMEDOUT:
        NN_RESULT_THROW(ResultCurlErrorOperationTimedout());
    case CURLE_RANGE_ERROR:
        NN_RESULT_THROW(ResultCurlErrorRangeError());
    case CURLE_HTTP_POST_ERROR:
        NN_RESULT_THROW(ResultCurlErrorHttpPostError());
    case CURLE_SSL_CONNECT_ERROR:
        NN_RESULT_THROW(ResultCurlErrorSslConnectError());
    case CURLE_BAD_DOWNLOAD_RESUME:
        NN_RESULT_THROW(ResultCurlErrorBadDownloadResume());
    case CURLE_FUNCTION_NOT_FOUND:
        NN_RESULT_THROW(ResultCurlErrorFunctionNotFound());
    case CURLE_ABORTED_BY_CALLBACK:
        NN_RESULT_THROW(ResultCurlErrorAbortedByCallback());
    case CURLE_BAD_FUNCTION_ARGUMENT:
        NN_RESULT_THROW(ResultCurlErrorBadFunctionArgument());
    case CURLE_INTERFACE_FAILED:
        NN_RESULT_THROW(ResultCurlErrorInterfaceFailed());
    case CURLE_TOO_MANY_REDIRECTS:
        NN_RESULT_THROW(ResultCurlErrorTooManyRedirects());
    case CURLE_UNKNOWN_OPTION:
        NN_RESULT_THROW(ResultCurlErrorUnknownOption());
    case CURLE_PEER_FAILED_VERIFICATION:
        NN_RESULT_THROW(ResultCurlErrorPeerFailedVerification());
    case CURLE_GOT_NOTHING:
        NN_RESULT_THROW(ResultCurlErrorGotNothing());
    case CURLE_SSL_ENGINE_NOTFOUND:
        NN_RESULT_THROW(ResultCurlErrorSslEngineNotFound());
    case CURLE_SSL_ENGINE_SETFAILED:
        NN_RESULT_THROW(ResultCurlErrorSslEngineSetFailed());
    case CURLE_SEND_ERROR:
        NN_RESULT_THROW(ResultCurlErrorSendError());
    case CURLE_RECV_ERROR:
        NN_RESULT_THROW(ResultCurlErrorRecvError());
    case CURLE_SSL_CERTPROBLEM:
        NN_RESULT_THROW(ResultCurlErrorSslCertProblem());
    case CURLE_SSL_CIPHER:
        NN_RESULT_THROW(ResultCurlErrorSslCipher());
    case CURLE_SSL_CACERT:
        NN_RESULT_THROW(ResultCurlErrorSslCaCert());
    case CURLE_BAD_CONTENT_ENCODING:
        NN_RESULT_THROW(ResultCurlErrorBadContentEncoding());
    case CURLE_FILESIZE_EXCEEDED:
        NN_RESULT_THROW(ResultCurlErrorFileSizeExceeded());
    case CURLE_USE_SSL_FAILED:
        NN_RESULT_THROW(ResultCurlErrorUseSslFailed());
    case CURLE_SEND_FAIL_REWIND:
        NN_RESULT_THROW(ResultCurlErrorSendFailRewind());
    case CURLE_SSL_ENGINE_INITFAILED:
        NN_RESULT_THROW(ResultCurlErrorSslEngineInitFailed());
    case CURLE_LOGIN_DENIED:
        NN_RESULT_THROW(ResultCurlErrorLoginDenied());
    case CURLE_CONV_FAILED:
        NN_RESULT_THROW(ResultCurlErrorConvFailed());
    case CURLE_CONV_REQD:
        NN_RESULT_THROW(ResultCurlErrorConvReqd());
    case CURLE_SSL_CACERT_BADFILE:
        NN_RESULT_THROW(ResultCurlErrorSslCaCertBadFile());
    case CURLE_SSL_SHUTDOWN_FAILED:
        NN_RESULT_THROW(ResultCurlErrorSslShutdownFailed());
    case CURLE_AGAIN:
        NN_RESULT_THROW(ResultCurlErrorAgain());
    case CURLE_SSL_CRL_BADFILE:
        NN_RESULT_THROW(ResultCurlErrorSslCrlBadFile());
    case CURLE_SSL_ISSUER_ERROR:
        NN_RESULT_THROW(ResultCurlErrorSslIssuerError());
    case CURLE_CHUNK_FAILED:
        NN_RESULT_THROW(ResultCurlErrorChunkFailed());
    case CURLE_NO_CONNECTION_AVAILABLE:
        NN_RESULT_THROW(ResultCurlErrorNoConnectionAvailable());
    case CURLE_SSL_PINNEDPUBKEYNOTMATCH:
        NN_RESULT_THROW(ResultCurlErrorSslPinnedPubkeyNotMatch());
    case CURLE_SSL_INVALIDCERTSTATUS:
        NN_RESULT_THROW(ResultCurlErrorSslInvalidCertStatus());
    case CURLE_SSL_CTX_FUNCTION_NOT_SET:
        NN_RESULT_THROW(ResultCurlErrorSslCtxFunctionNotSet());
    case CURLE_SSL_CTX_INVALID:
        NN_RESULT_THROW(ResultCurlErrorSslCtxInvalid());
    case CURLE_SSL_CTX_FATAL:
        NN_RESULT_THROW(ResultCurlErrorSslCtxFatal());
    case CURLE_SSL_ALREADY_CONNECTED:
        NN_RESULT_THROW(ResultCurlErrorSslAlreadyConnected());
    case CURLE_SSL_INVALID_REFERENCE:
        NN_RESULT_THROW(ResultCurlErrorSslInvalidReference());

    default:
        NN_RESULT_THROW(ResultFailedCurlPerform());
    }
} // NOLINT(readability/fn_size)

}}} // ~namespace nn::netdiag::detail
