﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <nn/nn_SdkAssert.h>
#include "lm_LogPacketParser.h"

namespace nn { namespace lm { namespace impl {

namespace
{
    const nn::Bit8* ParseUleb128(uint64_t* out, const nn::Bit8* pBegin, const nn::Bit8* pEnd) NN_NOEXCEPT
    {
        auto pCurrent = pBegin;
        uint64_t value = 0;
        const auto valueWidth = sizeof(value) * 8;
        auto base = 0;
        const auto baseUnit = 7;
        while (pCurrent < pEnd && base + baseUnit <= valueWidth)
        {
            value = value | (static_cast<uint64_t>(*pCurrent & 0x7F) << base);
            if ((*pCurrent & 0x80) == 0)
            {
                *out = value;
                return pCurrent;
            }
            pCurrent++;
            base += baseUnit;
        }
        return pEnd; // TORIAEZU: 終端に達した or uint64_t に収まらないときは pEnd を返す。
    }
}

bool LogPacketParser::FindDataChunk(const void** out, size_t* outSize, detail::LogDataChunkKey key, const void* pPacket, size_t packetSize) NN_NOEXCEPT
{
    struct FindDataChunkContext
    {
        const void* pChunk;
        size_t      chunkSize;
        bool        isFound;
        detail::LogDataChunkKey key;
    } context = { nullptr, 0u, false, key };

    LogPacketParser::ParsePacket(pPacket, packetSize,
        [](const detail::LogPacketHeader& header, const void* pPayload, size_t payloadSize, void* argument) -> bool
        {
            auto pContext = reinterpret_cast<FindDataChunkContext*>(argument);
            if (!header.IsHead())
            {
                // 先頭パケット以外は、データチャンクが分割されていて、
                // パースできない可能性があるため、データチャンクのパースは行わずに読み飛ばす。
                return true;
            }
            return LogPacketParser::ParseDataChunk(pPayload, payloadSize,
                [](detail::LogDataChunkKey foundKey, const void* pChunk, size_t chunkSize, void* argument) -> bool
                {
                    auto pContext = reinterpret_cast<FindDataChunkContext*>(argument);
                    if (foundKey == pContext->key)
                    {
                        pContext->pChunk = pChunk;
                        pContext->chunkSize = chunkSize;
                        pContext->isFound = true;
                        return false; // 指定されたキーが見つかったので、false を返して中断する。
                    }
                    return true;
                }, pContext);
        }, &context);
    if (context.isFound)
    {
        *out = context.pChunk;
        *outSize = context.chunkSize;
        return true;
    }
    else
    {
        return false;
    }
}

bool LogPacketParser::ParsePacket(const void* buffer, size_t bufferSize, ParsePacketCallback callback, void* argument) NN_NOEXCEPT
{
    auto pBegin = reinterpret_cast<const nn::Bit8*>(buffer);
    auto pCurrent = pBegin;
    auto pEnd = pBegin + bufferSize;
    auto GetRemainSize = [&]() -> uint64_t { return static_cast<size_t>(pEnd - pCurrent); };

    while (pCurrent < pEnd)
    {
        if (!(GetRemainSize() >= sizeof(detail::LogPacketHeader)))
        {
            NN_SDK_ASSERT(false);
            return false;
        }
        detail::LogPacketHeader header;
        std::memcpy(&header, pCurrent, sizeof(header)); // アンアラインドアクセスを避けるため、コピーしてアクセスする。
        pCurrent += sizeof(header);
        const auto payloadSize = header.GetPayloadSize();
        if (!(GetRemainSize() >= payloadSize))
        {
            NN_SDK_ASSERT(false);
            return false;
        }
        if (!callback(header, pCurrent, payloadSize, argument))
        {
            return false;
        }
        pCurrent += payloadSize;
    }
    NN_SDK_ASSERT_EQUAL(pCurrent, pEnd);

    return true;
}

bool LogPacketParser::ParseDataChunkWithoutSizeCheck(const void* pPayload, size_t payloadSize, ParseDataChunkCallback callback, void* argument) NN_NOEXCEPT
{
    auto pBegin = reinterpret_cast<const nn::Bit8*>(pPayload);
    auto pCurrent = pBegin;
    auto pEnd = pBegin + payloadSize;

    while (pCurrent < pEnd)
    {
        uint64_t key;
        const auto pKeyLast = ParseUleb128(&key, pCurrent, pEnd);
        if (!(pKeyLast < pEnd))
        {
            return false;
        }
        pCurrent = pKeyLast + 1;

        uint64_t size;
        const auto pSizeLast = ParseUleb128(&size, pCurrent, pEnd);
        if (!(pSizeLast < pEnd))
        {
            return false;
        }
        pCurrent = pSizeLast + 1;

        if (!callback(static_cast<detail::LogDataChunkKey>(key), pCurrent, static_cast<size_t>(size), argument))
        {
            return false;
        }

        pCurrent += size;
    }

    return true;
}

bool LogPacketParser::ParseDataChunk(const void* pPayload, size_t payloadSize, ParseDataChunkCallback callback, void* argument) NN_NOEXCEPT
{
    struct ParseDataChunkArgument
    {
        const char* pPayloadEnd;
        ParseDataChunkCallback originalCallback;
        void* originalArgument;
    } parseArg = { static_cast<const char*>(pPayload) + payloadSize, callback, argument };

    return ParseDataChunkWithoutSizeCheck(pPayload, payloadSize, [](detail::LogDataChunkKey key, const void* pChunk, size_t chunkSize, void* argument)
    {
        auto parseArg = reinterpret_cast<ParseDataChunkArgument*>(argument);
        if (static_cast<const char*>(pChunk) + chunkSize <= parseArg->pPayloadEnd)
        {
            return parseArg->originalCallback(key, pChunk, chunkSize, parseArg->originalArgument);
        }
        return true;
    }, &parseArg);
}

void LogPacketParser::ParseTextLogWithContext(const void* pPacket, size_t packetSize, ParseTextLogCallback callback, void* argument) NN_NOEXCEPT
{
    struct PreviousPacketInfo
    {
        uint64_t    processId;
        uint64_t    threadId;
        size_t      carrySize;
        bool        endsWithTextLog;
    };
    NN_FUNCTION_LOCAL_STATIC(PreviousPacketInfo, s_PreviousPacketInfo);

    auto pHeader     = static_cast<const detail::LogPacketHeader*>(pPacket);
    auto pPayload    = static_cast<const char*>(pPacket) + sizeof(detail::LogPacketHeader);
    auto payloadSize = packetSize - sizeof(detail::LogPacketHeader);

    const bool isFollowingPacket =
        !pHeader->IsHead() &&
        pHeader->GetProcessId() == s_PreviousPacketInfo.processId &&
        pHeader->GetThreadId() == s_PreviousPacketInfo.threadId;

    if (!(pHeader->IsHead() || isFollowingPacket))
    {
        return;
    }

    if (isFollowingPacket && s_PreviousPacketInfo.carrySize > 0)
    {
        const size_t outputSize = std::min(s_PreviousPacketInfo.carrySize, payloadSize);
        if (s_PreviousPacketInfo.endsWithTextLog)
        {
            callback(pPayload, outputSize, argument);
        }
        s_PreviousPacketInfo.carrySize -= outputSize;
        pPayload += outputSize;
        payloadSize -= outputSize;
    }

    if (!(payloadSize > 0))
    {
        return;
    }

    struct ParseDataChunkArgument
    {
        size_t                  carrySize;
        bool                    endsWithTextLog;
        const char*             pPayloadEnd;
        ParseTextLogCallback    originalCallback;
        void*                   originalArgument;
    } arg = { 0, false, pPayload + payloadSize, callback, argument };

    ParseDataChunkWithoutSizeCheck(pPayload, payloadSize,
        [](detail::LogDataChunkKey foundKey, const void* pChunk, size_t chunkSize, void* arg) -> bool
        {
            auto& parseArg = *static_cast<ParseDataChunkArgument*>(arg);

            const bool isTextLog = foundKey == detail::LogDataChunkKey_TextLog;
            const size_t restPayloadSize = parseArg.pPayloadEnd - reinterpret_cast<const char*>(pChunk);

            if (!(chunkSize < restPayloadSize)) // 最後のパケットのとき。
            {
                parseArg.carrySize = chunkSize - restPayloadSize;
                parseArg.endsWithTextLog = isTextLog;
            }

            if (isTextLog)
            {
                const size_t outputSize = std::min(chunkSize, restPayloadSize);
                parseArg.originalCallback(static_cast<const char*>(pChunk), outputSize, parseArg.originalArgument);
            }

            return true;
        }, &arg);

    if (!pHeader->IsTail())
    {
        s_PreviousPacketInfo.processId = pHeader->GetProcessId();
        s_PreviousPacketInfo.threadId  = pHeader->GetThreadId();
        s_PreviousPacketInfo.carrySize = arg.carrySize;
        s_PreviousPacketInfo.endsWithTextLog = arg.endsWithTextLog;
    }
}

size_t LogPacketParser::ParseModuleName(char* buffer, size_t bufferSize, const void* pPacket, size_t packetSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0u);
    NN_SDK_REQUIRES_NOT_NULL(pPacket);
    NN_SDK_REQUIRES_GREATER(packetSize, 0u);

    const void* pChunk;
    size_t chunkSize;
    const bool isFound = LogPacketParser::FindDataChunk(
        &pChunk, &chunkSize, detail::LogDataChunkKey_ModuleName, pPacket, packetSize);
    if (!(isFound && chunkSize > 0))
    {
        buffer[0] = '\0';
        return 0;
    }

    const size_t copySize = std::min(chunkSize, bufferSize - 1);
    std::memcpy(buffer, pChunk, copySize);
    buffer[copySize] = '\0';

    return chunkSize;
}

}}} // nn::lm::impl
