﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_Mutex.h>

namespace nn { namespace lm { namespace impl {

/*!
    @brief  イベントログの送信器です。
*/
class EventLogTransmitter
{
private:
    NN_DISALLOW_COPY( EventLogTransmitter );
    NN_DISALLOW_MOVE( EventLogTransmitter );

public:
    /*!
        @brief      フラッシュ関数です。

        @param[in]  message     メッセージのポインタ。
        @param[in]  messageSize メッセージのバイトサイズ。
    */
    typedef bool (*FlushFunction)(const uint8_t* message, size_t messageSize);

public:
    /*!
        @brief      コンストラクタです。

        @param[in]  function    フラッシュ関数。
    */
    explicit EventLogTransmitter(FlushFunction function) NN_NOEXCEPT;

public:
    /*!
        @brief      既定のインスタンスを取得します。

        @return     インスタンス。

        @return     成功したとき true、失敗したとき false。
    */
    static EventLogTransmitter& GetDefaultInstance() NN_NOEXCEPT;

public:
    /*!
        @brief      プロセスのログセッション開始を通知するログを送信します。

        @param[in]  processId   プロセス ID。

        @return     成功したとき true、失敗したとき false。
    */
    bool PushLogSessionBegin(uint64_t processId) NN_NOEXCEPT;

    /*!
        @brief      プロセスのログセッション終了を通知するログを送信します。

        @param[in]  processId   プロセス ID。
    */
    bool PushLogSessionEnd(uint64_t processId) NN_NOEXCEPT;

    /*!
        @brief      ログパケットのドロップ数を送信します。
    */
    bool PushLogPacketDropCountIfExists() NN_NOEXCEPT;

    /*!
        @brief      ログパケットのドロップ数をひとつ増やします。
    */
    void IncreaseLogPacketDropCount() NN_NOEXCEPT;

private:
    FlushFunction m_FlushFunction;
    //
    size_t m_LogPacketDropCount;
    //
    nn::os::Mutex m_LogPacketDropCountMutex;
};

}}} // nn::lm::impl
