﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <type_traits>
#include <new>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/psc.h>

#include <nn/lbl/server/lbl_LblHipcServer.h>

#include "../detail/lbl_ServiceType.h"
#include <nn/lbl/detail/lbl_ILblController.h>
#include <nn/lbl/detail/lbl_Log.h>
#include <nn/lbl/impl/lbl.h>
#include <nn/lbl/impl/lbl_Suspend.h>

#include "lbl_AutoControlThread.h"
#include "lbl_LblControllerImpl.h"
#include "lbl_CreateLblControllerByDfc.h"

namespace nn { namespace lbl { namespace server {

namespace {


// HipcSimpleAllInOneServerManager は HIPC のサーバのポートとセッションを一元管理する。
class MyServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<30, 1>
{};

// 繰り返しのサーバの起動と終了が可能となるように placement new で初期化を行う。
// 繰り返しの起動と終了が必要ない場合には MyServerManager は直接配置してもよい。
std::aligned_storage<sizeof(MyServerManager), NN_ALIGNOF(MyServerManager)>::type g_MyServerManagerStorage;
MyServerManager* g_pMyServerManager;

// スリープ等の通知を受け取るオブジェクト
nn::psc::PmModule g_PmModule;

}

nn::Result HandlePmEvent(nn::psc::PmState state, nn::psc::PmFlagSet flags)
{
    NN_UNUSED(flags);
    switch (state)
    {
    case nn::psc::PmState_FullAwake:
    case nn::psc::PmState_MinimumAwake:
        NN_DETAIL_LBL_TRACE("Resume\n");
        nn::lbl::impl::Resume();
        nn::lbl::server::StartAutoControlThread();
        break;

    case nn::psc::PmState_SleepReady:
        NN_DETAIL_LBL_TRACE("Suspend\n");
        nn::lbl::server::StopAutoControlThread();
        nn::lbl::impl::Suspend();
        break;

    default:
        break;
    }
    NN_RESULT_SUCCESS;
}

void LoopServer(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    nn::os::MultiWaitHolderType pmEventHolder;
    nn::os::InitializeMultiWaitHolder(
        &pmEventHolder,
        g_PmModule.GetEventPointer()->GetBase());
    nn::os::SetMultiWaitHolderUserData(
        &pmEventHolder,
        nn::psc::PmModuleId_Lbl); // PmModuleId は SF のタグと衝突しないことを保証している

    g_pMyServerManager->AddUserWaitHolder(&pmEventHolder);

    while (NN_STATIC_CONDITION(true))
    {
        nn::os::MultiWaitHolderType* p = g_pMyServerManager->Wait();

        switch (nn::os::GetMultiWaitHolderUserData(p))
        {
            // SF processing
        case MyServerManager::HipcSimpleAllInOneServerManager::AcceptTag:
        case MyServerManager::HipcSimpleAllInOneServerManager::InvokeTag:
            g_pMyServerManager->ProcessAuto(p);
            break;

        case nn::psc::PmModuleId_Lbl:
        {
            bool tryWaitResult = g_PmModule.GetEventPointer()->TryWait();
            NN_SDK_ASSERT(tryWaitResult);
            NN_UNUSED(tryWaitResult);
            g_PmModule.GetEventPointer()->Clear();
            g_pMyServerManager->AddUserWaitHolder(&pmEventHolder);

            nn::psc::PmState    state;
            nn::psc::PmFlagSet  flags;
            auto result = g_PmModule.GetRequest(&state, &flags);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            auto pmEventResult = HandlePmEvent(state, flags);

            g_PmModule.Acknowledge(state, pmEventResult);
        }
        break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

void StartServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pMyServerManager);

    // MyServerManager オブジェクトのコンストラクト
    g_pMyServerManager = new (&g_MyServerManagerStorage) MyServerManager;

    // サービス名の登録とポートの初期化
    // sessionCountMax は、現時点では十分に大きな値を指定しておけばよい。
    // 現在はどのプロセスからアクセスしても 1つのマネージャーにアクセスされる
    auto sessionCountMax = 30;
    g_pMyServerManager->RegisterObjectForPort(CreateLblControllerByDfc(), sessionCountMax, detail::LblServiceName);

    // 自動輝度調整スレッドの開始
    nn::lbl::server::StartAutoControlThread();

    nn::psc::PmModuleId dependencies[] = { nn::psc::PmModuleId_Gpio, nn::psc::PmModuleId_I2c, nn::psc::PmModuleId_Pwm };

    // PSC へのモジュール登録
    auto result = g_PmModule.Initialize(
        nn::psc::PmModuleId_Lbl,
        dependencies,
        sizeof(dependencies) / sizeof(dependencies[0]),
        nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // サーバマネージャの開始
    // ただし、実際のサーバ動作は、LoopAuto 関数等を呼び出すことで行う必要がある。
    g_pMyServerManager->Start();
}


void RequestStopServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    // 処理ループの停止リクエストを送り、LoopAuto 関数を返す
    g_pMyServerManager->RequestStop();
}

void FinalizeServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    // 自動輝度調整スレッドを終了
    nn::lbl::server::StopAutoControlThread();

    // MyServerManager のデストラクト
    // 登録したサービスなどはここで登録が解除される。
    g_pMyServerManager->~MyServerManager();
    g_pMyServerManager = nullptr;
}

}}}
