﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_IntUtil.h>

#include <nn/jit/jit_CommonTypes.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Result.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/svc/svc_Dd.h>

namespace nn { namespace jitsrv { namespace detail {

class AslrAllocator
{
public:
    NN_IMPLICIT AslrAllocator(nn::svc::Handle process) NN_NOEXCEPT;
    uint64_t GetAslrRegion(uint64_t mappingSize, int index = 0) NN_NOEXCEPT;
private:
    uint64_t m_HeapBegin;
    uint64_t m_HeapSize;
    uint64_t m_RsvBegin;
    uint64_t m_RsvSize;
    uint64_t m_AslrBegin;
    uint64_t m_AslrSize;
};

void StoreDataCacheAndMemoryBarrier(uintptr_t addr, size_t size) NN_NOEXCEPT;

template <typename Memory>
class MemoryHolder
    : private Memory
{
private:

    detail::AslrAllocator* m_pAddressAllocator = nullptr;
    jit::MemorySecurityMode m_SecurityMode = jit::MemorySecurityMode::MemorySecurityMode_Default;
    uintptr_t m_Address = 0;
    bool m_Started = false;
    bool m_Accessing = false;

    bool MapImpl(uintptr_t address) NN_NOEXCEPT
    {
        auto size = GetBody().GetSize();
        if (size == 0)
        {
            return 0;
        }
        bool success;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetBody().Map(&success, address));
        return success;
    }

    uintptr_t MapImplWithAslr(int index) NN_NOEXCEPT
    {
        auto size = GetBody().GetSize();
        for (;;)
        {
            auto address = m_pAddressAllocator->GetAslrRegion(size, index);
            if (MapImpl(address))
            {
                return static_cast<uintptr_t>(address);
            }
        }
    }

    void UnmapImpl(uintptr_t address) NN_NOEXCEPT
    {
        auto size = GetBody().GetSize();
        if (size == 0)
        {
            return;
        }
        GetBody().Unmap(address);
    }

public:

    Memory& GetBody() NN_NOEXCEPT
    {
        return *this;
    }

    void Initialize(detail::AslrAllocator* pAddressAllocator) NN_NOEXCEPT
    {
        this->m_pAddressAllocator = pAddressAllocator;
        this->m_Address = MapImplWithAslr(1);
    }

    ~MemoryHolder() NN_NOEXCEPT
    {
        NN_UNUSED(m_Accessing);
        NN_SDK_REQUIRES(!m_Accessing);
        if (m_Started)
        {
            using namespace jit;
            switch (m_SecurityMode)
            {
                case MemorySecurityMode_Default:
                case MemorySecurityMode_NoAslr:
                {
                    // nop
                    return;
                }
                case MemorySecurityMode_AlwaysMapped:
                {
                    UnmapImpl(m_Address);
                    return;
                }
                default: NN_UNEXPECTED_DEFAULT;
            }
        }
        else
        {
            UnmapImpl(m_Address);
        }
    }

    void Start(jit::MemorySecurityMode securityMode) NN_NOEXCEPT
    {
        this->m_SecurityMode = securityMode;
        this->m_Started = true;
        using namespace jit;
        switch (m_SecurityMode)
        {
            case MemorySecurityMode_Default:
            {
                UnmapImpl(m_Address);
                this->m_Address = 0;
                return;
            }
            case MemorySecurityMode_AlwaysMapped:
            {
                // nop
                return;
            }
            case MemorySecurityMode_NoAslr:
            {
                UnmapImpl(m_Address);
                return;
            }
            default: NN_UNEXPECTED_DEFAULT;
        }
    }

    void BeginAccess() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_Accessing);
        using namespace jit;
        switch (m_SecurityMode)
        {
            case MemorySecurityMode_Default:
            {
                this->m_Address = MapImplWithAslr(0);
                return;
            }
            case MemorySecurityMode_AlwaysMapped:
            {
                // nop
                return;
            }
            case MemorySecurityMode_NoAslr:
            {
                MapImpl(m_Address);
                return;
            }
            default: NN_UNEXPECTED_DEFAULT;
        }
    }

    void* GetAddress() NN_NOEXCEPT
    {
        return reinterpret_cast<void*>(m_Address);
    }

    void StoreDataCache(const jit::CodeRange& range) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(range.IsValid(GetBody().GetSize()));
        StoreDataCacheAndMemoryBarrier(m_Address + range.offset, range.size);
    }

    void EndAccess() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_Accessing);
        using namespace jit;
        switch (m_SecurityMode)
        {
            case MemorySecurityMode_Default:
            {
                UnmapImpl(m_Address);
                this->m_Address = 0;
                return;
            }
            case MemorySecurityMode_AlwaysMapped:
            {
                // nop
                return;
            }
            case MemorySecurityMode_NoAslr:
            {
                UnmapImpl(m_Address);
                return;
            }
            default: NN_UNEXPECTED_DEFAULT;
        }
    }

};

class CodeMemory
{
private:

    sf::NativeHandle m_SfHandle;
    uint64_t m_Size = 0;
    uint64_t m_OwnerAddress = 0;
    svc::MemoryPermission m_OwnerPermission = svc::MemoryPermission_None;

public:

    Result Initialize(detail::AslrAllocator* pOwnerAslrAllocator, sf::NativeHandle sfHandle, uint64_t size, svc::MemoryPermission ownerPermission) NN_NOEXCEPT
    {
        if (size == 0)
        {
            NN_RESULT_SUCCESS;
        }
        svc::Handle handle{sfHandle.GetOsHandle()};
    retry:
        uint64_t ownerAddress = pOwnerAslrAllocator->GetAslrRegion(size);
        NN_RESULT_TRY(svc::ControlCodeMemory(handle, svc::CodeMemoryOperation_MapToOwner, ownerAddress, size, ownerPermission))
            NN_RESULT_CATCH(svc::ResultInvalidCurrentMemory)
            {
                goto retry;
            }
            NN_RESULT_CATCH(nn::svc::ResultInvalidRegion)
            {
                // TORIAEZU: svc::ControlCodeMemory(svc::CodeMemoryOperation_MapToOwner) のチェックバグ回避
                goto retry;
            }
        NN_RESULT_END_TRY
        this->m_SfHandle = std::move(sfHandle);
        this->m_Size = size;
        this->m_OwnerAddress = ownerAddress;
        this->m_OwnerPermission = ownerPermission;
        NN_RESULT_SUCCESS;
    }

    ~CodeMemory() NN_NOEXCEPT
    {
        if (m_Size)
        {
            svc::Handle handle{m_SfHandle.GetOsHandle()};
            NN_ABORT_UNLESS_RESULT_SUCCESS(svc::ControlCodeMemory(handle, svc::CodeMemoryOperation_UnmapFromOwner, m_OwnerAddress, m_Size, svc::MemoryPermission_None));
        }
    }

    void swap(CodeMemory& other) NN_NOEXCEPT
    {
        using std::swap;
        m_SfHandle.swap(other.m_SfHandle);
        swap(m_Size, other.m_Size);
        swap(m_OwnerAddress, other.m_OwnerAddress);
        swap(m_OwnerPermission, other.m_OwnerPermission);
    }

    uint64_t GetOwnerAddress() const NN_NOEXCEPT
    {
        return m_OwnerAddress;
    }

    size_t GetSize() const NN_NOEXCEPT
    {
        return m_Size;
    }

    Result Map(bool* pSuccess, uintptr_t address) NN_NOEXCEPT
    {
        svc::Handle handle{m_SfHandle.GetOsHandle()};
        NN_RESULT_TRY(svc::ControlCodeMemory(handle, svc::CodeMemoryOperation_Map, address, m_Size, svc::MemoryPermission_ReadWrite))
            NN_RESULT_CATCH(svc::ResultInvalidCurrentMemory)
            {
                *pSuccess = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY
        *pSuccess = true;
        NN_RESULT_SUCCESS;
    }

    void Unmap(uintptr_t address) NN_NOEXCEPT
    {
        svc::Handle handle{m_SfHandle.GetOsHandle()};
        NN_ABORT_UNLESS_RESULT_SUCCESS(svc::ControlCodeMemory(handle, svc::CodeMemoryOperation_Unmap, address, m_Size, svc::MemoryPermission_None));
    }

};

class WorkMemory
{
private:

    sf::NativeHandle m_SfHandle;
    size_t m_Size = 0;

    static Result ClearMemory(const sf::NativeHandle& sfHandle, size_t size) NN_NOEXCEPT
    {
        if (size > 0)
        {
            os::TransferMemory transferMemory{size, sfHandle.GetOsHandle(), false};
            void* p;
            NN_RESULT_DO(transferMemory.Map(&p, os::MemoryPermission_None));
            std::memset(p, 0, size);
            StoreDataCacheAndMemoryBarrier(reinterpret_cast<uintptr_t>(p), size);
            transferMemory.Unmap();
        }
        NN_RESULT_SUCCESS;
    }

    static Result MapImpl(void** pOut, detail::AslrAllocator* pAslrAllocator, const sf::NativeHandle& sfHandle, size_t size) NN_NOEXCEPT
    {
        if (size == 0)
        {
            NN_RESULT_SUCCESS;
        }
        svc::Handle handle{sfHandle.GetOsHandle()};
    retry:
        uintptr_t address = static_cast<uintptr_t>(pAslrAllocator->GetAslrRegion(size));
        NN_RESULT_TRY(svc::MapTransferMemory(handle, address, size, svc::MemoryPermission_None))
            NN_RESULT_CATCH(svc::ResultInvalidCurrentMemory)
            {
                goto retry;
            }
        NN_RESULT_END_TRY
        *pOut = reinterpret_cast<void*>(address);
        NN_RESULT_SUCCESS;
    }

    static void UnmapImpl(const sf::NativeHandle& sfHandle, void* address, size_t size) NN_NOEXCEPT
    {
        if (size == 0)
        {
            return;
        }
        svc::Handle handle{sfHandle.GetOsHandle()};
        NN_ABORT_UNLESS_RESULT_SUCCESS(svc::UnmapTransferMemory(handle, reinterpret_cast<uintptr_t>(address), size));
    }

public:

    Result Initialize(sf::NativeHandle sfHandle, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_Size == 0);
        NN_RESULT_DO(ClearMemory(sfHandle, size));
        this->m_SfHandle = std::move(sfHandle);
        this->m_Size = size;
        NN_RESULT_SUCCESS;
    }

    void Finalize() NN_NOEXCEPT
    {
        if (m_Size)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(ClearMemory(m_SfHandle, m_Size));
            m_SfHandle.Reset();
        }
        this->m_Size = 0;
    }

    ~WorkMemory() NN_NOEXCEPT
    {
        Finalize();
    }

    size_t GetSize() const NN_NOEXCEPT
    {
        return m_Size;
    }

    Result Map(bool* pSuccess, uintptr_t address) NN_NOEXCEPT
    {
        svc::Handle handle{m_SfHandle.GetOsHandle()};
        NN_RESULT_TRY(svc::MapTransferMemory(handle, address, m_Size, svc::MemoryPermission_None))
            NN_RESULT_CATCH(svc::ResultInvalidCurrentMemory)
            {
                *pSuccess = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY
        *pSuccess = true;
        NN_RESULT_SUCCESS;
    }

    void Unmap(uintptr_t address) NN_NOEXCEPT
    {
        svc::Handle handle{m_SfHandle.GetOsHandle()};
        NN_ABORT_UNLESS_RESULT_SUCCESS(svc::UnmapTransferMemory(handle, address, m_Size));
    }

    void swap(WorkMemory& other) NN_NOEXCEPT
    {
        using std::swap;
        m_SfHandle.swap(other.m_SfHandle);
        swap(m_Size, other.m_Size);
    }

};

}}}
