﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/am/service/am_DisplayLayerControl.h>
#include <nn/lbl/lbl.h>
#include <nn/lbl/lbl_Switch.h>
#include <nn/idle/detail/idle_Log.h>
#include "idle_HandlerImplDimmer.h"
#include "idle_IHandler.h"

namespace nn { namespace idle { namespace server {

namespace {

    // 暗くする度合
    float g_dimLevelLcd = -.1f;
    float g_dimLevelTv = -.7f;

    void LoadDimLevelSettings() NN_NOEXCEPT
    {
        int dimLevelPercentLcd = 10;
        int dimLevelPercentTv = 70;
        settings::fwdbg::GetSettingsItemValue(&dimLevelPercentLcd, sizeof(dimLevelPercentLcd), "idle", "dim_level_percent_lcd");
        settings::fwdbg::GetSettingsItemValue(&dimLevelPercentTv, sizeof(dimLevelPercentTv), "idle", "dim_level_percent_tv");

        NN_DETAIL_IDLE_INFO("Firmware debug settings: dimLevelPercentLcd=%d [%%], dimLevelPercentTv=%d [%%]\n", dimLevelPercentLcd, dimLevelPercentTv);
        g_dimLevelLcd = -(static_cast<float>(dimLevelPercentLcd) / 100.0f);
        g_dimLevelTv = -(static_cast<float>(dimLevelPercentTv) / 100.0f);
    }

    // 無効化する場合は 0 を返す
    nn::TimeSpan GetTimeToTriggerBody(const PolicyParam& policyParam)
    {
        // fwdbg の指定が全てをオーバーライド
        if ( policyParam.isAutoSleepDisabledForDebug )
        {
            // デバッグ目的で自動スリープ無効時には低輝度化も一緒に無効にする
            return 0;
        }
        if ( policyParam.overrideAutoSleepTimeForDebugInSeconds > 0 )
        {
            // TORIAEZU: 自動スリープ発動 5 秒前に暗くする
            return nn::TimeSpan::FromSeconds(
                (policyParam.overrideAutoSleepTimeForDebugInSeconds > 5) ?
                (policyParam.overrideAutoSleepTimeForDebugInSeconds - 5) : 1
            );
        }

        // コンテキスト依存で無効化されていれば優先
        if ( policyParam.handlingContext.isDimmingDisabled )
        {
            return 0;
        }

        // 「メディア再生中に自動スリープする」設定がオフで、今メディア再生中なら、暗くしない
        if ( !policyParam.isAutoSleepInMediaPlaybackEnabled && policyParam.handlingContext.isInMediaPlayback )
        {
            return 0;
        }

        switch (policyParam.handlingContext.operationMode)
        {
            case nn::omm::OperationMode::Handheld:
            {
                // コンテキスト依存の一時的な設定上書きがあれば優先
                if ( policyParam.handlingContext.overrideDimmingTimeInHandheldInSeconds > 0 )
                {
                    return nn::TimeSpan::FromSeconds(policyParam.handlingContext.overrideDimmingTimeInHandheldInSeconds);
                }

                if ( !policyParam.handlingContext.isInMediaPlayback )
                {
                    // 通常時ポリシーを適用
                    switch ( policyParam.handheldSleepPlan )
                    {
                        case nn::settings::system::HandheldSleepPlan_1Min: return nn::TimeSpan::FromSeconds(45);
                        case nn::settings::system::HandheldSleepPlan_3Min: return nn::TimeSpan::FromMinutes(2);
                        case nn::settings::system::HandheldSleepPlan_5Min: return nn::TimeSpan::FromMinutes(4);
                        case nn::settings::system::HandheldSleepPlan_10Min: return nn::TimeSpan::FromMinutes(5);
                        case nn::settings::system::HandheldSleepPlan_30Min: return nn::TimeSpan::FromMinutes(5);
                        case nn::settings::system::HandheldSleepPlan_Never: return nn::TimeSpan::FromMinutes(5);
                    } // NOLINT(style/switch_default)
                }
                else
                {
                    // メディア再生中ポリシー
                    switch ( policyParam.handheldSleepPlan )
                    {
                        case nn::settings::system::HandheldSleepPlan_1Min: return 0; // 無効化
                        case nn::settings::system::HandheldSleepPlan_3Min: return 0; // 無効化
                        case nn::settings::system::HandheldSleepPlan_5Min: return 0; // 無効化
                        case nn::settings::system::HandheldSleepPlan_10Min: return 0; // 無効化
                        case nn::settings::system::HandheldSleepPlan_30Min: return 0; // 無効化
                        case nn::settings::system::HandheldSleepPlan_Never: return 0; // 無効化
                    } // NOLINT(style/switch_default)
                }
            }
            case nn::omm::OperationMode::Console:
            {
                // コンテキスト依存の一時的な設定上書きがあれば優先
                if ( policyParam.handlingContext.overrideDimmingTimeInConsoleInSeconds > 0 )
                {
                    return nn::TimeSpan::FromSeconds(policyParam.handlingContext.overrideDimmingTimeInConsoleInSeconds);
                }

                // TV 再生中の画面焼き付き防止は、本体設定によりオフできる
                if ( !policyParam.isTvDimmingEnabled )
                {
                    return 0;
                }

                if ( !policyParam.handlingContext.isInMediaPlayback )
                {
                    // 通常時ポリシーを適用
                    switch ( policyParam.consoleSleepPlan )
                    {
                        case nn::settings::system::ConsoleSleepPlan_1Hour: return nn::TimeSpan::FromMinutes(5);
                        case nn::settings::system::ConsoleSleepPlan_2Hour: return nn::TimeSpan::FromMinutes(5);
                        case nn::settings::system::ConsoleSleepPlan_3Hour: return nn::TimeSpan::FromMinutes(5);
                        case nn::settings::system::ConsoleSleepPlan_6Hour: return nn::TimeSpan::FromMinutes(5);
                        case nn::settings::system::ConsoleSleepPlan_12Hour: return nn::TimeSpan::FromMinutes(5);
                        case nn::settings::system::ConsoleSleepPlan_Never: return nn::TimeSpan::FromMinutes(5);
                    } // NOLINT(style/switch_default)
                }
                else
                {
                    // メディア再生中ポリシー
                    switch ( policyParam.consoleSleepPlan )
                    {
                        case nn::settings::system::ConsoleSleepPlan_1Hour: return 0; // 無効化
                        case nn::settings::system::ConsoleSleepPlan_2Hour: return 0; // 無効化
                        case nn::settings::system::ConsoleSleepPlan_3Hour: return 0; // 無効化
                        case nn::settings::system::ConsoleSleepPlan_6Hour: return 0; // 無効化
                        case nn::settings::system::ConsoleSleepPlan_12Hour: return 0; // 無効化
                        case nn::settings::system::ConsoleSleepPlan_Never: return 0; // 無効化
                    } // NOLINT(style/switch_default)
                }
            }
        } // NOLINT(style/switch_default)

        NN_SDK_ASSERT(false, "NEVER REACHED HERE");
        return 0;
    }

    void SetCmaOnDisplay(float targetLumaLcd, float targetLumaTv) NN_NOEXCEPT
    {
        // 動作モードに必要以上に依存させないため、現在の出力先がどちらでも無関係に全体に低輝度化をかける

        nn::Result result;

        // Tv
        result = nn::am::service::SetDisplayCmuLuma("External", targetLumaTv);
        if ( result.IsFailure() )
        {
            NN_DETAIL_IDLE_WARN("SetDisplayCmuLuma(Tv) returned: Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
        }

        // Lcd
        result = nn::am::service::SetDisplayCmuLuma("Internal", targetLumaLcd);
        if ( result.IsFailure() )
        {
            NN_DETAIL_IDLE_WARN("SetDisplayCmuLuma(Lcd) returned: Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
        }
    }

    nn::TimeSpan DelayTriggerTimeIfRequested(const nn::TimeSpan& newTime, const PolicyParam& policyParam)
    {
        switch ( policyParam.handlingContext.idleTimeDetectionDelayMode )
        {
            case IdleTimeDetectionExtension_Enabled:
            {
                const auto MaxDelayTime = nn::TimeSpan::FromMinutes(15);
                if ( 0 < newTime && newTime < MaxDelayTime )
                {
                    return MaxDelayTime;
                }
                break;
            }
            case IdleTimeDetectionExtension_EnabledUnsafe:
            {
                // 無効化
                return 0;
            }
            default:
                break;
        }
        return newTime;
    }
}

void HandlerImplDimmer::Initialize() NN_NOEXCEPT
{
    LoadDimLevelSettings();
    nn::lbl::Initialize();
}

// 無効化する場合は 0 を返す
nn::TimeSpan HandlerImplDimmer::GetTimeToTrigger(const PolicyParam& policyParam) NN_NOEXCEPT
{
    auto newTime = GetTimeToTriggerBody(policyParam);
    newTime = DelayTriggerTimeIfRequested(newTime, policyParam);
    NN_DETAIL_IDLE_INFO("Time to dim: %d sec\n", newTime.GetSeconds());
    return newTime;
}

void HandlerImplDimmer::HandleActiveState() NN_NOEXCEPT
{
    NN_DETAIL_IDLE_INFO("Dimming deactivated\n");

    SetCmaOnDisplay(.0f, .0f); // 0 で明るさが戻る

    nn::lbl::DisableDimming();
}

void HandlerImplDimmer::HandleIdleState() NN_NOEXCEPT
{
    NN_DETAIL_IDLE_INFO("Dimming activated\n");

    SetCmaOnDisplay(g_dimLevelLcd, g_dimLevelTv);

    nn::lbl::EnableDimming();
}

}}} // namespace nn::idle::server

