﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_DeviceCode.h>

#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/i2c/driver/i2c.h>
#include <nn/i2c/driver/i2c_BusDev.h>

#include <nn/i2c/i2c_IManager.sfdl.h>
#include <nn/i2c/i2c_ISession.sfdl.h>
#include <nn/i2c/i2c_ServiceType.h>

#include <nn/sf/impl/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/i2c/server/i2c_ManagerImpl.h>

namespace nn { namespace i2c { namespace server {

// ISession 実装クラス
class SessionImpl
{
public:
    // コンストラクタで親の SharedPointer を設定し、セッションに必要な情報を渡す
    // [Gen1] TODO: Deprecate
    SessionImpl(ManagerImpl* pParent, nn::i2c::driver::I2cSession internalSession) NN_NOEXCEPT
        : m_Parent(pParent, true)
        , m_InternalSession(internalSession)
    {}

    // [Gen2]
    explicit SessionImpl(ManagerImpl* pParent) NN_NOEXCEPT
        : m_Parent(pParent, true)
    {
        // Gen2 OpenSession は Result を返すのでメソッドを別に実装し、ここでは呼ばない‘
    }

    // Session のデストラクタで Session のクローズを行う
    ~SessionImpl()
    {
        nn::i2c::driver::CloseSession(m_InternalSession);
    }

    nn::Result OpenSession(nn::DeviceCode deviceCode) NN_NOEXCEPT;
    nn::Result Send(const nn::sf::InBuffer& inData, std::int32_t inOption) NN_NOEXCEPT;
    nn::Result Receive(const nn::sf::OutBuffer& outData, std::int32_t inOption) NN_NOEXCEPT;
    nn::Result ExecuteCommandList(const nn::sf::OutBuffer& receiveBuffer, const nn::sf::InArray<nn::i2c::I2cCommand>& commandList) NN_NOEXCEPT;
    nn::Result SetRetryPolicy(std::int32_t maxRetryCount, std::int32_t retryIntervalMicroSeconds) NN_NOEXCEPT;

    // 互換性維持のための API 定義
    nn::Result SendOld(const nn::sf::InBuffer& inData, std::int32_t inOption) NN_NOEXCEPT;
    nn::Result ReceiveOld(const nn::sf::OutBuffer& outData, std::int32_t inOption) NN_NOEXCEPT;
    nn::Result ExecuteCommandListOld(const nn::sf::OutBuffer& receiveBuffer, const nn::sf::InArray<nn::i2c::I2cCommand>& commandList) NN_NOEXCEPT;

private:

    // 親への SharedPointer
    nn::sf::SharedPointer<ManagerImpl> m_Parent;

    // I2C ドライバライブラリが持つ I2cSession 構造体
    nn::i2c::driver::I2cSession m_InternalSession;
};

// [GEN2]
nn::Result SessionImpl::OpenSession(nn::DeviceCode deviceCode) NN_NOEXCEPT
{
    return nn::i2c::driver::OpenSession(&m_InternalSession, deviceCode);
}

nn::Result SessionImpl::Send(const nn::sf::InBuffer& inData,
                                                std::int32_t inOption) NN_NOEXCEPT
{
    return nn::i2c::driver::Send(m_InternalSession,
                         inData.GetPointerUnsafe(),
                         inData.GetSize(),
                         static_cast<nn::i2c::TransactionOption>(inOption));
}

nn::Result SessionImpl::Receive(const nn::sf::OutBuffer& outData,
                                                   std::int32_t inOption) NN_NOEXCEPT
{
    return nn::i2c::driver::Receive(outData.GetPointerUnsafe(),
                            m_InternalSession,
                            outData.GetSize(),
                            static_cast<nn::i2c::TransactionOption>(inOption));
}

nn::Result SessionImpl::ExecuteCommandList(const nn::sf::OutBuffer& receiveBuffer,
                                                              const nn::sf::InArray<nn::i2c::I2cCommand>& commandList) NN_NOEXCEPT
{
    return nn::i2c::driver::ExecuteCommandList(receiveBuffer.GetPointerUnsafe(),
                                               receiveBuffer.GetSize(),
                                               m_InternalSession,
                                               commandList.GetData(),
                                               commandList.GetLength());
}

nn::Result SessionImpl::SetRetryPolicy(std::int32_t maxRetryCount, std::int32_t retryIntervalMicroSeconds) NN_NOEXCEPT
{
    return nn::i2c::driver::SetRetryPolicy(m_InternalSession, maxRetryCount, retryIntervalMicroSeconds);
}

// 互換性維持のための API 定義
nn::Result SessionImpl::SendOld(const nn::sf::InBuffer& inData,
    std::int32_t inOption) NN_NOEXCEPT
{
    return Send(inData, inOption);
}

nn::Result SessionImpl::ReceiveOld(const nn::sf::OutBuffer& outData,
    std::int32_t inOption) NN_NOEXCEPT
{
    return Receive(outData, inOption);
}

nn::Result SessionImpl::ExecuteCommandListOld(const nn::sf::OutBuffer& receiveBuffer,
    const nn::sf::InArray<nn::i2c::I2cCommand>& commandList) NN_NOEXCEPT
{
    return ExecuteCommandList(receiveBuffer, commandList);
}

// マネージャーの実装
ManagerImpl::ManagerImpl() NN_NOEXCEPT
{
    // コンストラクタ内で拡張ヒープを初期化
    m_HeapHandle = nn::lmem::CreateExpHeap(&m_HeapBuffer, sizeof(m_HeapBuffer), nn::lmem::CreationOption_NoOption);
    // アロケータにヒープハンドルをアタッチ
    m_Allocator.Attach(m_HeapHandle);
}

ManagerImpl::~ManagerImpl() NN_NOEXCEPT
{
    nn::lmem::DestroyExpHeap(m_HeapHandle);
}

nn::Result ManagerImpl::OpenSession2(nn::sf::Out<nn::sf::SharedPointer<nn::i2c::ISession>> outSession, nn::detail::DeviceCodeType name) NN_NOEXCEPT
{
    // ObjectFactory の CreateSharedEmpleced で Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返す
    // GetImpl したいので、Factory の戻りを auto ではなく EmplacedRef で受けておく
    typedef nn::sf::ObjectFactory<MyAllocator::Policy> Factory;
    nn::sf::EmplacedRef<ISession, SessionImpl> p = Factory::CreateSharedEmplaced<ISession, SessionImpl>(&m_Allocator, this);
    NN_ABORT_UNLESS(p != nullptr, "[I2c Server] Create session failed. I2c server can not create session any more\n");

    // 失敗時は p をさらに move しないまま破棄して関数が返るため、自動的にリソースは解放される
    NN_RESULT_DO(p.GetImpl().OpenSession(name));

    // std::move で outSession に生成したオブジェクトの共有ポインタを渡す
    *outSession = std::move(p);
    NN_RESULT_SUCCESS;
}

nn::Result ManagerImpl::OpenSessionForDev(nn::sf::Out<nn::sf::SharedPointer<nn::i2c::ISession>> outSession,
                                             std::int32_t busIdx, nn::Bit16 slaveAddress, std::int32_t addressingMode, std::int32_t speedMode) NN_NOEXCEPT
{
    // I2C ドライバライブラリに渡す I2cSession 構造体
    nn::i2c::driver::I2cSession internalSession;

    nn::i2c::driver::OpenSessionForDev(&internalSession,
                               static_cast<int>(busIdx),
                               slaveAddress,
                               static_cast<nn::i2c::AddressingMode>(addressingMode),
                               static_cast<nn::i2c::SpeedMode>(speedMode));


    // ObjectFactory の CreateSharedEmpleced で Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返す
    typedef nn::sf::ObjectFactory<MyAllocator::Policy>  Factory;
    auto p = Factory::CreateSharedEmplaced<ISession, SessionImpl>(&m_Allocator, this, internalSession);

    // std::move で outSession に生成したオブジェクトの共有ポインタを渡す
    *outSession = std::move(p);
    return nn::ResultSuccess();
}

nn::Result ManagerImpl::OpenSession(nn::sf::Out<nn::sf::SharedPointer<nn::i2c::ISession>> outSession, std::int32_t device) NN_NOEXCEPT
{
    // I2C ドライバライブラリに渡す I2cSession 構造体
    nn::i2c::driver::I2cSession internalSession;

    nn::i2c::driver::OpenSession(&internalSession, static_cast<nn::i2c::I2cDevice>(device));

    // ObjectFactory の CreateSharedEmpleced で Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返す
    typedef nn::sf::ObjectFactory<MyAllocator::Policy>  Factory;
    auto p = Factory::CreateSharedEmplaced<ISession, SessionImpl>(&m_Allocator, this, internalSession);

    // std::move で outSession に生成したオブジェクトの共有ポインタを渡す
    *outSession = std::move(p);
    return nn::ResultSuccess();
}

nn::Result ManagerImpl::HasDevice(nn::sf::Out<bool> hasDevice, std::int32_t device) NN_NOEXCEPT
{
    // SIGLO-25371 により HasDevice は削除されました。
    // Service Framework の互換性のために本関数は維持しますが、互換性を切ることのできる修正を行うときに削除することを推奨します。
    NN_UNUSED(hasDevice);
    NN_UNUSED(device);

    NN_RESULT_SUCCESS;
}

nn::Result ManagerImpl::HasDeviceForDev(nn::sf::Out<bool> hasDevice, std::int32_t busIdx, nn::Bit16 slaveAddress, std::int32_t addressingMode, std::int32_t speedMode) NN_NOEXCEPT
{
    // SIGLO-25371 により HasDeviceForDev は削除されました。
    // Service Framework の互換性のために本関数は維持しますが、互換性を切ることのできる修正を行うときに削除することを推奨します。
    NN_UNUSED(hasDevice);
    NN_UNUSED(busIdx);
    NN_UNUSED(slaveAddress);
    NN_UNUSED(addressingMode);
    NN_UNUSED(speedMode);

    NN_RESULT_SUCCESS;
}

}}}
