﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/i2c/driver/i2c.h>
#include <nn/i2c/i2c_ResultForPrivate.h>
#include <nn/os/os_Mutex.h>

#include "i2c_BusAccessor.h"
#include "i2c_Command.h"


namespace nn {
namespace i2c {
namespace driver {
namespace detail {

class Session
{
    NN_DISALLOW_COPY(Session);
    NN_DISALLOW_MOVE(Session);

public:
    Session() NN_NOEXCEPT :
        m_IsOpen(false),                               // クローズ状態
        m_pBusAccessor(nullptr),                       // バスアクセッサは未登録
        m_SlaveAddress(0x0000),                        // スレーブアドレスはゼロ埋め
        m_AddressingMode(AddressingMode_BitWidth7),    // 7 bit addressing mode
        m_BusAccessorMutex(false),                     // セッションのステート向けの Mutex
        m_RetryCount(0),
        m_RetryInterval(0)
        {}

    void      Open(int busIdx, Bit16 slaveAddress, AddressingMode addressingMode, SpeedMode speedMode, BusAccessor* pBusAccessor,
                   int retryCount, nn::TimeSpan retryInterval) NN_NOEXCEPT; // Session を開くときに呼ばれる。引数　device は m_I2cDevice に格納する
    void      Start()  NN_NOEXCEPT;                 // Session を開始するときに呼ばれる。
    void      Close() NN_NOEXCEPT;                  // Session を終了するときに呼ばれる。

    bool      IsOpened() const NN_NOEXCEPT;         // セッションの状態 m_IsOpen を返す

    Result    ExecuteTransaction( void* pReceiveData, const void* pSendData, size_t dataBytes, TransactionOption inOption, Command command ) NN_NOEXCEPT;
    Result    ExecuteTransactionWithRetry(void* pReceiveData, const void* pSendData, size_t dataBytes, TransactionOption inOption, Command command) NN_NOEXCEPT;

    void      Dump() NN_NOEXCEPT;

private:
    bool                        m_IsOpen;                  // セッションのステート。排他が必要
    BusAccessor*                m_pBusAccessor;            // 各Busのアクセッサへのポインタ。Open時に登録される。
    int                         m_BusIdx;                  // 各Busのインデックス
    Bit16                       m_SlaveAddress;            // スレーブアドレス
    AddressingMode              m_AddressingMode;          // アドレスモード
    nn::os::Mutex               m_BusAccessorMutex;        // ロック対象は m_pBusAccessor
    int                         m_RetryCount;              // Timeout 時の RetryCount
    nn::TimeSpan                m_RetryInterval;           // Timeout 時の RetryInterval

};

} // detail
} // driver
} // i2c
} // nn
