﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/http.h>

#include <nn/nn_SdkLog.h>
#include <curl/curl.h>

#include <cctype>
#include <mutex>

#include "http_Utility.h"

namespace nn {
namespace http {

class DestinationBuffer
{
public:
    DestinationBuffer(void* pBuffer, size_t size)
        : m_pBuffer(reinterpret_cast<uint8_t*>(pBuffer)), m_size(size), m_filledSize(0)
    {
    }

    size_t Put(const void* pSource, size_t size)
    {
        size_t sizeToFill = std::min<size_t>(size, m_size - m_filledSize);
        if (sizeToFill > 0)
        {
            std::memcpy(&m_pBuffer[m_filledSize], pSource, sizeToFill);
            m_filledSize += sizeToFill;
        }
        return sizeToFill;
    }

    size_t GetFilledSize() const
    {
        return m_filledSize;
    }

    size_t GetFreeSize() const
    {
        return m_size - m_filledSize;
    }

    bool IsFull() const
    {
        return GetFreeSize() == 0;
    }

private:
    uint8_t *m_pBuffer;
    size_t m_size;
    size_t m_filledSize;
};

Response::Response()
    : m_mutexRead(true)
    , m_pDestinationBuffer(nullptr)
    , m_pProgress(nullptr)
    , m_sizeCopied(0)
    , m_pCancelEvent(nullptr)
{
}

Result Response::ReadBody(size_t * outValue, void * pBuffer, size_t size, os::Event * pCancelEvent)
{
    NN_SDK_ASSERT(m_pCurlEasyHandle != nullptr, "This response is not attached to Request.");

    // リードを発行できるスレッドは一つに絞る
    std::lock_guard<os::Mutex> readBlocker(m_mutexRead);

    // WRITEFUNCTION の書き込み先バッファを準備
    DestinationBuffer destBuf(pBuffer, size);
    {
        std::lock_guard<os::Mutex> locker(m_mutex);
        m_pDestinationBuffer = &destBuf;
    }

    if (IsConnected())
    {
        // バッファを設定したのでコールバックの実行を再開
        m_pConnection->CurlPauseEasyHandle(m_pCurlEasyHandle, CURLPAUSE_RECV_CONT);
    }

    // バッファが満たされる、もしくは接続がエラー終了するのを待つ
    {
        std::lock_guard<os::Mutex> waitConditionBlocker(m_mutexForCond);

        while (!IsCompleted())
        {
            m_pConnection->PsuedoWaitCondition(m_mutexForCond, m_cond, pCancelEvent);
            if (destBuf.IsFull())
            {
                break;
            }
        }
    }

    // バッファの登録を解除
    {
        std::lock_guard<os::Mutex> locker(m_mutex);
        m_pDestinationBuffer = nullptr;
    }

    // エラーが発生していないか確認
    if (IsCompleted() && m_CurlCode != CURLE_OK)
    {
        return GetLastResult();
    }

    *outValue = destBuf.GetFilledSize();
    return ResultSuccess();
}

Result Response::ReadBody(size_t * outValue, void * pBuffer, size_t size)
{
    return ReadBody(outValue, pBuffer, size, nullptr);
}

void Response::SetProgressTarget(Progress * pProgress)
{
    m_pProgress = pProgress;
}

bool Response::OnFillBuffer(void * pSource, size_t size)
{
    std::lock_guard<os::Mutex> locker(m_mutex);

    if (!m_pDestinationBuffer)
    {
        return false;
    }

    m_sizeCopied += m_pDestinationBuffer->Put(
        &(reinterpret_cast<char*>(pSource)[m_sizeCopied]),
        size - m_sizeCopied
    );

    if (m_pDestinationBuffer->IsFull())
    {
        NotifyBufferFull();
        return false;
    }
    else
    {
        m_sizeCopied = 0;
        return true;
    }
}

void Response::OnTransfer(curl_off_t dlTotal, curl_off_t dlNow, curl_off_t ulTotal, curl_off_t ulNow)
{
    if (m_pProgress)
    {
        m_pProgress->Update(dlNow, dlTotal);
    }
}

void Response::NotifyBufferFull()
{
    m_cond.Broadcast();
}

}
} // ~namespace nn::http
