﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/http.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <curl/curl.h>

#include <mutex>

#include "http_Utility.h"

namespace nn { namespace http {

Request::Request(ConnectionBroker & broker)
    : m_mutexReadWrite(true)
    , m_MethodType(MethodType_Get)
    , m_Connection(broker)
    , m_pHeaderList(nullptr)
    , m_pEasyHandle(nullptr)
    , m_pUrl(nullptr)
    , m_pUrlBuffer(nullptr)
    , m_timeTimeout(TimeSpan::FromSeconds(TimeoutSeconds))
    , m_timeConnectTimeout(TimeSpan::FromSeconds(ConnectTimeoutSeconds))
    , m_bKeepAlive(true)
    , m_bSkipSslVerificationForDebug(false)
    , m_bVerbose(false)
    , m_sizeFormatBuffer(FormatBufferSizeDefault)
    , m_pPostData(nullptr)
    , m_sizePostData(0)
    , m_pPostDataBuffer(nullptr)
    , m_pCookieBuffer(nullptr)
    , m_pCancelEvent(nullptr)
{
    broker.ApplyDefaultOptions(*this);
}

Request::~Request()
{
    if (m_pHeaderList)
    {
        curl_slist_free_all(m_pHeaderList);
        m_pHeaderList = nullptr;
    }
    if (m_pEasyHandle)
    {
        CurlEasyCleanup(m_pEasyHandle);
        m_pEasyHandle = nullptr;
    }
    if (m_pUrlBuffer)
    {
        FreeMemory(m_pUrlBuffer);
        m_pUrlBuffer = nullptr;
    }
    if (m_pPostDataBuffer)
    {
        FreeMemory(m_pPostDataBuffer);
        m_pPostDataBuffer = nullptr;
    }
    if (m_pCookieBuffer)
    {
        FreeMemory(m_pCookieBuffer);
        m_pCookieBuffer = nullptr;
    }
}

void Request::SetUrlPointer(const char * pUrl)
{
    NN_SDK_REQUIRES_NOT_NULL(pUrl);
    m_pUrl = pUrl;
}

Result Request::SetUrlByFormatString(const char * format, ...)
{
    if (m_pUrlBuffer)
    {
        FreeMemory(m_pUrlBuffer);
        m_pUrlBuffer = nullptr;
        m_pUrl = nullptr;
    }

    std::va_list vlist;
    char* pUrlBuffer;

    va_start(vlist, format);

    Result result = AllocateAndFormatString(&pUrlBuffer, nullptr, format, vlist);

    va_end(vlist);

    if (result.IsFailure())
    {
        return result;
    }

    m_pUrlBuffer = pUrlBuffer;
    SetUrlPointer(m_pUrlBuffer);

    return ResultSuccess();
}

void Request::SetPostDataPointer(const char * pData, size_t size)
{
    m_pPostData = pData;
    m_sizePostData = size;
}

Result Request::SetPostDataByFormatString(const char * format, ...)
{
    if (m_pPostDataBuffer)
    {
        FreeMemory(m_pPostDataBuffer);
        m_pPostDataBuffer = nullptr;
        m_pPostData = nullptr;
        m_sizePostData = 0;
    }

    std::va_list vlist;
    char* pPostData;
    size_t sizePostData;

    va_start(vlist, format);

    Result result = AllocateAndFormatString(&pPostData, &sizePostData, format, vlist);

    va_end(vlist);

    if (result.IsFailure())
    {
        return result;
    }
    else
    {
        m_pPostDataBuffer = pPostData;
    }

    SetPostDataPointer(m_pPostDataBuffer, sizePostData);

    return ResultSuccess();
}

void Request::SetFormatBufferSize(size_t size)
{
    m_sizeFormatBuffer = size;
}

void Request::SetTimeout(const TimeSpan& timeout)
{
    m_timeTimeout = timeout;
}

void Request::SetConnectTimeout(const TimeSpan& timeout)
{
    m_timeConnectTimeout = timeout;
}

void Request::SetKeepAlive(bool bEnable)
{
    m_bKeepAlive = bEnable;
}

void Request::SetVerbose(bool bVerbose)
{
    m_bVerbose = bVerbose;
}

void Request::SetSkipSslVerificationForDebug(bool bSkip)
{
    m_bSkipSslVerificationForDebug = bSkip;
}

Result Request::AddHeaderLine(const char * pLine)
{
    curl_slist* headers = curl_slist_append(m_pHeaderList, pLine);
    if (!headers)
    {
        return ResultCurlSlistAppendFailed();
    }

    m_pHeaderList = headers;
    return nn::ResultSuccess();
}

Result Request::AddHeaderFormat(const char* format, ...)
{
    char* pLineBuffer;

    std::va_list vlist;

    va_start(vlist, format);

    Result result = AllocateAndFormatString(&pLineBuffer, nullptr, format, vlist);

    va_end(vlist);

    if (result.IsFailure())
    {
        return result;
    }

    result = AddHeaderLine(pLineBuffer);
    FreeMemory(pLineBuffer);
    return result;
}

Result Request::SetCookieFormat(const char* format, ...)
{
    if(m_pCookieBuffer)
    {
        FreeMemory(m_pCookieBuffer);
        m_pCookieBuffer = nullptr;
    }

    char* pLineBuffer;

    std::va_list vlist;

    va_start(vlist, format);

    Result result = AllocateAndFormatString(&pLineBuffer, nullptr, format, vlist);

    va_end(vlist);

    if (result.IsFailure())
    {
        return result;
    }

    m_pCookieBuffer = pLineBuffer;
    return result;
}

void Request::SetMethodType(MethodType type)
{
    NN_SDK_ASSERT_RANGE(type, MethodType_Get, MethodType_Max);
    m_MethodType = type;
}

Result Request::WriteBody(const char * pBody, size_t size)
{
    std::lock_guard<os::Mutex> blocker(m_mutexReadWrite);

    Result result = EnsureEasyHandleReady();
    NN_HTTP_RESULT_THROW_IF_FAILURE(result);

    return ResultSuccess();
}

Result Request::GetResponse(ResponseImpl* pResponse)
{
    NN_SDK_REQUIRES(m_pUrl != nullptr, "SetUrl must be called.");
    NN_SDK_REQUIRES_NOT_NULL(pResponse);
    {
        std::lock_guard<os::Mutex> blocker(m_mutexReadWrite);

        Result result = EnsureEasyHandleReady();
        NN_HTTP_RESULT_THROW_IF_FAILURE(result);

        pResponse->AssignHandle(m_pEasyHandle, m_Connection);
        pResponse->SetCancelEvent(m_pCancelEvent);

        if (m_MethodType != MethodType_Get && m_MethodType != MethodType_Head)
        {
            if (m_pPostData)
            {
                curl_easy_setopt(m_pEasyHandle, CURLOPT_POSTFIELDS, m_pPostData);
                curl_easy_setopt(m_pEasyHandle, CURLOPT_POSTFIELDSIZE, m_sizePostData);
            }
            else
            {
                curl_easy_setopt(m_pEasyHandle, CURLOPT_POSTFIELDS, "");
                curl_easy_setopt(m_pEasyHandle, CURLOPT_POSTFIELDSIZE, 0);
            }
        }

        m_pEasyHandle = nullptr;
    }

    if (!pResponse->WaitConnected() || pResponse->IsCompleted())
    {
        Result result = pResponse->GetLastResult();
        if (result.IsFailure())
        {
            NN_HTTP_WARN("Failed to get response for %s. (0x%08x)\n", m_pUrl, result.GetInnerValueForDebug());
        }
        return result;
    }

    return ResultSuccess();
}

void Request::SetCancelEvent(os::Event * pCancelEvent)
{
    m_pCancelEvent = pCancelEvent;
}

Result Request::EnsureEasyHandleReady()
{
    if (!m_pEasyHandle)
    {
        CURL* pCurlEasyHandle;

        Result result = CurlEasyInit(pCurlEasyHandle);
        NN_HTTP_RESULT_THROW_IF_FAILURE(result);

        result = ApplyOptionsToEasyHandle(pCurlEasyHandle);
        if (result.IsFailure())
        {
            CurlEasyCleanup(pCurlEasyHandle);
            NN_RESULT_THROW(result);
        }

        result = m_Connection.CurlAddEasyHandleToMulti(pCurlEasyHandle);
        if (result.IsFailure())
        {
            CurlEasyCleanup(pCurlEasyHandle);
            NN_RESULT_THROW(result);
        }

        m_pEasyHandle = pCurlEasyHandle;
    }
    return ResultSuccess();
}

Result Request::AllocateAndFormatString(char ** ppOut, size_t* pOutSize, const char* format, std::va_list vlist)
{
    NN_SDK_REQUIRES_NOT_NULL(format);

    char* pLineBuffer = static_cast<char*>(AllocateMemory(m_sizeFormatBuffer));
    if (!pLineBuffer)
    {
        NN_HTTP_ERROR("Failed to allocate memory. (%d bytes)\n", m_sizeFormatBuffer);
        return ResultOutOfMemory();
    }

    int length = util::VSNPrintf(pLineBuffer, m_sizeFormatBuffer, format, vlist);
    if (length >= static_cast<int>(m_sizeFormatBuffer))
    {
        NN_HTTP_ERROR("Too long string for formatting. Please increase format buffer. (current = %ld bytes)\n", m_sizeFormatBuffer);
        FreeMemory(pLineBuffer);
        return ResultTooLongArgument();
    }

    if (pOutSize)
    {
        *pOutSize = static_cast<size_t>(length);
    }
    *ppOut = pLineBuffer;
    return ResultSuccess();
}

Result Request::ApplyOptionsToEasyHandle(CURL * pCurlEasyHandle)
{
    Result result;

    switch (m_MethodType)
    {
    case MethodType_Get:
        break;
    case MethodType_Head:
        (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_NOBODY, 1);
        break;
    case MethodType_Post:
        (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_POST, 1);
        result = AddHeaderLine("Expect:");
        NN_HTTP_RESULT_THROW_IF_FAILURE(result);
        break;
    case MethodType_Put:
        (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_PUT, 1);
        break;
    case MethodType_Delete:
        result = ConvertCurlCodeToResult( curl_easy_setopt(pCurlEasyHandle, CURLOPT_CUSTOMREQUEST, "DELETE") );
        NN_HTTP_RESULT_THROW_IF_FAILURE(result);
        break;
    case MethodType_Patch:
        result = ConvertCurlCodeToResult( curl_easy_setopt(pCurlEasyHandle, CURLOPT_CUSTOMREQUEST, "PATCH") );
        NN_HTTP_RESULT_THROW_IF_FAILURE(result);
        break;
    default:
        return ResultNotImplemented();
    }

    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_VERBOSE,        m_bVerbose ? 1L : 0L);
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_NOPROGRESS,     0L);
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_FOLLOWLOCATION, 1L);
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_FORBID_REUSE,   m_bKeepAlive ? 0L : 1L);
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_MAXREDIRS,      RedirectCountMax);
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_LOW_SPEED_LIMIT, LowSpeedLimit);
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_LOW_SPEED_TIME, LowSpeedTimeSeconds);

    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_TIMEOUT,        static_cast<long>(m_timeTimeout.GetSeconds()));
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_CONNECTTIMEOUT, static_cast<long>(m_timeConnectTimeout.GetSeconds()));

#if 0
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_OPENSOCKETFUNCTION, m_Connection.CurlOpenSocketFunction);
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_OPENSOCKETDATA, static_cast<void*>(&m_Connection));
#else
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_SOCKOPTFUNCTION, m_Connection.CurlSocketOptionFunction);
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_SOCKOPTDATA,    static_cast<void*>(&m_Connection));
#endif

    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_DEBUGFUNCTION,  m_Connection.CurlDebugFunction);
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_DEBUGDATA,      static_cast<void*>(&m_Connection));

    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_SSL_VERIFYPEER, m_bSkipSslVerificationForDebug ? 0L : 1L);
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_SSL_VERIFYHOST, m_bSkipSslVerificationForDebug ? 0L : 2L);
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_SSL_VERIFYDATE, 0L);

    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_SSL_CTX_FUNCTION, m_Connection.CurlSslCtxFunction);
    (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_SSL_CTX_DATA,   static_cast<void*>(&m_Connection));

    result = ConvertCurlCodeToResult( curl_easy_setopt(pCurlEasyHandle, CURLOPT_URL, m_pUrl) );
    NN_HTTP_RESULT_THROW_IF_FAILURE(result);

    if (m_pHeaderList)
    {
        (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_HTTPHEADER, m_pHeaderList);
    }

    if (m_pCookieBuffer)
    {
        (void)curl_easy_setopt(pCurlEasyHandle, CURLOPT_COOKIE, m_pCookieBuffer);
    }

    curl_easy_pause(pCurlEasyHandle, CURLPAUSE_ALL);

    return ResultSuccess();
}


}} // ~namespace nn::http
