﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <memory>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_ThreadLocalStorage.h>
#include <nn/htc.h>
#include <nn/htcs/htcs_Library.h>
#include <nn/htcs/htcs_Socket.h>
#include <nn/htcs/htcs_Result.h>
#include <nn/htcs/htcs_Config.h>
#include <nn/sf/sf_HipcClient.h>
#include "htcs_Const.h"
#if defined(NN_BUILD_CONFIG_OS_WIN)
#include "../../Libraries/tmagent/htcs/htcs_HtcsManager.h"
#else
#include "htcs_HtcsManagerByHipc.h"
#endif
#include <cstdlib>
#include <nn/nn_SdkLog.h>
#include "htcs_VirtualSocketCollection.h"

#if USE_VIRTUAL_SOCKETS

#define HTCS_API_TRACE( channel, ... ) //NN_SDK_LOG( "[" ); NN_SDK_LOG( channel ); NN_SDK_LOG( "] - " ); NN_SDK_LOG( __VA_ARGS__ ); NN_SDK_LOG( "\n" )

namespace nn { namespace htcs { namespace detail {

enum
{
    INVALID_SOCKET = -1,
    MAX_LISTEN_QUEUE = 200,
};
//========================================================================
// The primitive calls.  Implemented in htcs_Socket.cpp
nn::sf::SharedPointer<nn::tma::ISocket> socket  ( int& LastError ) NN_NOEXCEPT;
int                                     close   ( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, int& LastError ) NN_NOEXCEPT;
int                                     connect ( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, const nn::htcs::SockAddrHtcs* address, int& LastError ) NN_NOEXCEPT;
int                                     bind    ( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, const nn::htcs::SockAddrHtcs* address, int& LastError ) NN_NOEXCEPT;
int                                     listen  ( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, int backlogCount, int& LastError ) NN_NOEXCEPT;
nn::sf::SharedPointer<nn::tma::ISocket> accept  ( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, nn::htcs::SockAddrHtcs* address, int& LastError ) NN_NOEXCEPT;
nn::htcs::ssize_t                       recv    ( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, void* buffer, size_t bufferByteSize, int flags, int& LastError ) NN_NOEXCEPT;
nn::htcs::ssize_t                       send    ( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, const void* buffer, size_t bufferByteSize, int flags, int& LastError ) NN_NOEXCEPT;
int                                     shutdown( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, int how, int& LastError ) NN_NOEXCEPT;
int                                     fcntl   ( nn::sf::SharedPointer<nn::tma::ISocket> descriptor, int command, int value, int& LastError ) NN_NOEXCEPT;

//==============================================================================
// Virtual struct to wrap the socket primitive.  Done to satisfy jira OASIS-371,
// "Socket and Bind should success even when TMS has not been connected".
struct virtual_socket
{
    virtual_socket ()
    {
        Init();
    }

    ~virtual_socket()
    {
    }

    //==================================================================

    void Init()
    {
        m_Id = INVALID_SOCKET;
        m_Socket = nullptr;

        m_Blocking = true;

        m_DoBind = false;
        memset(&m_Address, 0, sizeof(m_Address));

        m_ListenBacklogCount = -1;

        m_FcntlCommand = -1;
        m_FcntlValue = 0;
    }

    //==================================================================

    int Bind( const nn::htcs::SockAddrHtcs* pAddress, int& LastError )
    {
        m_DoBind = true;
        memcpy( &m_Address, pAddress, sizeof(nn::htcs::SockAddrHtcs) );
        LastError = 0;
        return 0;
    }

    //==================================================================

    int Listen( int backlogCount, int& LastError )
    {
        int Ret = -1;
        if( m_DoBind == false )
        {
            LastError = HTCS_EINVAL;
        }

        else if( m_ListenBacklogCount > 0 )
        {
            LastError = HTCS_EISCONN;
        }
        else
        {
            if( backlogCount < 1 )
            {
                m_ListenBacklogCount = 1;
            }
            else
            {
                m_ListenBacklogCount = backlogCount;
            }
            LastError = 0;
            Ret = 0;
        }
        return Ret;
    }

    //==================================================================

    int Fcntl( int Command, int Value, int& LastError )
    {
        int Ret = 0;
        LastError = 0;

        if ( Command == HTCS_F_SETFL )
        {
            m_FcntlCommand = Command;
            m_FcntlValue = Value;

            // Set the blocking flag
            m_Blocking = (Value & HTCS_O_NONBLOCK) == 0;
        }
        else if( Command == HTCS_F_GETFL )
        {
            Ret = m_FcntlValue;
        }
        else
        {
            //Don't recognize anything else, so this is an error
            LastError = HTCS_EINVAL;
            Ret = -1;
        }

        return Ret;
    }

    //==================================================================

    int SetSocket( nn::sf::SharedPointer<nn::tma::ISocket> SocketPrimitive, int& LastError )
    {
        int Ret = 0;
        if( !m_Socket && SocketPrimitive )
        {
            m_Socket = SocketPrimitive;

            //===============================================================
            // Do all the tasks we need to, now that we've got a real socket.
            //===============================================================
            if( m_DoBind )
            {
                Ret = bind( m_Socket, &m_Address, LastError );
            }

            if( (Ret == 0 ) && ( m_FcntlCommand != -1 ) )
            {
                Ret = fcntl( m_Socket, m_FcntlCommand, m_FcntlValue, LastError );
            }

            if( (Ret == 0 ) && ( m_ListenBacklogCount > 0 ) )
            {
                Ret = listen( m_Socket, m_ListenBacklogCount, LastError );
            }

        }
        return Ret;
    }

    //==============================================================================

    int                     m_Id;
    nn::sf::SharedPointer<nn::tma::ISocket> m_Socket;

    bool                    m_DoBind;
    nn::htcs::SockAddrHtcs  m_Address;

    int                     m_ListenBacklogCount;

    int                     m_FcntlCommand;
    int                     m_FcntlValue;

    bool                    m_Blocking;
};

//==============================================================================
// Virtual socket collection
//======================================================================================
// Collection of virtual socket primitives.  This is layer of virtual sockets fixes
// OASIS-371, "Some of htcs APIs should succeed even when TMS is not connected".
//======================================================================================
virtual_socket_collection::virtual_socket_collection()
{
    HTCS_API_TRACE( "virtual_socket_collection", "virtual_socket_collection" );
    nn::os::InitializeMutex( &m_Mutex, false, 0 );

    m_NextId = 1;
    m_ListCount = 0;
    m_ListSize = 0;
    m_pList = NULL;
}

//==============================================================================

virtual_socket_collection::~virtual_socket_collection()
{
    HTCS_API_TRACE( "virtual_socket_collection", "~virtual_socket_collection" );
    Clear();
    for (int i = 0; i < m_ListSize; i++)
    {
        m_pList[i].~virtual_socket();
    }
    nn::os::FinalizeMutex( &m_Mutex );
}

//==============================================================================

size_t virtual_socket_collection::GetWorkingMemorySize(int socketCountMax) NN_NOEXCEPT
{
    NN_SDK_ASSERT(socketCountMax <= nn::htcs::SocketCountMax);
    return socketCountMax * (sizeof(virtual_socket));
}

//==============================================================================

void virtual_socket_collection::Init(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    HTCS_API_TRACE( "virtual_socket_collection", "Init" );

    m_Buffer = buffer;
    m_BufferSize = bufferSize;

    m_ListSize = static_cast<int32_t>(m_BufferSize / (sizeof(virtual_socket)));
    if (m_ListSize > nn::htcs::SocketCountMax)
    {
        m_ListSize = nn::htcs::SocketCountMax;
    }

    m_pList = static_cast<virtual_socket*>(m_Buffer);
    virtual_socket* pCurrent = m_pList;
    for (int i = 0; i < m_ListSize; i++)
    {
        new (pCurrent) virtual_socket;
        pCurrent++;
    }
}

//==============================================================================

int virtual_socket_collection::CreateSocket( nn::sf::SharedPointer<nn::tma::ISocket> FromSocket, int& ErrorCode ) NN_NOEXCEPT
{
    ErrorCode = 0;
    int Id = -1;

    if (m_ListCount >= m_ListSize)
    {
        if (FromSocket)
        {
            int errorCode;
            close(FromSocket, errorCode);
        }

        ErrorCode = nn::htcs::HTCS_EMFILE;
        return -1;
    }

    Id = CreateId();

    Insert(Id, FromSocket);

    // Return it's Id
    return Id;
}

//==============================================================================
// List functionality
//==============================================================================

int virtual_socket_collection::Add( nn::sf::SharedPointer<nn::tma::ISocket> FromSocket ) NN_NOEXCEPT
{
    HTCS_API_TRACE( "virtual_socket_collection", "Add socket %d", FromSocket );

    //===================================================
    // If it's invalid, then we don't need to add it.
    //===================================================
    if( !FromSocket )
    {
        return -1;
    }

    //Create this socket.
    int ErrorCode;
    return CreateSocket( FromSocket, ErrorCode );
}
//==============================================================================

void virtual_socket_collection::Clear() NN_NOEXCEPT
{
    HTCS_API_TRACE( "virtual_socket_collection", "Clear" );
    nn::os::LockMutex( &m_Mutex );

    m_ListCount = 0;
    memset( m_pList, 0, m_ListSize * sizeof(virtual_socket) );
    nn::os::UnlockMutex( &m_Mutex );
}

//==============================================================================
// Done this way because we don't want to create Ids that are currently in use.
int virtual_socket_collection::CreateId() NN_NOEXCEPT
{
    HTCS_API_TRACE( "virtual_socket_collection", "CreateId" );
    nn::os::LockMutex( &m_Mutex );
    int Ret = 0;

    do
    {
        Ret = m_NextId;
        m_NextId += 1;
        if( m_NextId <= 0 )
        {
            m_NextId = 1;
        }
    }
    while( Find( Ret ) >= 0  );
    nn::os::UnlockMutex( &m_Mutex );

    return Ret;
}

//==============================================================================

void virtual_socket_collection::SetSize( int32_t Size ) NN_NOEXCEPT
{
    NN_UNUSED(Size);
}

//==============================================================================

void virtual_socket_collection::Insert( int Id, nn::sf::SharedPointer<nn::tma::ISocket> pSocket ) NN_NOEXCEPT
{
    HTCS_API_TRACE( "virtual_socket_collection", "Insert %d", pSocket->GetId() );
    nn::os::LockMutex( &m_Mutex );
    //SetSize( m_ListCount + 1 );

    if( m_ListCount == 0 )
    {
        m_pList[0].m_Id = Id;
        m_pList[0].m_Socket = pSocket;
    }
    else // m_ListCount > 0
    {
        //Find the first one that's smaller than this Id
        int32_t AtIndex = m_ListCount - 1;
        for( ; AtIndex >= 0; AtIndex -= 1 )
        {
            if( m_pList[AtIndex].m_Id < Id )
            {
                break;
            }

            //Move this guy further down the list.
            m_pList[AtIndex + 1] = m_pList[AtIndex];
        }

        // Here's our place in the list.
        m_pList[AtIndex + 1].m_Id = Id;
        m_pList[AtIndex + 1].m_Socket = pSocket;
    }

    //Keep track of how many we've got in our list.
    m_ListCount += 1;
    nn::os::UnlockMutex( &m_Mutex );
}

//==============================================================================

int virtual_socket_collection::Find( int Id, int* pErrorCode ) NN_NOEXCEPT
{
    HTCS_API_TRACE( "virtual_socket_collection", "Find %d", Id );
    if( m_ListCount > 0 )
    {
        //====================================================================
        // Binary search our list.
        //====================================================================
        int32_t Left   = 0;
        int32_t Right  = m_ListCount - 1;
        while ( Left <= Right )
        {
            int Middle = (Left + Right) / 2;
            virtual_socket* pSocket = &m_pList[Middle];
            if ( pSocket->m_Id == Id )
            {
                return Middle;
            }
            else if ( pSocket->m_Id > Id )
            {
                Right = Middle - 1;
            }
            else
            {
                Left = Middle + 1;
            }
        }
    }

    // Invalid socket
    if( pErrorCode != NULL )
    {
        *pErrorCode = nn::htcs::HTCS_EBADF;
    }

    return INVALID_SOCKET;
}

//==============================================================================

bool virtual_socket_collection::HasAddr(const nn::htcs::SockAddrHtcs* pAddress) NN_NOEXCEPT
{
    if (m_ListCount > 0)
    {
        for (int i = 0; i < m_ListCount; i++)
        {
            if ((pAddress->family == m_pList[i].m_Address.family)
                && (strcmp(pAddress->peerName.name, m_pList[i].m_Address.peerName.name) == 0)
                && (strcmp(pAddress->portName.name, m_pList[i].m_Address.portName.name) == 0))
            {
                return true;
            }
        }
    }

    return false;
}
//==============================================================================

nn::sf::SharedPointer<nn::tma::ISocket> virtual_socket_collection::GetSocket( int Id, int* pErrorCode ) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::tma::ISocket> Ret = nullptr;
    nn::os::LockMutex( &m_Mutex );
    int AtIndex = Find( Id, pErrorCode );
    if( AtIndex >= 0 )
    {
        Ret = m_pList[AtIndex].m_Socket;
    }
    nn::os::UnlockMutex( &m_Mutex );

    return Ret;
}

//==============================================================================

nn::sf::SharedPointer<nn::tma::ISocket> virtual_socket_collection::FetchSocket( int socketId, int& LastError ) NN_NOEXCEPT
{
    LastError = 0;
    nn::sf::SharedPointer<nn::tma::ISocket> FetchedSocket = GetSocket( socketId, &LastError );

    //If we've got a real socket, then we're done here.
    if( FetchedSocket )
    {
        return FetchedSocket;
    }

    // If we found this virtual socket, then --
    if( LastError == 0 )
    {
        // We've got a virtual without a real socket.  Try to fetch it now.
        FetchedSocket = socket( LastError );
        if( FetchedSocket )
        {
            //========================================================================
            // Assign this socket primitive to this virtual socket.
            //========================================================================
            nn::os::LockMutex( &m_Mutex );
            int AtIndex = Find( socketId, &LastError );
            if( AtIndex >= 0 )
            {
                m_pList[AtIndex].SetSocket( FetchedSocket, LastError );
            }
            nn::os::UnlockMutex( &m_Mutex );

            // Did someone delete this guy while we were fetching the socket??
            if( AtIndex < 0 )
            {
                int tempError = 0;
                close( FetchedSocket, tempError );
                FetchedSocket = nullptr;
            }
        }
    }
    return FetchedSocket;
}

//==============================================================================
// HTCS functionality.  What we live for
//==============================================================================

int virtual_socket_collection::Socket( int &ErrorCode ) NN_NOEXCEPT
{
    HTCS_API_TRACE( "virtual_socket_collection", "Socket" );
    ErrorCode = 0;

    //Try to create this socket.
    nn::sf::SharedPointer<nn::tma::ISocket> FromSocket = socket( ErrorCode );

    return CreateSocket( FromSocket, ErrorCode );
}

//==============================================================================

int virtual_socket_collection::Close( int Id, int &ErrorCode ) NN_NOEXCEPT
{
    HTCS_API_TRACE( "virtual_socket_collection", "Close %d", Id );
    ErrorCode = 0;
    nn::sf::SharedPointer<nn::tma::ISocket> SocketPrimitive = nullptr;
    int Ret = 0;

    //================================================================
    // Lock down our list while we hunt this guy down and delete him
    //================================================================
    nn::os::LockMutex( &m_Mutex );
    int AtIndex = Find( Id, &ErrorCode );
    if( AtIndex >= 0 )
    {
        // Close this guy.
        virtual_socket* pSocket = &m_pList[AtIndex];
        SocketPrimitive = pSocket->m_Socket;

        //Push everyone up in the list.
        for( ; AtIndex < m_ListCount - 1; AtIndex += 1 )
        {
            m_pList[AtIndex] = m_pList[AtIndex + 1];
        }
        m_pList[AtIndex].Init();

        m_ListCount -= 1;
    }
    nn::os::UnlockMutex( &m_Mutex );

    //===========================================
    // If it was a valid socket, close it.
    if( SocketPrimitive )
    {
        close( SocketPrimitive, ErrorCode );

        //=================================================
        // Because this is an "always succeed" call,
        // return success.  If it failed, TMA will detect
        // it and close it properly next chance it gets.
        //=================================================
        Ret = ErrorCode = 0;
    }

    // Return whether or not we could find this socket.
    return ( AtIndex < 0 ) ? -1 : Ret;
}

//==============================================================================

int virtual_socket_collection::Bind( int Id, const nn::htcs::SockAddrHtcs* pAddr, int &ErrorCode ) NN_NOEXCEPT
{
    int Ret = -1;
    ErrorCode = 0;
    nn::sf::SharedPointer<nn::tma::ISocket> Socket = GetSocket( Id, &ErrorCode );

    if( Socket )
    {
        Ret = bind( Socket, pAddr, ErrorCode );
    }
    // Does this socket actually exist?
    else if ( ErrorCode != nn::htcs::HTCS_EBADF )
    {
        //==================================================
        // Is this one already bound?
        nn::os::LockMutex( &m_Mutex );
        int AtIndex = Find( Id );
        if( AtIndex >= 0 )
        {
            bool Exists = HasAddr(pAddr);

            if (m_pList[AtIndex].m_DoBind)
            {
                ErrorCode = HTCS_EINVAL;
                Ret = -1;
            }
            else if (Exists)
            {
                ErrorCode = HTCS_EADDRINUSE;
                Ret = -1;
            }
            else
            {
                Ret = m_pList[AtIndex].Bind(pAddr, ErrorCode);
            }
        }
        else
        {
            // Made it here, then the socket has been deleted by another thread.
            ErrorCode = nn::htcs::HTCS_EBADF;
        }

        nn::os::UnlockMutex( &m_Mutex );
    }

    return Ret;
}

//==============================================================================

int virtual_socket_collection::Listen( int Id, int backlogCount, int& ErrorCode ) NN_NOEXCEPT
{
    int Ret = -1;
    ErrorCode = 0;
    nn::sf::SharedPointer<nn::tma::ISocket> Socket = GetSocket( Id, &ErrorCode );

    if( Socket )
    {
        Ret = listen( Socket, backlogCount, ErrorCode );
    }
    // Does this socket actually exist?
    else if ( ErrorCode != nn::htcs::HTCS_EBADF )
    {
        nn::os::LockMutex( &m_Mutex );
        int AtIndex = Find( Id );
        if( AtIndex >= 0 )
        {
            Ret = m_pList[AtIndex].Listen( backlogCount, ErrorCode );
        }
        nn::os::UnlockMutex( &m_Mutex );
    }

    return Ret;
}

//==============================================================================

int virtual_socket_collection::Fcntl( int Id, int command, int value, int& ErrorCode ) NN_NOEXCEPT
{
    int Ret = -1;
    ErrorCode = 0;
    nn::sf::SharedPointer<nn::tma::ISocket> Socket = GetSocket( Id, &ErrorCode );

    if( Socket )
    {
        Ret = fcntl( Socket, command, value, ErrorCode );
    }
    // Does this socket actually exist?
    else if ( ErrorCode != nn::htcs::HTCS_EBADF )
    {
        nn::os::LockMutex( &m_Mutex );
        int AtIndex = Find( Id );
        if( AtIndex >= 0 )
        {
            Ret = m_pList[AtIndex].Fcntl( command, value, ErrorCode );
        }

        nn::os::UnlockMutex( &m_Mutex );
    }

    return Ret;
}

//==============================================================================

int virtual_socket_collection::Accept( int Id, nn::htcs::SockAddrHtcs* pAddress, int& ErrorCode ) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::tma::ISocket> Ret = nullptr;
    int result = -1;
    ErrorCode = 0;
    nn::sf::SharedPointer<nn::tma::ISocket> Socket = FetchSocket( Id, ErrorCode );

    if( Socket )
    {
        if (ErrorCode != 0)
        {
            return -1;
        }
        Ret = accept(Socket, pAddress, ErrorCode);
    }
    // Does this socket actually exist?
    else if ( ErrorCode != nn::htcs::HTCS_EBADF )
    {
        do
        {
            //==========================================================
            //Is this a blocker?
            //==========================================================
            bool BlockUntilDone = true;
            bool IsListened = false;

            nn::os::LockMutex( &m_Mutex );
            int AtIndex = Find( Id, &ErrorCode );
            if( AtIndex >= 0 )
            {
                BlockUntilDone = m_pList[AtIndex].m_Blocking;
                IsListened = (m_pList[AtIndex].m_ListenBacklogCount > 0);
            }
            nn::os::UnlockMutex( &m_Mutex );

            //Did this socket get deleted?
            if( AtIndex < 0 )
            {
                ErrorCode = nn::htcs::HTCS_EINTR;
                return -1;
            }

            if (!IsListened)
            {
                ErrorCode = nn::htcs::HTCS_EINVAL;
                return -1;
            }

            if( BlockUntilDone == false )
            {
                ErrorCode = nn::htcs::HTCS_EWOULDBLOCK;
                return -1;
            }

            //Sleep a bit and try again.
            os::SleepThread( nn::TimeSpan::FromMilliSeconds(500) );
            Socket  = FetchSocket( Id, ErrorCode );
        }
        //==========================================================
        // Park it here until we get our notification.
        //==========================================================
        while( !Socket );

        if (ErrorCode != 0)
        {
            return -1;
        }
        Ret = accept(Socket, pAddress, ErrorCode);
    }

    //====================================================================
    // Accept is the only call that creates a socket.  Virtualize it
    // so we can find it next time someone hands it in.
    //====================================================================
    if( Ret )
    {
        result = CreateSocket( Ret, ErrorCode );
        if (result < 0)
        {
            int errorCode;
            close(Ret, errorCode);
        }
    }

    return result;
}


//===============================================================================
// These can fail if the network is down, but only because they rely on
// other functionality that can fail if the network is down.
//==============================================================================

int virtual_socket_collection::Shutdown( int Id, int how, int& ErrorCode ) NN_NOEXCEPT
{
    int Ret = -1;
    ErrorCode = 0;
    nn::sf::SharedPointer<nn::tma::ISocket> Socket = GetSocket( Id, &ErrorCode );

    if( Socket )
    {
        Ret = shutdown( Socket, how, ErrorCode );
    }
    else if ( ErrorCode != nn::htcs::HTCS_EBADF )
    {
        //=================================================
        // If we don't have a valid socket, it's because
        // the connect call failed, so therefore we're
        // not connected.
        //=================================================
        ErrorCode = HTCS_ENOTCONN;
    }
    return Ret;
}

//===============================================================================
// These can fail if the network is down.
//==============================================================================

nn::htcs::ssize_t virtual_socket_collection::Recv( int Id, void* buffer, size_t bufferByteSize, int flags, int& ErrorCode ) NN_NOEXCEPT
{
    nn::htcs::ssize_t Ret = -1;
    ErrorCode = 0;
    nn::sf::SharedPointer<nn::tma::ISocket> Socket = FetchSocket( Id, ErrorCode );

    if( Socket )
    {
        if (ErrorCode != 0)
        {
            return -1;
        }
        Ret = recv( Socket, buffer, bufferByteSize, flags, ErrorCode );
    }
    else if ( ErrorCode != nn::htcs::HTCS_EBADF )
    {
        ErrorCode = HTCS_ENOTCONN;
    }
    return Ret;
}

//==============================================================================

nn::htcs::ssize_t virtual_socket_collection::Send( int Id, const void* buffer, size_t bufferByteSize, int flags, int& ErrorCode ) NN_NOEXCEPT
{
    nn::htcs::ssize_t Ret = -1;
    ErrorCode = 0;
    nn::sf::SharedPointer<nn::tma::ISocket> Socket = FetchSocket( Id, ErrorCode );

    if( Socket )
    {
        if (ErrorCode != 0)
        {
            return -1;
        }
        Ret = send( Socket, buffer, bufferByteSize, flags, ErrorCode );
    }
    else if ( ErrorCode != nn::htcs::HTCS_EBADF )
    {
        ErrorCode = HTCS_ENOTCONN;
    }
    return Ret;
}

//==============================================================================

int virtual_socket_collection::Connect( int Id, const nn::htcs::SockAddrHtcs* pAddress, int& ErrorCode ) NN_NOEXCEPT
{
    int Ret = -1;
    ErrorCode = 0;
    nn::sf::SharedPointer<nn::tma::ISocket> Socket = FetchSocket( Id, ErrorCode );

    if( Socket )
    {
        if (ErrorCode != 0)
        {
            return -1;
        }
        Ret = connect( Socket, pAddress, ErrorCode );
    }
    else if ( ErrorCode != nn::htcs::HTCS_EBADF )
    {
        ErrorCode = HTCS_EADDRNOTAVAIL;
    }
    return Ret;
}

//==============================================================================

//==============================================================================
}}} // nn::htcs::detail
//==============================================================================
#endif //#if USE_VIRTUAL_SOCKETS
