﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/os.h>

#include <nn/htclow/htclow_ResultPrivate.h>
#include <nn/htclow/detail/htclow_Log.h>

#include "../htclow_Print.h"
#include "htclow_Mux.h"

namespace nn { namespace htclow { namespace server { namespace mux {

    Mux::Mux() NN_NOEXCEPT
        : m_ApiSendReadyEvent(nn::os::EventClearMode_ManualClear, true)
        , m_AddSendPacketEvent(nn::os::EventClearMode_ManualClear)
        , m_pCancelEvent(nullptr)
    {
        m_ApiSendReadyEvent.Signal();
        m_AddSendPacketEvent.Clear();
    }

    Result Mux::Open(detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);

        NN_RESULT_DO(m_ChannelResources.AddChannel(channel));
        NN_RESULT_SUCCESS;
    }

    void Mux::Close(detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);

        if (m_ChannelResources.Exists(channel))
        {
            m_ChannelResources.RemoveChannel(channel);
        }
    }

    Result Mux::Connect(detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);

        if (!m_ChannelResources.Exists(channel))
        {
            return ResultChannelClosed();
        }

        auto& resource = m_ChannelResources[channel];
        if (resource.GetState() != ChannelState::Opened)
        {
            return ResultChannelStateError();
        }

        auto packet = resource.GetSendPacketFactory()->MakeSynPacket(channel);
        if (!packet)
        {
            return ResultOutOfMemory();
        }

        NN_RESULT_DO(resource.GetSendBuffer()->AddPacket(std::move(packet)));

        resource.SetState(ChannelState::SynSent);

        m_AddSendPacketEvent.Signal();

        NN_RESULT_SUCCESS;
    }

    Result Mux::Shutdown(detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);

        if (!m_ChannelResources.Exists(channel))
        {
            return ResultChannelClosed();
        }

        auto& resource = m_ChannelResources[channel];

        switch (resource.GetState())
        {
        case ChannelState::Established:
            resource.SetState(ChannelState::FinWait1);
            break;
        case ChannelState::CloseWait:
            resource.SetState(ChannelState::LastAck);
            break;
        default:
            return ResultChannelStateError();
        }

        auto packet = resource.GetSendPacketFactory()->MakeFinPacket(channel);
        if (!packet)
        {
            return ResultOutOfMemory();
        }

        NN_RESULT_DO(resource.GetSendBuffer()->AddPacket(std::move(packet)));

        m_AddSendPacketEvent.Signal();

        NN_RESULT_SUCCESS;
    }

    Result Mux::Receive(int* pOutSize, void* buffer, int bufferSize, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        if (bufferSize < 0)
        {
            return ResultInvalidArgument();
        }

        if (bufferSize == 0)
        {
            *pOutSize = 0;
            NN_RESULT_SUCCESS;
        }

        std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);

        if (!m_ChannelResources.Exists(channel))
        {
            return ResultChannelClosed();
        }

        auto& resource = m_ChannelResources[channel];

        if (resource.GetState() == ChannelState::Opened ||
            resource.GetState() == ChannelState::SynSent)
        {
            return ResultChannelStateError();
        }

        const auto readableSize = resource.GetReceiveBuffer()->GetReadableBodySize();

        if (readableSize <= 0)
        {
            // 受信バッファにデータがない
            if (resource.GetState() == ChannelState::Established ||
                resource.GetState() == ChannelState::FinWait1 ||
                resource.GetState() == ChannelState::FinWait2)
            {
                // まだデータを受信する可能性がある
                return ResultChannelReceiveBufferEmpty();
            }
            else
            {
                // もうデータを受信する見込みは無い
                return ResultConnectionFailure();
            }
        }

        // 受信バッファにデータがある
        const auto readSize = std::min(readableSize, bufferSize);
        NN_RESULT_DO(resource.GetReceiveBuffer()->ReadBody(buffer, readSize));

        *pOutSize = readSize;
        NN_RESULT_SUCCESS;
    }

    Result Mux::Send(int* pOutSize, const void* buffer, int bufferSize, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        if (bufferSize < 0)
        {
            return ResultInvalidArgument();
        }

        std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);

        if (!m_ChannelResources.Exists(channel))
        {
            return ResultChannelClosed();
        }

        auto& resource = m_ChannelResources[channel];

        if (resource.GetState() == ChannelState::Opened ||
            resource.GetState() == ChannelState::SynSent)
        {
            return ResultChannelStateError();
        }

        if (resource.GetState() == ChannelState::FinWait1 ||
            resource.GetState() == ChannelState::FinWait2 ||
            resource.GetState() == ChannelState::Closing ||
            resource.GetState() == ChannelState::LastAck ||
            resource.GetState() == ChannelState::Closed)
        {
            return ResultConnectionFailure();
        }

        if (bufferSize == 0)
        {
            *pOutSize = 0;
            NN_RESULT_SUCCESS;
        }

        // 最大ボディサイズを考慮して、データを分割しながらパケットを作成
        // TODO: バッファコピーの削減を検討 (Phase 2 以降)
        int totalSendSize = 0;
        while (totalSendSize < bufferSize)
        {
            const auto bodySize = std::min(bufferSize - totalSendSize, MaxBodySize);
            auto body = reinterpret_cast<const uint8_t*>(buffer) + totalSendSize;

            auto packet = resource.GetSendPacketFactory()->MakeDataPacket(channel, body, bodySize);
            if (!packet)
            {
                if (totalSendSize == 0)
                {
                    m_ApiSendReadyEvent.Clear();
                    return ResultOutOfMemory();
                }

                // 送信キューに積んだデータがある場合は、ひとまず成功扱い
                break;
            }

            NN_RESULT_DO(resource.GetSendBuffer()->AddPacket(std::move(packet)));

            totalSendSize += bodySize;
        }

        m_AddSendPacketEvent.Signal();

        *pOutSize = totalSendSize;
        NN_RESULT_SUCCESS;
    }

    Result Mux::GetClosedEventHandle(nn::os::NativeHandle* pOutHandle, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);

        if (!m_ChannelResources.Exists(channel))
        {
            return ResultChannelClosed();
        }

        auto& resource = m_ChannelResources[channel];
        *pOutHandle = resource.GetClosedEvent()->GetReadableHandle();

        NN_RESULT_SUCCESS;
    }

    Result Mux::GetEstablishedEventHandle(nn::os::NativeHandle* pOutHandle, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);

        if (!m_ChannelResources.Exists(channel))
        {
            return ResultChannelClosed();
        }

        auto& resource = m_ChannelResources[channel];
        *pOutHandle = resource.GetEstablishedEvent()->GetReadableHandle();

        NN_RESULT_SUCCESS;
    }

    Result Mux::GetSendReadyEventHandle(nn::os::NativeHandle* pOutHandle, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        // TORIAEZU: グローバルな ApiSendReady イベントを使う
        NN_UNUSED(channel);

        *pOutHandle = m_ApiSendReadyEvent.GetReadableHandle();
        NN_RESULT_SUCCESS;
    }

    Result Mux::GetReceiveReadyEventHandle(nn::os::NativeHandle* pOutHandle, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);

        if (!m_ChannelResources.Exists(channel))
        {
            return ResultChannelClosed();
        }

        auto& resource = m_ChannelResources[channel];
        *pOutHandle = resource.GetReceiveReadyEvent()->GetReadableHandle();

        NN_RESULT_SUCCESS;
    }

    Result Mux::GetSendCompleteEventHandle(nn::os::NativeHandle* pOutHandle, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);

        if (!m_ChannelResources.Exists(channel))
        {
            return ResultChannelClosed();
        }

        auto& resource = m_ChannelResources[channel];
        *pOutHandle = resource.GetSendCompleteEvent()->GetReadableHandle();

        NN_RESULT_SUCCESS;
    }

    void Mux::CancelAllChannel() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);

        for (const auto& pair: m_ChannelResources.GetMap())
        {
            auto& resource = m_ChannelResources.GetChannelResource(pair.second);
            if (resource.GetState() != ChannelState::Opened)
            {
                resource.SetState(ChannelState::Closed);
            }
        }
    }

    void Mux::SetRetransmitEnabled(bool enabled) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);
        m_ChannelResources.SetRetransmitEnabled(enabled);
    }

    void Mux::SetCancelEvent(nn::os::Event* pEvent) NN_NOEXCEPT
    {
        m_pCancelEvent = pEvent;
    }

    void Mux::RemoveAckPacket(detail::ChannelInternalType channel, int64_t sequenceId) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);

        if (!m_ChannelResources.Exists(channel))
        {
            return;
        }

        auto& resource = m_ChannelResources[channel];

        resource.GetSendBuffer()->RemoveAckPacket(sequenceId);

        // パケットが削除されたら送信可能になるかも
        m_ApiSendReadyEvent.Signal();
    }

    Result Mux::QuerySendPacket(SendPacket** ppOutSendPacket) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(ppOutSendPacket);
        NN_SDK_ASSERT_NOT_NULL(m_pCancelEvent);

        for (;;)
        {
            m_AddSendPacketEvent.Clear();

            {
                std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);

                for (auto pair : m_ChannelResources.GetMap())
                {
                    auto& resource = m_ChannelResources.GetChannelResource(pair.second);

                    auto pPacket = resource.GetSendBuffer()->GetNextPacket();
                    if (pPacket != nullptr)
                    {
                        *ppOutSendPacket = pPacket;
                        NN_RESULT_SUCCESS;
                    }
                }
            }

            // TODO: 最適なタイムアウト時間を計算 (Phase 2)
            const auto trigger = nn::os::TimedWaitAny(RetransmitTimeout, m_AddSendPacketEvent.GetBase(), m_pCancelEvent->GetBase());
            if (trigger == 1)
            {
                return ResultCancelled();
            }
        }
    }

    Result Mux::ProcessReceivePacket(std::unique_ptr<ReceivePacket> packet) NN_NOEXCEPT
    {
        switch (packet->GetHeader()->packetType)
        {
        case PacketType::Data:
            return ProcessReceiveDataPacket(std::move(packet));
        case PacketType::DataAck:
            return ProcessReceiveDataAckPacket(std::move(packet));
        case PacketType::Syn:
            return ResultProtocolError();
        case PacketType::SynAck:
            return ProcessReceiveSynAckPacket(std::move(packet));
        case PacketType::Fin:
            return ProcessReceiveFinPacket(std::move(packet));
        case PacketType::FinAck:
            return ProcessReceiveFinAckPacket(std::move(packet));
        default:
            return ResultProtocolError();
        }
    }

    Result Mux::ProcessReceiveDataPacket(std::unique_ptr<ReceivePacket> packet) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);

        const auto channel = packet->GetHeader()->channel;
        const auto sequenceId = packet->GetHeader()->sequenceId;

        if (!m_ChannelResources.Exists(channel))
        {
            return ResultChannelClosed();
        }

        auto& resource = m_ChannelResources[channel];

        if (resource.GetState() != ChannelState::Established &&
            resource.GetState() != ChannelState::FinWait1 &&
            resource.GetState() != ChannelState::FinWait2)
        {
            return ResultChannelStateError();
        }


        NN_RESULT_DO(resource.CheckPacketVersion(packet.get()));
        NN_RESULT_DO(resource.GetReceiveBuffer()->AddPacket(std::move(packet)));

        auto ackPacket = resource.GetSendPacketFactory()->MakeDataAckPacket(channel, sequenceId);
        if (!ackPacket)
        {
            return ResultOutOfMemory();
        }

        NN_RESULT_DO(resource.GetSendBuffer()->AddAckPacket(std::move(ackPacket)));

        m_AddSendPacketEvent.Signal();

        NN_RESULT_SUCCESS;
    }

    Result Mux::ProcessReceiveDataAckPacket(std::unique_ptr<ReceivePacket> packet) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);

        const auto channel = packet->GetHeader()->channel;
        const auto sequenceId = packet->GetHeader()->sequenceId;

        if (!m_ChannelResources.Exists(channel))
        {
            return ResultChannelClosed();
        }

        auto& resource = m_ChannelResources[channel];
        NN_RESULT_DO(resource.CheckPacketVersion(packet.get()));
        resource.GetSendBuffer()->RemovePacket(sequenceId);

        // パケットが削除されたら送信可能になるかも
        m_ApiSendReadyEvent.Signal();

        NN_RESULT_SUCCESS;
    }

    Result Mux::ProcessReceiveSynAckPacket(std::unique_ptr<ReceivePacket> packet) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);

        const auto channel = packet->GetHeader()->channel;

        if (!m_ChannelResources.Exists(channel))
        {
            return ResultChannelClosed();
        }

        auto& resource = m_ChannelResources[channel];

        if (resource.GetState() != ChannelState::SynSent)
        {
            return ResultChannelStateError();
        }

        // セッションで使用するバージョンの決定
        auto version = std::min(MaxVersion, packet->GetHeader()->version);
        resource.SetVersion(version);
        resource.SetState(ChannelState::Established);

        resource.GetSendBuffer()->RemovePacket(packet->GetHeader()->sequenceId);

        NN_RESULT_SUCCESS;
    }

    Result Mux::ProcessReceiveFinPacket(std::unique_ptr<ReceivePacket> packet) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);

        const auto channel = packet->GetHeader()->channel;
        const auto sequenceId = packet->GetHeader()->sequenceId;

        if (!m_ChannelResources.Exists(channel))
        {
            return ResultChannelClosed();
        }

        auto& resource = m_ChannelResources[channel];
        NN_RESULT_DO(resource.CheckPacketVersion(packet.get()));

        switch (resource.GetState())
        {
        case ChannelState::Established:
            resource.SetState(ChannelState::CloseWait);
            break;
        case ChannelState::FinWait1:
            resource.SetState(ChannelState::Closing);
            break;
        case ChannelState::FinWait2:
            resource.SetState(ChannelState::Closed);
            break;
        default:
            return ResultChannelStateError();
        }

        auto ackPacket = resource.GetSendPacketFactory()->MakeFinAckPacket(channel, sequenceId);
        if (!ackPacket)
        {
            return ResultOutOfMemory();
        }

        NN_RESULT_DO(resource.GetSendBuffer()->AddAckPacket(std::move(ackPacket)));
        m_AddSendPacketEvent.Signal();

        NN_RESULT_SUCCESS;
    }

    Result Mux::ProcessReceiveFinAckPacket(std::unique_ptr<ReceivePacket> packet) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_ChannelResourceMutex)> lock(m_ChannelResourceMutex);

        const auto channel = packet->GetHeader()->channel;

        if (!m_ChannelResources.Exists(channel))
        {
            return ResultChannelClosed();
        }

        auto& resource = m_ChannelResources[channel];

        NN_RESULT_DO(resource.CheckPacketVersion(packet.get()));

        switch (resource.GetState())
        {
            case ChannelState::FinWait1:
                resource.SetState(ChannelState::FinWait2);
                break;
            case ChannelState::Closing:
            case ChannelState::LastAck:
                resource.SetState(ChannelState::Closed);
                break;
            default:
                return ResultChannelStateError();
        }

        resource.GetSendBuffer()->RemovePacket(packet->GetHeader()->sequenceId);

        NN_RESULT_SUCCESS;
    }

}}}}
