﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/htclow/htclow_ResultPrivate.h>

#include <nn/htclow/detail/htclow_Log.h>

#include "htclow_ChannelResource.h"

//#define NN_DETAIL_HTCLOW_ENABLE_PRINT

namespace nn { namespace htclow { namespace server { namespace mux {

namespace{

#if defined(NN_DETAIL_HTCLOW_ENABLE_PRINT)
    const char* GetStateString(ChannelState state)
    {
#define NNT_DETAIL_HTCLOW_CASE(_state) case ChannelState::_state : {return #_state;}
        switch (state)
        {
        NNT_DETAIL_HTCLOW_CASE(Opened);
        NNT_DETAIL_HTCLOW_CASE(SynSent);
        NNT_DETAIL_HTCLOW_CASE(Established);
        NNT_DETAIL_HTCLOW_CASE(CloseWait);
        NNT_DETAIL_HTCLOW_CASE(LastAck);
        NNT_DETAIL_HTCLOW_CASE(FinWait1);
        NNT_DETAIL_HTCLOW_CASE(FinWait2);
        NNT_DETAIL_HTCLOW_CASE(Closing);
        NNT_DETAIL_HTCLOW_CASE(Closed);
        default:
            NN_UNEXPECTED_DEFAULT;
        }
#undef NNT_DETAIL_HTCLOW_CASE
    }
#endif

    void PrintStateTransition(ChannelState current, ChannelState next)
    {
#if defined(NN_DETAIL_HTCLOW_ENABLE_PRINT)
        NN_DETAIL_HTCLOW_INFO("ChannelState: %s -> %s\n", GetStateString(current), GetStateString(next));
#else
        NN_UNUSED(current);
        NN_UNUSED(next);
#endif
    }
}

    Result ChannelResource::CheckPacketVersion(BasePacket* packet) NN_NOEXCEPT
    {
        if (packet->GetHeader()->version != m_Version)
        {
            return ResultChannelVersionNotMatched();
        }
        NN_RESULT_SUCCESS;
    }

    void ChannelResource::SetState(ChannelState state) NN_NOEXCEPT
    {
        PrintStateTransition(m_State, state);
        if (!IsStateTransitionAllowed(m_State, state))
        {
            NN_ABORT("許可されていない状態遷移です。\n");
        }

        m_State = state;

        if (m_State == ChannelState::Established)
        {
            m_EstablishedEvent.Signal();
        }
        if (m_State == ChannelState::Closed)
        {
            m_ClosedEvent.Signal();
        }
    }

    void ChannelResource::SetVersion(int16_t version) NN_NOEXCEPT
    {
        m_Version = version;
        m_SendPacketFactory.SetVersion(version);
    }

    bool ChannelResource::IsStateTransitionAllowed(ChannelState current, ChannelState next) NN_NOEXCEPT
    {
        switch (current)
        {
        case ChannelState::Opened:
            return next == ChannelState::SynSent;
        case ChannelState::SynSent:
            return next == ChannelState::Established || next == ChannelState::Closed;
        case ChannelState::Established:
            return next == ChannelState::FinWait1 || next == ChannelState::CloseWait || next == ChannelState::Closed;
        case ChannelState::CloseWait:
            return next == ChannelState::LastAck || next == ChannelState::Closed;
        case ChannelState::LastAck:
            return next == ChannelState::Closed;
        case ChannelState::FinWait1:
            return next == ChannelState::FinWait2 || next == ChannelState::Closing || next == ChannelState::Closed;
        case ChannelState::FinWait2:
            return next == ChannelState::Closed;
        case ChannelState::Closing:
            return next == ChannelState::Closed;
        case ChannelState::Closed:
            return next == ChannelState::Closed;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }


}}}}
