﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>

#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>

#include <nn/os.h>
#include <nn/util/util_IntrusiveList.h>

#include <nn/htclow/detail/htclow_InternalTypes.h>

#include "htclow_Allocator.h"

namespace nn { namespace htclow { namespace server {

enum class PacketType : int16_t
{
    Data    = 0x0000,
    DataAck = 0x0001,
    Syn     = 0x0002,
    SynAck  = 0x0003,
    Fin     = 0x0004,
    FinAck  = 0x0005,
};

struct PacketHeader
{
    int16_t protocol;
    int16_t version;
    int16_t reserved;
    PacketType packetType;
    detail::ChannelInternalType channel;
    int32_t bodySize;
    int64_t sequenceId;
};

NN_STATIC_ASSERT(sizeof(PacketHeader) == 24);

const int16_t ProtocolId = 256;
const int16_t MaxVersion = 1;
const int MaxBodySize = 65536;

// パケット再送信タイムアウト
const TimeSpan RetransmitTimeout = TimeSpan::FromMilliSeconds(100);

// ユーティリティ関数
NN_FORCEINLINE bool IsAckPacket(PacketType packetType)
{
    return packetType == PacketType::DataAck || packetType == PacketType::FinAck || packetType == PacketType::SynAck;
}

/*
 * @brief パケットの基底クラス
 */
class BasePacket
{
    NN_DISALLOW_COPY(BasePacket);
    NN_DISALLOW_MOVE(BasePacket);

public:
    /*
     * @brief 指定したサイズのバッファを持つパケットを作成
     */
    BasePacket(int bufferSize = sizeof(PacketHeader)) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(bufferSize, static_cast<int>(sizeof(PacketHeader)));

        m_pBuffer = reinterpret_cast<uint8_t*>(AllocateDefault(bufferSize));
        m_pBufferSize = bufferSize;
    }

    virtual ~BasePacket() NN_NOEXCEPT
    {
        if (m_pBuffer != nullptr)
        {
            FreeDefault(m_pBuffer);
        }
    }

    static void* operator new(size_t size) NN_NOEXCEPT
    {
        return AllocateDefault(size);
    }

    static void operator delete(void* p, size_t) NN_NOEXCEPT
    {
        if (p != nullptr)
        {
            FreeDefault(p);
        }
    }

    /*
     * @brief 指定されたヘッダをパケットのバッファにコピーする
     */
    void CopyHeader(const PacketHeader* pHeader) NN_NOEXCEPT
    {
        std::memcpy(GetHeader(), pHeader, sizeof(PacketHeader));
    }

    /*
     * @brief 指定されたボディをパケットのバッファにコピーする
     */
    void CopyBody(const void* pBody, int bodySize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_GREATER_EQUAL(GetBodySize(), 0);
        std::memcpy(GetBody(), pBody, bodySize);
    }

    /*
     * @brief バッファへのポインタを取得する
     */
    void* GetBuffer() NN_NOEXCEPT
    {
        return m_pBuffer;
    }

    /*
     * @brief バッファへの const ポインタを取得する
     */
    const void* GetBuffer() const NN_NOEXCEPT
    {
        return m_pBuffer;
    }

    /*
     * @brief ヘッダへのポインタを取得する
     */
    PacketHeader* GetHeader() NN_NOEXCEPT
    {
        return reinterpret_cast<PacketHeader*>(m_pBuffer);
    }

    /*
     * @brief ヘッダへの const ポインタを取得する
     */
    const PacketHeader* GetHeader() const NN_NOEXCEPT
    {
        return reinterpret_cast<PacketHeader*>(m_pBuffer);
    }

    /*
     * @brief ボディへのポインタを取得する
     */
    uint8_t* GetBody() NN_NOEXCEPT
    {
        if (m_pBufferSize > sizeof(PacketHeader))
        {
            return &m_pBuffer[sizeof(PacketHeader)];
        }
        else
        {
            return nullptr;
        }
    }

    /*
     * @brief ボディへの const ポインタを取得する
     */
    const uint8_t* GetBody() const NN_NOEXCEPT
    {
        if (m_pBufferSize > sizeof(PacketHeader))
        {
            return &m_pBuffer[sizeof(PacketHeader)];
        }
        else
        {
            return nullptr;
        }
    }

    /*
     * @brief バッファのサイズを取得する
     */
    int GetBufferSize() const NN_NOEXCEPT
    {
        return m_pBufferSize;
    }

    /*
     * @brief ボディのサイズを取得する
     */
    int GetBodySize() const NN_NOEXCEPT
    {
        return m_pBufferSize - sizeof(PacketHeader);
    }

    /*
    * @brief メンバのアロケートに成功したか
    */
    bool IsAllocationSucceeded() const NN_NOEXCEPT
    {
        return m_pBuffer != nullptr;
    }

private:
    uint8_t* m_pBuffer;
    int m_pBufferSize;
};

class SendPacket
    : public nn::util::IntrusiveListBaseNode<SendPacket>
    , public BasePacket
{
    NN_DISALLOW_COPY(SendPacket);
    NN_DISALLOW_MOVE(SendPacket);

public:
    SendPacket(int bufferSize = sizeof(PacketHeader)) NN_NOEXCEPT
        : BasePacket(bufferSize)
        , m_SendFlag(false)
        , m_SendTime(nn::os::Tick(0))
    {
    }

    virtual ~SendPacket() NN_NOEXCEPT
    {
    }

    bool GetSendFlag() const NN_NOEXCEPT
    {
        return m_SendFlag;
    }

    void SetSendFlag(bool flag) NN_NOEXCEPT
    {
        m_SendFlag = flag;
    }

    nn::os::Tick GetSendTime() const NN_NOEXCEPT
    {
        return m_SendTime;
    }

    void SetSendTime(nn::os::Tick sendTime) NN_NOEXCEPT
    {
        m_SendTime = sendTime;
    }

private:
    bool m_SendFlag;
    nn::os::Tick m_SendTime;
};

/*
 * @brief 受信パケット
 */
class ReceivePacket
    : public nn::util::IntrusiveListBaseNode<ReceivePacket>
    , public BasePacket
{
    NN_DISALLOW_COPY(ReceivePacket);
    NN_DISALLOW_MOVE(ReceivePacket);

public:
    ReceivePacket(int bufferSize = sizeof(PacketHeader)) NN_NOEXCEPT
        : BasePacket(bufferSize)
    {
    }

    virtual ~ReceivePacket() NN_NOEXCEPT
    {
    }
};

}}}
