﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>

#include <nn/htclow.h>
#include <nn/htclow/htclow_ResultPrivate.h>

#include "htclow_DriverManager.h"

namespace nn { namespace htclow { namespace server { namespace driver {

IDriver* DriverManager::GetCurrentDriver() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return m_pCurrentDriver;
}

nn::Result DriverManager::OpenDriver(detail::DriverType driverType) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (m_pCurrentDriver != nullptr)
    {
        return ResultDriverOpened();
    }

    switch (driverType)
    {
    case detail::DriverType::Debug:
#if defined(NNT_DETAIL_HTCLOW_ENABLE_DEBUG_DRIVER)
        NN_RESULT_DO(m_DebugDriver.Open());
        m_pCurrentDriver = &m_DebugDriver;
        break;
#else
        return ResultUnsupportedDriverType();
#endif

    case detail::DriverType::Socket:
#if defined(NNT_DETAIL_HTCLOW_ENABLE_SOCKET_DRIVER)
        NN_RESULT_DO(m_SocketDriver.Open());
        m_pCurrentDriver = &m_SocketDriver;
        break;
#else
        return ResultUnsupportedDriverType();
#endif

    default:
        return ResultUnknownDriverType();
    }

    NN_RESULT_SUCCESS;
}

void DriverManager::CloseDriver() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (m_pCurrentDriver == nullptr)
    {
        return;
    }

    m_pCurrentDriver->Close();
    m_pCurrentDriver = nullptr;
}

}}}}
