﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <limits>
#include <mutex>

#include <nn/nn_Common.h>

#include <nn/htclow.h>

#include <nn/htcfs/htcfs_Result.h>
#include <nn/htcfs/htcfs_ResultPrivate.h>

#include "htcfs_Impl.h"
#include "htcfs_ResultConversion.h"

// HtcfsImpl 専用の NN_RESULT_DO
// 失敗の result が返ってきたら FinalizeChannel を呼んでからリターンする
#define NN_DETAIL_HTCFS_DO(_r) \
{ \
    ::nn::Result _nnHtcfsResult = (_r); \
    if (_nnHtcfsResult.IsFailure()) \
    { \
        FinalizeChannel(); \
        return _nnHtcfsResult; \
    } \
}

// condition が false なら FinalizeChannel を呼んでからリターンする
#define NN_DETAIL_HTCFS_CHECK(_condition, _result) \
{ \
    bool _nnHtcfsCondition = (_condition); \
    ::nn::Result _nnHtcfsResult = (_result); \
    if (!(_nnHtcfsCondition)) \
    { \
        FinalizeChannel(); \
        return _nnHtcfsResult; \
    } \
}

namespace nn { namespace htcfs { namespace server {

namespace {
    nn::os::SdkMutexType g_InitializeMutex = NN_OS_SDK_MUTEX_INITIALIZER();
    HtcfsImpl* g_pImpl = nullptr;
}

HtcfsImpl* HtcfsImpl::GetInstance() NN_NOEXCEPT
{
    // Double checking lock
    if (g_pImpl != nullptr)
    {
        return g_pImpl;
    }

    std::lock_guard<decltype(g_InitializeMutex)> lock(g_InitializeMutex);

    if (g_pImpl != nullptr)
    {
        return g_pImpl;
    }

    static HtcfsImpl s_Impl;
    g_pImpl = &s_Impl;

    return g_pImpl;
}

nn::Result HtcfsImpl::GetEntryType(nn::fs::DirectoryEntryType* pOutType, const char* pPath) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    const int64_t pathSize = std::strlen(pPath);

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::GetEntryType, pathSize);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));
    NN_DETAIL_HTCFS_DO(SendToHtclow(pPath, pathSize));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    NN_RESULT_DO(ConvertNativeResult(response.param1));
    *pOutType = static_cast<nn::fs::DirectoryEntryType>(response.param2);

    NN_RESULT_SUCCESS;
}

nn::Result HtcfsImpl::OpenFile(int32_t* pOutHandle, const char* pPath, nn::fs::OpenMode mode) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    const int64_t pathSize = std::strlen(pPath);

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::OpenFile, pathSize, mode);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));
    NN_DETAIL_HTCFS_DO(SendToHtclow(pPath, pathSize));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    NN_RESULT_DO(ConvertNativeResult(response.param1));
    *pOutHandle = static_cast<int32_t>(response.param2);

    NN_RESULT_SUCCESS;
}

nn::Result HtcfsImpl::CloseFile(int32_t handle) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::CloseFile, 0, handle);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    return ConvertHtcfsResult(response.param0);
}

nn::Result HtcfsImpl::GetPriorityForFile(int32_t* pOutPriority, int32_t handle) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::GetPriorityForFile, 0, handle);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    NN_RESULT_DO(ConvertNativeResult(response.param1));
    *pOutPriority = static_cast<int32_t>(response.param2);

    NN_RESULT_SUCCESS;
}

nn::Result HtcfsImpl::SetPriorityForFile(int32_t priority, int32_t handle) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::SetPriorityForFile, 0, handle, priority);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    return ConvertHtcfsResult(response.param0);
}

nn::Result HtcfsImpl::CreateFile(const char* pPath, int64_t size) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    const int64_t pathSize = std::strlen(pPath);

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::CreateFile, pathSize, size);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));
    NN_DETAIL_HTCFS_DO(SendToHtclow(pPath, pathSize));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    return ConvertNativeResult(response.param1);
}

nn::Result HtcfsImpl::DeleteFile(const char* pPath) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    const int64_t pathSize = std::strlen(pPath);

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::DeleteFile, pathSize);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));
    NN_DETAIL_HTCFS_DO(SendToHtclow(pPath, pathSize));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    return ConvertNativeResult(response.param1);
}

nn::Result HtcfsImpl::RenameFile(const char* pFromName, const char* pToName) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    const int64_t fromNameSize = std::strlen(pFromName);
    const int64_t toNameSize = std::strlen(pToName);

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::RenameFile, fromNameSize + toNameSize, fromNameSize, toNameSize);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));
    NN_DETAIL_HTCFS_DO(SendToHtclow(pFromName, fromNameSize));
    NN_DETAIL_HTCFS_DO(SendToHtclow(pToName, toNameSize));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    return ConvertNativeResult(response.param1);
}

nn::Result HtcfsImpl::FileExists(bool* pOutExists, const char* pPath) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    const int64_t pathSize = std::strlen(pPath);

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::FileExists, pathSize);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));
    NN_DETAIL_HTCFS_DO(SendToHtclow(pPath, pathSize));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    NN_RESULT_DO(ConvertNativeResult(response.param1));
    *pOutExists = (response.param2 != 0);

    NN_RESULT_SUCCESS;
}

nn::Result HtcfsImpl::ReadFile(int64_t* pOutSize, void* pOutBuffer, int32_t handle, int64_t offset, int64_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT
{
    NN_UNUSED(option);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::ReadFile, 0, handle, offset, size);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType));

    auto result = ConvertHtcfsResult(response.param0);
    if (result.IsFailure())
    {
        NN_DETAIL_HTCFS_CHECK(response.bodySize == 0, ResultUnexpectedResponseBodySize());
        return result;
    }

    auto nativeResult = ConvertNativeResult(response.param1);
    if (nativeResult.IsFailure())
    {
        NN_DETAIL_HTCFS_CHECK(response.bodySize == 0, ResultUnexpectedResponseBodySize());
        return nativeResult;
    }

    // ファイルの内容を取得
    NN_DETAIL_HTCFS_CHECK(response.bodySize <= size, ResultUnexpectedResponseBodySize());
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(pOutBuffer, response.bodySize));

    *pOutSize = response.bodySize;

    NN_RESULT_SUCCESS;
}

nn::Result HtcfsImpl::WriteFile(const void* pBuffer, int32_t handle, int64_t offset, int64_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::WriteFile, size, handle, option.flags, offset);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));
    NN_DETAIL_HTCFS_DO(SendToHtclow(pBuffer, size));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    return ConvertNativeResult(response.param1);
}

nn::Result HtcfsImpl::FlushFile(int32_t handle) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::FlushFile, 0, handle);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    return ConvertNativeResult(response.param1);
}

nn::Result HtcfsImpl::GetFileTimeStamp(uint64_t* pOutCreateTime, uint64_t* pOutAccessTime, uint64_t* pOutModifyTime, const char* pPath) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    const int64_t pathSize = std::strlen(pPath);

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::GetFileTimeStamp, pathSize);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));
    NN_DETAIL_HTCFS_DO(SendToHtclow(pPath, pathSize));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    NN_RESULT_DO(ConvertNativeResult(response.param1));

    *pOutCreateTime = response.param2;
    *pOutAccessTime = response.param3;
    *pOutModifyTime = response.param4;

    NN_RESULT_SUCCESS;
}

nn::Result HtcfsImpl::GetFileSize(int64_t* pOutSize, int32_t handle) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::GetFileSize, 0, handle);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    NN_RESULT_DO(ConvertNativeResult(response.param1));
    *pOutSize = response.param2;

    NN_RESULT_SUCCESS;
}

nn::Result HtcfsImpl::SetFileSize(int64_t size, int32_t handle) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::SetFileSize, 0, handle, size);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    return ConvertNativeResult(response.param1);
}

nn::Result HtcfsImpl::OpenDirectory(int32_t* pOutHandle, const char* pPath, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    const int64_t pathSize = std::strlen(pPath);

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::OpenDirectory, pathSize, mode);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));
    NN_DETAIL_HTCFS_DO(SendToHtclow(pPath, pathSize));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    NN_RESULT_DO(ConvertNativeResult(response.param1));
    *pOutHandle = static_cast<int32_t>(response.param2);

    NN_RESULT_SUCCESS;
}

nn::Result HtcfsImpl::CloseDirectory(int32_t handle) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::CloseDirectory, 0, handle);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    return ConvertNativeResult(response.param1);
}

nn::Result HtcfsImpl::GetPriorityForDirectory(int32_t* pOutPriority, int32_t handle) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::GetPriorityForDirectory, 0, handle);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    NN_RESULT_DO(ConvertNativeResult(response.param1));
    *pOutPriority = static_cast<int32_t>(response.param2);

    NN_RESULT_SUCCESS;
}

nn::Result HtcfsImpl::SetPriorityForDirectory(int32_t priority, int32_t handle) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::SetPriorityForDirectory, 0, handle, priority);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    return ConvertNativeResult(response.param1);
}

nn::Result HtcfsImpl::CreateDirectory(const char* pPath) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    const int64_t pathSize = std::strlen(pPath);

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::CreateDirectory, pathSize);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));
    NN_DETAIL_HTCFS_DO(SendToHtclow(pPath, pathSize));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    return ConvertNativeResult(response.param1);
}

nn::Result HtcfsImpl::DeleteDirectory(const char* pPath, bool recursively) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    const int64_t pathSize = std::strlen(pPath);

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::DeleteDirectory, pathSize, recursively);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));
    NN_DETAIL_HTCFS_DO(SendToHtclow(pPath, pathSize));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    return ConvertNativeResult(response.param1);
}

nn::Result HtcfsImpl::RenameDirectory(const char* pFromName, const char* pToName) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    const int64_t fromNameSize = std::strlen(pFromName);
    const int64_t toNameSize = std::strlen(pToName);

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::RenameDirectory, fromNameSize + toNameSize, fromNameSize, toNameSize);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));
    NN_DETAIL_HTCFS_DO(SendToHtclow(pFromName, fromNameSize));
    NN_DETAIL_HTCFS_DO(SendToHtclow(pToName, toNameSize));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    return ConvertNativeResult(response.param1);
}

nn::Result HtcfsImpl::DirectoryExists(bool* pOutExists, const char* pPath) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    const int64_t pathSize = std::strlen(pPath);

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::DirectoryExists, pathSize);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));
    NN_DETAIL_HTCFS_DO(SendToHtclow(pPath, pathSize));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    NN_RESULT_DO(ConvertNativeResult(response.param1));
    *pOutExists = (response.param2 != 0);

    NN_RESULT_SUCCESS;
}

nn::Result HtcfsImpl::ReadDirectory(int64_t* pOutCount, nn::fs::DirectoryEntry* pOutBuffer, int64_t count, int32_t handle) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::ReadDirectory, 0, handle, count);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    NN_RESULT_DO(ConvertNativeResult(response.param1));

    const int64_t responseEntryCount = response.param2;

    NN_DETAIL_HTCFS_CHECK(response.bodySize == responseEntryCount * static_cast<int>(sizeof(nn::fs::DirectoryEntry)), ResultUnexpectedResponseBodySize());

    if (responseEntryCount > count)
    {
        return ResultUnexpectedResponseBody();
    }

    *pOutCount = responseEntryCount;
    NN_RESULT_DO(ReceiveFromHtclow(pOutBuffer, response.bodySize));

    NN_RESULT_SUCCESS;
}

nn::Result HtcfsImpl::GetEntryCount(int64_t* pOutCount, int32_t handle) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    NN_DETAIL_HTCFS_DO(InitializeChannel());

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::GetEntryCount, 0, handle);
    NN_DETAIL_HTCFS_DO(SendToHtclow(&request, sizeof(request)));

    Header response;
    NN_DETAIL_HTCFS_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeader(response, request.packetType, 0));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));
    NN_RESULT_DO(ConvertNativeResult(response.param1));
    *pOutCount = response.param2;

    NN_RESULT_SUCCESS;
}

nn::Result HtcfsImpl::InitializeChannel() NN_NOEXCEPT
{
    if (m_IsChannelInitialized)
    {
        NN_RESULT_SUCCESS;
    }

    nn::htclow::Module module(nn::htclow::ModuleId::Fs);
    NN_RESULT_DO(m_Channel.Open(&module, HtclowChannelId));
    NN_RESULT_DO(m_Channel.Connect());

    // バージョンの設定
    nn::Result result;
    int16_t hostMaxVersion;

    result = GetMaxProtocolVersion(&hostMaxVersion);
    if (result.IsFailure())
    {
        m_Channel.Shutdown();
        m_Channel.Close();
        return result;
    }

    if (hostMaxVersion < 0)
    {
        m_Channel.Shutdown();
        m_Channel.Close();
        return ResultUnsupportedProtocolVersion();
    }

    auto version = std::min(MaxVersion, hostMaxVersion);

    result = SetProtocolVersion(version);
    if (result.IsFailure())
    {
        m_Channel.Shutdown();
        m_Channel.Close();
        return result;
    }

    m_Factory.SetVersion(version);

    m_IsChannelInitialized = true;

    NN_RESULT_SUCCESS;
}

void HtcfsImpl::FinalizeChannel() NN_NOEXCEPT
{
    // 既に channel が切断された状態なので、nn::htclow::Shutdown() は呼ばない
    m_Channel.Close();

    m_Factory.SetVersion(0);

    m_IsChannelInitialized = false;
}

nn::Result HtcfsImpl::GetMaxProtocolVersion(int16_t* pOutVersion) NN_NOEXCEPT
{
    // InitializeChannel() から呼ばれるので、m_Mutex はロックせず NN_DETAIL_HTCFS_DO は使わない

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::GetMaxProtocolVersion);

    NN_RESULT_DO(SendToHtclow(&request, sizeof(request)));

    Header response;
    NN_RESULT_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeaderWithoutVersion(response, request.packetType));

    NN_RESULT_DO(ConvertHtcfsResult(response.param0));

    *pOutVersion = static_cast<int16_t>(response.param1);
    NN_RESULT_SUCCESS;
}

nn::Result HtcfsImpl::SetProtocolVersion(int16_t version) NN_NOEXCEPT
{
    // InitializeChannel() から呼ばれるので、m_Mutex はロックせず NN_DETAIL_HTCFS_DO は使わない

    Header request;
    m_Factory.MakeRequestHeader(&request, PacketType::SetProtocolVersion, 0, version);

    NN_RESULT_DO(SendToHtclow(&request, sizeof(request)));

    Header response;
    NN_RESULT_DO(ReceiveFromHtclow(&response, sizeof(response)));
    NN_DETAIL_HTCFS_DO(CheckResponseHeaderWithoutVersion(response, request.packetType));

    return ConvertHtcfsResult(response.param0);
}

nn::Result HtcfsImpl::SendToHtclow(const void* pBuffer, int64_t size) NN_NOEXCEPT
{
    if (size < 0)
    {
        return ResultInvalidArgument();
    }

    // TODO: キャストが多発して危険なので、htclow のシグネチャ変更 (size を size_t => int64_t に変更) で改善できないか検討
    int64_t totalSendSize = 0;
    while (totalSendSize < size)
    {
        const size_t maxSendSize = std::numeric_limits<size_t>::max();
        const size_t sendSize = static_cast<size_t>(std::min(static_cast<uint64_t>(maxSendSize), static_cast<uint64_t>(size - totalSendSize)));

        size_t actualSendSize;
        NN_RESULT_DO(m_Channel.Send(&actualSendSize, reinterpret_cast<const uint8_t*>(pBuffer) + totalSendSize, sendSize));

        if (actualSendSize != sendSize)
        {
            return ResultHtclowChannelClosed();
        }
        totalSendSize += static_cast<int64_t>(actualSendSize);
    }

    NN_RESULT_SUCCESS;
}

nn::Result HtcfsImpl::ReceiveFromHtclow(void* pOutBuffer, int64_t size) NN_NOEXCEPT
{
    if (size < 0)
    {
        return ResultInvalidArgument();
    }

    // TODO: キャストが多発して危険なので、htclow のシグネチャ変更 (size を size_t => int64_t に変更) で改善できないか検討
    int64_t totalReceiveSize = 0;
    while (totalReceiveSize < size)
    {
        const size_t maxReceiveSize = std::numeric_limits<size_t>::max();
        const size_t receiveSize = static_cast<size_t>(std::min(static_cast<uint64_t>(maxReceiveSize), static_cast<uint64_t>(size - totalReceiveSize)));

        size_t actualReceiveSize;
        NN_RESULT_DO(m_Channel.Receive(&actualReceiveSize, reinterpret_cast<uint8_t*>(pOutBuffer) + totalReceiveSize, receiveSize));

        if (actualReceiveSize != receiveSize)
        {
            return ResultHtclowChannelClosed();
        }
        totalReceiveSize += static_cast<int64_t>(actualReceiveSize);
    }

    NN_RESULT_SUCCESS;
}

nn::Result HtcfsImpl::CheckResponseHeaderWithoutVersion(const Header& header, PacketType expectPacketType) NN_NOEXCEPT
{
    if (header.protocol != HtcfsProtocol)
    {
        return ::nn::htcfs::ResultUnexpectedResponseProtocolId();
    }
    if (header.packetCategory != PacketCategory::Response)
    {
        return ::nn::htcfs::ResultUnexpectedResponsePacketCategory();
    }
    if (header.packetType != expectPacketType)
    {
        return ::nn::htcfs::ResultUnexpectedResponsePacketType();
    }

    NN_RESULT_SUCCESS;
}

nn::Result HtcfsImpl::CheckResponseHeader(const Header& header, PacketType expectPacketType) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckResponseHeaderWithoutVersion(header, expectPacketType));

    if (header.version != m_Factory.GetVersion())
    {
        return ::nn::htcfs::ResultUnexpectedResponseProtocolVersion();
    }

    NN_RESULT_SUCCESS;
}

nn::Result HtcfsImpl::CheckResponseHeader(const Header& header, PacketType expectPacketType, int64_t expectBodySize) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckResponseHeader(header, expectPacketType));

    if (header.bodySize != expectBodySize)
    {
        return ::nn::htcfs::ResultUnexpectedResponseBodySize();
    }

    NN_RESULT_SUCCESS;
}

}}}
