﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/htcfs/htcfs_Result.h>
#include <nn/htcfs/htcfs_ResultPrivate.h>

#include "htcfs_FileServiceObject.h"
#include "htcfs_Impl.h"

namespace nn { namespace htcfs { namespace server {

FileServiceObject::FileServiceObject(int32_t handle) NN_NOEXCEPT
    : m_Handle(handle)
{
}

FileServiceObject::~FileServiceObject() NN_NOEXCEPT
{
    HtcfsImpl::GetInstance()->CloseFile(m_Handle);
}

nn::Result FileServiceObject::GetPriorityForFile(nn::sf::Out<std::int32_t> outPriority) NN_NOEXCEPT
{
    int32_t priority;
    NN_RESULT_DO(HtcfsImpl::GetInstance()->GetPriorityForFile(&priority, m_Handle));
    *outPriority = priority;
    NN_RESULT_SUCCESS;
}

nn::Result FileServiceObject::SetPriorityForFile(std::int32_t priority) NN_NOEXCEPT
{
    return HtcfsImpl::GetInstance()->SetPriorityForFile(priority, m_Handle);
}

nn::Result FileServiceObject::ReadFile(nn::sf::Out<std::int64_t> outSize, const nn::sf::OutBuffer& outBuffer, std::int64_t offset, nn::fs::ReadOption option) NN_NOEXCEPT
{
    if (offset < 0)
    {
        return ResultInvalidArgument();
    }

    int64_t size;
    NN_RESULT_DO(HtcfsImpl::GetInstance()->ReadFile(&size, outBuffer.GetPointerUnsafe(), m_Handle, offset, outBuffer.GetSize(), option));
    *outSize = size;
    NN_RESULT_SUCCESS;
}

nn::Result FileServiceObject::WriteFile(const nn::sf::InBuffer& buffer, std::int64_t offset, nn::fs::WriteOption option) NN_NOEXCEPT
{
    if (offset < 0)
    {
        return ResultInvalidArgument();
    }

    return HtcfsImpl::GetInstance()->WriteFile(buffer.GetPointerUnsafe(), m_Handle, offset, buffer.GetSize(), option);
}

nn::Result FileServiceObject::FlushFile() NN_NOEXCEPT
{
    return HtcfsImpl::GetInstance()->FlushFile(m_Handle);
}

nn::Result FileServiceObject::GetFileSize(nn::sf::Out<std::int64_t> outSize) NN_NOEXCEPT
{
    int64_t size;
    NN_RESULT_DO(HtcfsImpl::GetInstance()->GetFileSize(&size, m_Handle));
    *outSize = size;
    NN_RESULT_SUCCESS;
}

nn::Result FileServiceObject::SetFileSize(std::int64_t size) NN_NOEXCEPT
{
    if (size < 0)
    {
        return ResultInvalidArgument();
    }

    return HtcfsImpl::GetInstance()->SetFileSize(size, m_Handle);
}

}}}
