﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/psc.h>

#include <nn/gpio/gpio_ServiceName.h>
#include <nn/gpio/gpio_IManager.sfdl.h>
#include <nn/gpio/driver/gpio_Suspend.h>

#include <nn/gpio/server/gpio_ManagerImpl.h>
#include <nn/gpio/server/gpio_Server.h>

#include "gpio_ServerUtil.h"

namespace nn { namespace gpio { namespace server {

namespace {

    // サーバー側から発行する全セッションの Max
    // TORIAEZU : 今予定されているピンの数(55) + α(10)
    const int GpioSessionCountMax = 65;

    // TORIAEZU : 今はマネージャーは 1つだけ
    class GpioServerManager
        : public nn::sf::HipcSimpleAllInOneServerManager<GpioSessionCountMax, 1>
    {};

    GpioServerManager g_ServerManager;

    nn::sf::UnmanagedServiceObject<nn::gpio::IManager, nn::gpio::server::ManagerImpl>   g_ServiceObject;

    // スリープ等の通知を受け取るオブジェクト
    nn::psc::PmModule g_PmModule;
    nn::psc::PmModule g_LowPmModule;

    // MultiWait するための各種変数
    nn::os::MultiWaitType           g_PmEventWaiter;
    nn::os::MultiWaitHolderType     g_PmEventHolder;
    nn::os::MultiWaitHolderType     g_LowPmEventHolder;

    bool g_IsHipcServerRegistered = false;
    bool g_IsHipcServerStarted = false;
}

nn::sf::SharedPointer<nn::gpio::IManager> GetServiceObject() NN_NOEXCEPT
{
    return g_ServiceObject.GetShared();
}

void RegisterHipcServer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!g_IsHipcServerRegistered);

    // サービス名の登録とポートの初期化
    // 現在はどのプロセスからアクセスしても 1つのマネージャーにアクセスされる
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ServerManager.RegisterObjectForPort(g_ServiceObject.GetShared(), GpioSessionCountMax, nn::gpio::GpioServiceName));

    g_IsHipcServerRegistered = true;
}

void StartHipcServer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_IsHipcServerRegistered);
    NN_SDK_REQUIRES(!g_IsHipcServerStarted);

    // PSC へのモジュール登録
    // GPIO
    auto result = g_PmModule.Initialize(
        nn::psc::PmModuleId_Gpio,
        GpioDependencies,
        sizeof(GpioDependencies) / sizeof(GpioDependencies[0]),
        nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // GPIOLow
    auto resultLow = g_LowPmModule.Initialize(
        nn::psc::PmModuleId_GpioLow,
        nullptr,
        0,
        nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(resultLow);

    // 通常の GPIO と GPIO Low を 1 つのスレッドで受けるために MultiWait する
    nn::os::InitializeMultiWait(&g_PmEventWaiter);

    // GPIO
    nn::os::InitializeMultiWaitHolder(
        &g_PmEventHolder,
        g_PmModule.GetEventPointer()->GetBase());
    nn::os::SetMultiWaitHolderUserData(
        &g_PmEventHolder,
        nn::psc::PmModuleId_Gpio);
    nn::os::LinkMultiWaitHolder(&g_PmEventWaiter, &g_PmEventHolder);

    // GPIO Low
    nn::os::InitializeMultiWaitHolder(
        &g_LowPmEventHolder,
        g_LowPmModule.GetEventPointer()->GetBase());
    nn::os::SetMultiWaitHolderUserData(
        &g_LowPmEventHolder,
        nn::psc::PmModuleId_GpioLow);
    nn::os::LinkMultiWaitHolder(&g_PmEventWaiter, &g_LowPmEventHolder);

    g_ServerManager.Start();

    g_IsHipcServerStarted = true;
}

void LoopHipcServer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_IsHipcServerRegistered);
    NN_SDK_REQUIRES(g_IsHipcServerStarted);

    nn::psc::PmState    gpioState = nn::psc::PmState_Unknown;
    nn::psc::PmState    gpioPreviousState = nn::psc::PmState_Unknown;
    nn::psc::PmFlagSet  gpioFlags;
    nn::psc::PmState    gpioLowState = nn::psc::PmState_Unknown;
    nn::psc::PmState    gpioLowPreviousState = nn::psc::PmState_Unknown;
    nn::psc::PmFlagSet  gpioLowFlags;
    nn::psc::PmModuleId receivedModuleId = nn::psc::PmModuleId_Gpio;
    nn::os::MultiWaitHolderType* pEventHolder;

    bool ipcEnable = true;

    while ( NN_STATIC_CONDITION(true) )
    {
        if ( ipcEnable )
        {
            // IPC 許可状態
            NN_SDK_LOG("[gpio] Enter IPC Loop\n");
            IpcLoopForGpio<GpioServerManager>(&g_PmModule,
                &gpioPreviousState, &gpioState, &gpioFlags,
                &g_LowPmModule,
                &gpioLowPreviousState, &gpioLowState, &gpioLowFlags,
                &g_ServerManager);
            ipcEnable = false;
        }
        else
        {
            // IPC 禁止状態 (PSC のイベントのみ待ち受ける)
            pEventHolder = nn::os::WaitAny(&g_PmEventWaiter);
            switch ( nn::os::GetMultiWaitHolderUserData(pEventHolder) )
            {
                case nn::psc::PmModuleId_Gpio:
                    g_PmModule.GetEventPointer()->Clear();
                    NN_ABORT_UNLESS_RESULT_SUCCESS(g_PmModule.GetRequest(&gpioState, &gpioFlags));
                    receivedModuleId = nn::psc::PmModuleId_Gpio;
                    break;

                case nn::psc::PmModuleId_GpioLow:
                    g_LowPmModule.GetEventPointer()->Clear();
                    NN_ABORT_UNLESS_RESULT_SUCCESS(g_LowPmModule.GetRequest(&gpioLowState, &gpioLowFlags));
                    receivedModuleId = nn::psc::PmModuleId_GpioLow;
                    break;

                default: NN_UNEXPECTED_DEFAULT;
            }
        }

        StateEdge stateEdge;
        if ( receivedModuleId == nn::psc::PmModuleId_Gpio )
        {
            stateEdge = GetStateEdge(gpioPreviousState, gpioState);
        }
        else
        {
            stateEdge = GetStateEdge(gpioLowPreviousState, gpioLowState);
        }

        switch ( stateEdge )
        {
            case StateEdge_SleepReadyToEssentialServicesSleepReady:

                if ( receivedModuleId == nn::psc::PmModuleId_Gpio )
                {
                    // GPIO ドライバのサスペンド処理
                    nn::gpio::driver::Suspend();
                    g_PmModule.Acknowledge(gpioState, nn::ResultSuccess());
                    gpioPreviousState = gpioState;
                }
                else
                {
                    // GPIO ドライバの Low サスペンド処理
                    nn::gpio::driver::SuspendLow();
                    g_LowPmModule.Acknowledge(gpioLowState, nn::ResultSuccess());
                    gpioLowPreviousState = gpioLowState;
                }
                break;


            case StateEdge_EssentialServicesSleepReadyToEssentialServicesAwake:

                if ( receivedModuleId == nn::psc::PmModuleId_Gpio )
                {
                    // GPIO ドライバのレジューム処理
                    nn::gpio::driver::Resume();
                    g_PmModule.Acknowledge(gpioState, nn::ResultSuccess());
                    gpioPreviousState = gpioState;
                    receivedModuleId = nn::psc::PmModuleId_Gpio;
                    ipcEnable = true;
                }
                else
                {
                    // GPIO ドライバの Low レジューム処理
                    nn::gpio::driver::ResumeLow();
                    g_LowPmModule.Acknowledge(gpioLowState, nn::ResultSuccess());
                    gpioLowPreviousState = gpioLowState;
                }
                break;

            default:
                // その他の State 変化時は IPC 可能なので IpcLoop へ
                if ( receivedModuleId == nn::psc::PmModuleId_Gpio )
                {
                    g_PmModule.Acknowledge(gpioState, nn::ResultSuccess());
                    gpioPreviousState = gpioState;
                    receivedModuleId = nn::psc::PmModuleId_Gpio;
                    ipcEnable = true;
                }
                else
                {
                    // その他の State 変化時は Ackowledge するだけ
                    g_LowPmModule.Acknowledge(gpioLowState, nn::ResultSuccess());
                    gpioLowPreviousState = gpioLowState;
                }
                break;
        }
    }
}

void StopHipcServer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_IsHipcServerRegistered);
    NN_SDK_REQUIRES(g_IsHipcServerStarted);

    g_ServerManager.RequestStop();

    g_IsHipcServerStarted = false;
}

}}}
